 // used in checkArgs below
plot::plotObjectNumber:=0:


/** plot::createPlotDomain:
 *
 *   Creates the plot domain plotDom in accordance with the model 
 *   defined above and sets some default entries.  
 *
 *   Arguments:     objType -- name of the domain to be created
 *                  infoStr -- a descriptive string
 *   Return value:  --
 */

plot::createPlotDomain :=
proc(objType : DOM_STRING, infoStr : DOM_STRING, dim,
     definition = [] : DOM_LIST, elemType = "Obj")
  local plotDom, i, entry, automaticAttrib, attrName;
  option escape;
begin
  // check if str is in plot::objectHierachy !!!!
  assert(contains(plot::objectHierarchy, objType) = TRUE or args(0) >= 4);

  if not contains(plot::objectHierarchy, objType) then
    // add objType to object model
    plot::addToObjectHierarchy(objType, dim, elemType);

    // build inspector entry
    plot::buildInspectorEntry(objType, dim, definition, elemType);

    if elemType = "Obj" then
      automaticAttrib :=
      [ParameterName, ParameterBegin, ParameterEnd, Name, AffectViewingBox,
       Visible, Frames,
       TimeBegin, TimeEnd, VisibleBeforeBegin, VisibleAfterEnd,
       Title, TitlePositionX, TitlePositionY,
       if dim = 3 then TitlePositionZ else null() end,
       Legend, LegendEntry, LegendText, TitleFont, TitleAlignment];
    elif elemType = "Transform" then
      automaticAttrib :=
      [ParameterName, ParameterBegin, ParameterEnd, Name, AffectViewingBox,
       TimeBegin, TimeEnd, Frames]
    else
      // not known
      error("unknown element type")
    end_if;
    
    // add attributes to attributes table
    for i in definition.automaticAttrib do
      if domtype(i) = DOM_LIST then
        attrName := i[1]
      else
        attrName := i
      end_if;
      if contains(plot::attributes, attrName) then
        entry := plot::attributes[attrName];
        sysassign(plot::attributes[attrName],
                  subsop(entry, 4=op(entry, 4) union {objType}));
      else
        error("Invalid attribute name: ".expr2text(attrName)."");
      end_if;
    end_for;
      
    
    // update global tables
    plot::updateLibraryAttributes(objType,
                                  {op(automaticAttrib), op(definition)});
    plot::computeLegalAttributes();
    if testargs() = TRUE then
      // disabled, because of warnings in the development version
//      plot::checkAttributesTable();
    end_if;
  end_if;

  if not contains(table(op(plot)), objType) or
    domtype(op(select(op(plot), x -> op(x, 1)=objType), 2)) = stdlib::LoadProc then
    // create the domain
    plotDom := newDomain(hold(slot)(plot, objType));
  else
    error("plot::".objType." already exists");
  end_if;
    
  // assign the domain to the corresponding slot in plot
  sysassign(slot(plot, objType), plotDom);

  /* Set Default Domain Entries */

  // entry "dimension", iff appropriate (Canvas has no dimension set!)
  if contains(plot::all2dObjects, objType) then
    plotDom::dimension := 2;
  elif contains(plot::all3dObjects, objType) then
    plotDom::dimension := 3;
  end_if;

  // entry "info"
  plotDom::info := expr2text(plotDom::key)." -- ".infoStr;

  // entry "print"
  plotDom::print :=
  obj -> subsop(hold(hold)(op(obj::children),
                           dom::printAttributes(obj)),
                0 = plotDom, Unsimplified):
  
  // entry "printAttributes"
  // prints the attributes stored in obj as sequence of equations;
  // the attributes from the set in the 2nd argument are excluded
  plotDom::printAttributes :=
  proc(obj, set={})
    local rep, res, i, j, param, colorName;
  begin
    colorName := proc(rgb)
                 begin
                   if contains(RGB::rgbrev, rgb) then
                     hold(slot)(RGB, RGB::rgbrev[rgb])
                   else
                     rgb
                   end_if;
                 end_proc;
    rep := extop(obj, 1);
    // throw all known values into a table ...
    res := table(rep::inherited, rep::legal,
                 map(rep::legalExpr, eval@plot::ExprAttribute::getExpr), rep::hints,
                 rep::styleEntries);
    
    param := null();
    if obj::VisibleFromTo <> FAIL then
      param := param, VisibleFromTo = obj::VisibleFromTo;
      set := set union {VisibleFromTo, TimeRange}
    end_if;
    for i in plot::legalAttributes[objType][4] minus dom::hidden minus
             {ParameterRange, XRange, YRange, ZRange, Color} minus set do
      if slot(obj, "".i) <> FAIL then
        param := param, i = slot(obj, "".i);
        set := set union {i}
      end_if;
    end_for;
    
    for i in set do
      delete res[i];
      if plot::attributes[i][1] = "Library" then
        for j in indets(plot::attributes[i][6]) do
          delete res[j];
        end_for;
      end_if;
    end_for;
    for i in dom::hidden do
      delete res[i];
    end_for;
    for i in dom::hints do
      if res[lhs(i)] = rhs(i) then
        // a hint value is unchanged
        delete res[lhs(i)];
      end_if;
    end_for;
    
    if contains(res, Colors) then
      param := Colors = map(res[Colors], colorName@plot::ExprAttribute::getExpr), param;
      delete res[Colors]
    end_if;
    if contains(res, ParameterName) and
       contains(res, ParameterBegin) and
       contains(res, ParameterEnd) then
      param := res[ParameterName] = res[ParameterBegin]..res[ParameterEnd], param;
      delete res[ParameterName], res[ParameterBegin], res[ParameterEnd];
    end_if;
    if contains(res, ZName) and
       contains(res, ZMin) and
       contains(res, ZMax) then
      param := res[ZName] = res[ZMin]..res[ZMax], param;
      delete res[ZName], res[ZMin], res[ZMax];
    end_if;
    if contains(res, YName) and
       contains(res, YMin) and
       contains(res, YMax) then
      param := res[YName] = res[YMin]..res[YMax], param;
      delete res[YName], res[YMin], res[YMax];
    end_if;
    if contains(res, XName) and
       contains(res, XMin) and
       contains(res, XMax) then
      param := res[XName] = res[XMin]..res[XMax], param;
      delete res[XName], res[XMin], res[XMax];
    end_if;
    param, op(map(res, colorName@eval@plot::ExprAttribute::getExpr))
  end_proc;
  
  // entry "expr2text"
  plotDom::expr2text :=
  proc(obj)
    local res, rep, children;
  begin
    rep := extop(obj, 1);
    res := table(rep::inherited, rep::legal,
                 map(rep::legalExpr, plot::ExprAttribute::getExpr), rep::hints,
                 rep::styleEntries);
    children:=expr2text(op(rep::children));
    res := expr2text(op(res));
    if children <> "" and res <> "" then
      children := children.", "
    end_if;
    _concat("plot::", objType, "(", children, res, ")")
  end_proc;
  
  // default MuPlotML defaulting to doPlotStatic
  plotDom::MuPlotML := 
  (object, attributes, inheritedAttributes) -> 
    plotDom::doPlotStatic(plot::MuPlotML, args()):
  
  // entry "type"
  plotDom::type := "plotObject";

  // entries "op", "nops", "subsop", "append" :  access children
  if contains(plot::animatedObjects minus
              plot::allTransformations, objType) then
    // graphical primitives have no children and do nothing
    plotDom::op         := () -> if args(0) = 1 then null() else FAIL end;
    plotDom::nops       := () -> 0;
    plotDom::subsop     := FAIL;
    plotDom::append     := FAIL;
    plotDom::_index     := FAIL;
    plotDom::_set_index := FAIL;
  else
    plotDom::op :=
    proc(obj, n)
    begin
      if testargs() then
        if args(0) < 1 or args(0) > 2 then
          error("invalid number of arguments");
        end_if;
      end_if;  /* testargs */
      
      obj:= (extop(obj, 1))::children;
      if args(0) = 1 then op(obj) else op(obj, n) end_if
    end_proc;
    plotDom::nops :=
    proc(obj)
    begin
      if testargs() then
        if args(0) <> 1 then
          error("invalid number of arguments");
        end_if;
      end_if;  /* testargs */
      
      nops((extop(obj, 1))::children);
    end_proc;
                       
    plotDom::_index     :=
    proc(obj, n)
      local children;
    begin
      if testargs() then
        if args(0) <> 2 then
          error("invalid number of arguments");
        end_if;
      end_if;  /* testargs */
      
      children := (extop(obj, 1))::children;
      if domtype(n) = DOM_INT and n > 0 and n <= nops(children) or
        type(n) = "_range" and op(n, 1) > 0 and nops(children) >= op(n, 1) and
        op(n, 2) > 0 and nops(children) >= op(n, 2) then
        children[n]
      else
        hold(_index)(obj, n)
      end_if
    end_proc;
    plotDom::set_index :=
    proc(obj, n, value)
      local children, rep;
    begin
      if testargs() then
        if args(0) <> 3 then
          error("invalid number of arguments");
        end_if;
      end_if;  /* testargs */
      
      if args(0) = 3 and (args(3) = NIL or
        value = null() or value = FAIL) then
        value := null()
      end_if;
       
      children := (extop(obj, 1))::children;
      if domtype(n) = DOM_INT and n > 0 and nops(children) >= n then
        rep := extop(obj, 1);
        rep::children[n] := value;
        obj
      else
        FAIL
      end_if
    end_proc;  
  end_if;

  // entry "slot": access to attributes + "other"
  plotDom::slot     :=
  proc(obj, Index, value)
    name slot;
    local res, rep, slotName;
  begin
    if args(0) = 2 then
      // read access to slot
      if contains({"other", "children"}, Index) then
        // read access to the special slots
        slot(extop(obj, 1), Index);
      else
        slotName := hold(``).Index;
        rep := extop(obj, 1);
        // throw all known values into a table ...
        res := table(rep::inherited, rep::legal,
                     map(rep::legalExpr, eval@plot::ExprAttribute::getExpr), rep::hints,
                     rep::extra, rep::styleEntries);

        // and look if entry is there
        if contains(res, slotName."_") then
          // get internal value
          return(res[slotName."_"])
        end;
        if contains(res, slotName) then
          res := res[slotName];
          if domtype(res) = plot::ExprAttribute then
            eval@plot::ExprAttribute::getExpr(res);
          elif contains(plot::allFontAttributes, slotName) then
            plot::font2list(res)
          else
            if contains(plot::outputSizeAttributes, slotName) then
              // convert the stored value in MM into the prefeenced user
              // unit given by attributes[OutputUnits][2]
              unit::convert(res*unit::mm, plot::attributes[OutputUnits][2])
            else
              res
            end_if
          end_if;
        else
          // look if slotName is a library option
          if contains(plot::attributes, slotName) and
            plot::attributes[slotName][1] = "Library" then
            // call library read function
            plot::attributes[slotName][5](obj, slotName, res)
          else
            // nothing known about slotName, return FAIL
            FAIL
          end_if
        end_if
      end_if;      /* end read access to slot */
    else       
      // write access to slot; check notifier
      if (extop(obj, 1))::notify = TRUE and
        obj::dom::changeNotifier <> FAIL then
        res := obj::dom::changeNotifier(obj, Index = value);
        if not contains({TRUE, FALSE}, res) then
          if domtype(res) <> DOM_STRING then
            error("wrong type of return value of ".
                    expr2text(plotDom)."::changeNotifier");
            return(obj);
          end_if;
          error(res);
          return(obj);
        end_if;
        if res = TRUE then
          // accepted by changeNotifier, do it
          obj::dom::extslot(obj, Index, value);
        else
          // res = FALSE; assignment was redirected;
          // do nothing and raise no error
        end_if;
        return(obj)
      end_if;

      // accepted by changeNotifier, do it
      obj::dom::extslot(obj, Index, value);
    end_if;
  end_proc;                       /*  end of method "slot" */

  // things needed for distinction in extslot
  plotDom::ownAttributes := plot::getOwnAttributeNames(objType);
  plotDom::knownAttributes := plot::getAttributeNames(objType);
  plotDom::allKnownAttributes := plot::getAttributeNames();

  if not contains({"Canvas", "Scene2d", "CoordinateSystem2d", "Group2d",
            "Scene3d", "CoordinateSystem3d", "Group3d"}, objType) then
    // update knownAttributes of Canvas etc
    for i in {"Canvas", "Scene2d", "CoordinateSystem2d", "Group2d",
              "Scene3d", "CoordinateSystem3d", "Group3d"} do
      if slot(plot, i) <> FAIL then
        sysassign(slot(slot(plot, i), "knownAttributes"),
                  plot::getAttributeNames(i));
        sysassign(slot(slot(plot, i), "allKnownAttributes"),
                  plot::getAttributeNames());
      end_if;
    end_for;
  end_if;

  for i in plotDom::ownAttributes minus {Name} do
    slot(plotDom, "".i, plot::StyleSheetEntry(objType, i))
  end_for;

  plotDom::interface := plotDom::ownAttributes minus {LegendColor, Name};

  plotDom::hidden := {};
  // slots which are not visible for the user
  // in TAB-Completion, interface and documentation
  plotDom::hiddenSlots :=
  proc(hidden : DOM_SET)
  begin
    hidden := plotDom::hidden union hidden;
    sysassign(plotDom::interface, plotDom::interface minus hidden);
    sysassign(plotDom::hidden, hidden);
  end_proc;
  
  // styleSheet is a table of AttributeName = value
  plotDom::styleSheet := table();
  // hints is a set of AttributeName = value or AttributeName
  plotDom::hints := {};

  // set type of Attributes Color, Mesh, SubMesh
  plotDom::libInterface := table();
  plotDom::libInterface[Color] := 
  proc()
    local colors, tmp, res;
  begin
    colors := (plot::legalAttributes[objType][2] union
               plot::legalAttributes[objType][3]) intersect
              {FillColor, LineColor, PointColor, Colors, LightColor};
                       
    tmp := slot(plot, objType);
    if tmp::primaryColor <> FAIL then
      res := tmp::primaryColor
    elif contains(colors, Colors) then
      res := Colors
    elif contains(colors, FillColor) then
      res := FillColor
    elif contains(colors, LineColor) then
      res := LineColor
    elif contains(colors, PointColor) then
      res := PointColor
    elif contains(colors, LightColor) then
      res := LightColor
    else
      res := NIL
    end_if;
    if contains({FillColor, LineColor, PointColor}, res) then
      sysassign(plotDom::legendColor, res);
    end_if;
    res;
  end_proc():
  plotDom::libInterface[Mesh] :=
  proc()
    local meshvars;
  begin
    meshvars := plot::legalAttributes[objType][3] intersect
                {UMesh, VMesh, XMesh, YMesh, ZMesh};
    meshvars := DOM_SET::sort(meshvars);
    if nops(meshvars) = 0 then
      NIL;
    elif nops(meshvars) = 1 then
      op(meshvars);
    else
      meshvars
    end_if;
  end_proc():
  plotDom::libInterface[Submesh] := 
  proc()
    local meshvars;
  begin
    meshvars := plot::legalAttributes[objType][3] intersect
                {USubmesh, VSubmesh, XSubmesh, YSubmesh};
    meshvars := DOM_SET::sort(meshvars);
    if nops(meshvars) = 0 then
      NIL;
    elif nops(meshvars) = 1 then
      op(meshvars);
    else
      meshvars
    end_if;
  end_proc():

  plotDom::setPrimaryColor :=
  proc(colorAttribute)
  begin
    if not contains({FillColor, LineColor, PointColor, Colors, LightColor},
                    colorAttribute) then
 //     error("unexpected color attribute name '".expr2text(colorAttribute)."'")
    end_if;
    
    // update plotDom::primaryColor
    sysassign(plotDom::primaryColor, colorAttribute);

    // update plotDom::libInterface[Color] for getDefault
    sysassign(plotDom::libInterface[Color], colorAttribute);

    // update LegendColor
    if contains({FillColor, LineColor, PointColor}, colorAttribute) then
      sysassign(plotDom::legendColor, colorAttribute);
    else
      // delete existing entry
      sysassign(plotDom::legendColor, FAIL);
    end_if;    
  end:
  
  // entry "extslot" : write access to attributes
  plotDom::extslot  :=
  proc(obj, Ind, value)
    name extslot;
    local rep, slotName;
  begin
    // get the internal representation
    rep := extop(obj, 1);
    slotName := hold(``).Ind;
    
    // special entry  "children"
    if Ind = "children" then
      rep::children := value;
      return(obj);
    end_if;

    if contains({FAIL, NIL}, value) <> TRUE then
      // split arguments into the usual types of attributes
      if contains(plotDom::allKnownAttributes, slotName) then
        obj::dom::setValue(rep, slotName = value);
      else
        // neither legal nor legalExpr nor hints
        // not allowed in slot calls, only from direct extslot calls
        if stdlib::interactive() or context(hold(procname)) <> hold(slot) then
          rep::extra[slotName] := value;
        else
          context(hold(error)("assignment to invalid attribute '".
                              expr2text(slotName)."' in ".expr2text(dom)));
        end_if;
      end_if;
    else
      // value is FAIL or NIL, delete the corresponding entry
      if contains(plot::legalAttributes[objType][3], slotName) then
        delete rep::inherited[slotName];
      elif contains(plot::legalAttributes[objType][2] 
              union plot::legalAttributes[objType][1], slotName) then
        if contains(rep::legal, slotName) then
          delete rep::legal[slotName];
        elif contains(rep::legalExpr, slotName) then
          delete rep::legalExpr[slotName];
        end_if;
      elif contains(rep::hints, slotName) then
        delete rep::hints[slotName];
      elif contains(rep::extra, slotName) then
        delete rep::extra[slotName];
      elif contains(plot::legalAttributes[objType][4], slotName) then
        // library attribute?
        map(indets(plot::attributes[slotName][6]), x -> slot(obj, "".x, NIL))
      else
        // don't know, ignore
      end_if;
      if contains(rep::extra, slotName."_") then
        // remove internal value of Data slots as well
        delete rep::extra[slotName."_"];
      end_if;
    end_if;

    // return the updated object
    obj;
  end_proc;                            /*  end of method "extslot" */
//------------------------------------------------------------------------

  // entries "copy", "modify"
  plotDom::copy   :=
  proc(obj)
    local newObj, extra;
  begin
    sysassign(plot::plotObjectNumber, plot::plotObjectNumber+1);
    newObj := newDomain("plotObject".plot::plotObjectNumber, extop(obj,1));
    newObj::id := newObj::key;
    if extnops(obj) > 1 then
      extra := extop(obj, 2..extnops(obj))
    else
      extra := null()
    end_if;
    new(plotDom, newObj, extra);
  end_proc:
    
  plotDom::modify :=
  proc(obj)
    local i;
  begin
    obj := obj::dom::copy(obj);
    for i in [args(2..args(0))] do
      slot(obj, expr2text(op(i, 1)), op(i, 2))
    end_for;
    obj;
  end_proc:

  // used by extslot, checkArgs and checkObject to set one value into
  // the internal tables
  // the domain rep of the internal representaion is cheanged as a side effect
  plotDom::setValue :=
  proc(rep : DOM_DOMAIN, eq : "_equal")
    local j, i, op1, doSet, childTypes;
  begin
    doSet :=
    proc(slotName, eq)
    begin
      if op(eq, 2) = NIL then
        // delete entry
        delete slot(rep, slotName)[op(eq, 1)];
      else
        // set entry
        slot(rep, slotName)[op(eq, 1)] := op(j, 2);
      end_if;
    end_proc;

    if op(eq, 2) = null() then
      context(hold(error)("illegal value 'null()' for attribute '".
                          expr2text(op(eq, 1))."'"))
    end_if;
    
    if op(eq, 2) = NIL or op(eq, 2) = FAIL then
      i := table(op(eq, 1) = NIL);
    elif contains(plot::allLibraryAttributes, op(eq, 1)) and
      not contains(plot::legalAttributes[objType][4], op(eq, 1)) and
      nops(rep::children) > 0 then
      // Library interface used where where it can only be inherited
      // e.g. Color in a Group
      childTypes := _union(op(map(map(rep::children, extop),
                                  slot, "structs")));
      // keep original input for the case that it is a library
      // interface hint like  AxesVisible set in Group for a
      // CoordinateSystem
      i := eq;
      for j in childTypes do
        if contains(plot::legalAttributes[j][4], op(eq, 1)) or
          plot::attributes[op(eq,1)][1] = "Inherited" then
         i := i, plot::StyleSheetEntry(j, op(eq,1)) = op(eq, 2);
        end;
      end_for;
      i := plot::checkOptions(objType, [i]);
    else
      i := plot::checkOptions(objType, [eq]);
    end_if;
    
    // i is now a table which can contain more than one value
    for j in i do
      op1 := op(j, 1);
      if contains(plotDom::knownAttributes, op1) = TRUE then
        // attribute is legal here
        
        // check if attribute is of expression type
        if domtype(op(j, 2)) = plot::ExprAttribute then
          // add to legalExpr
          doSet("legalExpr", j);
        else
          // add to legal or inherited;
          if contains(plot::legalAttributes[objType][3], op1) = TRUE then
            if contains(plotDom::ownAttributes, op1) = TRUE then
              doSet("inherited", j);
            else
              doSet("styleEntries", j);
            end_if;
          elif contains(plot::legalAttributes[objType][4], op1) = TRUE then
            doSet("extra", j);
          else
            doSet("legal", j);
          end_if;
        end_if;
      elif domtype(op1) = plot::StyleSheetEntry then
        // add to style entries
        doSet("styleEntries", j);
      elif contains(plot::hintAllowed, op1) or
        contains({"Canvas", "Scene2d", "Scene3d",
                  "CoordinateSystem2d", "CoordinateSystem3d"}, dom::objType) then
        // attribute not valid here, put it to hints
        doSet("hints", j);
      elif expr2text(op1)[-1] = "_" then
        // internal _-slot, add to extra entries
        doSet("extra", j);
      else
        if not plot::inReplot then
          context(hold(warning)("attribute '".expr2text(op1)."' not allowed in ".expr2text(dom).", ignored"));
        end_if;
      end_if;
    end_for;  /* j */
    rep
  end_proc:

  // entry "checkArgs"
  plotDom::checkArgs :=
/** checkArgs:
 * 
 *  Checks the arguments of a plot object and creates a domain for
 *  the later internal representation of the plot object.
 *   
 *
 *   Arguments:    1. ranges  -- list which describes the interpretation of
 *                               arguments of the form identifier = range
 *                               Valid list entries are strings and a list
 *                               of strings, where the list denotes
 *                               optional parameter ranges.  Thus a value
 *                               ["U", ["V"]]
 *                               means that 1 to 2 parameter ranges are
 *                               accepted.  The first range is tranlated
 *                               into attributes UName, UMin, UMax and so on
 *                               with VName ... .
 *                               For all animable objects an additional
 *                               range may be given.  It will automatically
 *                               be translated into the Parameter range by
 *                               this function.
 *                 2. ...     -- the complete argument list of a primitive
 *   Return value:  a domain with the following entries:
 *
 *       id        - the uniqe ID of the plot object
 *       children  - list of children plot objects
 *       inherited - attributes which are inherited and valid here,
 *                   these are XML-attributes of the object
 *       legal     - attributes which are optional or mandatory, but no
 *                   expressions, these are XML-attributes of the object
 *       legalExpr - attributes which are printed here and are expressions,
 *                   these attributes are elements from the XML point of view
 *       hints     - attributes which are not printed here, but which are
 *                   hints for parent objects, like AxesInFront used in
 *                   2D-Primitives is a hint for the CoordinateSystem2d
 *       other     - all not mentioned above;  these are typically special
 *                   primitive arguments which have to be processed by the
 *                   primitives themselves
 *       extra     - empty list in which plot::additionalAttribute can
 *                   store additional values
 */
proc(ranges : DOM_LIST)
  name checkArgs;
  local Args, children, other, dummy,
        equations, i, op1, eqTab, op_1,
        rangeArgs, addRange, rep, xName, implicitTrue;
begin
  Args := [args(2..args(0))];

  if contains({"Canvas", "Scene2d", "Scene3d",
               "CoordinateSystem2d", "CoordinateSystem3d",
               "Group2d", "Group3d"}, objType) then
    // flatten out lists and sets
    while map({op(Args)}, domtype) intersect {DOM_SET, DOM_LIST} <> {} do
      Args := map(Args,
                  x -> if contains({DOM_SET, DOM_LIST}, domtype(x)) then
                         op(x)
                       else
                         x
                       end);
    end_while;
  end_if;
  
  // first split off all equations, 
  [equations, Args, dummy] := split(Args, testtype, "_equal");
  assert(dummy=[]);
  [implicitTrue, Args, dummy] := split(Args, x ->
                                    domtype(x) = plot::StyleSheetEntry or
                                    contains(plotDom::allKnownAttributes, x));
  assert(dummy=[]);

  equations :=  equations.map(implicitTrue, _equal, TRUE);
  
  // then split off all plot Objects, 
  [children, Args, dummy] := split(Args, x -> type(x) = "plotObject");
  assert(dummy=[]);

  equations := map(equations, x -> if testtype(x, "_equal") then x else x=TRUE end);
  
  eqTab := table(equations);
  // now we create a domain which is to be used as internal
  // representation in the plot primitives
  if contains(eqTab, id) then
    rep := newDomain(eqTab[id]);
    // get entries from existing domain

    // throw away all inherited attributes
    // they will be sent again by the frontend, iff needed
    // This is necessary for the synchronisation with the frontend
    rep::inherited := table():
    if domtype(rep::legal) <> DOM_TABLE then
      rep::legal := table():
    end_if;
    if domtype(rep::legalExpr) <> DOM_TABLE then
      rep::legalExpr := table():
    end_if;
    if domtype(rep::hints) <> DOM_TABLE then
      // merge in hints given in object type
      rep::hints :=
      table(op(select((slot(plot, objType))::hints, testtype, "_equal"))):
    end_if;
    if domtype(rep::extra) <> DOM_TABLE then
      rep::extra := table():
    end_if;
    if domtype(rep::styleEntries) <> DOM_TABLE then
      rep::styleEntries := table():
    end_if;
  else
    sysassign(plot::plotObjectNumber, plot::plotObjectNumber+1);
    rep := newDomain("plotObject".plot::plotObjectNumber);
    rep::inherited := table():
    rep::legal     := table():
    rep::legalExpr := table():
    // merge in hints given in object type
    rep::hints     :=
    table(op(select((slot(plot, objType))::hints, testtype, "_equal"))):
    rep::extra     := table():
    rep::styleEntries:= table():
  end_if;

  rep::structs   := _union(op(map(children, x -> (extop(x, 1))::structs)), {objType}) minus {"View2d", "View3d"};
  rep::objType   := objType;
  rep::id := rep::key;
  rep::children  := children;

  other := [];  rangeArgs := [];
  // insert defaults for mandatory attributes from style sheets
  equations:=
     [op(select((slot(plot, objType))::styleSheet,
                x -> contains(plot::allMandatoryAttributes, op(x, 1))))].
     equations;
  for i in equations do
    op1 := op(i, 1);
    if op1 = id then
      next;
    end_if;

    if domtype(op1) = plot::StyleSheetEntry then
      if contains((rep::structs union {"Scene2d", "Scene3d", "Camera",
                    "CoordinateSystem2d", "CoordinateSystem3d"})
                  minus {objType}, op1::dom::objType(op1)) then
        op_1 := op1::dom::attribute(op1)
      elif op1::dom::objType(op1) = objType then
        // no need for style sheet entry;
        // convert it into a simple attribute
        op_1 := op1::dom::attribute(op1);
        i := op_1 = op(i, 2);
      else
        if not plot::inReplot then
          context(hold(warning)("'".expr2text(op1)."' makes no sense in plot::".objType.", ignored"));
        end_if;
        next;
      end_if;
    else
      op_1 := op1
    end_if;

    if contains(plotDom::allKnownAttributes, op_1) then
      if contains(plotDom::knownAttributes union plot::hintAllowed, op_1) then
        // apply check function
        plotDom::setValue(rep, i);
      else
        if not plot::inReplot then
          context(hold(warning)("'".expr2text(op1)."' makes no sense in plot::".objType.", ignored"));
        end_if;
      end_if;
    else
      if testtype(i, "_equal") and testtype(op1, Type::Indeterminate) and
        testtype(op(i, 2), "_range") then
        rangeArgs := rangeArgs.[i];
      else
        // attributes not known, put to other
        other:= other.[i];
      end_if;
    end_if;
  end_for;
  
  // now scan rangeArgs and translate them into proper attribute
  // values
  addRange :=
  proc(xname     : DOM_STRING,
       range     : Type::Equation(Type::Indeterminate, "_range"))
  begin
   rep::legalExpr := table(rep::legalExpr, 
                      plot::checkOptions(objType,
                           [hold(``).xname."Name"  = op(range, 1),
                            hold(``).xname."Range" =   op(range, 2)]));
  end_proc;

  if not contains(plot::animatedObjects, objType) and nops(rangeArgs) > 0 then
      context(hold(error)("unexpected range argument: ".expr2text(op(rangeArgs))))
  end_if;
  
  for i from 1 to nops(ranges) do
    if i > nops(rangeArgs) then break; end_if;
    if domtype(ranges[i]) = DOM_STRING then
      xName := ranges[i]
    else
      assert(domtype(ranges[i]) = DOM_LIST and
             nops(ranges[i]) = 1 and
             domtype(ranges[i][1]) = DOM_STRING);
      xName := ranges[i][1]
    end_if;
    addRange(xName, rangeArgs[i]);
  end_for;
  if nops(rangeArgs) > nops(ranges) and
    contains(plot::animatedObjects, objType) = TRUE then
    // one more parameter range is interpreted as animation
    // parameter;  following ranges are ignored
    addRange("Parameter", rangeArgs[i]);
  end_if;
  if nops(rangeArgs) > i then
    context(hold(error)("unexpected range argument: ".expr2text(rangeArgs[i+1])))
  end_if;
     
  // the leftover Args and unprocessed equations have to be processed by
  // the primitive itself
  other := Args.other; 

  rep::other     := other;
  rep::notify    := FALSE;  // flag that controls the notifier usage

  new(plotDom, rep);
end_proc;                                 /*  end of method "checkArgs" */
//------------------------------------------------------------------------

  plotDom::checkObjectRecursive :=
  subsop(proc(obj)
         begin
           map(obj::children, x -> x::dom::checkObjectRecursive(x));
           dom::checkObject(obj, TRUE);
         end_proc, 6 = plotDom):
  
  // entry "checkObject"
  plotDom::checkObject :=
  proc(obj, recursive = FALSE)
    name checkObject;
    local rep, attribTable, i, j, identSet, idents, hurga, varName, equations;
  begin
    // get internal representation: a domaiin
    rep := extop(obj, 1);

    if map({op(rep::children)}, x -> (extop(x))::objType) minus 
                            plot::objectHierarchy[objType] <> {} then
      // ignore Hatch and Sweep which can have children
      if not contains({"Hatch", "Sweep", "Integral"}, objType) then
        context(hold(error)("invalid child type '".
              select(map(rep::children, x->(extop(x))::objType), 
                     y -> not y in plot::objectHierarchy[objType])[1]."'"))
      end
    end;
    
    // recompute structs for the case that children have been
    // added implicitly, e.g. Scene* in Canvas
    rep::structs   := _union(op(map(rep::children,
                                    x -> (extop(x, 1))::structs)), {objType})
                      minus {"View2d", "View3d"};

    if not recursive then
      // hint processing: look for hints in all children, combine them
      // with the hint combining function from the attribute table and
      // set own values accordingly
      equations := plot::combineHints(rep::children);
      for i in equations do
        if slot(obj, "".op(i, 1)) = FAIL then
          // only use hint, iff no value was given explicitly
          plotDom::setValue(rep, i);
        end_if;
      end_for;
    end_if;
    
    // have all mandatory attributes for this object a value?
    attribTable := table(rep::legalExpr, rep::legal);
    for i in plot::legalAttributes[objType][1] do
      if contains(attribTable, i) <> TRUE then
        // last try: get value from style sheet
        if contains(rep::extra, i."_") then
          // huge data slot si there, OK
        else
          // pop up help page
//          help("plot::".obj::dom::objType);
          context(hold(error)("mandatory attribute '".expr2text(i).
                            "' missing in ".expr2text(plotDom)))
        end_if;
      end_if;
    end_for;

    // compare the set of free variables in expression attributes
    // with the identifiers for which ranges are given
    identSet := {UName, VName, XName, YName, ZName, ParameterName} intersect
                (plot::legalAttributes[objType][1] union
                 plot::legalAttributes[objType][2]);

    // Check that for all variables ...Name which have a value
    // the range is there, too.
    for i in identSet do
      if contains(attribTable, i) = TRUE then
        // make sure that the corresponding range is OK
        if i = ParameterName then
          if not contains(attribTable, ParameterBegin) then
            context(hold(error)("ParameterName given, but ParameterBegin missing"))
          elif not contains(attribTable, ParameterEnd) then
            context(hold(error)("ParameterName given, but ParameterEnd missing"))
          end_if;
        else
          // UName, VName, XName, YName, ZName
          varName := stringlib::remove(expr2text(i), "Name");
          if not contains(attribTable, hold(``).varName."Min") then
            context(hold(error)("".i." given, but ".varName."Min missing"))
          elif not contains(attribTable, hold(``).varName."Max") then
            context(hold(error)("".i." given, but ".varName."Max missing"))
          end_if;          
        end_if;
      end_if;
    end_for;
    
    // after the loop idents contains all identifiers bound by ranges
    idents := {};
    for i in identSet do
      if contains(attribTable, i) = TRUE then
        idents := idents union {plot::ExprAttribute::getExpr(attribTable[i])}
      end_if;
    end_for;

    if nops(identSet) <> nops(idents) then
      // range identifier used twice or more
      hurga := DOM_SET::sort(identSet);  // get sorted list
      hurga := sort(map(hurga, i -> plot::ExprAttribute::getExpr(attribTable[i])));
      for i from 1 to nops(hurga) do
        varName := hurga[i];
        for j from i+1 to nops(hurga) do
          if hurga[j] = varName then
            context(hold(error)("duplicate identifiers '".expr2text(varName).
                                "' in range specifications"))
          end_if
        end_for
      end_for;
    end_if;

    // idents in obj::legalIdents are explicitly allowed
    if obj::legalIdents = FAIL then
      obj::dom::extslot(obj, "legalIdents", {});
    end_if;                       
    // idents may contain indexed identifiers.  remove them first
    identSet := map(map({op(rep::legalExpr)}, op, 2), plot::ExprAttribute::getExpr) minus
                idents;
    if indets(identSet, All) = {} then
      identSet := {};
    else
      // numeric::indets doesn't find F in F(a) therefore use indets with option
      // All to find 0th operands, but avoid to find y[i] 
      identSet := ((indets(identSet, All) minus indets(identSet))
                   union numeric::indets(identSet)) minus
                   Type::ConstantIdents minus stdlib::OPTIONS minus idents;
      // remove those identifiers which have a value
      identSet := select(identSet, x -> x = eval(x)) minus obj::legalIdents;
      if identSet <> {} then
        context(hold(error)("unbound identifier(s) '".
                            expr2text(op(identSet))."' found"))
      end_if;
    end_if;

    // parameters which may only depend on the animation parameter
    identSet := {TitlePositionX, TitlePositionY, TitlePositionZ};
    if contains(rep::legalExpr, ParameterName) then
      varName := {plot::ExprAttribute::getExpr(rep::legalExpr[ParameterName])};
    else
      varName := {};
    end;
    for i in identSet do
      if contains(rep::legalExpr, i) and
         indets(eval(rep::legalExpr[i])) minus
         Type::ConstantIdents minus varName <> {} then
        context(hold(error)(expr2text(i)." depends on '".expr2text(op(indets(eval(rep::legalExpr[i])) minus varName))."'"))
      end_if;
    end;
    
    // switch notifier on
    rep::notify := TRUE;
    
    // return object
    obj
  end_proc;

  /* internal entries  */

  // entry "animable"
  plotDom::animable := contains(plot::animatedObjects, objType);

  // entry "create_dom"
  plotDom::create_dom := hold(slot)(plot, objType);

  // entry "graphPrim"
  plotDom::graphPrim:=  contains(plot::allGraphPrim, objType);

  // entry "complete" used for TAB completion
  plotDom::complete := () -> plotDom::interface union
                                plot::standardHints union {Name};
  
  // entry "objType"
  plotDom::objType := objType;

  plotDom;
end_proc:
