//--------------------------------------------------------------
// The model of the plot library of MuPAD 3.0 and later.
//
// The table plot::objectHierarchy sets the tree structure of a 
// graphics that the library needs to build up before it starts 
// writing XML code. 
// The root is a Canvas object.
// The children of a Canvas are Scene2d or Scene3d objects.
// The children of a Scene3d object are CoordinateSystem3d
//      and, possibly, a Camera, Lights, and a Clippingbox.
// Etc.
// Finally, there are graphical primitives without children,
//    characterized by sets of numbers as table entries.
//    The numbers represent the dimension in which these
//    primitives may occur.  
//--------------------------------------------------------------
// How to add a new graphical Primitive:
// Add a string to one of the sets below.
//--------------------------------------------------------------

//set containig all 2d graphical primitives
plot::allGraphPrim2d :=
    {"Arrow2d", "Circle2d", "Curve2d", "Ellipse2d", "Function2d",
     "Hatch", "Implicit2d", "Line2d", "Point2d", "Polygon2d",
     "Rectangle", "VectorField2d"}:

// set containig all 3d graphical primitives
plot::allGraphPrim3d :=
    {"Arrow3d", "Box",   "Circle3d", "Cone", "Curve3d", "Function3d",
     "Implicit3d", "Line3d", "Point3d" , "Polygon3d", "Sphere",
     "Surface", "VectorField3d"}:


//==============================================================
//  No changes needed below this point.
//==============================================================

// set containig all graphical primitives
plot::allGraphPrim := plot::allGraphPrim2d union plot::allGraphPrim3d:

// set containig all light types
plot::allLights :=
    {"AmbientLight", "PointLight", "DistantLight", "SpotLight"}:

// set containig all 2d transformation types
plot::allTransformations2d :=
    {"Translate2d", "Rotate2d", "Scale2d", "Transform2d"}:

// set containig all 2d transformation types
plot::allTransformations3d :=
    {"Translate3d", "Rotate3d", "Scale3d", "Transform3d"}:

// set containig all transformation types
plot::allTransformations :=
    plot::allTransformations2d union plot::allTransformations3d:


// set containig all 2d objects
plot::all2dObjects :=
    plot::allGraphPrim2d union
    {"Scene2d", "CoordinateSystem2d", "Group2d", "View2d"} union
    plot::allTransformations2d:

// set containig all 3d objects
plot::all3dObjects :=
    plot::allGraphPrim3d union
    {"Scene3d", "CoordinateSystem3d", "Group3d", "ClippingBox", "Camera", "View3d"}
    union plot::allLights union plot::allTransformations3d:


plot::objectHierarchy:= table(
  "Canvas"  = {"Scene2d", "Scene3d"},
  "Scene2d" = {"CoordinateSystem2d", "View2d"},
  "Scene3d" = {"CoordinateSystem3d", "View3d"},
                              
  // no dimension since no graphical primitives
   "Camera"      = {},
   "View2d"      = {},
   "View3d"      = {},
   "ClippingBox" = {}
                              
  //------------------------------------------------
  // everything else is inserted automatically below
  //------------------------------------------------
):


// insert graphical primitives in to plot::objectHierarchy
(plot::objectHierarchy[i] := {2}) $ i in plot::allGraphPrim2d:
(plot::objectHierarchy[i] := {3}) $ i in plot::allGraphPrim3d:

// insert groups and coordinatesystems
plot::objectHierarchy["Group2d"]:= plot::allGraphPrim2d union {"Group2d"} union plot::allTransformations2d:
plot::objectHierarchy["Group3d"]:= plot::allGraphPrim3d union {"Group3d"} union plot::allTransformations3d:

// insert transformations in to plot::objectHierarchy
(plot::objectHierarchy[i] := plot::objectHierarchy["Group2d"]) $ 
   i in plot::allTransformations2d:
(plot::objectHierarchy[i] := plot::objectHierarchy["Group3d"]) $ 
   i in plot::allTransformations3d:

// insert lights in to plot::objectHierarchy
(plot::objectHierarchy[i] := {}) $ i in plot::allLights:


plot::objectHierarchy["CoordinateSystem2d"]:= 
   plot::objectHierarchy["Group2d"] union plot::allTransformations2d :
plot::objectHierarchy["CoordinateSystem3d"]:= plot::objectHierarchy["Group3d"]
  union plot::allLights union {"Camera", "ClippingBox"} union
plot::allTransformations3d :


// set containing all object types defined above
plot::allObjectTypes := map({op(plot::objectHierarchy)}, op, 1):

// set containing all object types which can be animated
plot::animatedObjects := plot::allObjectTypes minus
{"Canvas", "Scene2d", "Scene3d", "CoordinateSystem2d", "CoordinateSystem3d",
 "Group2d", "Group3d", "View2d", "View3d"}:

plot::addToObjectHierarchy:=
proc(objType, dim, elemType)
  local i;
begin
  sysassign(plot::objectHierarchy[objType], {dim});
  if dim = 2 then
    sysassign(plot::objectHierarchy["Group2d"],
              plot::objectHierarchy["Group2d"] union {objType});
    sysassign(plot::objectHierarchy[i], plot::objectHierarchy["Group2d"]) $ 
      i in plot::allTransformations2d:
    sysassign(plot::objectHierarchy["CoordinateSystem2d"],
              plot::objectHierarchy["CoordinateSystem2d"] union {objType});
    if elemType = "Obj" then
      sysassign(plot::allGraphPrim2d,
                plot::allGraphPrim2d union {objType});
    elif elemType = "Transform" then
      sysassign(plot::allTransformations2d,
                plot::allTransformations2d union {objType});
    else
      // not known
      error("unknown element type")
    end_if;
    sysassign(plot::all2dObjects,
              plot::all2dObjects union {objType});
  else
    sysassign(plot::objectHierarchy["Group3d"],
              plot::objectHierarchy["Group3d"] union {objType});
    sysassign(plot::objectHierarchy[i], plot::objectHierarchy["Group3d"]) $ 
      i in plot::allTransformations3d:
    sysassign(plot::objectHierarchy["CoordinateSystem3d"],
              plot::objectHierarchy["CoordinateSystem3d"] union {objType});
    if elemType = "Obj" then
      sysassign(plot::allGraphPrim3d,
                plot::allGraphPrim3d union {objType});
    elif elemType = "Transform" then
      sysassign(plot::allTransformations3d,
                plot::allTransformations3d union {objType});
    else
      // not known
      error("unknown element type")
    end_if;
    sysassign(plot::all3dObjects,
              plot::all3dObjects union {objType});
  end_if;
  if elemType = "Obj" then
    sysassign(plot::allGraphPrim,
              plot::allGraphPrim union {objType});
  elif elemType = "Transform" then
    sysassign(plot::allTransformations,
              plot::allTransformations union {objType});
  else
    // not known
    error("unknown element type")
  end_if;

  sysassign(plot::allObjectTypes,
            plot::allObjectTypes union {objType});
  sysassign(plot::animatedObjects,
            plot::animatedObjects union {objType});
end_proc:

