/* faclib::algfactor(a,flag) factors the polynomial a in F(alpha)[z]
  or tests it for irreducibility only, depending on flag 
  
  Input: a is a bivariate polynomial in z and alpha, 
         flag: TRUE or FALSE
  Output: if flag <> FALSE then list of factors of a
          if flag is FALSE then TRUE if a is irreducible
				FALSE if a is reducible

  Reference: Geddes et al, Algorithms for Computer Algebra, algorithm
  AlgebraicFactorization page 383
*/

faclib::algfactor :=
proc(a:DOM_POLY, flag=TRUE:DOM_BOOL)
  local s,Norm,x,b,i,n,F,m2,m,alpha,z,Falpha,aa,g,Fone, Falphanew,
  makeGF: DOM_PROC;
begin

  // makeGF - try to make F into a Dom::GaloisField
  makeGF:=
  proc(F)
    local grF;
  begin
    if F::constructor <> Dom::AlgebraicExtension then
      return(F)
    end_if;
    grF:= makeGF(F::groundField);
    if grF::constructor = Dom::IntegerMod or
      grF::constructor = Dom::GaloisField then
      return(Dom::GaloisField(grF, degree(F::minpoly), F::minpoly))
    end_if; 
    F
  end_proc;

  
  z:=op(a,2);
  if nops(z)<>1 then
    error("univariate polynomial expected")
  end_if;
  z:=op(z);
  userinfo(3, "faclib::algfactor called with input of degree ".
           expr2text(degree(a)));
  Falpha:=op(a,3);
  if Falpha::constructor<>Dom::AlgebraicExtension then
    error("polynomial over an algebraic extension expected")
  end_if;

  // for finite fields, we have to do something entirely different
  Falphanew:= makeGF(Falpha);
  if Falphanew <> Falpha then
    assert(Falphanew::constructor = Dom::GaloisField);
    a:= poly(a, Falphanew);
    if flag then
      b:= faclib::factorgf(a);
      for i from 2 to nops(b) step 2 do
        b[i]:= poly(expr(b[i]), [z], Falpha)
      end_for;
      return(b)
    else
      return(irreducible(a))
    end_if
  end_if;

  
  b:=Factored::convert_to( polylib::sqrfree(a, "Recollect"=FALSE),
                          DOM_LIST );  
  if nops(b)<>3 or op(b,3)>1 then // non square-free 
    userinfo(2, "Polynomial is not squarefree");
    if not flag then
      return(FALSE);
    end_if;
    s:=[b[1]];
    for i from 1 to nops(b) div 2 do
      m:=faclib::algfactor(b[2*i],TRUE);
      s[1]:=s[1]*m[1]^b[2*i+1];
      s:=append(s,(m[2*n],b[2*i+1])$n=1..nops(m) div 2);
    end_for;
    return(s)
  end_if;
  x:=Falpha::variable;
  m:=Falpha::minpoly; // minimal polynomial 
  F:=Falpha::groundField; // base field 
  if type(m)<>DOM_POLY then
    if type(m)="_equal" then
      m:=op(m,1)-op(m,2)
    end_if;
    m:=poly(m,[x],F);
  end_if;
  Fone:=F::one;
  if F=Dom::Rational then
    F:=Expr
  end_if; // to have system representation 
  s:=0;
  if x=z then
    error("choose a different variable for the minimal polynomial of the algebraic extension")
  end_if;
  alpha:=Falpha(x); // algebraic number 
  aa:=poly(subs(expr(a),expr(alpha)=x),[z,x],F);
  // look for a square-free norm 
  userinfo(2,"Looking for square-free norm");
  m2:=poly(m,[z,x],F); // bivariate form 
  repeat
    Norm:=polylib::resultant(m2,aa,x);
    // Norm is a polynom in z over F 
    userinfo(4,"Trying s=".expr2text(s));
    g:=gcd(Norm,D([1],Norm));
    if degree(g)=0 then
      userinfo(2,"Succeeded for s=".expr2text(s));
      break
    end_if;
    s:=s+1;
    // evalp(aa,z=z-x) does not work is F<>Expr 
    aa:=poly(subs(expr(aa),z=z-x),[z,x],F);
  until FALSE end_repeat;
  // factor Norm in F[z] and lift result to F(alpha)[z] 
  b:=factor(Norm);
  b:=Factored::convert_to(b,DOM_LIST);
  n:=nops(b) div 2; // number of irreducible factors 
  userinfo(2,"Norm has ".expr2text(n)." factors");
  if n=1 then
    if flag then
      [lcoeff(a), multcoeffs(a, _invert(lcoeff(a))), 1] // irreducible 
    else
      TRUE
    end_if
  else
    if flag then
      aa:=poly(aa,[z],Falpha);
      for i from 1 to n do
        // b[2*i+1] should be 1
        assert(b[2*i+1] = 1);
        userinfo(3,"Handling ".expr2text(i)."th factor of the norm");
        b[2*i]:=poly(b[2*i],Falpha);
        userinfo(4,"Computing gcd");
        g:=gcd(b[2*i],aa);
        // b[2*i]:=evalp(g,z=z+s*al) does not work 
        userinfo(4,"Substituting");
        b[2*i]:=poly(subs(expr(g),expr(alpha)=x,z=z+s*x),[z],Falpha);
        // make monic
        b[1]:= b[1] * lcoeff(b[2*i]);
        b[2*i]:= multcoeffs(b[2*i], _invert(lcoeff(b[2*i])))
      end_for;
      /* the following guarantees that the product of the unit by the
           factors equals the original polynomial */
      b[1]:=coeff(divide(a,_mult(b[2*i]$i=1..n),Quo),0);
      b
    else
      FALSE
    end_if
  end_if;
end_proc:
