//   


/*++
faclib::factor_expr -- factor expression with given factorizer

faclib::factor_expr(p, options, squarefree)

p   - polynomial expression over the rationals
options - table of options
squarefree - DOM_BOOL. 

Returns a list of the form [c, f1, e1,...,fn, en] such that p=c*f1^e1*...
*fn^en and c is the content of p. 
Returns the squarefree factorization if squarefree = TRUE; otherwise returns the complete factorization.

++*/

faclib::factor_expr:=
proc(p, options = table(MaxDegree = 1, Adjoin = {}): DOM_TABLE, squarefree = FALSE)
  local l, oldl, i, j, f, e, c, newp, fac: DOM_PROC;
begin

  if squarefree then
    fac:= faclib::sqrfree_poly
  else
    fac:= 
    proc(x)
    begin
      Factored::convert_to(factor(x, options), DOM_LIST)
    end_proc;
  end_if;

  // special cases which may save some time...
  case type(p)
    of DOM_INT do
    of DOM_RAT do
      return([p]);

    of "_power" do
      j:= op(p,2);
      if domtype(j) <> DOM_INT then
        return([1, p, 1])
      end_if;
      l:= faclib::factor_expr(op(p,1), options, squarefree);
      return([ l[1]^j, (l[2*i], l[2*i+1]*j) $ i=1..(nops(l) div 2) ]);

    of "_mult" do
      c:= FAIL;
      f:= table();
      // find common extension containing all coefficients
      // of all factors, as well as all elements of options[Adjoin]
      newp:= polylib::makerat([op(p), op(options[Adjoin])], options[MaxDegree]);
      options[Adjoin]:= {};
      newp:= [[op(op(newp, 1), 1..nops(p))], [op(op(newp, 2), 1..nops(p))],
             op(newp, 3)];
      if type(op(newp,[1,1]))<>DOM_POLY then
        // all indets are algebraic over Q, each el. of op(newp,1) is a unit
        return([p])
      end_if;

      
    // divide common factors off
      for i from 1 to nops(op(newp, 1)) do
        for j from 1 to nops(op(newp, 2)) do
           e:= gcd(newp[1][i], newp[2][j]);
           if degree(e) > 0 then
             newp[1][i]:=  divide(newp[1][i], e, Exact);
             newp[2][j]:=  divide(newp[2][j], e, Exact)
           end_if;  
        end_for;
      end_for;  
        
    // factors of the numerator
      
      for e in op(newp,1) do
        e:= fac(e);
        c:= if c=FAIL then e[1] else c*e[1] end_if;
        if iszero(c) then return( [0] ) end_if;

        // collect factors in f:
        for j from 2 to nops(e) step 2 do
          if contains(f, e[j]) then
            f[e[j]]:= f[e[j]] + e[j+1]
          else
            f[e[j]]:= e[j+1]
          end_if
        end_for;
      end_for;

      // factors of the denominator
      // denoted by negative multiplicities
      
      for e in op(newp,2) do
        e:= fac(e);
        c:= c/ e[1];
        if iszero(c) then
          error("division by zero")
        end_if;

        for j from 2 to nops(e) step 2 do
          if contains(f, e[j]) then
            // this may well happen:
            // the product had not been normalized
            f[e[j]]:= f[e[j]] - e[j+1]
          else
            f[e[j]]:= -e[j+1]
          end_if
        end_for;
      end_for;
        
      // deleting zero entries
      f:=select(f, x-> not iszero(op(x,2)));
      l:= map([ c, op(op(f,i)) $ i=1..nops(f) ], expr);
      repeat
        oldl:= l;
        l:= subs(l, op(newp,3))
      until l=oldl end_repeat;
      // this substituitions may have introduced zeroes; which we handle below
      
      break

  otherwise

      if has(p, infinity) then
        return([1, p, 1])
      end_if;
      newp:=normal(p, Recursive = FALSE);
      if contains({DOM_INT, DOM_RAT, "_power", "_mult"}, type(newp)) then
    // simple case after normalizing, start from top
        return(faclib::factor_expr(newp, options, squarefree))
      end_if;
      newp:=polylib::makerat([newp, op(options[Adjoin])], options[MaxDegree]);
      options[Adjoin]:= {};
      newp:= subsop(newp, 1=op(newp, [1, 1]), 2=op(newp, [2, 1]));

      if type(op(newp,1))<>DOM_POLY then
    // all indets are algebraic over Q, p is a unit
        return([p])
      end_if;
    
       // eventually factor using fac
      f:= fac(op(newp,1));
      e:= fac(op(newp,2));
      l:= f . [(op(e,2*i), -op(e,2*i+1)) $ i=1..nops(e) div 2];
      // adjust unit
      l[1]:=l[1]/op(e,1);
        // back substitution:
      
      l:= subs(map(l, expr));
    
      repeat
        oldl:= l;
        l:= subs(l, op(op(newp,3)))
      until l=oldl end_repeat;
      // this substituitions may have introduced zeroes; which we handle below
      
  end_case;

 
  // adjust signs. After back substitution, previously positive
  // factors may now have
  // a negative sign, therefore we have to exchange signs
  for j from 2 to nops(l) step 2 do
    // it may happen that we have detected a constant zero through factoring
    if l[j] = 0 then
      return([0])
    end_if;
    f:=poly(l[j]);
    if f = FAIL then
      e:= stdlib::hasmsign(l[j])
    else
      e:= sign(lcoeff(f)) = -1
    end_if;
    if e then
      l[j]:= - l[j];
      // if n is even, x^n may be replaced by (-x)^n,
      // otherwise we have to adjust the unit
      if l[j+1] mod 2 = 1 then
        l[1]:= -l[1]
      end_if;
    end_if
  end_for;

  return(l)

end_proc:

