// File contains: faclib::heulift
/*-------------------------------------------------------------------

heulift -- Try to guess the correct coefficients of the factors.

Call -- heulift(f, g, l, t)

Parameters -- f - a squarefree multivariate polynomial,
                  f is expected to be DOM_POLY.
                  Any coefficent-ring that supports
                  divide(*,*,Exact) is valid.
                  The constant coeff w.r.t. x is assumed to
                  be nonzero.
              g - the factors of the univariate f(x,a2,..,an)
                  for some a2,..,an in Z.
                  The g are expected to be DOM_POLY.
              l - a list with the correct leading coefficients 
                  of the g's -- or [FAIL] or [] if they were 
                  not computed.
              t - a list with the correct trailing coefficients 
                  of the g's -- or [FAIL] or [] if they were 
                  not computed.
              
Return value -- a list with the correct multivariate factors of 
                f or FAIL if they could not be computed by this
                algorithm.
                
Description -- We have g[1]*..*g[r] = f(x,a2,..,an) for some 
               a2,..,an in Z.
               For each g[i] = c0 + c1 x + c2 x^2 + .. + cs x^s 
               we will replace the nonzero ci by indeterminates Zi and
               call the result gt[i]. 
               
               Then we will let F be the product of the gt[i] and insert 
               the passed correct lcoeffs and tcoeffs. Then we look
               at f = F and try to solve the system of equations
               by coefficient comparison.
                              
               There is no real reference for this, it was inspired 
               by a paper about Maple's factorizations algorithm:
               L. Bernardin, M.B. Monagan: "Efficient Multivariate
               Factorization Over Finite Fiels", AAECC 12, Springer 
               1997, pp. 15--28. 
               Compare especially p.20f "2.5 Sparse Heuristic".
-------------------------------------------------------------------*/

faclib::heulift := proc(f : DOM_POLY, 
		g : DOM_LIST, 
		l : DOM_LIST, 
		t = [FAIL] : DOM_LIST)
local XX, x, R, // Variables
	i, j, // Indices, index-sets
	n, r, d, md, // 'sizes'	
	F, v, pd, st,
	lico, // vectors with the degrees
	Z, // Matrices 
	gt,  // Polynomials
	tmp, t1, t2, t3;
begin
	userinfo(1,"Heuristic lifting called");
	st := time();
	if l = [] then l := [FAIL]; end_if;
	if t = [] then t := [FAIL]; end_if;
	// if l and t are empty, we have no chance.
	if l=[FAIL] and t=[FAIL] then
		return(FAIL);
	end_if;
	//-----------------------------------------------------
	// Check the arguments
	//-----------------------------------------------------
	if testargs() then
		if (DOM_SET@op@map)(g, domtype) <> {DOM_POLY} then
			warning("The homomorphic factors must be DOM_POLY.");
			return(FAIL);
		end_if;
		// get and check the variables to the factors
		tmp := (DOM_SET@op@map)(g, op, 2);
		if nops(tmp) <> 1 then
			warning("All homomorphic factors must have the ".
				" same variable");
			return(FAIL);
		elif nops(op(tmp)) <> 1 then
			warning("The homomorphic factors must be univariate.");
			return(FAIL);
		end_if;
	end_if; // testargs()
	
	//-----------------------------------------------------
	// At first we will do some initializations
	//-----------------------------------------------------
	// get the variables
	XX := op(f,2); // all variables of the multivariate f
	R := op(f,3); // the field over which f lives
	x := op(op(g[1],2), 1); // the variable of the factored 
		// homomorphic image
	n := nops(XX); // the number of variables
	r := nops(g); // the number of factors
	d := map(g,degree,x); // the max x-degree-vector for the g's
	md := max(op(d)); // the maximum degree of the factors
	
	//-----------------------------------------------------
	// Prepare for substitution.
	// Write each g[i] as
	// g[i] = sum(c[i,j]*x^j, j=0..d[i]) where c[i,j] in F 
	// and replace all _nonzero_ c[i,j] by new indets
	// stored in Z[i,j]. So we get
	// gt[i] = sum(Z[i,j]*x^j, j=0..d[i]) 
	// The gt will be polys with _all_ indets of f. 
	//-----------------------------------------------------
	userinfo(9, "Sorting away the coefficients into Matrices and ".
		"inserting passed coefficients.");
	Z := Dom::/*Sparse*/Matrix()(r,md+1); // s.a.
	for i from 1 to r do // iterate over all factors
		for j from 0 to d[i] do // iterate over all terms
			if j = d[i] and l <> [FAIL] then
				Z[i,j+1] := l[i];
			elif j = 0 and t <> [FAIL] then
				Z[i,j+1] := t[i];
			elif not iszero(coeff(g[i],x,j)) then
				Z[i,j+1] := genident("Z");
			end_if; // nontrivial coefficient 
		end_for; // j=0..d[i]
	end_for; // i=1..r
	gt := [poly(_plus((Z[i,j+1]*x^j) $ j=0..d[i]),XX,R) $ i=1..r];

	//-----------------------------------------------------
	// Now multiply the substituted polynomials
	//-----------------------------------------------------
	F := _mult(gt[i] $ i=1..r);
	
	//-----------------------------------------------------
	// we will now look for equations from which we can 
	// reconstruct the correct coefficients of the
	// factors
	//-----------------------------------------------------
	userinfo(9, "Trying to find linear equations and solve them.");

	pd := {$ 0..degree(F)}; // all possible occurring degrees.

	repeat // until nops(lico) = 0
		//-----------------------------------------------------
		// find all linear equations defined by 
		// coefficient-comparison.
		// Therefore we look for those degrees where only one
		// variable (apart from those in XX) occurs.
		// lico means LinearCoefficients.
		//-----------------------------------------------------
		lico := select(pd,
		   q -> (nops((indets@expr)(coeff(F,x,q)) minus {op(XX)}) = 1));		
		
		userinfo(9, "these coefficients of F are linear: ".expr2text(lico));
		
		for j in lico do
			t1 := coeff(F,x,j);
			v := op(indets(expr(t1)) minus {op(XX)});
			if nops(v) = 0 then next; end_if; // this var was 
				// substituted before
			// we now define the linear equation for v
			t2 := poly(coeff(f,x,j),XX,R);
			t1 := poly(t1, XX.[v],R);
			// and solve it
			t3 := divide(t2-coeff(t1,v,0),coeff(t1,v,1),Exact);

			userinfo(9, "just found: ".expr2text(v)." = ".
				expr2text(expr(t3)));
									
			if t3 = FAIL then
				userinfo(9, "There was an unsolvable ".
					"equation, aborting.");
				return (FAIL);
			end_if;

			// insert the newly found coefficient into all gt...
			gt := map(gt, eval@subs, [v = expr(t3)]);
			// ...and into F
			F := (eval@subs)(F, [v = expr(t3)]);
			pd := pd minus {j};
		end_for; // j in lico
	until lico = {} end_repeat;

	//-----------------------------------------------------
	// The loop was finished, therefore there are no more linear
	// dependencies.
	//-----------------------------------------------------
	// Are there dependencies of higher degree left?
	tmp := select(pd, 
		q -> (nops((indets@expr)(coeff(F,x,q)) minus {op(XX)}) > 0));
	// If yes, the algorithm did not succeed
	if tmp <> {} then
		userinfo(9, "There were dependencies of higher degree, ".
                	expr2text(tmp)." -- aborting.");
		return (FAIL);
	end_if;

	// check whether the calculated coefficients were all right.
	if not iszero(f-_mult(op(gt))) then
		userinfo(9, "seems like I calculated something wrong.");
		return(FAIL); // this should happen very rarely (hopefully never)
	end_if;
	
	// everythig's OK, we did it!
	userinfo(9, "faclib::heulift succeeded in ".expr2text(time()-st)." ms");
	return(gt);
end_proc: // faclib::heulift

