/*--
faclib::true_fs -- get the true factor of a(x) from reconstructing g(x) in Z. 
                   Output is an empty list, if it is not true, otherwise, it's
                   given in form [true factor in polynomial form, (the input
                   polynomial)/(the true factor)].
                    
faclib::true_fs(a,lc,g,p,m2)
a - a primitive square free polynomial in Z
lc - lcoeff(a)
g - a polynomial in Z
p -a positive integer number
m2 - a positive real number

faclib::mod_rat -- compute a unique rational number a/b (igcd(a,b)=1) such that
                   c=a/b mod p and |a(b)|<sqrt(p/2), if this number does not
                   exist, then return an empty list
 
faclib::mod_rat(c,p,m2)
c, m - a positive integer number
m2 -a positive real number

the algorithms of faclib::true_fs and faclib::mod_rat can be found on page 2-3
of P. S. Wang, M. Guy, J. Davenport, p-adic reconstruction of rational numbers, ACM SIGSAM Bulletin Vol.16, May 1982 and on page 212-217 of Wang P. S., A p-adic
algorithm for univariate partial fractions, Proceedinfs of the 1981 ACM
symposium on symbolic and algebraic computation, Snowbird Utah, August 5-7, 1981

faclib::mod_rat may be called by faclib::true_fs many times with the same input,
so set "option remember"  
--*/

faclib::true_fs:=proc(a,lc,gg,p,m2)
local c, d, i, n, g, new_coeff, r, tf, u;
begin
    g:=gg;
    // if g(x) is a monic polynomial, then just divide a(x) to test, otherwise 
    // at first using faclib::mod_rat to reconstruct g(x) and then test        
    if (lc)=1 then 
       if tcoeff(a) mod tcoeff(g)=0 then
          if (u:=divide(a,g,Exact))<>FAIL then
             return([g,u]); 
          end_if;
       end_if;
    elif (n:=nterms(g))>1 then 
         new_coeff[1]:=[1,1];
         g:=mapcoeffs(multcoeffs(g,1/lcoeff(g)),modp,p);
         // if for every coefficient of g(x), there exists the rational 
         // number, then the polynomial constructed from these rational 
         // numbers may be a true factor of a(x) in Z                   
         for i from 2 to n do
             if (c:=nthcoeff(g,i))>m2 then 
                if (r:=faclib::mod_rat(c,p,m2))=[] then 
                   return([]);
                else if lc mod r[2]=0 then 
                        new_coeff[i]:=r; 
                     else return([]);
                     end_if; 
                end_if;
             else new_coeff[i]:=[c,1]; 
             end_if;
         end_for;
         // simplify 
         d:=ilcm(new_coeff[i][2]$i=1..n);
         tf:=_plus(multcoeffs(nthterm(g,i),\
                   d*new_coeff[i][1]/new_coeff[i][2])$i=1..n);
         // test 
         if lc mod lcoeff(tf)=0 then
            if tcoeff(a) mod tcoeff(tf)=0 then
               if (u:=divide(a,tf,Exact))<>FAIL then 
                  return([tf,u]);           
               end_if;
            end_if;
         end_if;
    end_if;  
    [];  
end_proc:




faclib::mod_rat:=proc(c,p,m2)
local ra;
option remember;
begin
    [0,p];
    [1,c];
    repeat [%2[1]-(%2[2] div %[2])*%[1],modp(%2[2],%[2])]; 
    until %[2]<m2 end_repeat;
    ra:=%;
    if abs(ra[1])<m2 and c mod p=ra[2]/ra[1] mod p then 
    [ra[2],ra[1]]; 
    else []; 
    end_if;    
end_proc:

