//    
/*
gcdlib::gcdexRational(a, b)

 computes the gcdex of two univariate polynomials with
 rational coefficients

 returns a sequence g, s, t where
     g = gcd(a, b) 
     sa + tb = g

 Method: we use a "primitive" Euclidean algorithm with pseudo-division
 instead of division with remainder

 
*/

gcdlib::gcdexRational:=
proc(A: DOM_POLY, B: DOM_POLY)
 local cA, cB, r;
begin
  cA:= icontent(A);
  cB:= icontent(B);
  if iszero(cB) then
    cB:= 1
  end_if;
  A:= multcoeffs(A, 1/cA);
  B:= multcoeffs(B, 1/cB);
  
  // compute integer gcdex  
  r:= gcdlib::gcdexInteger(A, B);
  // and multiply back by content
  op(r, 1), op(r, 2)/cA, op(r, 3)/cB
end_proc:  
  

/*
 By von zur Gathen/Gerhard, Corollary 6.50, the result is bounded by norm_2(f)^degree(g) * norm_2(g)^degree(f)
 It seems that only for results with more than about 3000 bits, the modular method is better



*/

gcdlib::gcdexInteger:=
proc(A: DOM_POLY, B: DOM_POLY)
   local  resultsize;
 begin
  resultsize:= norm(A, 2)^degree(B) * norm(B, 2)^degree(A);
  if not iszero(resultsize) then 
    resultsize:= resultsize/specfunc::ln(2.0)
  end_if;  
  userinfo(25, "Size of result may be around ", resultsize);
  
  if resultsize < 3000 then 
    userinfo(25, "Using Euclidean algorithm");
    gcdlib::gcdexIntegerPrimitive(A, B)
  else
    userinfo(25, "Using modular algorithm");
    gcdlib::gcdexIntegerModular(A, B)
  end_if  
end_proc:  
  

gcdlib::gcdexIntegerPrimitive:=
proc(A, B)
local a, b, c, newc, xl, R, s, t, u, v, q, r, lc;
begin   
  a:= A;
  b:= B;
  xl:= op(a, 2);
  R:= op(a, 3);
  s:= poly(1, xl, R);
  t:= poly(0, xl, R);
  u:= t;
  v:= s;
  newc:= 1;
// loop invariant : s*A + t*B = a
//                  u*A + v*B = b 
  while not iszero(b) do
    userinfo(10, "Applying Euclidean step");
    [c, q, r]:= [pdivide(a, b)];
    userinfo(10, expr2text(expr(c))."*".expr2text(expr(a))." = ".
             expr2text(expr(q))."*".expr2text(expr(b))." + ".
             expr2text(expr(r)));
    // ca = qb + r
    r:= multcoeffs(r, 1/newc);
    // c*a = q*b + newc*r
    // thus q(u*A + v*B) + newc*r = (c*s)*A + (c*t)*B
    // hence (c*s - q*u)*A + (c*t - q*v)*B = newc*r
    // divide everything by newc
    [a, b]:= [b, r];
    [s, t, u, v]:= [u, v, multcoeffs(multcoeffs(s, c)-q*u, 1/newc),
                    multcoeffs(multcoeffs(t, c)-q*v, 1/newc)];
    newc:= c;
    // loop invariant holds again:
    assert(s*A + t*B = a)
  end_while;
  lc:= 1/lcoeff(a); // we want the gcd to be monic
  // multiply everything by lc:
  // (s*lc)*A + (t*lc)*B = a*lc
  multcoeffs(a, lc), multcoeffs(s, lc), multcoeffs(t, lc)
end_proc:


// modular method
// 
// gcdlib::gcdexIntegerModular(f, g)
// f, g: polynomials over the integers
// experimentally, 2^511 is a good size for the modulus
// since nextprime is quite expensive, we have a list of precomputed primes

gcdlib::gcdexIntegerModular:=
proc(f: DOM_POLY, g: DOM_POLY)
  local x, res, h, p, s, t, ss, tt, pprod, d, i, j, dg;
begin
  x:= op(f, [2, 1]); 
  
  h:= gcd(f, g);
  f:= divide(f, h, Exact);
  g:= divide(g, h, Exact);
  
  res:= polylib::resultant(f, g, x);
  
  // solve s*f + t*g = res 
  // p:= INITIALP;
  pprod:= 1;
  s:= poly(0, [x]);
  t:= poly(0, [x]);
  j:= 0;  

  repeat
    if (j:= j+1) <= nops(numlib::primelist511) then
      p:= numlib::primelist511[j]
    else
      p:= nextprime(p+1)
    end_if;
    [d, ss, tt]:= [gcdlib::gcdexEuclidean(poly(f, IntMod(p)), poly(g, IntMod(p)), x)];
    if degree(d) > 0 then next; end_if;
    ss:= ss*res;
    tt:= tt*res;
    dg:= max(degree(ss), degree(s));
    s:= poly(_plus(numlib::ichrem([coeff(s, i), coeff(ss, i)], [pprod, p]) * x^i $i=0..dg), [x]);
    dg:= max(degree(tt), degree(t));
    t:= poly(_plus(numlib::ichrem([coeff(t, i), coeff(tt, i)], [pprod, p]) * x^i $i=0..dg), [x]);
    pprod:= pprod*p;
    s:= mapcoeffs(s, mods, pprod);
    t:= mapcoeffs(t, mods, pprod);
  until s*f + t*g = poly(res, [x]) end_repeat;
  
  // s*f + t*g = res
  // Thus
  // (s/res/cf) * cf*f*h + (t/res/cg)* cg*g*h = h
  // as desired
  
  h, multcoeffs(s, 1/res), multcoeffs(t, 1/res)
end_proc:




// end of file
