//    

// kg, 09/12/93 

/*--
gcdlib::heu_gcd -- compute heuristic gcd of 2 polynomials over the integers

gcdlib::heu_gcd(p1, p2)

p1,p2 - primitive non-zero polynomials over the integers

gcdlib::heu_gcd computes the gcd of two non-zero primitive polynomials with
integer coeffizients by a heuristic method. The procedure returns the gcd or
FAIL resp. FALSE if the heuristic fails. The output must be made primitive!

See K.O.Geddes et al "Alg. for Computer Algebra", Kluwer 1992, pp320
This algorithm has also been analyzed  in Liao/Fateman, "Evaluation of the Heuristic Polynomial GCD"


--*/

// magic numbers
/*
experiments indicate that the univariate gcd is best computed using the
heuristic gcd if the number of decimal digits of the coefficients is less than
about 170, i.e., their ln is less than 400, for 64 bit Linux systems
For Windows systems, the appropriate value seems to be 140

*/

if sysname(Arch) = "glnxa64" then
  alias(COEFFMAX = 400.0)
else
  alias(COEFFMAX = 140.0)
end_if;

/*
  experiments indicate that the bivariate gcd can be quickly computed using the heuristic gcd, 
  but another algorithm in the recursive call for univariate polynomials, provided that either the 
  coefficients are not too large or the degree is small. We use the bounds below
*/
alias(DEGBOUND_BIVARIATE = 20):
alias(COEFFBOUND_BIVARIATE = 70):

/*
   experiments indicate that in our current test base, the first try succeeds in most cases; the second try succeeds in most of the remaining cases; and so on.
   Since mostly small univariate examples survive many tries and such input can be handled slightly more quickly by other algorithms, 
   NUMBEROFTRIES = 4 might mean a neglegible improvement or maybe not.
*/
alias(NUMBEROFTRIES = 6):


/*
   The following multiplier is suggested in most papers. It may be varied, but should be neither too small (too many tries needed) 
   nor too large (algorithm fails due to coefficient size soon). In addition, the continued fraction expansions of its positive integer powers should have small coefficients.

*/
alias(MULTIPLIER = 73794 / 27011):


gcdlib::heu_gcd:=
proc(a, b)
  local x, xv, g, tries, bound, ca, cb, aa, bb;
begin
  x:= op(a,[2,1]);
  bound:= max(degree(a,x), degree(b,x));
// xv is large enough only if a and b are really primitive polynomials!!
// otherwise, see counterexample in lib/STDLIB/TEST/gcd.tst
  xv:= 2*min(norm(a), norm(b)) + 2;
  
  for tries from 1 to NUMBEROFTRIES do
    if nops(op(a,2)) = 1 then
      if specfunc::ln(xv) > COEFFMAX then
        return(FALSE)
      end_if;
      g:= igcd(evalp(a,x=xv), evalp(b,x=xv))
    else
      if specfunc::ln(xv) > COEFFMAX then 
        return(FALSE)
      end_if;
      aa:= evalp(a, x=xv);
      bb:= evalp(b, x=xv);
      if iszero(aa) then
        g:= bb
      elif iszero(bb) then
        g:= aa
      elif nops(op(a, 2)) = 2 and (bound < DEGBOUND_BIVARIATE or specfunc::ln(xv) < COEFFBOUND_BIVARIATE)
      then 
        // int_gcd can handle non-primitive polynomials
        g:= gcdlib::int_gcd(aa, bb)
      else  
        // use heu_gcd, but take care that the polynomials must be primitive
        ca:= icontent(aa);
        cb:= icontent(bb);
        aa:= multcoeffs(aa, 1/ca);
        bb:= multcoeffs(bb, 1/cb);
        g:= gcdlib::heu_gcd(aa, bb);
        if g = FALSE then 
          return(FALSE) 
        elif g <> FAIL then
          g:= multcoeffs(g, igcd(ca, cb)) 
        end_if  
      end_if;  
      
    end_if;
    if g <> FAIL then
      g:= genpoly(g, xv, x);
      g:= multcoeffs(g, 1/icontent(g));
      if divide(a, g, hold(Exact)) <> FAIL then
        if divide(b, g, hold(Exact)) <> FAIL then
          return(g)
        end_if
      end_if
    end_if;
    /* some authors also suggest to do 
     xv:= trunc(xv^(5/4) * MULTIPLIER); 
     in this moment */
    xv:= trunc(xv * MULTIPLIER);
  end_for;
  FAIL
end_proc:

// end of file 
