

/* 
polylib::CPRES(A, B, x, maxdeg, bez)

Compute the resultant of A and B w.r.t. x using a modular method (Collins' algorithm, see Collins(1971))

A, B - multivariate polynomials 
   x - variable (one element of op(A, 2))
maxdeg - precomputed list of number of necessary interpolation points, or FAIL. 
         Let vars2 be the list op(A, 2) after x has been removed. 
         maxdeg[i] gives the number of different values we need to plug into vars2[i]
  bez  - Bezout matrix of A and B, or FAIL. The desired result is the determinant of bez (possibly times -1). 
         We need to decide whether we want to start a determinant computation or not.
*/



polylib::CPRES:=
proc(A: DOM_POLY, B: DOM_POLY, x, maxdeg = FAIL, bez = FAIL)
  local vars: DOM_LIST, // all variables
        vars2: DOM_LIST, // all variables, except x
           X, // a particular variable (unequal to x) into which we want to plug in
        vars3: DOM_LIST, // all remaining variables (unequal to x and X)       
        F, C, Cstar, Astar, Bstar, i, j, k, DD, b, m: DOM_INT, n: DOM_INT, bound1, euclideanBound, degX, Xindex, ratio, maxratio;
begin   
  vars:= op(A, 2); // all variables
  F:= op(A, 3);
  m:= degree(A, x);
  n:= degree(B, x);
  assert(m >= n);
  
  // by plugging in values, we could have arrived at this very special case
  if nterms(A) = 1 then
    // resultant(a*x^m, B) = a^n * resultant(x, B)^m 
    // note that resultant(x, B) = coeff(B, x, 0) 
    return(coeff(A, x, m)^n* coeff(B, x, 0)^m)
  end_if;
  if nterms(B) = 1 then 
    return(coeff(B, x, n)^m* coeff(A, x, 0)^n)
  end_if;
  
  
  if nops(vars) = 1 then
    return(polylib::presultant(A, B, x, _divide))
  end_if;

  
  vars2:= select(vars, _unequal, x); // variables unequal x
    
  if maxdeg = FAIL then
    assert(bez = FAIL);
  // we are on the top of the stack and have to compute the degree bounds
  // We compute the maximum degree the resultant can have with respect to each variable
  // one bound is clear from looking at the Sylvester matrix: m rows correspond to coefficients of B, 
  // while n rows correspond to the coefficients of A (depending on the reference, some authors take these as colums instead, e.g., von zur Gathen/Gerhard)
    maxdeg:= map(vars2, X -> m * degree(B, X) + n*degree(A, X));  
// We could now use the columnwise maximum of degrees of the Sylvester matrix as an alternative bound.   
// We prefer to set up the Bezout matrix (because this is cheap) instead of using a formula. 
// The advantage is also that we can try to guess whether computing the determinant of it 
// would be cheap.  We do not do this if the matrix created would be too large
    if m < 30 then
   	 bez:= linalg::bezout(A, B, x);
	// for each variable, we now take the sum of the maximal degrees in the rows and the columns
     for i from 1 to nops(vars2) do
       X:= vars2[i];
     	 degX:= matrix(map(bez, degree, X));
       b:= _plus(max(degX[k, j] $j=1..m) $k=1..m);
       if b < maxdeg[i] then maxdeg[i]:= b end_if;
       // the same columnwise, if the matrix is not symmetric (m<>n)
       if m<>n then
         b:= _plus(max(degX[j, k] $j=1..m) $k=1..m); 
         if b < maxdeg[i] then maxdeg[i]:= b end_if
       end_if  
      end_for;  
    end_if; // m < 30
  end_if;  // maxdeg = FAIL
  


// At this point, we decide whether we prefer Bezout resultants. They have to be computed by Gaussian elimination of an m x m matrix.  
// This should be done if the number of recursive interpolations is still high, but we have no exact analysis of the Gaussian elimination algorithm.
// Thus we just use an estimate of the form m^3 and compare it to the number of necessary interpolations

  if bez <> FAIL and m^3 * degree(A) * degree(B) < _mult(op(maxdeg)) then
    // which determinant algorithm is the best ??
    userinfo(25, "Using Bezout resultant method");
    if nops(vars2) <= 4 then
      return((-1)^((m mod 4) div 2) * poly(numeric::det(map(bez, expr), Symbolic), vars2))
    else
      return((-1)^((m mod 4) div 2) * linalg::det(bez))
    end_if
  end_if; 
 
  
// Next, we determine how efficient the Euclidean algorithm would be on this input.
// The existence of an additional variable would slow down the Euclidean 
// algorithm by a factor for which it is difficult 
// to give a closed formula. Usually, the last division is more expensive than
// all others together; a rough estimate is that it takes 
// ( (m-2) * degree(B, X) + (n-2)*degree(A, X) ) * ( (m-1) * degree(B, X) + (n-1)*degree(A, X) )
// operations (pseudo - division of the second by the first subresultant)
// There is another constant factor involved which is difficult to estimate; we set it to 4 in 
// favour of the Euclidean algorithm
  euclideanBound:=  [( (m-2) * degree(B, vars2[i]) + (n-2)*degree(A, vars2[i]) ) * ( (m-1) * degree(B, vars2[i]) + (n-1)*degree(A, vars2[i]) ) $i=1..nops(vars2)];

  ratio:= zip(euclideanBound, maxdeg+1, _divide);
  maxratio:= max(op(ratio));
  userinfo(25, "Expected ratios of running times are: ", ratio);
  
  
  if maxratio < 25 then
  // the Euclidean algorithm will probably be faster than the modular algorithm
  // fall back to other method
    userinfo(25, "Estimated product of degrees of sub-resultants in last division: ", euclideanBound);
    userinfo(25, "Falling back to non-modular method");
    return(polylib::nonModularResultant(A, B, x))
  end_if;  
  
  
  Xindex:= contains(ratio, maxratio);
  X:= vars2[Xindex];
  vars3:= select(vars2, _unequal, X);
  bound1:= maxdeg[Xindex];
  userinfo(25, "Using interpolation variable ".expr2text(X));
  userinfo(25, "Degree bound: ", bound1); 

// Experiments show that bound1 is quite sharp, and we have no reason to choose a probabilistic algorithm.
// The literature recommends that we should decrease the elements of maxdeg 
// if a smaller degree has shown up for some variable in several (say, four) consecutive recursive calls


  delete maxdeg[Xindex];  // for the recursive calls

  C:= poly(0, vars2, F);
  DD:= poly(1, [X], F);
  b:= -1;
  while degree(DD) <= bound1 do
    repeat 
      b:= b+1;
      Astar:= evalp(A, X = b);
      Bstar:= evalp(B, X = b)
    until degree(Astar, x) = m and degree(Bstar, x) = n end_repeat;
    if nops(vars) = 2 then // do not need to update maxdeg and bez
      Cstar:= polylib::CPRES(Astar, Bstar, x, FAIL, FAIL)
    elif bez = FAIL then
      Cstar:= polylib::CPRES(Astar, Bstar, x, maxdeg, FAIL)
    else
      Cstar:= polylib::CPRES(Astar, Bstar, x, maxdeg, map(bez, evalp, X=b))
    end_if;
  // interpolate
    C:= poly(Cstar - evalp(C, X = b), vars2, F) * 
        poly(DD/evalp(DD, X = b), vars2, F) 
        + C;
    DD:= DD*poly(X - b, [X], F)
  end_while;
  C
end_proc:


/*
   polylib::nonModularResultant

   our fallback if we decide we cannot use the Bezout matrix, and also do not want to
   use the modular method from CPRES
  

*/
polylib::nonModularResultant:=
proc(f, g, x)
  local vars, R, T, conv;
begin
  vars:= select(op(f, 2), _unequal, x);
  R:= op(f, 3);
  
  if nops(vars) = 0 then
    conv:= expr
  else
    conv:= X -> poly(X, vars, R)
  end_if;  
  
  T:= polylib::Poly(vars, R);
  f:= poly(f, [x], T);
  g:= poly(g, [x], T);
  
  conv(polylib::euclideanResultant(f, g, x, (X, Y) -> divide(X, Y, Quo)))
end_proc:


// end of file
