/*--
polylib::presultant -- compute the resultant of two polynomials

presultant(p, q, x, df)

p, q - polynomials
 x   - variable
df   - coefficient division function

The polynomials must have the same variables over the same coefficient ring.
x must be among the variables.
If the coefficient ring is a domain then the domain must have the
method '_divide'.
--*/

polylib::presultant:= 
proc(pp: DOM_POLY, qq: DOM_POLY, x, df = _divide)
  local result, n: DOM_INT, m: DOM_INT, p: DOM_POLY, q: DOM_POLY, 
  i: DOM_INT, xindex: DOM_INT, d, contp, contq, rat, subst,
  plist: DOM_LIST, qlist: DOM_LIST, pexpos: DOM_LIST, qexpos: DOM_LIST, 
  one, zero, R, originalvars: DOM_LIST, newvars: DOM_SET, vars: DOM_LIST;
begin
  p:=pp; q:=qq;   // to avoid warnings 
  
  
  R:= op(p, 3);
  assert(R = op(q, 3));
  vars:= originalvars:= op(p, 2);

  
  assert(vars = op(q, 2));
  xindex:= contains(vars, x);
  assert(xindex > 0);
  
  if domtype(R) = DOM_DOMAIN then 
    one:= R::one;
    zero:= R::zero
  else 
    one:= 1;
    zero:= 0;
  end_if;
  
  // zero polynomials or 0 degree ? 
  if iszero(p) or iszero(q) then
    return(zero)
  end_if;
  m:= degree(p, x);
  n:= degree(q, x);
  

  
  if n > m then
    p:= polylib::presultant(q, p, x, df);
    return((if ((n*m) mod 2) = 1 then -p else p end_if));
  end_if;
  if m = 0 then
    return(one)
  end_if;
  if n = 0 then
    return(coeff(q, x, 0)^m)
  end_if;

    // given the high frequency with which these occur, we use a precomputed formula for small degrees
  case [m, n]
    of [2, 2] do
      return(subs(#a[0]^2*#b[2]^2 - #a[0]*#a[1]*#b[1]*#b[2] - 2*#a[0]*#a[2]*#b[0]*#b[2] + 
      #a[0]*#a[2]*#b[1]^2 + #a[1]^2*#b[0]*#b[2] - #a[1]*#a[2]*#b[0]*#b[1] + #a[2]^2*#b[0]^2,
      [(#a[i] = coeff(p, x, i)) $i=0..2, (#b[i] = coeff(q, x, i)) $i=0..2], EvalChanges))
    of [2, 1] do
      return(coeff(p, x, 2)*coeff(q, x, 0)^2 - coeff(p, x, 1)*coeff(q, x, 0)*coeff(q, x, 1) + coeff(p, x, 0)*coeff(q, x, 1)^2)
    of[1, 1] do
      return(coeff(p, x, 1)*coeff(q, x, 0) - coeff(q, x, 1)*coeff(p, x, 0))
  end_case;  
  
  
  plist:= poly2list(p);
  qlist:= poly2list(q);
 
  if nops(vars) = 1 then
    pexpos:= map(plist, op, 2);
    qexpos:= map(qlist, op, 2)
  else
    pexpos:= map(plist, op, [2, xindex]);
    qexpos:= map(qlist, op, [2, xindex])
  end_if;  
  
    // is the degree of a trailing term > 0 ? 
  d:= min(op(pexpos));
  if d > 0 then
    return(coeff(q, x, 0)^d *
    polylib::presultant(divide(p, poly(x^d, vars, R), Exact), q, x, df));
  end_if;
  d:= min(op(qexpos));
  if d > 0 then
    p:= coeff(p,x,0)^d *
    polylib::presultant(p, divide(q, poly(x^d, vars, R), Exact), x, df);
    return((if ((m*d) mod 2) = 1 then -p else p end_if));
  end_if;

  // is there a non-trivial gcd of the degrees of the terms ? 
  d:= igcd(op(pexpos), op(qexpos));
  if d > 1 then
    p:= poly(subs(expr(p), x=x^(1/d)), op(p,2..3));
    q:= poly(subs(expr(q), x=x^(1/d)), op(q,2..3));
    return(polylib::presultant(p, q, x, df)^d);
  end_if;
  
  
  
  if R = Expr then
    // rationalize the coefficients
    rat:= rationalize([plist, qlist], FindRelations = ["_power"]);
    p:= poly(op(rat, [1, 1]), vars, R);
    q:= poly(op(rat, [1, 2]), vars, R);
    subst:= op(rat, 2);
    
    [contp, p]:= gcdlib::extractDenom(p); 
    [contq, q]:= gcdlib::extractDenom(q);
    
    newvars:= indets([coeff(p), coeff(q)], RatExpr);
    assert(newvars = indets([coeff(p), coeff(q)], PolyExpr));
    if nops(newvars) > 0 then
      vars:= [op(newvars)].originalvars;
      p:= poly(p, vars, R);
      q:= poly(q, vars, R)
    end_if;   
  else 
    subst:= {};
    contp:= 1;
    contq:= 1;
  end_if;  
  
  
  if nops(vars) = 1 then
    result:= polylib::univariateResultant(p, q, x, df)
  else
    result:= polylib::multivariateResultant(p, q, x, df)
  end_if;
  
  if nops(vars) > nops(originalvars) then
    assert(R = Expr);
    if nops(originalvars) = 1 then
      result:= op(result, 1)
    else
      result:= poly(result, select(originalvars, _unequal, x)) 
    end_if
  end_if;
  
  if nops(subst) > 0 then
    result:= subs(result, subst, EvalChanges);
    contp:= subs(contp, subst, EvalChanges);
    contq:= subs(contq, subst, EvalChanges)
  end_if;  
  
  if contp <> 1 or contq <> 1 then
    result:= result * contp^n * contq^m
  end_if;  

  result
end_proc:

// end of file