
/*
Ref: Polynomial Decomposition Algorithms, by David R. Barton and Richard Zippel,
JSC 1, 1985, pages 159-168
*/

/*
>> use( polylib,decompose ):
>> decompose(x^12+13*x^8+17*x^4+9);

                          2    3       2   2
               17 x + 13 x  + x  + 9, x , x 

>> decompose(x^6+6*x^4+x^3+9*x^2+3*x-5);

                                2             3
                           x + x  - 5, 3 x + x 

>> decompose( x^6-9*x^5+27*x^4-27*x^3-2*x^2*y+6*x*y+1, x );

                                  3                2
                       - 2 x y + x  + 1 , - 3 x + x 

>> decompose( x^4-3*x^3-x+5);

                              4          3
                             x  - x - 3 x  + 5

>> decompose(poly(x^4-3*x^3-x+5,[x],IntMod(5)));

         /        2                  \       /      2                 \
    poly \ - x + x  , [x], IntMod(5) /, poly \ x + x , [x], IntMod(5) /

>> decompose(2985984*x^28 + 1);

                                   7       2   2
                          2985984 x  + 1, x , x
*/

// algorithm U page 166 : uses only univariate factorization 

polylib::decompose := 
proc(p, x)
local f,g,gt,i,l,k,j,n,l1;
begin
  if args(0)=0 or args(0) >= 3 then
    error("Wrong number of arguments")
  end_if;

  if p::dom::decompose<>FAIL then
    return(p::dom::decompose(args()))
  end_if;

  // check arguments 
  if type(p)=DOM_POLY then 
    if nops(op(p, 2)) <> 1 then
      if args(0) = 1 then
        error("Polynomial must be univariate")
      else
        f:= poly(p, [x])
      end_if
    else
      f:=p 
    end_if
  elif args(0) = 2 then
    f:=poly(p, [x])
  else
    f:= poly(p);
    if f = FAIL or nops(op(f, 2)) <> 1 then
      error("Polynomial expression in one variable expected")
    end_if  
  end_if;

  if f = FAIL then 
    error("Polynomial expression expected")
  end_if;
  
  if args(0)=1 then
    x:=op(f, [2,1]) 
  end_if;

  
  n:=degree(f);
  if n=0 or n=1 or isprime(n) then return(p) end_if;
   // check for f(x^k) 
   l:=igcd(degree(nthterm(f,i))$i=1..nterms(f));
   l:={op(numlib::primedivisors(l))} minus {n};
   if l<>{} then
      l:=min(op(l));
      g:=[polylib::decompose(subsop(subs(f,x=x^(1/l)),2=[x])),poly(x^l,[x])];
   else
     k:=[x],op(f,3); // coefficient field 
     // step U1: initialization 
     g[0]:=poly(x,k); 
     gt[0]:=g[0];
     // step U2: factorization 
     l:=f-poly(coeff(f,0),k);
     l:=factor(l);
     if nops(l) = 1 then
        return(p)
     end_if;
     // Anpassung an Umstellung von Factored::_index (Walter 29.12.04)
     l1:=coerce(l, DOM_LIST)[1];
     l:=polylib::divisors(l);
     userinfo(2,"list of possible candidates is",l);
     g:=polylib::decomp(l,g,gt,1,n);
     if g=FAIL then 
       return(p)
     else 
       i:=nops(g)-1
     end_if;
     g[i]:=multcoeffs(g[i],l1)+poly(f(0),k);
     g:=[g[i-j]$ j=0..i-1];
   end_if;
   if domtype(p)=DOM_EXPR then g:=map(g,expr) end_if;
   op(g)
end_proc:

polylib::decomp := 
proc(l,g,gt,i,n) 
  local ll,j,q, r;
begin
  // step U3: find candidate 
  ll:=[];
  for j in l do
    [q, r]:= [divide(j,gt[i-1])];
    if degree(q)>0 and iszero(mapcoeffs(r, normal)) then 
      ll:=append(ll,j) 
    end_if
  end_for;
  ll:= prog::sort(ll, degree);
  userinfo(2,"list of possible candidates is",ll);
  for q in ll do
    // step U4: p-adic expansion 
    gt[i]:=q;
    userinfo(2,"gt[".i."]=",gt[i]);
    /* here there was a typo in the paper: we have to divide gt[i] by gt[i-1] and not g[i-1] */
    g[i]:=polylib::decomp_compute_g(gt[i],gt[i-1]);
    userinfo(2,"find divisor g[".i."]:",g[i]);
    if g[i]<>FAIL then // try to recurse 
      if degree(gt[i])=n then 
        return(g)
      else 
        q:=polylib::decomp(ll,g,gt,i+1,n)
      end_if;
      if q<>FAIL then 
        return(q) 
      end_if
    end_if;
  end_for;
  FAIL
end_proc:

// compute g such that f(x)=g(h(x)) using the algorithm of section 2.1 
polylib::decomp_compute_g := 
proc(f, h)
  local q, r, i, x, j;
begin
  x:=op(h,[2,1]); 
  if op(h,1)=x then 
    return(f) 
  end_if;
  i:=0; 
  q[0]:=f; 
  repeat
    [q[i+1], r]:= [divide(q[i]-subsop(q[i],1=q[i](0)),h)];
    i:=i+1;
    if not iszero(mapcoeffs(r, normal)) then 
      return(FAIL) 
    end_if;
  until degree(q[i])=0 end_repeat;
  subsop(h,1=_plus(q[j](0)*x^j$j=0..i))
end_proc:
