/* --------------------------------------------------------------- 

   realroots -- isolate real roots of a real univariate polynomial

Call:    polylib::realroots(p);
         polylib::realroots(p, eps);

Parameter:  p -- a univariate polynomial (DOM_EXPR or DOM_POLY).  
                 Its coefficients must be integer or rational 
                 or convertible to floats via MuPADs float command.

           eps -- a positive real number (DOM_INT, DOM_RAT, DOM_FLOAT).
                  Also numerical symbols such as sqrt(2), PI etc. are
                  accepted.

Return value: a list [ [l1,u1], [l2,u2], ... ] of lists [l.i, u.i]
              with rational l.i and u.i.

Synopsis:    
              The returned lists [l.i, u.i] represent isolating intervals
              for the real roots of p.

              If l.i=u.i, then  u.i=l.i is a root of p.

              If l.i < u.i, then there is exactly one real root x of p
              in the open interval:  l.i < x < u.i.
              (the endpoints l.i, u.i may also be roots).

              The number of intervals returned by realroots is
              exactly the number of real roots of p. Multiplicities
              of the roots are not counted.

              If realroots is called with only one argument, then
              the algorithm stops once isolating intervals are found.
              These intervals may be large.

              With the second argument eps the algorithm continues
              to decrease the length of the intervals, until the
              roots are isolated with relative precision given by eps,
              i.e.,
   
                     u.i-l.i <= eps * (u.i+l.i)/2.

              If u.i+l.i = 0, then absolute precision eps will be achieved: 

                     u.i-l.i <= eps .


              Non-rational coefficients are converted to floats.
              These floats are then approximated by a rational 
              within relative precision given 10^(-DIGITS).
              realroots is then applied to this rationalized polynomial.
              Warning: this rationalization might change degenerate roots
                       (or badly isolated roots) drastically!

              Remark: refining the isolating intervals via the second
                      argument eps usually takes much more time than
                      just computing isolating intervals (using only
                      one argument p).


Examples:

  >>   polylib::realroots( poly(x^2-2) );

                          [[-4, 0], [0, 4]]

  >>   polylib::realroots( poly(x^2-2), 10^(-5) );

    [[-46341/32768, -185363/131072], [185363/131072, 46341/32768]]

  >>   polylib::realroots( (x+1)^2*(x-1)^5*(x-2)*(x-10^10)^2);

       [[-34359738368, 0], [0, 2], [2, 2], [8589934592, 17179869184]]

  >>   polylib::realroots( (x+1)^2*(x-1)^5*(x-2)*(x-10^10)^2, 0.1);

        [[-1, -1], [1, 1], [2, 2], [9663676416, 10200547328]]

  >>   polylib::realroots( orthpoly::legendre(10,x) );

      [[-1, -7/8], [-7/8, -3/4], [-3/4, -1/2], [-1/2, -1/4], [-1/4, 0],

         [0, 1/4], [1/4, 1/2], [1/2, 3/4], [3/4, 7/8], [7/8, 1]]
              
  # Isolating two close roots x = 1 +- 1/10^10: #
  >>   polylib::realroots( x*(x-1+1/10^10)*(x-1-1/10^10)*(x-2) );
                   [[0, 0], [0, 1], [1, 2], [2, 2]]

  # The following close roots x = 1 +- sqrt(2)/10^10 are
    not isolated because of float conversion. The rationalized
    polynomials has one double root at x = 1: #
  >>   polylib::realroots( x*(x-1+sqrt(2)/10^10)*(x-1-sqrt(2)/10^10)*(x-2) );

                        [[0, 0], [0, 2], [2, 2]]

  # Need a more accurate rationalization of the coefficients to isolate 
    the roots: #
  >> DIGITS:= 20:
  >> polylib::realroots( x*(x-1+sqrt(2)/10^10)*(x-1-sqrt(2)/10^10)*(x-2) );

                 [[0, 0], [0, 1], [1, 3/2], [3/2, 2]]


-----------------------------------------------------*/

// The module "uspensky" which should be
// in the standard installation of MuPAD. It provides
// fast module versions of the cost critical functions
//     polylib::Uspensky_X2XP1
//     polylib::Uspensky_VarX2XP1R
// If, for some reasons, this module cannot be loaded,
// polylib::realroots falls back to the corresponding
// library functions defined in this file.

          
//--------------------------------------------
// realroots uses global procedures
//     polylib::Uspensky_X2XP1
//     polylib::Uspensky_VarX2XP1R
// that may be provided by a module "uspensky"
//--------------------------------------------

polylib::realroots:= proc(p)
local p0,T,TT,n,x,l,pf,x2xp1,i,coefftypes, eps;
begin
    if args(0)<1 then error("expecting at least one argument") end_if;
    if args(0)>2 then error("expecting at most two arguments") end_if;
    if args(0)= 2 then
       eps:= args(2);
       if {domtype(eps)} minus {DOM_INT, DOM_RAT, DOM_FLOAT} <> {}
          then eps:= float(eps);
               if domtype(eps)<> DOM_FLOAT then
                  error("second argument must be numerical")
               end_if;
       end_if;
       if eps<=0 then error("second argument must be positive") end_if;
    end_if;

   p:= subs(p, float(0)=0);
   if domtype(p) <> DOM_POLY
   then x:= indets(p, PolyExpr);
        x:= x minus select(x,testtype,Type::Constant);
        if nops(x)>1 then error("input must be a univariate polynomial") end_if;
        if nops(x)<1 then 
           if iszero(p) then return([[-infinity,infinity]])
                        else return([])
           end_if;
        end_if;
        x:= op(x,1);
        if not testtype(p, Type::PolyExpr(x))
           then error("input must be a polynomial"); 
        end_if;
        p:=poly(p, [x]); // convert p to a polynomial
        if p=FAIL then error("input must be a polynomial"); end_if;
   else x:= op(p, 2);
        if nops(x)>1 then error("input must be a univariate polynomial") end_if;
        x:= op(x,1);
   end_if;
     
   if iszero(p) then return([[-infinity,infinity]]) end_if;
   n:=degree(p);
   if n=0 then return([]) end_if;

   // Additional code for treating polynomials with non-rational
   // coefficients:
   // 1) Convert poly to list representation. poly2list is
   //    now a kernel function in MuPAD1.5 !!!
   //    Note that the list returned by poly2list is sparse,
   //    i.e., we still need the dense polylib::Uspensky_tolist below.
   // 2) Convert expressions such as sqrt(2), sin(1) etc to floats.
   //    Also convert constants such as PI to floats.
   // 3) rationalize floats
   //
   // Need to document that for non-rational coefficients the
   // resulting separating intervals are not verified !!
   // They can be marred by the intermediate float steps !!!!!!!
   // (This is ok, I guess, because this is just extending the
   //  functionality of realroots);

   l:= poly2list(p):
   coefftypes:= {op(map(l, proc(c) begin domtype(c[1]); end_proc))};
   // Convert coeffs such as sqrt(2) etc. to floats.
   // Then convert floats to rationals:
   if coefftypes minus {DOM_INT, DOM_RAT} <> {} 
   then l:= map(l, proc(c) 
                   local mantissa, exponent;
                   begin 
                   if has({DOM_EXPR, DOM_IDENT}, domtype(c[1]))
                      then c[1]:= float(c[1]);
                   end_if;
                   if iszero(c[1]) then return([0,c[2]]) end_if;
                   if domtype(c[1]) = DOM_FLOAT //brute force rationalization
                     then if c[1]<0 
                          then exponent:= round(ln(-c[1])/ln(10.0));
                               mantissa:= -c[1]/10^exponent;
                               return([-round(mantissa*10^(DIGITS))*
                                        10^(exponent-DIGITS), c[2]])
                          else exponent:= round(ln(c[1])/ln(10.0));
                               mantissa:= c[1]/10^exponent;
                               return([round(mantissa*10^(DIGITS))*
                                       10^(exponent-DIGITS), c[2]])
                          end_if;
                     else return(c);
                   end_if
                 end_proc);
        l:= subs(l, float(0)=0); // remove numerical garbage
        //reconvert to poly for compatibility with original code below
        p:= poly(l, [x]);
   end_if;
   userinfo(1,"polynomial rationalized to ".expr2text(p));

   // double check types of coefficients
   coefftypes:= {op(map(l, proc(c) begin domtype(c[1]); end_proc))};
   if coefftypes minus {DOM_INT, DOM_RAT} <> {} then
      error("failed to rationalize the coefficients");
   end_if;

   // reduce p to squarefree poly:
   p:= p/gcd(p, diff(p,x));
   userinfo(1,"polynomial reduced to squarefree polynomial ".expr2text(p));
   if domtype(p)<> DOM_POLY then error("something is wrong with gcd") end_if;

   // turn rational coeffs into integers:
   p:= mapcoeffs(p, _divide, icontent(p));
   userinfo(1,"polynomial reduced to integer coefficient polynomial ".
               expr2text(p));

   n:=degree(p);
   if n=1 then
      x:= -coeff(p,0)/coeff(p,1); return([[x,x]]);
   end_if;
   p0:= p; // copy of p for late refinement of intervals
   pf:=mapcoeffs(p,float);
   T :=2*max(abs(coeff(pf,n-i)/coeff(pf,n))^(1/i) $ i=1..n);
   TT:=  max(abs(coeff(pf,0)/coeff(pf,n)),
                  1+abs(coeff(pf,i)/coeff(pf,n)) $ i=1..n-1);
   T:= (1.0+10^(2-DIGITS))*min(T,TT);
   if iszero(T) then return([[0,0]]); end_if;
   T:=2^ceil(ln(T)/ln(2.0)); // crucial with rational coefficients/bounds 
   if T<1 then T:=1 end_if; // to avoid rationals 
   userinfo(1,"bound is ".expr2text(T));
   pf:=time();
   // If there is a module uspensky, then use it. 
   // Otherwise, use the library routine in this file.
   x2xp1:=(if type(stdlib::uspensky)=DOM_DOMAIN then
	     stdlib::uspensky::X2XP1
           else polylib::Uspensky_X2XP1
           end_if);
   p:= [coeff(p,All)]; // convert poly to dense list 
   p:=polylib::Uspensky_H(p, -T); // p(-T*x) 
   p:=x2xp1(p); // p(-T*(x+1)) 
   p:=polylib::Uspensky_H(p, -2); // p(T*(2*x-1)) 
   userinfo(1,"initial change of variable took ".
              expr2text((time()-pf))." msec");
   l:=polylib::Uspensky_Kconst(p);
   l:=map(l,proc(t) begin [T*(2*t[1]/2^t[3]-1),T*(2*t[2]/2^t[3]-1)] end_proc);
   if args(0)=2 then
    pf:= time():
    l:= map(l, proc(t) begin polylib::Uspensky_refine(p0, eps, t[1], t[2]) end_proc);
    userinfo(1,"refining intervals by bisectioning took ".expr2text((time()-pf))." msec");
   end_if;
   sort(l,proc(x,y) begin x[1]+x[2]<y[1]+y[2] end_proc)
end_proc:

// ----------------------------------------------------------------------
// Let p = poly(..) be a square free univariate polynomial,
// Let [a,b] be an interval such that
// either   a = b = root of p
// or       the open interval (a,b) contains exactly one root
//          (however, a and b may also be roots).
// polylib::Uspensky_refine does bisectioning of [a,b]  and returns a smaller
// interval [A,B] containing the root.
// Either
//          A=-B  and abs(B-A)<= eps
// (i.e, [-B,B] locates small zero with absolute precision eps)
// or     
//          abs(B-A)<= eps*((A+B)/2)  with (A+B)/2<>0)
// (i.e, [A,B] locates the zero with relative precision eps)
// ----------------------------------------------------------------------
polylib::Uspensky_refine:= proc(p, eps, a, b)
local mid, sm, x, sa;
begin x:= op(p,[2,1]);
      while TRUE do
        mid:=(a+b)/2;
        if mid<>0 then if specfunc::abs(a-b) <= eps* specfunc::abs(mid)
                         then return([a,b]); end_if;
                  else if specfunc::abs(a-b) <= eps
                         then return([a,b]); end_if;
        end_if;
        sm:= specfunc::sign(evalp(p, x=mid));
        if sm=0 then return([mid,mid]) end_if;
        sa:= specfunc::sign(evalp(p, x=a));
        if sa=0 then sa:= specfunc::sign(evalp(diff(p,x), x=a)); end_if;
        if sa*sm<0 then b:= mid; else a:= mid; sa:= sm; end_if;
      end_while;
end_proc:
// ----------------------------------------------------------------------
polylib::Uspensky_icontentlist := proc(l) local i,g;
begin
   g:=l[1]; for i from 2 to nops(l) do
      g:=igcd(g,l[i]); if g=1 then return(g) end_if
   end_for;
   g
end_proc:
// ----------------------------------------------------------------------
// uses global procedures
//     polylib::Uspensky_X2XP1
//     polylib::Uspensky_VarX2XP1R
//     stdlib::uspensky  (this can be a domain given by a module)
polylib::Uspensky_Kconst := proc(Q)
local todo,lastC,c,k,res,lastk,st,i,Q0,nr,st0,x2xp1,nodes,var,tr,s,sc,sa,sb,c0;
begin
   Q:=map(Q,_mult,1/polylib::Uspensky_icontentlist(Q));
   sa:=specfunc::sign(Q[1]); 
   sb:=specfunc::sign(_plus(op(Q)));
   todo:=[[0,0,sa,sb]]; res:=[]; lastk:=0; lastC:=0; Q0:=Q; nr:=0; 
   st0:=time();
   for s from 0 to 4 do nodes[s]:=0 end_for;
   // If there is a module stdlib::uspensky, then use it.
   // Otherwise, use the library routines in this file.
   if type(stdlib::uspensky)=DOM_DOMAIN then
      x2xp1:=stdlib::uspensky::X2XP1; 
      var:=stdlib::uspensky::VarX2XP1R;
    else
      x2xp1:=polylib::Uspensky_X2XP1; 
      var:=polylib::Uspensky_VarX2XP1R;
    end_if;
   tr:=0; sc:=0;
   while todo<>[] do
      ([c,k,sa,sb]):=todo[1];
      delete todo[1]; 
      userinfo(2,"dealing with ".expr2text([c,k])); 
      st:=time();
      if k<>lastk then
         userinfo(1,"start k=".k." at time=".expr2text((time()-st0)),nodes);
         st:=time();
         Q:=polylib::Uspensky_H2(Q0); Q0:=[];
         userinfo(2,"x <- x/2 took ".expr2text(time()-st)." msec");
         st:=time();
         lastk:=k; 
         i:=polylib::Uspensky_icontentlist(Q);
         Q:=map(Q,_mult,1/i);
         userinfo(2,"removing icontent of length ".length(i)." took ".
                    expr2text(time()-st)." msec");
      else 
         Q:=polylib::Uspensky_Translate(Q,c-lastC,x2xp1);
         i:=time()-st;
         userinfo(2,"translation took ".expr2text(i)." msec"); 
         tr:=tr+i;
      end_if;
      lastC:=c;
      if Q[1]=0 then // found root c/2^k 
         res:=append(res,[c,c,k]); 
         delete Q[1];
         sa:=Q[1];
         if Q0<>[] then Q0:=polylib::Uspensky_Translate(Q,c0-c,x2xp1) end_if; // update Q0 
         // and update signs in todo 
         todo:=map(todo, proc(t,x) begin [t[1],t[2],t[3]*specfunc::sign(t[1]/2^t[2]-x),
	   t[4]*specfunc::sign((t[1]+1)/2^t[2]-x)] end_proc, c/2^k);
      end_if;
      st:=time(); 
      s:=var(Q,sa*sb); 
      i:=time()-st;
      if (sa*sb>0 and s mod 2=1) or (sa*sb<0 and s mod 2=0) then s:=s+1 end_if;
      userinfo(2,"number of sign changes is ".s." (".expr2text(i)." msec)"); sc:=sc+i;
      nodes[s]:=nodes[s]+1;
      if s=1 then
         userinfo(1,"*** isolated root ".expr2text((nr:=nr+1)).
                    " at time ".expr2text(time()-st0)." msec"); 
         res:=append(res,[c,c+1,k]);
      elif s>1 then
         if todo=[] or todo[nops(todo)][2]=k then Q0:=Q; c0:=c end_if;
         i:=polylib::Uspensky_sign_middle(Q);
         if [s,sa,i]=[2,sb,-sa] then
         userinfo(1,"*** isolated roots ".expr2text((nr:=nr+1)).
                                  " and ".expr2text((nr:=nr+1)).
                    " at time ".expr2text(time()-st0)." msec");
              res:=append(res,[2*c,2*c+1,k+1],[2*c+1,2*c+2,k+1]);
         else todo:=append(todo,[2*c,k+1,sa,i],[2*c+1,k+1,i,sb]) end_if
      end_if
   end_while;
   userinfo(1,"number of visited nodes is ".expr2text(_plus(nodes[s]$s=0..4)),nodes);
   userinfo(1,"total translation time is ".expr2text(tr)." msec");
   userinfo(1,"total time for sign changes ".expr2text(sc)." msec");
   res
end_proc:
// ----------------------------------------------------------------------
// sign(P(1/2)) 
polylib::Uspensky_sign_middle := proc(P) local i,n; begin
   n:=nops(P)-1; specfunc::sign(_plus(2^(n-i)*P[i+1] $ i=0..n));
end_proc:
// ----------------------------------------------------------------------
// P(c*x) 
polylib::Uspensky_H := proc(P,c) local n,t,i; begin
   n:=nops(P)-1; t:=1/c; [(t:=t*c)*P[i+1] $ i=0..n]
end_proc:
// ----------------------------------------------------------------------
// H(P,1/2) i.e. 2^n*P(x/2) 
polylib::Uspensky_H2 := proc(l) local i,n; begin
   n:=nops(l)-1; [2^(n-i)*l[i+1] $ i=0..n]
end_proc:
// ----------------------------------------------------------------------
// returns P(x+c) 
polylib::Uspensky_Translate := proc(P,c,x2xp1) begin
   if c=0 then P
   elif c=1 then x2xp1(P)
   else polylib::Uspensky_H(x2xp1(polylib::Uspensky_H(P,c)),1/c)
   end_if
end_proc:
// ----------------------------------------------------------------------
polylib::Uspensky_X2XP1 := proc(l) local i,j,n,lj; begin
   n:=nops(l)-1;
   for i from 0 to n-1 do
      lj:=l[n+1]; // invariant: lj = l[j+1] 
      for j from n downto i+1 do
         lj:=l[j]+lj; 
         l[j]:=lj;//lj;
      end_for
   end_for;
   l
end_proc:
// ----------------------------------------------------------------------
// polylib::Uspensky_Var(polylib::Uspensky_X2XP1(R(P))), d is the product of signs at both bounds 
polylib::Uspensky_VarX2XP1R := proc(l,d) local i,j,n,v,c,ls,n1,lj;
begin
   n:=nops(l)-1; v:=0; c:=0;
   n1:=1; while (ls:=specfunc::sign(l[n1]))=0 do n1:=n1+1 end_while;
   // l[1]..l[n1-1] are zero or of sign ls 
   for i from n+1 downto 2 do
      lj:=l[1];
      for j from 2 to i do
         lj:=l[j]+lj; l[j]:=lj;
      end_for;
      // now l[i] won't change anymore 
      if c=0 then 
         c:=specfunc::sign(l[i])
      elif specfunc::sign(l[i])*c<0 then
         v:=v+1; 
         if v>=3 or (v>=2 and d<0) then
           return(v);
         end_if;
         c:=specfunc::sign(l[i]);
      end_if;
      // check if all l[1]..l[i-1] are of same sign 
      while n1<i and specfunc::sign(l[n1])*ls>=0 do n1:=n1+1 end_while;
      if n1=i then
         if ls*c<0 then v:=v+1 end_if;
         return(v);
      end_if
   end_for;
   // check leading coeff 
   if c*specfunc::sign(l[1])<0 then v:=v+1 end_if;
   v;
end_proc:
// end of file
