/* --------------------------------------------------------------- 

   rootbound -- returns a bound on the absolute value of a
                polynomials roots.

Call:    polylib::rootbound(p)

Parameter:  p -- a univariate polynomial (DOM_POLY).  
                 Its coefficients must be integers or rationals 
                 or convertible to floats via MuPADs float command.
           

Return value: a float value.

Synopsis:    
              The return value is an upper bound on the absolute value
              for the roots of the polynomial p, i.e.:
              
              p(x) = 0 ==> |x| <= polylib::rootbound(p)
              
              For details see W. Oevel: "Einfuehrung in die numerische
              Mathematik", Springer 1996, pp 69--70.
              
Remark:       The computation uses floating-point arithmetic.
              
Examples:

  >> polylib::rootbound(poly(x^5+2*x^4+3*x^3+4*x^2+5*x+6,[x]));

                                    4.0
  >> polylib::rootbound(poly(x^5+x^3+x-1,[x]));

                                    2.0
  >> polylib::rootbound(poly((x-3)*(x-1)*(x+5),[x]));

                                8.246211251
  >> polylib::rootbound(poly(x^2+10*x+300,[x]));

                                   30.0
--------------------------------------------------------------- */

polylib::rootbound := 
proc(p : DOM_POLY)
  local b1, b2, b3, p2, b4, L, n, i, xdiv, u, l, t, x;
  save DIGITS;
begin
  userinfo(9,"polylib::rootbound called with polynomial of degree ".degree(p));
  
  if testargs() then
    if nops(op(p, 2)) <> 1 then
      error("The polynomial must be univariate.");
    end_if;
    p2 := map(poly2list(p), x -> float(op(x,1)));
    if {op(map (p2, X -> testtype(X,Type::Numeric)))} <> {TRUE} then
      error("The polynomial must have proper numbers as coefficients.");
    end_if;
  end_if;
  
  x := op(op(p,2),1);
  
  userinfo(9,"calculating the tree 'tradional' rootbounds");
        	// get the coefficient-vector of p 
  n := degree(p);
  L := array(0..n, map([coeff(p,All)], abs));
  
        	// (a) first bound
        	// max{|p0/pn|, 1+|p1/pn|,...,1+|pn-1/pn|}
  b1 := max(L[0]/L[n], 1+L[i]/L[n] $ i=1..n-1);
  userinfo(1,"polylib::rootbound: First bound: ", float(b1));
  
        	// (b) second bound
        	// max{|p0/p1|, 2*|p1/p2|,...,2*|pn-1/pn|}
  xdiv := proc(a,b) begin if iszero(b) then return(infinity) else
  return(a/b) end_if; end_proc;
  b2 := max(xdiv(L[0],L[1]), 2*xdiv(L[i],L[i+1]) $ i=1..n-1);
  userinfo(1,"polylib::rootbound: Second bound: ", float(b2));
  
        	// (c) third bound
        	// 2*max{|pn-1/pn|, |p0/pn|^(1/n),...,|pn-2/pn|^(1/2)}
  b3 := 2*max(float(L[n-1]/L[n]), float(L[i]/L[n])^(1/(n-i)) $ i=0..n-2);
  userinfo(1,"polylib::rootbound: Third bound: ", float(b3));
  
        	// (d) fourth bound
        	// the positive real root of ...
  p2 := poly(x^n - _plus(L[i]/L[n]*x^i $ i=0..n-1));
  DIGITS := 5;
  u := float(min(b1,b2,b3)); l := 0.0;
  repeat
    t := (u+l)/2;
    if p2(t) > 0 then
      u := t;
    else
      l := t;
    end_if;
  until (u-l) < 0.5 end_repeat;
  b4 := u;
  userinfo(1,"polylib::rootbound: Fourth bound: ", b4);
  
  // return the smallest one of the three.
  return(float(min(b1,b2,b3,b4)));
  
end_proc;
