/*    */


/*++ 

	polylib::elemSym ([x1,...,xn], k)

	- x1, ..., xn : identifiers
	- k	      : integer

	computes the k-th elementary symmetric polynomial
	in the variables x1,...,xn

        (That one is defined as the sum of all binomial(n,k)
	 products over every k-element subset of the n
	 variables.)

++*/

polylib::elemSym:=
proc(varlist:DOM_LIST, n:Type::PosInt)

begin
  if n> nops(varlist) then
    return(undefined)
  end_if;
  if n=nops(varlist) then
    _mult(op(varlist))
  else
    op(combinat::choose({op(varlist)}, n));
    _plus(map(%, _mult@op))
  end_if;
  poly(%, varlist)
end_proc:


/*++

	polylib::representByElemSym(f, [s1,..,sn])

	f         - symmetric polynomial in n variables
	s1,.., sn - identifiers

	computes a polynomial g \in K[s1,...,sn] such
	that g(e1,..,en)=f, where ei is the i-th elementary
	symmetric polynomial. 

++*/ 

polylib::representByElemSym:=
proc(f, symbollist: DOM_LIST)
local F, i, elsym, varlist, g, result, lm, n;

begin
  F:=op(f,3);
  varlist:=op(f,2);
  if (n:=nops(varlist))<>nops(symbollist) then
    error("List of symbols must have as many operands as there are variables");
  end_if;
  elsym:=map([polylib::elemSym(varlist,i) $i=1..n],expr);
  result:=poly(0, symbollist, F);
  while not iszero(f) do
    lm:=lmonomial(f);
    g:= expr(lcoeff(f))*
        _mult(symbollist[i]^(degree(lm,varlist[i])-degree(lm,varlist[i+1]))
              $i=1..n-1)*symbollist[n]^degree(lm, varlist[n]);
    result:= result + poly(g, symbollist, F);
    if result = FAIL then
      userinfo(5, "Input was not a symmetric polynomial");
      return(FAIL)
    end_if;
    g:= subs(g, symbollist[i]=elsym[i] $i=1..n);
    f:=f-poly(g, varlist, F);
    userinfo(10, "Result so far is ".expr2text(result));
    userinfo(10, "Polynomial left is ".expr2text(f))
end_while;
result
end_proc:


