/*
prog::profile should calculate sums of the time spend
in all subroutines.  But we have to take care for recursive
calls.  In particular indirekt recursive calls should
be counted correctly.  Here are some test examples.

reset():
f:= proc(i) local t; begin if i = 1 then t := time(); while time() < t + 99 do end else g(i-1); f(i-1) end end_proc:
g := n -> ((f(i): g(i-1)) $ i = 1..n):
p := () -> (g(2)):
prog::profile(p()):
p := () -> (g(3)):
prog::profile(p()):
p := () -> (g(1); f(2); g(2)):
prog::profile(p()):
*/

alias(ENTRYPOINT = "procedure entry point"):

prog::profile :=
  proc(stmt)
    option hold, escape;
    save MAXDEPTH, GPROF, fprint;
    local OPT, f, k, t, kk,
          CALLS,     // table: recursion depth -> list of childs
          CALLS2,    // table: read from a file
          INFO,      // table: function -> all timing information
          INFO2,     // table: read from a file
          TIMEforall,// time used for whole
          NUMBER,    // table with numbered procedures
          result,    // filename for kernel profiling file
          html;      // is this actually HTML output?
  begin
    ///////////////////////////////////////////////////////////////////////
    //// O P T I O N S ////////////////////////////////////////////////////
    ///////////////////////////////////////////////////////////////////////
    // options and arguments

    if args(0) = 0 then
      return()
    end_if;

    OPT := prog::getOptions(2, context([FAIL, args(2..args(0))]),
                            table(hold(Last) = FALSE,
                                  Style = 0,
                                  Output = 0,
                                  Quiet = FALSE,
                                  hold(SaveTo) = 0,
                                  hold(Collect) = []), TRUE)[1];

    // open output file
    html := FALSE;
    if OPT[Output] <> 0 then
      if testtype(OPT[Output], DOM_STRING) and strmatch(OPT[Output], "\\.html$") = TRUE then
        html := TRUE;
      end_if;
      if (OPT[Output] := prog::checkFile(OPT[Output], Write, Text)) = FAIL then
        error("cannot write to given file")
      end_if;
      if html then
        fprint(Unquoted, OPT[Output],
          prog::profile::HTMLprologue,
          "<h1>Profile call</h1>\n<pre>",
          stringlib::subs(expr2text(stmt), "&"="&amp;", "<"="&lt;"),
          "</pre>\n\n");
      else
        fprint(Unquoted, OPT[Output],
               "Profile call: prog::profile(", args(), ")\n\n");
      end_if;
    end_if;

    ///////////////////////////////////////////////////////////////////////
    //// E V A L //////////////////////////////////////////////////////////
    ///////////////////////////////////////////////////////////////////////
    // eval stmt and create gprof file
    [INFO, CALLS, result] := stdlib::gprof(context(stmt));
    
    INFO := table(map([op(INFO)], eq -> expr2text(op(eq, 1)) = op(eq, 2)));
    CALLS := table(map([op(CALLS)],
      eq -> expr2text(op(eq, 1)) =
        table(map([op(op(eq, 2))], eqq -> expr2text(op(eqq, 1)) = op(eqq, 2)))));
    
    INFO[ENTRYPOINT] := INFO["#TOPLEVEL"];
    delete INFO["#TOPLEVEL"];
    [INFO[ENTRYPOINT][1], INFO[ENTRYPOINT][2]] :=
      [INFO[ENTRYPOINT][2], INFO[ENTRYPOINT][1]];
    TIMEforall := INFO[ENTRYPOINT][1];
  
    delete CALLS["#TOPLEVEL"]["#TOPLEVEL"];
    CALLS[0] := CALLS["#TOPLEVEL"];
    delete CALLS["#TOPLEVEL"];
    
    ///////////////////////////////////////////////////////////////////////
    //// O U T P U T //////////////////////////////////////////////////////
    ///////////////////////////////////////////////////////////////////////
    // output information

    //  %   cumulative   self              self     total
    //time   seconds   seconds    calls  ms/call  ms/call  name

    // INFO - [TS, TC, calls, remembers, errors]

//fprint(0, "\n%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% INFO %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%");
//output::tableForm(INFO);
//
//fprint(0, "\n%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% CALLS %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%");
//output::tableForm(CALLS);


    ///////////////////////////////////////////////////////////////////////
    //// C O L L E C T ////////////////////////////////////////////////////
    ///////////////////////////////////////////////////////////////////////

    if OPT[hold(Collect)] <> [] then // collect information from files
      // INFO and CALLS are tables, possibly filled
      if nops(INFO) = 1 and op(INFO, [1, 2]) = [0, 0, 1, 0, 0] then
        // remove this invalid call
        INFO := table(ENTRYPOINT = [0, 0, 0, 0, 0]);
        CALLS := table()
      end_if;
      for f in OPT[hold(Collect)] do
        if (OPT[hold(File)] := prog::checkFile(f, Read)) = FAIL then
          warning("skip file '".prog::getname(f)."' (unable to open)");
        else
          finput(OPT[hold(File)], stmt, INFO2, CALLS2);
//fprint(0, "%%%% INFO %%%%");
//output::tableForm(INFO2);
          fclose(OPT[hold(File)]);

          // collect information
          for k in INFO2 do
            if contains(INFO, op(k, 1)) then
              INFO[op(k, 1)] := zip(INFO[op(k, 1)], op(k, 2), _plus)
            else
              INFO[op(k, 1)] := op(k, 2)
            end_if
          end_for;
          for k in CALLS2 do // name = table(name = [n, n, n])
            if contains(CALLS, op(k, 1)) then
              t := CALLS[op(k, 1)]
            else
              t := table()
            end_if;
            for kk in op(k, 2) do // each entry of the sub-table
              if contains(t, op(kk, 1)) then
                t[op(kk, 1)] := zip(t[op(kk, 1)], op(kk, 2), _plus)
              else
                t[op(kk, 1)] := op(kk, 2)
              end_if
            end_for;
            CALLS[op(k, 1)] := t
          end_for
        end_if
      end_for;
      // new TIMEforall
      TIMEforall := INFO[ENTRYPOINT][1]
    end_if; // end of modes: collect or compute information

    if OPT[hold(SaveTo)] <> 0 then
      if (OPT[hold(File)] := prog::checkFile(OPT[hold(SaveTo)], Bin, Write)) = FAIL then
        error("cannot open file '".prog::getname(OPT[hold(SaveTo)])."' for writing")
      end_if;
      fprint(OPT[hold(File)], "call: ".expr2text(stmt), INFO, CALLS, result);
      fclose(OPT[hold(File)]);
      if OPT[Quiet] = TRUE then
        return()
      end_if
    end_if;
    
    case OPT[Style]
      ////////////////////////////////////////////////////////////
      of 0 do
        NUMBER := 
          prog::profileOutput
          (INFO, 
           ["percent usage of all",
            "time self per single call",
            "time self",
            "time children per single call",
            "time children",
            "calls/normal exit", "calls/remember exit", "calls/errors",
            "[index] function name"],
           html,
           op(OPT), // options
           Sort =
             proc(LIST)
             begin
               float(100*(LIST[1])/(.0001 + TIMEforall))
             end_proc,
           Separator = "  ",
           hold(Align) = [Right, Right, Right, Right, Right, Right, Right, Right, Left, Left],
           Format = proc(INFO) //
                    begin
                      stringlib::formatf(float(100*(INFO[1])/(.0001 + TIMEforall)), 1), // percent time
                      stringlib::formatf(float((INFO[1])/(INFO[3] + INFO[4] + INFO[5])), 1), // self time per call
                      stringlib::formatf(float(INFO[1]), 1), // time self
                      stringlib::formatf(float(INFO[2]/(INFO[3]  + INFO[4] + INFO[5])), 1), // children time per call
                      stringlib::formatf(float(INFO[2]), 1), // children time 
                      expr2text(INFO[3]), // normal return
                      expr2text(INFO[4]), // remember table
                      expr2text(INFO[5]), // error
                      expr2text([INFO[6]]), // Index
                      INFO[7]  // Name
                    end_proc);
        // vspace
        fprint(Unquoted, OPT[Output], "\n");
        
        prog::profileGraph
        (INFO, CALLS, TIMEforall,
         ["index", "%time", "self", "children", "called", "[index] name"],
         NUMBER, // numbering of procedures, returned by profileOutput
         html,
         op(OPT), // options
         Sort =
                proc(INFO)
                begin
                  float(100*(INFO[1])/(.0001 + TIMEforall))
                end_proc,
         Separator = "  ",
         hold(Align) = [Right, Right, Right, Right, Right, Left]
         // Format procedure statically
         );
        break
      ////////////////////////////////////////////////////////////
      of 1 do
        // prog::profileOutput
        // (INFO,
        //  ["Percent usage of all",
        //   "Time self args", "Time self body",
        //   "Time cumulative args", "Time cumulative body",
        //   "Calls", "Remember", "Error",
        //   "Function name"],
        //  Sort = proc(INFO)
        //         begin
        //           float(100*(INFO[1] + INFO[2])/(1 + TIMEforall))
        //         end_proc,
        //  Separator = "  ",
        //  Align = [Right, Right, Right, Right, Right, Right, Right, Left],
        //  Format = proc(LIST) //
        //           begin
        //           end_proc);
        break;
      otherwise
        error("unknown style number")
    end_case;

    if html then
      fprint(Unquoted, OPT[Output], "\n<p>Time sum: ", TIMEforall, " ms</p>");
    else
      fprint(Unquoted, OPT[Output], "\nTime sum: ", TIMEforall, " ms");
    end_if;
    //fprint(Unquoted, 0, "Time for analysis: ", time() - STIME);

    if OPT[Quiet] <> TRUE then
      if OPT[Output] <> 0 then
        fprint(Unquoted, 0, "Time sum: ", TIMEforall, " ms\n",
                            "Info: Output written to given file.");
        if html then
          fprint(Unquoted, OPT[Output], "\n<h2>RESULT</h2>\n<pre>",
            stringlib::subs(strprint(result), "&"="&amp;", "<"="&lt;"), "</pre>",
            prog::profile::HTMLend);
        else
          fprint(Unquoted, OPT[Output], "\nRESULT:\n", result);
        end_if;
        if type(OPT[Output]) = DOM_STRING then
          fclose(OPT[Output])
        end_if
      end_if;
      result
    else
      if OPT[Output] <> 0 then
        if html then
          fprint(Unquoted, OPT[Output], "\n<h2>RESULT</h2>\n<pre>",
            stringlib::subs(strprint(result), "&"="&amp;", "<"="&lt;"), "</pre>",
            prog::profile::HTMLend);
        else
          fprint(Unquoted, OPT[Output], "\nRESULT:\n", result);
        end_if;
        if type(OPT[Output]) = DOM_STRING then
          fclose(OPT[Output])
        end_if
      end_if;
      null()
    end_if
  end_proc:

//
    /////////////////////////////////////////////////////////////////
    //  timing table  ///////////////////////////////////////////////
    /////////////////////////////////////////////////////////////////

// INFO is a table, each entry is a list with fixed number of entries
// HEAD  is a list of headings
//
// Options
//     Sort = num         : sorting by num-th entry of a line
//                        : default = 0 -> sort by index
//     Sort = proc        : procedure to compute a sortable index! for each
//                          table line, called with complete line (! Index)
//     Skip = num|list    : forget num-th entries
//                          list is a list of nums -> forget all num-th entries
//     Format = proc      : procs, that formats the k entries of a line, given as list
//     Align = list       : list of k 'Left|Center|Right' for all k entries

prog::profileOutput :=
  proc(INFO, HEAD = [], html)
    local k,       // 
          n,       //
          rows,    // number of indices
          OPT,     // table of options
          width,   // list of column width's
          SORTtab, // table sorting entry = index
          SINDEX,  // list of entries, that should be sorted by
          //SKIP,    // list of indices, that should be skipped
          SEP,     // Separator between columns
          FORMAT,  // format procedure
          NUMBER,  // table with number
          i;
  begin
    OPT := prog::getOptions(2, [args()],
                            table(//hold(Last) = FALSE,
                                  // options given to profile
                                  Output = 0,
                                  
                                  Sort = 0,
                                  hold(Format) = id,
                                  hold(Align) = [],
                                  Separator = " "), FALSE)[1];
    SEP := OPT[Separator];

    // number of rows
    rows := nops(INFO);

    // Sorting
    // SINDEX contains all sorting indices
    // SORTtab contains for each sorting index all entries with the _same_ index
    SINDEX := [];
    SORTtab := table();
    if type(OPT[Sort]) = DOM_PROC then
      for i from 1 to nops(INFO) do
        k := OPT[Sort](op(INFO, [i, 2]) . [op(INFO, [i, 1])]);
        if contains(SORTtab, k) then
          SORTtab[k] := append(SORTtab[k], op(INFO, [i, 1]))
        else
          SORTtab[k] := [op(INFO, [i, 1])]
        end_if
      end_for;
      SINDEX := sort(map([op(SORTtab)], op, 1))
    elif type(OPT[Sort]) = DOM_INT and OPT[Sort] > 0 then
      // sorting by n-th entry
      for k in INFO do
        i := op(k, [2, OPT[Sort]]);
        if contains(SORTtab, i) then
          SORTtab[i] := append(SORTtab[i], op(k, 1))
        else
          SORTtab[i] := [op(k, 1)]
        end_if
      end_for;
      SINDEX := sort(map([op(SORTtab)], op, 1))
    else
      // sorting by index
      SINDEX := sort(map([op(INFO)], op, 1))
    end_if;
    
    // numbering by sorting
    INFO[ENTRYPOINT] := append(INFO[ENTRYPOINT], 0);
    NUMBER := table();
    width := 1;
    if OPT[Sort] <> 0 then
      for i from nops(SINDEX) downto 1 do
        n := SORTtab[op(SINDEX, i)];
        for k in sort(n) do
          if k <> ENTRYPOINT then
            INFO[k] := append(INFO[k], width);
            NUMBER[width] := k;
            width := width + 1
          end_if            
        end_for
      end_for
    else
      for i from rows downto 1 do
        INFO[k] := append(INFO[k], width);
        NUMBER[width] := k;
        width := width + 1
      end_for
    end_if;
    
    // Format procedure for one row
    if OPT[hold(Format)] = id then
      // keep entries, only separate it
      FORMAT := proc(LIST)
                  local k;
                begin
                  (case type(LIST[k])
                     of DOM_FLOAT do
                       stringlib::formatf(LIST[k], 1); break
                     of DOM_STRING do
                       LIST[k]; break
                     otherwise
                       expr2text(LIST[k])
                   end_case) $ k = 1..nops(LIST)
                end_proc
    else
      FORMAT := OPT[hold(Format)]
    end_if;

    // maximum of all entries per column
    width := [0 $ nops(op(INFO, [1, 2])) + 1];
    for i from 1 to rows do
      INFO[op(INFO, [i, 1])] := [FORMAT(op(INFO, [i, 2]) . [op(INFO, [i, 1])])];
      width := zip(width, map(INFO[op(INFO, [i, 1])], length), max, 0)
    end_for;
    //width := [op(width, 1..nops(width)-1), length(max(map(op(NUMBER), op, 2))) + 3, op(nops(width))];

    if html then
      fprint(Unquoted, OPT[Output],
        prog::profile::Output_HTMLHeader);
      // table lines
      FORMAT := proc(i)
        local data, nam, num;
      begin
        nam := if iszero(i) then ENTRYPOINT else NUMBER[i] end;
        data := INFO[nam];
        fprint(Unquoted, OPT[Output],
          "<tr",
          if i=0 then " class='top bot'"
          elif i=1 then " class='top'"
          elif i=nops(NUMBER) then " class='bot'"
          end_if,
          ">",
          ("<td>", if iszero(num) or (testtype(num, DOM_STRING) and strmatch(num, "^[0\\. ]*$")) then "." else num end, "</td>") $ num in data[1..-3],
          "<td><a href='#f-", i, "'>[", i, "]</a></td><td class='funcname'>",
          stringlib::subs(nam, "&"="&amp;", "<"="&lt;"), "</td></tr>"
          );
      end_proc;
      for i from 0 to nops(NUMBER) do
        FORMAT(i);
      end_for;
      fprint(Unquoted, OPT[Output],
        prog::profile::Output_HTMLFooter);
    else
      // format one line
      if nops(OPT[hold(Align)]) < nops(width) then
        OPT[hold(Align)] := OPT[hold(Align)].[hold(Left) $ nops(width)]
      end_if;
      FORMAT := proc(LIST)
                  local k;
                begin
                  stringlib::format((case LIST[k]
                                       of " 0.0" do ". "; break
                                       of "0" do "."; break
                                       otherwise LIST[k]
                                     end_case),
                                    width[k], op(OPT[hold(Align)], k), hold(Separat))
                  . SEP
                  $ k = 1..nops(LIST)
                end_proc;
    
      /////////////////////////////////////////////////////////////////
      //  timing table  ///////////////////////////////////////////////
      /////////////////////////////////////////////////////////////////

      // print headings
      if OPT[hold(Headings)] <> hold(Plain) then
        for i from 1 to nops(HEAD) do
          fprint(Unquoted, OPT[Output],
                 (_if(width[k-1] <= 2, " " $ width[k-1] - 1, " " $ width[k-1] - 2),
                  "|",
                  _if(width[k-1] <= 2, "", " "),
                  SEP)
                 $ k = 2..i,
                 _if(i < nops(HEAD), " " $ (width[i] div 2) - 1, ""),
                 HEAD[i])
        end_for
      else
        // fit into column width?
        for i from 1 to nops(width) do
          width := zip(width, map(HEAD, length), max)
        end_for;
        fprint(Unquoted, OPT[Output], FORMAT(HEAD))
      end_if;
      // heading separator
      fprint(Unquoted, OPT[Output], "-" $ _plus(op(width), length(SEP)*nops(width)));
      // print table
      // first ENTRYPOINT with additional separator
      fprint(Unquoted, OPT[Output], FORMAT(INFO[ENTRYPOINT]));
      fprint(Unquoted, OPT[Output], "-" $ _plus(op(width), length(SEP)*nops(width)));
      for k from 1 to nops(NUMBER) do
        fprint(Unquoted, OPT[Output], FORMAT(INFO[NUMBER[k]]))
      end_for;
      // closing separator
      if nops(NUMBER) >= 1 then
        fprint(Unquoted, OPT[Output], "-" $ _plus(op(width), length(SEP)*nops(width)))
      end_if;
    end_if;

    // return numbering/order of procedures
    NUMBER
  end_proc:

//
    /////////////////////////////////////////////////////////////////
    //  call graph  /////////////////////////////////////////////////
    /////////////////////////////////////////////////////////////////

// INFO is a table, each entry is a list with fixed number of entries
// (only needed for whole time of a procedure)
// CALLS is a table, each entry is a list with fixed number of entries
// HEAD  is a list of headings
//
// Options
//     Sort = proc        : procedure to compute a sortable index! for each
//                          table line, called with complete line (! Index)
//     Align = list       : list of k 'Left|Center|Right' for all k entries

prog::profileGraph :=
  proc(INFO, CALLS, TIMEforall, HEAD = [], NUMBER = [], html)
    local k,       // 
          n, ch,   //
          OPT,     // table of options
          width,   // list of column width's
          SINDEX,  // list of entries, that should be sorted by
          TIME,    // table of times
          // NUMBER is given by profileOutput
          //NUMBER,  // table of numbering
          SEP,     // Separator between columns
          FORMAT,  // format procedure
          CALLEDBY,// temp table
          i, sorted, top;
  begin
    OPT := prog::getOptions(4, [args()], 
                            table(//hold(Last) = FALSE,
                                  // options given to profile
                                  Output = 0,
                                  
                                  Sort = 0,
                                  hold(Align) = [],
                                  Separator = " "), FALSE)[1];
    SEP := OPT[Separator];

    /////////////////////////////////////////////////////////////////
    //  call graph output  //////////////////////////////////////////
    /////////////////////////////////////////////////////////////////

    // static format
    // index %time   self children called name
    // --------------------------------------
    // [1]     4.0   0.0  100.0       1   assume [1]
    //              12.1  350.0       5       stdlib::syseval [2]
    // --------------------------------------
    //                                5       assume [1]
    // [2]    60.0  12.1  350.0       1   stdlib::syseval [2]
    // --------------------------------------

    // CALLS is a table with an entry NAME for each procedure
    // CALLS[NAME] is a table with an entry FUNC for each procedure called by NAME
    //     with entries [num, time self, time children]
    //     num - number of calls (how often calls NAME the procedure FUNC)
    //     time self - how many time uses FUNC for its args/body self
    //     time children - how many time uses FUNC's children for its args/body self
    // ! extract also functions, that did not call any other function !
    //   such functions are only in CALLEDBY
    
    // extract "called by" information
    CALLEDBY := table();
    TIME := table();
    for k in CALLS do // op(k, 1) is the second index of CALLEDBY
      if op(k, 1) = 0 then
        k := (ENTRYPOINT = op(k, 2))
      end_if;
      for i in op(k, 2) do // op(i, 1) is the first index of CALLEDBY
        if contains(CALLEDBY, op(i, 1)) then
          if contains(CALLEDBY[op(i, 1)], op(k, 1)) then
            CALLEDBY[op(i, 1)] := zip(CALLEDBY[op(i, 1)][op(k, 1)], op(i, 2), _plus, 0)
          else
            CALLEDBY[op(i, 1)][op(k, 1)] := op(i, 2);
          end_if
        else // new
          if op(k, 1) <> ENTRYPOINT then
            CALLEDBY[op(i, 1)] := table(op(k, 1) = op(i, 2))
          end_if;
          TIME[op(i, 1)] := float(100*(INFO[op(i, 1)][1])/(.0001 + TIMEforall));
        end_if
      end_for
    end_for;

//output::tableForm(TIME);
//fprint(0, "----");
//output::tableForm(CALLS);
//fprint(0, "----");
//output::tableForm(CALLEDBY);
//fprint(0, "----");
//output::tableForm(TIME);

    // NUMBER procedures
    if NUMBER = [] then
      // call index by CALLEDBY[4] - percent time
      // Sorting
      // SINDEX contains all sorting indices
      // SORTtab contains for each sorting index all entries with the _same_ index
      SINDEX := map([op(TIME)], op, 1);
      SINDEX := sort(SINDEX, (X,Y) -> if TIME[X] = TIME[Y] then
                                        sysorder(Y,X)
                                      else
                                        bool(TIME[X] < TIME[Y])
                                      end_if);
      n := nops(SINDEX) + 1;

      NUMBER := table(ENTRYPOINT = 0, k = n - contains(SINDEX, k) $ k in SINDEX)
    else
      // NUMBER given by prog::profile in reverse order
      SINDEX := revert([NUMBER[k] $ k = 1..nops(NUMBER)]);
      NUMBER := revert(NUMBER)
    end_if;

    for k in SINDEX do
      assert(contains(NUMBER, k))
    end_for;

    // special entries for procedure entry
    if contains(CALLS, 0) then
      CALLS[ENTRYPOINT] := CALLS[0]
    else
      CALLS[ENTRYPOINT] := []
    end_if;
    TIME[ENTRYPOINT] := 100;
    NUMBER[ENTRYPOINT] := 0;
    SINDEX := append(SINDEX, ENTRYPOINT);
    

//output::tableForm(CALLS);
//fprint(0, "----");
//output::tableForm(SINDEX);
//fprint(0, "----");
//output::tableForm(NUMBER);
    
    if html then
      fprint(Unquoted, OPT[Output],
        prog::profile::Graph_HTMLHeader);
      for i from nops(SINDEX) downto 1 do // !!!! 0 excluded temporarily
        n := op(SINDEX, i); // function name
        // 1. all parents
        // extract indices and sort by index
        if contains(CALLEDBY, n) then
          top := " class='top'";
          for k in sort([op(CALLEDBY[n])],
                        (X,Y) -> _plus(10^5*op(X, [2, 2]), 10^-5*op(X, [2, 3]))
                                   <= _plus(10^5*op(Y, [2, 2]), 10^-5*op(Y, [2, 3]))) do
                             //(X,Y) -> NUMBER[op(X, 1)] < NUMBER[op(Y, 1)]) do
            fprint(Unquoted, OPT[Output],
              "<tr", top, "><td/><td/>", // index, time,
              "<td>", op(k, [2, 2]), "</td>", // self
              "<td>", op(k, [2, 3]), "</td>", // children
              "<td>", op(k, [2, 1]), "</td>", // called
              "<td><a href='#f-", NUMBER[op(k, 1)], "'>[", NUMBER[op(k, 1)], "]</a></td>", // index
              "<td class='funcname'>", op(k, 1), "</td>"  // name
            );
            top := "";
          end_for;
        end_if;
        //fprint(Unquoted, 0, "%%", TIME[n]);
        // 2. function itself
        if contains(CALLS, n) and n <> ENTRYPOINT then
          ch := _plus(op(k, [2, 2..3]) $ k in CALLS[n])
        else
          ch := 0
        end_if;
        fprint(Unquoted, OPT[Output],
          "<tr class='",
          if not contains(CALLEDBY, n) then "top " end,
          if not contains(CALLS, n) or CALLS[n] = [] then "bot" end,
          "'><td id='f-", NUMBER[n], "'>[", NUMBER[n], "]</td>", // index
          "<td>", stringlib::formatf(float(TIME[n]), 1), "</td>", // %time
          "<td>", INFO[n][1], "</td>", // self
          "<td>", ch, "</td>", // children
          "<td>", INFO[n][3] + INFO[n][4] + INFO[n][5], "</td>", // exits
          "<td colspan='2' class='funcname main'>", n, "</td></tr>");
        // 3. all called functions
        if not contains(CALLS, n) or CALLS[n] = [] then
          // the function does not call any other function
        else
          // need to detect last entry
          sorted := sort([op(CALLS[n])],
                        (X,Y) -> _plus(10^5*op(X, [2, 2]), 10^-5*op(X, [2, 3]))
                                   > _plus(10^5*op(Y, [2, 2]), 10^-5*op(Y, [2, 3])));
          for k from 1 to nops(sorted) do
            fprint(Unquoted, OPT[Output],
              "<tr", if k=nops(sorted) then " class='bot'" end, ">",
              "<td/><td/>", // index, %time
              "<td>", op(sorted[k], [2, 2]), "</td>", // self
              "<td>", op(sorted[k], [2, 3]), "</td>", // children
              "<td>", op(sorted[k], [2, 1]), "</td>", // called
              "<td><a href='#f-", NUMBER[op(sorted[k], 1)], "'>[", NUMBER[op(sorted[k], 1)], "]</a></td>", // index
              "<td class='funcname'>", op(sorted[k], 1), "</td></tr>"  // name
            );
          end_for
        end_if
      end_for;
      fprint(Unquoted, OPT[Output],
        prog::profile::Graph_HTMLFooter);
    else
      // format each entry to find the widest entry
      //FORMAT := proc(NAME, Centry) // one line from CALLEDBY
      //          begin
      //            stringlib::formatf(float(Centry[4]), 1), // percent time
      //            stringlib::formatf(float(Centry[2]), 1), // self time per call
      //            stringlib::formatf(float(Centry[3]), 1), // cumulative time per call
      //            expr2text(Centry[1]), // calls
      //            expr2text(NAME) // name
      //          end_proc;

      // maximum of all entries per column -- fixed
      width := [5, 8, 8, 8, 9, 2, length(nops(NUMBER)) + 2];
      // 
      //width := map(HEAD, length); // width of the headings
      //for k in INDEX do // all functions
      //  for i in CALLS[k] do
      //    // replace table line with formatted line
      //    CALLS
      //    CALLEDBY[CALLS[op(k, 1)][op(i, 1)]] := [FORMAT(CALLS, [i, 1]), op(CALLS, [i, 2]))];
      //    width := zip(width, map(INFO[op(INFO, [i, 1])], length), max, 0)
      //  end_for
      //end_for;

      // format (print) one line
      if nops(OPT[hold(Align)]) < nops(width) then
        OPT[hold(Align)] := OPT[hold(Align)].[hold(Left) $ nops(width)]
      end_if;
      //FORMAT := proc(TYPE, NAME, LIST) // see last FORMAT procedure for LIST
      //            local k, INDEX;
      //          begin
      //            INDEX := expr2text([CALLEDBY[n][hold(CALLEDBY_Index_)]]);
      //            case TYPE
      //              of -1 do
      //                fprint(Unquoted, 0, " " $ _plus(op(width, 1..3), length(SEP)*3),
      //                                 LIST[1], SEP, "    ", NAME, " ", INDEX); // parents
      //                break
      //              of 0 do
      //                fprint(Unquoted, 0,
      //                       stringlib::format(INDEX, width[1]),
      //                       stringlib::format(INDEX, width[1]));  // function itself
      //                break
      //              of 1 do
      //                fprint(Unquoted, 0, " " $ _plus(op(width, 1..3), length(SEP)*3),
      //                                 LIST[1], SEP, "    ", NAME, " ", INDEX); // parents
      //                break
      //            end_case
      //          end_proc;

      FORMAT := proc()
                begin
                  (if k <= nops(width) then
                     stringlib::format(args(k), width[k], op(OPT[hold(Align)], k), hold(Separat))
                   else
                     args(k)
                   end_if)
                  . SEP
                  $ k = 1..(args(0) - 1),
                  args(args(0)) // the name without alignment, should be left
                end_proc;

    
  //fprint(Unquoted, 0, "\n\n", output::tableForm(CALLS));

  //fprint(Unquoted, 0, "\n\n", output::tableForm(CALLEDBY));

      // print space
      fprint(Unquoted, OPT[Output], FORMAT(op(HEAD)));

      // print graph
      // index %time   self children called name
      for i from nops(SINDEX) downto 1 do // !!!! 0 excluded temporary
        n := op(SINDEX, i); // function name
        // separator
        fprint(Unquoted, OPT[Output], "-" $ _plus(36, op(width), length(SEP)*nops(width)));
        // 1. all parents
        // extract indices and sort by index
        if not contains(CALLEDBY, n) then
          // the function does not call any other function
        else
          for k in sort([op(CALLEDBY[n])],
                        (X,Y) -> _plus(10^5*op(X, [2, 2]), 10^-5*op(X, [2, 3]))
                                   <= _plus(10^5*op(Y, [2, 2]), 10^-5*op(Y, [2, 3]))) do
                             //(X,Y) -> NUMBER[op(X, 1)] < NUMBER[op(Y, 1)]) do
            fprint(Unquoted, OPT[Output],
                   FORMAT("", // index
                          "", // %time
                          expr2text(op(k, [2, 2])), // self
                          expr2text(op(k, [2, 3])), // children
                          expr2text(op(k, [2, 1])), // called
                          "  ", // indent
                          expr2text([NUMBER[op(k, 1)]]), // index
                          op(k, 1)  // name
                          ))
          end_for
        end_if;
        //fprint(Unquoted, 0, "%%", TIME[n]);
        // 2. function itself
        if contains(CALLS, n) and n <> ENTRYPOINT then
          ch := _plus(op(k, [2, 2..3]) $ k in CALLS[n])
        else
          ch := 0
        end_if;
        fprint(Unquoted, OPT[Output],
               FORMAT(expr2text([NUMBER[n]]), // index
                      stringlib::formatf(float(TIME[n]), 1), // %time
                      expr2text(INFO[n][1]), // self
                      expr2text(ch), // children
                      expr2text(INFO[n][3] + INFO[n][4] + INFO[n][5]), // exits
                      /*"[31m".*/ n /*."[0m"*/  // name
                      //expr2text([NUMBER[n]]) // index not necessary?
                      ));
        // 3. all called functions
        if not contains(CALLS, n) or CALLS[n] = [] then
          // the function does not call any other function
        else
          for k in sort([op(CALLS[n])],
                        (X,Y) -> _plus(10^5*op(X, [2, 2]), 10^-5*op(X, [2, 3]))
                                   > _plus(10^5*op(Y, [2, 2]), 10^-5*op(Y, [2, 3]))) do
            //(X,Y) -> NUMBER[op(X, 1)] < NUMBER[op(Y, 1)]) do
            fprint(Unquoted, OPT[Output], FORMAT("", // index
                                       "", // %time
                                       expr2text(op(k, [2, 2])), // self
                                       expr2text(op(k, [2, 3])), // children
                                       expr2text(op(k, [2, 1])), // called
                                       "  ", // indent
                                       expr2text([NUMBER[op(k, 1)]]), // index
                                       op(k, 1)  // name
                                       ));
          end_for
        end_if
      end_for;
      fprint(Unquoted, OPT[Output], "-" $ _plus(36, op(width), length(SEP)*nops(width)));
    end_if; // html, else branch
  end_proc:


prog::profile := funcenv(prog::profile):

prog::profile::HTMLprologue :=
"<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\"
  \"http://www.w3.org/TR/html4/strict.dtd\">
<html>
  <head>
    <style type=\"text/css\">
      table {
        margin-bottom: 20px;
      }
      th {
        text-align: left;
        border: 1px solid #999;
        border-bottom: 0px;
        border-right: 0px;
      }
      td {
        border-left: 1px solid #999;
        text-align: right;
      }
      tbody {
        height: 70ex;
        overflow: auto;
      }
      tbody tr {
        overflow: hidden;
      }
      .funcname {
        font-family: monospace;
        border-right: 1px solid #999;
        text-align: left;
      }
      .main {
        border-top: 1px solid #999;
        border-bottom: 1px solid #999;
      }
      .last {
        border-right: 1px solid #999;
      }
      .rsp {
        border-bottom: 1px solid #999;
        border-top: 0px;
        border-right: 0px;
      }
      .bot td, .bot th {
        border-bottom: 1px solid #999;
      }
      .top td {
        border-top: 1px solid #999;
      }
    </style>
  </head>
  <body>":

prog::profile::HTMLend :=
"  </body>
</html>":

prog::profile::Output_HTMLHeader :=
"    <table cellspacing='0' cellpadding='5'>
      <thead>
        <tr><th colspan='10'>percent usage of all</th></tr>
        <tr><th rowspan='8' class='rsp'></th><th colspan='9' class='last'>time self per single call</th></tr>
        <tr><th rowspan='7' class='rsp'></th><th colspan='8' class='last'>time self</th></tr>
        <tr><th rowspan='6' class='rsp'></th><th colspan='7' class='last'>time children per single call</th></tr>
        <tr><th rowspan='5' class='rsp'></th><th colspan='6' class='last'>time children</th></tr>
        <tr><th rowspan='4' class='rsp'></th><th colspan='5' class='last'>calls with normal exit</th></tr>
        <tr><th rowspan='3' class='rsp'></th><th colspan='4' class='last'>calls with remember exit</th></tr>
        <tr><th rowspan='2' class='rsp'></th><th colspan='3' class='last'>calls with errors</th></tr>
        <tr class='bot'>
          <th rowspan='1' class='rsp'></th>
          <th>[index]</th><th class='last'>function name</th>
        </tr>
      </thead>
      <!-- =============================== -->
      <tbody>":
      
prog::profile::Output_HTMLFooter :=
"      </tbody>
    </table>
":

prog::profile::Graph_HTMLHeader :=
"    <table cellspacing='0' cellpadding='5'>
      <thead>
        <tr class='bot'><th>index</th><th>% time</th><th>time self</th><th>time children</th><th>calls</th><th>[index]</th><th class='last'>name</th></tr>
      </thead>
      <tbody>
":

prog::profile::Graph_HTMLFooter :=
"      </tbody>
    </table>
":
