/*
*** MuPAD Library Test Coverage?

"MuPAD Library Test Coverage" (tcov) means counting the number of visits of (debug) nodes 
(= MuPAD statements) while executing MuPAD library code. The coverage shows which MuPAD
statements have been visited and which not.

Counting the number of visits of (debug) nodes (= MuPAD statements) is done by the MuPAD 
kernel when it is started in the tcov mode (-t command line option). The kernel writes 
this information to a tcov data file. The MuPAD kernel can also read such tcov data files
from previous MuPAD runs in order to merge them.

The tcov data file consists of a filename section and a data section separated by a line
'-1:-1' and has the following format:

"filename":index:
...
-1:-1:
fileid:line:column:hidden:passes:
...

*** What does this function?

1. Create a copy of MuPAD source files adding information about how often a line (= statement) 
   has been visited. This information can be printed to the screen or to an HTML formatted file.
2. Generate, display and export statistical information about the MuPAD Library Test Coverage.
3. Merge tcov data files from several MuPAD runs.

*** User Interface

prog::tcov(Reset)        - Starten einer neuen Session
prog::tcov(Info)         - Anzeigen einer Statistik
prog::tcov(Write, file)  - Schreiben einer log-Datei (fuer Append oder externe Tools)
prog::tcov(Export, file) - Exportieren der Statistik

prog::tcov(Annotate)     - Erstellen von 'tcov-Dateien', Source-Dateien mit Anmerkungen ueber Duchlaeufe
  OPTION  (Path = dir)   - schreibt die tcov-Dateien in das Verzeichnis dir statt in das lib-Verzeichnis

prog::tcov(file, Append) - fuegt die log-Daten des tcov data files zu der aktuellen Session hinzu

prog::tcov(stmt)         - evaluiert die gegebene Anweisung stmt

  OPTION  (Comment)      - Kommentar fuer kommentierte Source-Dateien und HTML-Export
  OPTION  (Graphical)    - Erstellt graphische Anzeigen fuer Export
  OPTION  (Filter)       - ...
  OPTION  (Lines)        - ...
  OPTION  (Summary)      - ...
  OPTION  (All)          - beruecksichtigt alle Debug-Knoten
  OPTION  (Unused)       - beruecksichtigt auch unbesuchte Debug-Knoten
  OPTION  (Hidden)       - beruecksichtigt auch versteckte Debug-Knoten


CSS stylesheet file for HTML export


*/
// 
prog::tcov_stylesheet :=
  ["/*  tcov main file */", "\n",
   ".tcov-summary         { font-family:monospace; }", "\n",
   ".tcov-summary h1      { font-size:15pt; }", "\n",
   ".tcov-summary h2      { font-size:12pt; }", "\n",
   ".tcov-summary-comment { }", "\n",

   "\n",
   ".tcov-summary-table          { font-size:11pt; background-color:#F2F2F2; border:1px solid gray; }", "\n",
   ".tcov-summary-table .cb      { width:550px; }", "\n",
   ".tcov-summary-table .cb-proc { width:550px; }", "\n",
   ".tcov-summary-table-small    { font-size:8pt; background-color:#F2F2F2; border:1px solid gray; }", "\n",
   ".tcov-summary-lib            { font-family:sans-serif; font-weight:bold; }", "\n",

   "\n",
   ".tcov-summary-libraries { }", "\n",
   ".tcov-summary-library   { vertical-align:top; border-top:2px solid #AAAAAA; padding:4px 10px; margin-top:10px; width:100%; }", "\n",
   
   "\n",
   ".tcov-summary-row        { border-top:1px solid #F2F2FF; }", "\n",
   ".tcov-summary-col1       { width:254px; vertical-align:top; }", "\n",
   ".tcov-summary-col2       { min-width:400px; }", "\n",
   ".tcov-summary-source-ref { font-family:monospace; }", "\n",
   ".tcov-summary-file       { font-size:8pt; }", "\n",
   ".tcov-summary-files      { background:#F8F8FF; vertical-align:top; border:1px solid #CCCCEE; padding:4px 10px; margin:4px 0px 10px 10px; }", "\n",
   ".tcov-summary-files-tab  { border:0px; }", "\n",
   
   "\n",
   ".tcov-summary-files-tab td                 { border-top:1px solid #DDDDFF; }", "\n", 
   ".tcov-summary-files-tab .tcov-summary-col1 { width:244px; font-size:10pt; }", "\n",
   ".tcov-summary-files-tab .tcov-summary-col2 { min-width:400px; }", "\n",

   "\n",
   ".tcov-use-index     { font-weight:bold; }", "\n",
   ".tcov-folder-toggle { text-decoration:none; }", "\n",

   "\n",
   "/* tcov source file */", "\n",
   ".tcov-filename    { font-family:monospace; font-weight:bold; border-bottom:1px solid #AAAAAA; }", "\n",
   ".button-hide      { font-family:monospace; font-weight:bold; font-size:80%; border:1px solid gray; padding:1px 2px 0px 2px; color:black; background:#D2D2D2; }", "\n",
   ".button-back      { font-family:monospace; font-weight:bold; font-size:80%; border:1px solid gray; padding:1px 2px 0px 2px; color:black; background:#D2D2D2; }", "\n",
   ".l-table          { font-family:monospace; white-space:pre; }", "\n",
   ".l-table tr       { }", "\n",
   ".l-table td       { }", "\n",
   ".l-table .l-hide  { padding-left:4px; margin-right:1px; }", "\n",
   ".l-thide .l-hide  { padding-left:4px; margin-right:1px; display:none; }", "\n",
   ".l-up             { text-align:center; color:gray; border-left:1px solid #D2D2D2; padding-left:0px; padding-right:0px; }", "\n",
   ".l-up a           { text-align:center; color:gray; font-weight:bold; color:black; background:#D2D2D2; }", "\n",
   ".l-num            { cursor:pointer; min-width:4em; text-align:right; color:gray;  border-left:1px solid #D2D2D2; border-right:1px solid #D2D2D2; padding-right:2px; background:#F2F2F2; }", "\n",
   ".l-num-proc       { cursor:pointer; min-width:4em; text-align:right; color:black; border-left:1px solid #D2D2D2; border-right:1px solid #D2D2D2; padding-right:2px; background:#FFFF82; font-weight:bold; }", "\n",
   ".l-num-proc-nouse { cursor:pointer; min-width:4em; text-align:right; color:black; border-left:1px solid #D2D2D2; border-right:1px solid #D2D2D2; padding-right:2px; background:#FF8282; font-weight:bold; }", "\n",
   ".l-num-inproc     { cursor:pointer; min-width:4em; text-align:right; color:gray;  border-left:1px solid #D2D2D2; border-right:1px solid #D2D2D2; padding-right:2px; background:#F2F2F2; }", "\n",
   ".l-num-endproc    { cursor:pointer; min-width:4em; text-align:right; color:black; border-left:1px solid #D2D2D2; border-right:1px solid #D2D2D2; padding-right:2px; background:#FFFF82; font-weight:bold; }", "\n",
   ".l-proc-def       { }", "\n",
   ".l-idx-tab        { border:1px solid gray; }", "\n",
   ".l-idx-tab td     { border:0px; }", "\n",
   ".l-idx-row        { cursor:pointer; min-height:6px;  height:6px;  background:white; }", "\n",
   ".l-idx-row-big    { cursor:pointer; min-height:12px; height:12px; background:white; }", "\n",
   ".l-idx            { }", "\n",
   ".l-space          { width:4px; }", "\n",
   ".l-src            { padding-left:4px; margin-right:1px; }", "\n",
   ".l-src-high-one   { padding-left:4px; margin-right:1px; background-color:#E0E0FF; border-left:1px solid blue; border-right:1px solid blue; border-top:1px solid blue; border-bottom:1px solid blue; }", "\n",
   ".l-src-high-top   { padding-left:4px; margin-right:1px; background-color:#E0E0FF; border-left:1px solid blue; border-right:1px solid blue; border-top:1px solid blue; }", "\n",
   ".l-src-high-mid   { padding-left:4px; margin-right:1px; background-color:#E0E0FF; border-left:1px solid blue; border-right:1px solid blue; }", "\n",
   ".l-src-high-bot   { padding-left:4px; margin-right:1px; background-color:#E0E0FF; border-left:1px solid blue; border-right:1px solid blue; border-bottom:1px solid blue; }", "\n",
   ".l-ndn            { color:#AAAAAA; }", "\n",

   "\n",
   "/* tcov highlight files when it contains highlighted lines */", "\n",
   ".l-highlighted-fil{ background-color:#E0E0FF; border:1px solid blue; }", "\n",
   
   "\n",
   "/* node coverage pass levels for colored source lines */", "\n",
   ".l-0 { color:#FF0000;                 } /* =    0 */", "\n",
   ".l-1 { color:#EEBB00;                 } /* <    3 */", "\n",
   ".l-2 {                                } /* <   10 */", "\n",
   ".l-3 {                                } /* <  100 */", "\n",
   ".l-4 {                                } /* < 1000 */", "\n",
   ".l-5 { font-weight:bold;              } /* > 1000 */", "\n",
   "/* node coverage pass levels for colored index */", "\n",
   ".b-0 { width:1px; background:#FF0000; } /* =    0 */", "\n",
   ".b-1 { width:1px; background:#EEBB00; } /* <    3 */", "\n",
   ".b-2 { width:1px; background:#DDDDDD; } /* <   10 */", "\n",
   ".b-3 { width:1px; background:#DDDDDD; } /* <  100 */", "\n",
   ".b-4 { width:1px; background:#DDDDDD; } /* < 1000 */", "\n",
   ".b-5 { width:1px; background:#D2D2D2; } /* > 1000 */", "\n",
   ".b-h { width:1px; background:#C0C0FF; } /* highlighted */", "\n",

   "\n",
   "/* coverage bars in summary tables */", "\n",
   ".cb       { width:300px; background:#DDDDDD; border:1px solid #AAAAAA; vertical-align:middle; font-weight:bold; padding:1px; }", "\n",
   ".cb0      { width:00%;  border:1px solid #ff0000; background:#ff0000; }", "\n",
   ".cb5      { width:05%;  border:1px solid #ff0000; background:#ff0000; }", "\n",
   ".cb10     { width:10%;  border:1px solid #ff0000; background:#ff0000; }", "\n",
   ".cb15     { width:15%;  border:1px solid #ff0000; background:#ff0000; }", "\n",
   ".cb20     { width:20%;  border:1px solid #ff0000; background:#ff0000; }", "\n",
   ".cb25     { width:25%;  border:1px solid #ff0000; background:#ff0000; }", "\n",
   ".cb30     { width:30%;  border:1px solid #ff0000; background:#ff0000; }", "\n",
   ".cb35     { width:35%;  border:1px solid #ff0000; background:#ff0000; }", "\n",
   ".cb40     { width:40%;  border:1px solid #ff6600; background:#ff0000; }", "\n",
   ".cb45     { width:45%;  border:1px solid #ff6600; background:#ff0000; }", "\n",
   ".cb50     { width:50%;  border:1px solid #ff6600; background:#ff6600; }", "\n",
   ".cb55     { width:55%;  border:1px solid #ff6600; background:#ff6600; }", "\n",
   ".cb60     { width:60%;  border:1px solid #88ff88; background:#88ff88; }", "\n",
   ".cb65     { width:65%;  border:1px solid #88ff88; background:#88ff88; }", "\n",
   ".cb70     { width:70%;  border:1px solid #88ff88; background:#88ff88; }", "\n",
   ".cb75     { width:75%;  border:1px solid #88ff88; background:#88ff88; }", "\n",
   ".cb80     { width:80%;  border:1px solid #00ff00; background:#00ff00; }", "\n",
   ".cb85     { width:85%;  border:1px solid #00ff00; background:#00ff00; }", "\n",
   ".cb90     { width:90%;  border:1px solid #00ff00; background:#00ff00; }", "\n",
   ".cb95     { width:95%;  border:1px solid #00ff00; background:#00ff00; }", "\n",
   ".cb100    { width:99.8%; border:1px solid #00ff00; background:#00ff00; }", "\n",

   ".cb-small { width: 30px; background:#DDDDDD; border:1px solid #AAAAAA; vertical-align:middle; font-size:8pt;    padding:1px 2px 1px 1px; margin-right:5px; margin-top:2px; float:left; }", "\n",

   ".cb-proc           { width:300px; background:#CCCCCC; border:1px solid #AAAAAA; vertical-align:middle; font-weight:bold; padding:1px; }", "\n",
   ".cb-proc .cb0      { width:00%;  border:1px solid #ff0000; background:#ff0000; }", "\n",
   ".cb-proc .cb5      { width:05%;  border:1px solid #ff0000; background:#ff0000; }", "\n",
   ".cb-proc .cb10     { width:10%;  border:1px solid #ff0000; background:#ff0000; }", "\n",
   ".cb-proc .cb15     { width:15%;  border:1px solid #ff0000; background:#ff0000; }", "\n",
   ".cb-proc .cb20     { width:20%;  border:1px solid #ff0000; background:#ff0000; }", "\n",
   ".cb-proc .cb25     { width:25%;  border:1px solid #ff0000; background:#ff0000; }", "\n",
   ".cb-proc .cb30     { width:30%;  border:1px solid #ff0000; background:#ff0000; }", "\n",
   ".cb-proc .cb35     { width:35%;  border:1px solid #ff0000; background:#ff0000; }", "\n",
   ".cb-proc .cb40     { width:40%;  border:1px solid #ff6600; background:#ff0000; }", "\n",
   ".cb-proc .cb45     { width:45%;  border:1px solid #ff6600; background:#ff0000; }", "\n",
   ".cb-proc .cb50     { width:50%;  border:1px solid #ff6600; background:#ff0000; }", "\n",
   ".cb-proc .cb55     { width:55%;  border:1px solid #ff6600; background:#ff0000; }", "\n",
   ".cb-proc .cb60     { width:60%;  border:1px solid #ff6600; background:#ff6600; }", "\n",
   ".cb-proc .cb65     { width:65%;  border:1px solid #ff6600; background:#ff6600; }", "\n",
   ".cb-proc .cb70     { width:70%;  border:1px solid #ff6600; background:#ff6600; }", "\n",
   ".cb-proc .cb75     { width:75%;  border:1px solid #ff6600; background:#ff6600; }", "\n",
   ".cb-proc .cb80     { width:80%;  border:1px solid #88ff88; background:#88ff88; }", "\n",
   ".cb-proc .cb85     { width:85%;  border:1px solid #88ff88; background:#88ff88; }", "\n",
   ".cb-proc .cb90     { width:90%;  border:1px solid #88ff88; background:#88ff88; }", "\n",
   ".cb-proc .cb95     { width:95%;  border:1px solid #00ff00; background:#00ff00; }", "\n",
   ".cb-proc .cb100    { width:99.8%; border:1px solid #00ff00; background:#00ff00; }", "\n",

   "\n",
   "a:link    { text-decoration:none; }", "\n",
   "a:visited { text-decoration:none; }", "\n",
   "a:focus   { text-decoration:none; }", "\n",
   "a:hover   { text-decoration:none; }", "\n",
   "a:active  { text-decoration:none; }", "\n",

   "\n",
   "/* preferences for priniting */", "\n",
   "@media print {", "\n",
   ".l-idx-tab { visibility:hidden; }", "\n",
   "}", "\n",
   ""]:

prog::tcov_javascript := 
  ["    <script language=\"javascript\">", "\n",
   "      function toggle (elt) {", "\n",
   "         el = document.getElementById(elt)", "\n",
   "         if (el.style.display == \"none\") {", "\n",
   "           el.style.display = \"\";", "\n",
   "         } else {", "\n",
   "           el.style.display = \"none\";", "\n",
   "         };", "\n",
   "      }", "\n",
   "      function setclass (elt, classname1, classname2) {", "\n",
   "         el = document.getElementById(elt)", "\n",
   "         if (el.className == classname1) {", "\n",
   "           el.className = classname2;", "\n",
   "         } else {", "\n",
   "           el.className = classname1;", "\n",
   "         };", "\n",
   "      }", "\n",
   "      lastf='';", "\n",
   "      lastl='';", "\n",
   "      function go (f, l) {", "\n",
   "        if (f != '' && l != '') {", "\n",
   "          lastf=f;", "\n",
   "          lastl=l;", "\n",
   "          window.location=f + \".tcov.html#l\" + l;", "\n",
   "        }", "\n",
   "      }", "\n",
   "      function ugo (f, l) {", "\n",
   "        setclass('src-table','l-table','l-table');", "\n",
   "        if (f != '' && l != '') {", "\n",
   "          lastf=f;", "\n",
   "          lastl=l;", "\n",
   "          window.location=f + \".tcov.html#l\" + l;", "\n",
   "        }", "\n",
   "      }", "\n",
   "      function wo (f, l) {", "\n",
   "        window.open(f + \".tcov.html#l\" + l);", "\n",
   "      }", "\n",
   "    </script>", "\n",
   ""]:

prog::tcov :=
proc(STMT)
  local allOptions, OPT, CLENGTH, OFILE, PLENGTH, baseName, c, chName, cnt, data,
        fd, file, filedata, fileline, filename, finame, graphIndex, h, hidden, path2css,
        i, l, lib, libName, library, line, linepart, lines, max_file, max_pass, minColor,
        writeStylesheet, nd, nodes, p, passes, sort_files, sort_libs, ud, unused, 
        useIndex, useProcs, procsInLine, ExcludeTab, exRa, highlightedSourceCode,
        val2class, EVALMODE, removeLibpath, fname, bname, mkdirRec, fl, fc, val2index, str2html,
        allNodes, allProcs, hiddenProcs, procList, procedure, unusedNodes, unusedProcs,
        inProc, nextProcStart, endproc, Pathname, tr_src_class, td_src_class, dbg,
        first, Args;
  option hold;
begin
  if args(0) = 0 then // Reset or Stat?
    error("wrong number of arguments")
  end_if;

  if debug() <> TRUE then
    error("MuPAD kernel must be started with option -t")
  end_if;
    
  allOptions := table(
                 Reset=FALSE,          // start (new) tcov session
                 Append=FAIL,          // append pass info from other sessions (written with Write)    
                 Write="",             // write log file
                 Info=FAIL,            // print statistics
                 Export="",            // export info to HTML file
                 
                 Annotate=FALSE,       // write source files with annotations
                 Path=FAIL,            // path for annotated source files

                 Comment="Created on ".external("util","date")(), // comment for annotation and export
                 Graphical=FALSE,      // graphical indices in HTML report
                 hold(GraphicalInAnnotate)=FALSE,  // graphical indices in annotated source files
                 hold(DataFrom)="",          // get data from file
                 hold(Exclude)="",           // exclude files from coverage (they do not go into the statistic!)
                 hold(ListExcludedFiles)=FALSE,// list all excluded files in HTML report
                 hold(Filter)="",            // *all* files in statistic, but details only for files that match Filter
                 hold(FilterInStatsOnly)=FALSE,// statistics and details only for files that match Filter
                 hold(Highlight)=[],         // Highlight specified ranges in source files
                 hold(HighlightInStatsOnly)=FALSE,// Exclude all nodes that are not highlighted from coverage/statistic
                 Lines=FALSE,          // show each line
                 Summary=FALSE,        // show only summary
                 All=FALSE,            // show all nodes
                 Unused=FALSE,         // show also un-passed nodes
                 hold(UnusedOnly)=FALSE,     // show un-passed nodes only
                 Hidden=FALSE,         // show also hidden nodes
                 hold(SubsumeSubfolders)=FALSE,// show also hidden nodes
                 null()
                );

  // special eval mode - before option check
  if type(STMT) <> "_equal" and contains(allOptions, STMT) = 0 then
    userinfo(1, "Enter eval mode.");
    EVALMODE := shell::tempFilename();
    stdlib::tcov(Reset);
    context(STMT);
    stdlib::tcov(hold(Writel), EVALMODE); // filter out passes in PROG/tcov.mu
    Args := [FAIL, context(args(2..args(0)))];
    first := 2;
  else
    EVALMODE := FALSE;
    Args :=context([args()]);
    first := 1;
  end_if;  

  // get options with error, skip first argument
  OPT:= prog::getOptions(first, Args, allOptions, TRUE,
                         table(Reset               = DOM_BOOL,
                               Append              = Type::Union(DOM_BOOL, DOM_STRING),
                               Write               = DOM_STRING,
                               Info                = Type::Union(DOM_BOOL, DOM_STRING),
                               Export              = DOM_STRING,

                               Annotate            = DOM_BOOL,

                               Comment             = DOM_STRING,
                               Graphical           = DOM_BOOL,
                               hold(GraphicalInAnnotate) = DOM_BOOL,
                               hold(DataFrom)            = DOM_STRING,
                               Path                = DOM_STRING,
                               hold(Exclude)             = DOM_STRING,
                               hold(ListExcludedFiles)   = DOM_BOOL,
                               hold(Filter)              = DOM_STRING,
                               hold(FilterInStatsOnly)   = DOM_BOOL,
                               hold(Highlight)           = Type::ListOf(DOM_LIST),
                               hold(HighlightInStatsOnly)= DOM_BOOL,
                               Lines               = DOM_BOOL,
                               Summary             = DOM_BOOL,
                               All                 = DOM_BOOL,
                               Unused              = DOM_BOOL,
                               hold(UnusedOnly)          = DOM_BOOL,
                               Hidden              = DOM_BOOL,
                               hold(SubsumeSubfolders)   = DOM_BOOL,
                               null()
                              ))[1];

  if OPT[Path] <> FAIL and OPT[Path] = "" then
    error("given Path must not be empty")
  end_if;

  if OPT[hold(UnusedOnly)] = TRUE then
     OPT[Unused]:= TRUE;
  end_if;

  if OPT[hold(Highlight)] <> [] then
     if {op(map(map(OPT[hold(Highlight)],op,1),domtype))} <> {DOM_STRING} then // 1st elements must be strings
       error("Invalid format in Highlight=[[file, l1..l2, ...], ...]: string expected.");
     end_if;
     data:= {op(map(map(OPT[hold(Highlight)],op@_index,2..-1),type))};
     if data <> {} then
       if data <> {"_range"} then // rest elements must be ranges
         error("Invalid format in Highlight=[[file, l1..l2, ...], ...]: range expected.");
       end_if;
       if {op(map(map(map(OPT[hold(Highlight)],op@_index,2..-1),op),testtype,Type::PosInt))} <> {TRUE} then // of positive integers with
         error("Invalid format in Highlight=[[file, l1..l2, ...], ...]: positive integers expected.");
       end_if;
       if {op(map(map(OPT[hold(Highlight)],op@_index,2..-1),(o)->bool(op(o,1)<=op(o,2))))} <> {TRUE} then // l1 <= l2
         error("Invalid format in Highlight=[[file, l1..l2, ...], ...]: l1 <= l2 expected.");
       end_if;  
     end_if;
  end_if;

  // remove library path from name
  removeLibpath:=
    proc(filename)
      local p;
    begin
      for p in [LIBPATH].[PACKAGEPATH] do
        // remove absolute path from name, mask \\ for windows
        if strmatch(filename, "^".stringlib::subs(p, "\\" = "\\\\")) then
          filename := filename[length(p)+1..-1];
          break
        end_if
      end_for;
      if filename[1] = stdlib::PathSep then
        filename[2..-1]
      else
        filename
      end_if
    end_proc;
  
  // return LIB
  libName:=
    proc(filename)
      local p;
    begin
      filename := removeLibpath(filename);
      filename := stringlib::subs(filename, stdlib::PathSep = "/");
      p := stringlib::split(filename, "/");
      if nops(p) > 1 then
        if OPT[hold(SubsumeSubfolders)] then 
          p[1];                                              // baseName includes subfolders
        else
          _concat(op(map(p[1..-2], _exprseq, "/")))[1..-2];  // libName  includes subfolders
        end_if;
      else
        "/";
      end_if;
    end_proc;
  
  // return LIB
  baseName:=
    proc(filename)
      local p;
    begin
      filename := removeLibpath(filename);
      filename := stringlib::subs(filename, stdlib::PathSep = "/");
      p := stringlib::split(filename, "/");
      if nops(p) > 1 then
        if OPT[hold(SubsumeSubfolders)] then 
          _concat(op(map(p[2..-2], _exprseq, "/")), p[-1]);  // baseName includes subfolders
        else
          p[-1];                                             // libName  includes subfolders
        end_if;
      else
        filename;
      end_if;
    end_proc;
  
  // change given filename
  chName:=
    proc(X, ch = null(), appendPath = TRUE)
    begin
      // change all windows paths to unix pathes
      X := stringlib::subs(X, stdlib::PathSep = "/");
      (if (appendPath
           or
           stdlib::PathSep = "\\" and strmatch(X, "^.:") = TRUE
           or
           strmatch(X, "^/") = TRUE) and OPT[Path] <> FAIL then
         X := stringlib::subs(X, ":" = ""); // remove volume sep
         stringlib::subs(OPT[Path], stdlib::PathSep = "/") . (if OPT[Path][-1] <> "/" then "/" else "" end_if)
       else
         ""
       end_if)
      . (if type(ch) = "_equal" then
           if strmatch(X, op(ch, 1)) = TRUE then
             stringlib::subs_regex(X, ch)
           else
             if strmatch(X, "^/") = TRUE then
               X := X[2..-1]
             end_if;
             X.op(ch, 2)
           end_if
         else
           X
         end_if)
    end_proc;

  mkdirRec:=
    proc(filename)
      local n, p;
    begin
      p := stringlib::split(filename, "/")[1..-2];
      for n from 1 to nops(p) do
        if shell::makeDir(Pathname([op(p, 1..n)])) <> TRUE and shell::exist(Pathname([op(p, 1..n)])) <> TRUE then
          warning("cannot make dir ".Pathname([op(p, 1..n)]))
        end_if
      end_for;
      TRUE
    end_proc;

  Pathname:=
    proc(list)
    begin
      // remove : inside path name
      for i from 2 to nops(list) do
        list[i] := stringlib::subs(list[i], ":" = "")
      end_for;
      _concat(op(zip(list, [], _concat, stdlib::PathSep)))
    end_proc:
  
  // return relative path to css file for annotated HTML source files
  path2css:=
    proc(filename = "")
      local n;
    begin
      (if filename <> "" then
         filename := stringlib::subs(filename, "\\" = "/");
         if (n := strmatch(filename[2..-1], "/", Index, All)) <> FALSE then
           _concat("../" $ nops(n))
         else
           ""
         end_if
       else
         ""
       end_if) . stringlib::split(chName(OPT[Export], ".html" = ".css"), "/")[-1] // exclude path
    end_proc;

  writeStylesheet:=
    proc(filename)
      local fd, n;
    begin
      if traperror((fd := fopen(Text, filename, Write))) > 0 or fd = FAIL then
        error("cannot open file '".filename."' for writing")
      end_if;
      for n in prog::tcov_stylesheet do
        fprint(Unquoted, NoNL, fd, n)
      end_for;
      fclose(fd)
    end_proc;
  
  // make graphical index from pass data
  minColor := 0.3;
  graphIndex:=
    proc(data, line = [], stayInWindow=FALSE)
      local maxval, maxnum, f, jsfun, i, class, rowclass, ret;
    begin
      if not OPT[Graphical] then
        return([""])
      end_if;
      rowclass:= "l-idx-row";
      jsfun:= "wo";
      f:= chName(removeLibpath(filename[file]), ".mu$" = "", FALSE);
      if stayInWindow then
        rowclass:= "l-idx-row-big";
        jsfun:= "go";
        f:= baseName(f);
      end_if;
      
      maxval := max(op(data), 1);
      maxnum := nops(data);
      ret := [];
      // normalize data
      //data := map(data, _divide, maxval);
      ret := ret . ["\n",
                    "<TABLE class=\"l-idx-tab\" cellspacing=\"0\" cellpadding=\"0\">", // important!
                    "<TR class=\"".rowclass."\"><!--#GI#-->", "\n"]; // mark each line for stripping!
      
      for i from 1 to maxnum do
        if line <> [] and OPT[hold(HighlightInStatsOnly)] <> TRUE and OPT[hold(Highlight)] <> [] and highlightedSourceCode(filename[file],line[i][1],TRUE)<>"" then
           class:= "b-h";
        else
           class:= val2class(data[i], "b-");
        end_if;
        ret := ret . ["<TD class=\"", class, "\"",
                      " title=\"",
                      (if line <> [] then "line:".line[i][1]."  col:".line[i][2]."  " else null() end_if),
                      "passes:",
                      data[i], "\"",
                      (if line <> [] and OPT[Annotate] then
                         " onClick=\"".jsfun."('", f, "','", line[i][1], "');\""
                       else
                         null()
                       end_if),
                      "></TD><!--#GI#-->", "\n"];
      end_for;

      ret.["</TR></TABLE>", "<!--#GI#-->", "\n"]//.["<SPAN style=\"font-size:8pt;\">".expr2text(data)."</SPAN>"]
    end_proc;

  // color depending on number of passes
  val2class :=
    proc(val, class = "")
    begin
      class.(if val = 0 then
               0
             elif val < 3 then // andi: changed from 5 to 3
               1
             elif val < 10 then
               2
             elif val < 100 then
               3
             elif val < 1000 then
               4
             else
               5
             end_if)
    end_proc;

  // create a HTML coverage index from given value
  val2index :=
    proc(val, class = "", coveragetype="")
      local rval;
    begin
      // round to 5
      rval := floor(val*2/10)*10/2;
      if coveragetype <> "" then
        coveragetype:= " title=\"".coveragetype." coverage\"";
      end_if;
      ["<div ".coveragetype."class=\"cb", class, "\"><div class=\"cb", rval, "\">", floor(val), "%</div></div>"]
    end_proc;

  // mask html entities
  str2html :=
    proc(str)
    begin
      str := stringlib::subs_regex(str, "&" = "&amp;");
      str := stringlib::subs_regex(str, "<" = "&lt;");
      str := stringlib::subs_regex(str, ">" = "&gt;");
      str
    end_proc;
    
  // highlightedSourceCode
  // line =-1: returns all ranges of the given file in a string.
  // line = 0: returns class ...-fil if current file matches a 'Highlight' file
  // line > 0: returns class ...-src-high... if current line also matches a corresponding 'Highlight' range.
  highlightedSourceCode := 
    proc(file,line=0,EmptyIfNoMatch=FALSE)
      local h, r;
    begin
      // look for file match
      // Format: Highlight=[[file, line1..line2, ...], ...]
      for h in OPT[hold(Highlight)] do
        if strmatch(file, h[1]) then
          // request for all corresponding ranges in a string?
          if line = -1 then
            return(expr2text(op(h[2..-1])));
          end_if;
          // request for file match only?
          if line = 0 then
            return(" class=\"l-highlighted-fil\" title=\"has highlighted lines\"");
          end_if;
          // check ranges of matched file
          for r in h[2..-1] do
            if line < op(r,1) or line > op(r,2) then
              next;
            end_if;
            if line = op(r,1) and line = op(r,2) then
              return(" class=\"l-src-high-one\" title=\"highlighted lines\"");
            end_if;
            if line = op(r,1) then
              return(" class=\"l-src-high-top\" title=\"highlighted lines\"");
            end_if;
            if line = op(r,2) then
              return(" class=\"l-src-high-bot\" title=\"highlighted lines\"");
            end_if;
            return(" class=\"l-src-high-mid\" title=\"highlighted lines\"");
          end_for;
          // file matched but no ranges means
          if h[2..-1] = [] then                      // if no range is given
            if line = 1 then
              return(" class=\"l-src-high-top\" title=\"highlighted lines\""); // highlight first line
            end_if;
            return(" class=\"l-src-high-mid\" title=\"highlighted lines\"");   // highlight all lines
          end_if;
          if EmptyIfNoMatch then
            return("");
          end_if;
          return(" class=\"l-src\""); // no line match: do not highlight this line
        end_if;
      end_for;
      return("");
    end_proc;

  /////////////////////////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////////////////////////
  // M O D U S 
  /////////////////////////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////////////////////////
  if OPT[Reset] then
    stdlib::tcov(Reset)
  elif OPT[Append] <> FAIL then
    stdlib::tcov(Append, OPT[Append]);
  elif OPT[Write] <> "" then
    if EVALMODE <> FALSE then
      // copy file, because of filter out prog::tcov
      shell::rename(EVALMODE, OPT[Write]);
    else
      stdlib::tcov(Write, OPT[Write]);
    end_if
  elif OPT[Info] <> FAIL or OPT[Export] <> "" or OPT[Annotate] then

    /////////////////////////////////////////////////////////////////////////////
    // write out and read in debug node pass information
    /////////////////////////////////////////////////////////////////////////////
    if EVALMODE <> FALSE then
      finame := EVALMODE;
      userinfo(2, "tcov log file '".finame."' opened from evalation.");
    elif OPT[hold(DataFrom)] <> "" then
      finame := OPT[hold(DataFrom)];
      userinfo(5, "get tcov data from file '".finame."'.");
    else
      // first write tcov info to temporary file
      finame := shell::tempFilename();
      stdlib::tcov(Write, finame);
      userinfo(5, "tcov log file '".finame."' written.");
    end_if;
    
    // now read in info and store it in table
    if (fd := fopen(finame)) = FAIL then
      error("cannot open temporary file '".finame."' for reading")
    end_if;
  
    filename := table(); // contains entry 'FileID = filename' for each file
    passes   := table(); 
    filedata := table();
    procedure:= table();
    endproc  := table();
    cnt      := table(); // contains a table for each file, each table contains one entry 'line = passes' for each passed line
    library  := table();

    // read tcov data file: FILE NAME SECTION 
    max_file := -1; // indexed from 0
    while finput(fd, file, i) <> null() and file <> -1 do
      max_file    := max_file + 1;
      filename[i] := file;
      passes[i]   := 0;
      nodes[i]    := 0;
      hidden[i]   := 0;
      unused[i]   := 0;
      procedure[i]:= [];
      endproc[i]  := table();
    end_while;

    if max_file < 0 then // no files read
      fprint(Unquoted, 0, "Info: No nodes passed [stdlib::tcov].");
      return();
    end_if;
    
    // read tcov data file: DATA SECTION 
    l := 0; // count lines for the progress report
    ExcludeTab:= table();
    while finput(fd, file, line, c, h, p, fl, fc) <> null() do
      // file = fileid
      // line = line
      // c    = column
      // h    = hidden: hidden=bit-1, end_proc=bit-2
      // p    = passes
      // fl   = first line   of the proc this line belongs to
      // fc   = first column of the proc this line belongs to
      
      if EVALMODE <> FALSE then // should not happen
        // filter out prog::tcov in eval mode (currently needed)
        if strmatch(filename[file], "PROG.tcov.mu$") then
          next;
        end_if;
      end_if;
     
      // Filter out all files which must not be considered in the test coverage.
      // These are MuPAD test files with the suffix '.tst'.
      if strmatch(filename[file], ".tst$") then
        // Debugging
        //if not contains(ExcludeTab, filename[file]) then
        //  print(Unquoted, "\nExcluded file: '".filename[file]."'\n");
        //end_if;
        //ExcludeTab[filename[file]]:= TRUE;
        next;
      end_if;
      
      // Exclude files from test coverage as specified by the user.
      if OPT[hold(Exclude)] <> "" and strmatch(filename[file], OPT[hold(Exclude)]) then
        // Debugging
        //if not contains(ExcludeTab, filename[file]) then
        //  print(Unquoted, "\nExcluded file: '".filename[file]."'\n");
        //end_if;
        ExcludeTab[filename[file]]:= "Fully";
        next;
      end_if;

      // Exclude files from test coverage that does not match the Filter (specified by the user).
      if OPT[hold(FilterInStatsOnly)] = TRUE and OPT[hold(Filter)] <> "" and not strmatch(filename[file], OPT[hold(Filter)]) then
        ExcludeTab[filename[file]]:= "Fully";
        next;
      end_if;

      // Exclude files from test coverage that does not match the Filter (specified by the user).
      if OPT[hold(HighlightInStatsOnly)] = TRUE and contains(OPT, hold(Highlight)) and highlightedSourceCode(filename[file],line,TRUE)="" then
        ExcludeTab[filename[file]]:= "Partially?"; // may be partially excluded!
        next;
      end_if;

      // progress report on screen
      l := l + 1;
      if l = 1000 then
        fprint(Unquoted, NoNL, 0, "Info: Status file progress (per 1000 lines) ..");
      elif l mod 1000 = 0 then
        fprint(Unquoted, NoNL, 0, ".");
      end_if;
      
      // init cnt entry when file appears for the first time
      if not contains(cnt, file) then
        cnt[file] := table()
      end_if;
      
      // skip 'end_proc' nodes: these are used in the debugger to stop at the end
      // of a proc before leaving it. They are of no use for the statistics. They
      // have to be removed!!!
      if h mod 4 >= 2 then // bit-2 is set => end_proc
        if contains(endproc[file], line) then
          endproc[file][line] := endproc[file][line] . [c]
        else
          endproc[file][line] := [c]
        end_if;
        next; // skip!
      end_if;
      
      // all other debug nodes (NO end_proc nodes)
      // init cnt->line entry when line appears for the first time
      if not contains(cnt[file], line) then
        cnt[file][line] = table()
      end_if;
      
      cnt[file][line][c] := [h, p, fl, fc]; // keep hidden/passes/first node in procedure?
      nodes[file] := nodes[file] + 1;
      
      if h = 0 or OPT[Hidden] = TRUE or OPT[All] = TRUE or OPT[Export] <> "" then
        passes[file] := passes[file] + p
      end_if;
      if h = 1 then
        hidden[file] := hidden[file] + 1
      end_if;
      if p = 0 then
        unused[file] := unused[file] + 1
      end_if;
      
      if fl > 0 then // procedure start resp. first debug node
        procedure[file] := procedure[file] . [[fl, fc, h, p, line, c]]; // save passes
      end_if;
    end_while; // read tcov data file: DATA SECTION
    
    if l > 1000 then
      fprint(Unquoted, 0, "done."); // newline after progress dots
    end_if;
    
    fclose(fd);
    userinfo(5, "Tcov log file '".finame."' read (".expr2text(l)." lines).");

    /////////////////////////////////////////////////////////////////////////////
    // S T A T I S T I C
    /////////////////////////////////////////////////////////////////////////////
    if OPT[Info] <> FAIL or OPT[Export] <> "" then

      // info for test coverage
      // - what files were read
      // - usage index for each file
      // - (Long) passed lines

      //
      for file in map([op(cnt)], op, 1) do
        if contains(cnt, file) and cnt[file] <> table() then
          data := [];
          lines := [];
          max_pass := 0;
          if passes[file] > 0 or OPT[Export] <> "" then
            if OPT[hold(Filter)] = "" or strmatch(filename[file], OPT[hold(Filter)]) then
              for line in sort([op(map([op(cnt[file])], op, 1))]) do
                for c in sort([op(map([op(cnt[file][line])], op, 1))]) do
                  if OPT[Export] <> FAIL or
                     ((cnt[file][line][c][1] = 0 or OPT[Hidden] = TRUE) and (cnt[file][line][c][2] > 0 or OPT[Unused] = TRUE)) or
                     (OPT[All] = TRUE) then
                    if max_pass < cnt[file][line][c][2] then
                      max_pass := cnt[file][line][c][2]
                    end_if;
                    data := data . [cnt[file][line][c][2]];
                    lines := lines . [[line, c]];
                  end_if
                end_for
              end_for
            end_if
          end_if;
          filedata[file] := data; // keep data
          fileline[file] := lines;
          if not contains(library, libName(filename[file])) then
            library[libName(filename[file])] := table();
          end_if;
          library[libName(filename[file])][file] := baseName(filename[file]);
        end_if
      end_for;
      userinfo(2, "File data collected.");

      //for file in map([op(cnt)], op, 1) do
      //end_for;
  
      sort_libs := sort(map([op(library)], op, 1)); // libraries
      sort_files := table();
      for lib in sort_libs do
        sort_files[lib] := sort(map([op(library[lib])], op, 1), (X, Y) -> sysorder(library[lib][X], library[lib][Y]))
      end_for;

      /////////////////////////////////////////////////////////////////////////////
      // H T M L - E X P O R T
      /////////////////////////////////////////////////////////////////////////////
      if OPT[Export] <> "" then

        // HTML export
        mkdirRec(chName(OPT[Export]));
        if traperror((fd := fopen(Text, chName(OPT[Export]), Write))) > 0 or fd = FAIL then
          error("cannot open file ".chName(OPT[Export])." for writing")
        end_if;

        allNodes    := _plus(op(map([op(nodes)], op, 2)));
        unusedNodes := _plus(op(map([op(unused)], op, 2)));
        allProcs    := _plus(op(map(map([op(procedure)], op, 2), nops)));
        unusedProcs := nops(select(map(map([op(procedure)], op, 2), op), X -> X[4] = 0));
        
        if allProcs = 0 then useProcs:= 1.0;
        else                 useProcs:= float((allProcs - unusedProcs)/allProcs); end_if;
        if allNodes = 0 then useIndex:= 1.0;
        else                 useIndex:= float((allNodes - unusedNodes)/allNodes); end_if;

        fprint(Unquoted, NoNL, 0, "Info: Export file '".chName(OPT[Export])."' progress (per examined libraries) .");
        
        // MAIN: index page top
        fprint(Unquoted, fd,
               "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\" \"http://www.w3.org/TR/html4/strict.dtd\">", "\n",
               "<HTML>", "\n",
               "<HEAD>","\n",
               "  <TITLE>MuPAD Library Test Coverage Report</TITLE>", "\n",
               "  <link rel=\"stylesheet\" type=\"text/css\" href=\"".path2css()."\">", "\n",
               op(prog::tcov_javascript),
               "</HEAD>", "\n",
               "\n",
               "<BODY class=\"tcov-summary\">", "\n",
               "\n",
               "<H1>MuPAD Library Test Coverage Report</H1>", "\n",
               "\n",
               "<H2><DIV class=\"tcov-summary-comment\">", OPT[Comment], "</DIV></H2>\n",
               "\n",
               "<TABLE class=\"tcov-summary-table\">\n",
               "  <TR><TD>Folders: </TD> <TD>",
               nops(library),
               "</TD></TR>\n",
               "  <TR><TD>Files:</TD> <TD>",
               nops(map(map([op(sort_files)], op, 2), op)),
               (if nops(ExcludeTab) > 0 or OPT[hold(ListExcludedFiles)] = TRUE then " (", nops(ExcludeTab), " excluded)" else "" end_if),
               "</TD></TR>\n",
               "  <TR><TD>Procs:</TD> <TD>",
                  nops(select(map(map([op(procedure)], op, 2), op), X -> X[4] > 0)), // passed procedures
                  " / ",
                  _plus(op(map(map([op(procedure)], op, 2), nops))), // all procedures
               "<SPAN ", "title=\"", "procedures with option noDebug", "\"> (",
               nops(select(map(map([op(procedure)], op, 2), op), X -> X[3] <> 0)), " hidden)</SPAN>",
               "</TD></TR>", "\n",
               "  <TR><TD>Nodes:</TD> <TD>", 
                  (_plus(op(map([op(nodes)], op, 2))) - _plus(op(map([op(unused)], op, 2)))), 
                  " / ",
                  _plus(op(map([op(nodes)], op, 2))), " (", _plus(op(map([op(hidden)], op, 2))), " hidden)", "</TD></TR>", "\n",
               "  <TR><TD>Proc Coverage:</TD><TD>", op(val2index(useProcs*100, "-proc")), "</TD></TR>", "\n",
               "  <TR><TD>Node Coverage:</TD><TD>", op(val2index(useIndex*100, "")), "</TD></TR>", "\n",
               "</TABLE>", "\n",
               "\n",
               "<H2>Folders and files:</H2>",
               "\n");

        fprint(Unquoted, fd, "<P class=\"tcov-summary-libraries\">\n");

        if OPT[hold(ListExcludedFiles)] = TRUE then
          fprint(Unquoted, fd,
               "\n",
               "<TABLE class=\"tcov-summary-library\"><TR>\n",
               "<TD class=\"tcov-summary-col1\">\n",
               "<A class='tcov-folder-toggle' href='javascript:toggle(\"ExcludedFiles\")'>", // for folder toggle
               "<SPAN class=\"tcov-summary-lib\">Files excluded from statistic:</SPAN>",
               "</TD>\n",
               "<TD class=\"tcov-summary-col2\">\n",
               "<TABLE class=\"tcov-summary-table-small\">\n",
               "  <TR><TD>Files:</TD> <TD>",
               "<SPAN title=\"Number of files excluded from the statistic.\">", nops(ExcludeTab), "</SPAN>", "</TD></TR>", "\n",
               "</TABLE>\n",
               "</TD\n",
               "</TR></TABLE>\n");
          fprint(Unquoted, fd, 
               "<DIV id=\"ExcludedFiles\" style=\"display:none;\" class=\"tcov-summary-files\">",
               "<TABLE class=\"tcov-summary-files-tab\">\n"); // for folder toggle
          for lib in sort(lhs(ExcludeTab)) do
               //exRa:= ExcludeTab[lib];
               exRa:= highlightedSourceCode(lib,-1);
               if exRa = "" then
                 exRa:= "Fully";
               else
                 exRa:= "Partially, not line ".exRa;
               end_if;
               fprint(Unquoted, fd,"  <TR class=\"tcov-summary-row\"><TD>".lib."</TD><TD> : </TD><TD>".exRa."&nbsp;</TD></TR>\n");
          end_for;
          fprint(Unquoted, fd, "</TABLE></DIV>\n"); // for folder toggle
        end_if;
        for lib in sort_libs do
          fprint(Unquoted, NoNL, 0, ".");
          if OPT[hold(Filter)] <> "" and select(map(sort_files[lib],o->filename[o]),strmatch,OPT[hold(Filter)])=[] then
            //print("Skipped statistics for library '".lib."'");
            next;
          end_if;

          // compute useIndex and useProcs of library
          allNodes := 0;
          unusedNodes := 0;
          allProcs := 0;
          unusedProcs := 0;
          hiddenProcs := 0;
          for file in sort_files[lib] do
            allNodes    := allNodes    + _plus(op(nodes[file]));
            unusedNodes := unusedNodes + _plus(op(unused[file]));
            allProcs    := allProcs    + nops(procedure[file]);
            unusedProcs := unusedProcs + nops(select(procedure[file], X -> X[4] = 0));
            hiddenProcs := hiddenProcs + nops(select(procedure[file], X -> X[3] <> 0));
          end_for;
         
          if allProcs = 0 then useProcs:= 1.0;
          else                 useProcs:= float((allProcs - unusedProcs)/allProcs); end_if;
          if allNodes = 0 then useIndex:= 1.0;
          else                 useIndex:= float((allNodes - unusedNodes)/allNodes); end_if;

          // MAIN index library entries
          fprint(Unquoted, fd,
                 "\n",
                 "<TABLE class=\"tcov-summary-library\"><TR>\n",
                 "<TD class=\"tcov-summary-col1\">\n",
                 "<A class='tcov-folder-toggle' href='javascript:toggle(\"".(if lib = "" then "tmp" elif lib = "/" then "Root" else lib end_if)."\")'>", // for folder toggle
                 "<SPAN class=\"tcov-summary-lib\">",
                 (if lib = "" then "Temporary files" elif lib = "/" then "&nbsp; . &nbsp;" else "", stringlib::subs(lib, "/"="<BR>/") end_if),
                 "</SPAN></A>\n",
                 "</TD>\n",
                 "<TD class=\"tcov-summary-col2\">\n",
                 "<TABLE class=\"tcov-summary-table-small\">", "\n",
                 "  <TR><TD>Files:</TD> <TD>",
                 "<SPAN title=\"number of files in this library\">", nops(sort_files[lib]), "</SPAN>", "</TD></TR>", "\n",
                 "  <TR><TD>Procs:</TD> <TD>",
                 "<SPAN title=\"used procedures\">", allProcs - unusedProcs, "</SPAN> / <SPAN title=\"all procedures\">", allProcs, "</SPAN>",
                 "<SPAN title=\"procedures with option noDebug\"> (", hiddenProcs, " hidden)</SPAN>", "</TD></TR>", "\n",
                 "  <TR><TD>Nodes:</TD><TD>",
                 "<SPAN title=\"passed debug nodes\">", allNodes-unusedNodes, "</SPAN> / <SPAN title=\"all debug nodes\">", allNodes, "</SPAN>",
                 "<SPAN title=\"debug nodes in procedures with option noDebug\"> (", _plus(op(hidden[file])), " hidden)</SPAN>", "\n",
                 "  <TR><TD>Proc Coverage:</TD><TD>", op(val2index(useProcs*100, "-proc")), "</TD></TR>", "\n",
                 "  <TR><TD>Node Coverage:</TD><TD>", op(val2index(useIndex*100, "")), "</TD></TR>", "\n",
                 "</TABLE>\n",
                 "</TD\n",
                 "</TR></TABLE>\n");

          fprint(Unquoted, fd, 
                 "<DIV id=\"",
                 (if lib = "" then "tmp" elif lib = "/" then "Root" else lib end_if),
                 "\" style=\"display:none;\" class=\"tcov-summary-files\">",
                 "<TABLE class=\"tcov-summary-files-tab\">\n"); // for folder toggle

          for file in sort_files[lib] do
            if OPT[hold(Filter)] <> "" and not strmatch(filename[file], OPT[hold(Filter)]) then
              next;
            end_if;
            allNodes    := _plus(op(nodes[file]));
            unusedNodes := _plus(op(unused[file]));
            allProcs    := nops(procedure[file]);
            unusedProcs := nops(select(procedure[file], X -> X[4] = 0));
            hiddenProcs := nops(select(procedure[file], X -> X[3] <> 0));

            if allProcs = 0 then useProcs:= 1.0;
            else                 useProcs:= float((allProcs - unusedProcs)/allProcs); end_if;
            if allNodes = 0 then useIndex:= 1.0;
            else                 useIndex:= float((allNodes - unusedNodes)/allNodes); end_if;

            fprint(Unquoted, fd,
                   "\n",
                   "<TR class=\"tcov-summary-row\">\n",
                   "<TD class=\"tcov-summary-col1\">",
                   "<SPAN".highlightedSourceCode(filename[file]).">",
                   "<A class=\"tcov-summary-source-ref\" ".(if OPT[Annotate] then "h" else "" end)."ref=\"", 
                   chName(removeLibpath(filename[file]), ".mu$" = ".tcov.html", FALSE), "\" target=\"_blank\">",
                   baseName(filename[file]), "</A>",
                   "</SPAN>\n",
                   "</TD>\n",
                   "<TD class=\"tcov-summary-col2\">",
                   op(val2index(useProcs*100, "-small", "Proc")),
                   op(val2index(useIndex*100, "-small", "Node")),
                   "<DIV class=\"tcov-summary-file\">",
                   "<SPAN title=\"passed / all - procedures\">Procs: ", allProcs - unusedProcs,
                   "/", allProcs, "</SPAN>",
                   " (<SPAN title=\"procedures with option noDebug\">", hiddenProcs, " hidden</SPAN>)",
                   " <SPAN title=\"passed / all - debug nodes\">Nodes: ", allNodes - unusedNodes, "/", allNodes,
                   "</SPAN></DIV>\n",
                   op(graphIndex(filedata[file], fileline[file])),
                   "</TD>\n",
                   "</TR>\n");
          end_for;

          fprint(Unquoted, fd, "</TABLE></DIV>\n"); // for folder toggle

        end_for;
        
        fprint(Unquoted, fd,
               "<DIV class=\"tcov-summary-library\"></DIV>", "\n",
               "</P>", "\n",

               "<H2>End of Report</H2>", "\n",
               
               "</BODY>", "\n",
               "</HTML>");
        
        fclose(fd);
        writeStylesheet(chName(OPT[Export], ".html" = ".css"));
        fprint(Unquoted, 0, "done.");
        
      /////////////////////////////////////////////////////////////////////////////
      // S C R E E N - O U T P U T
      /////////////////////////////////////////////////////////////////////////////
      else
        useIndex := float((_plus(op(map([op(nodes)], op, 2))) - _plus(op(map([op(unused)], op, 2))))/_plus(op(map([op(nodes)], op, 2))));
        useProcs := float(nops(select(map(map([op(procedure)], op, 2), op), X -> X[4] > 0)) / _plus(op(map(map([op(procedure)], op, 2), nops))));
        
        fprint(Unquoted, 0, "\nSUMMARY");
        fprint(Unquoted, 0, "  Folders : ", nops(library));
        fprint(Unquoted, 0, "  Files   : ", nops(map(map([op(sort_files)], op, 2), op)));
        fprint(Unquoted, 0, "  Procs   : ", nops(select(map(map([op(procedure)], op, 2), op), X -> X[4] > 0)), "/", _plus(op(map(map([op(procedure)], op, 2), nops))), " (", nops(select(map(map([op(procedure)], op, 2), op), X -> X[3] <> 0)), " hidden)");
        fprint(Unquoted, 0, "  Nodes   : ", (_plus(op(map([op(nodes)], op, 2))) - _plus(op(map([op(unused)], op, 2)))), "/", _plus(op(map([op(nodes)], op, 2))), " (", _plus(op(map([op(hidden)], op, 2))), " hidden)");
        
        fprint(Unquoted, 0, "  Proc Coverage: ", (if _plus(op(map(map([op(procedure)], op, 2), nops))) > 0 then floor(useProcs*100) else "0" end_if), "%");
        fprint(Unquoted, 0, "  Node Coverage: ", (if _plus(op(map([op(nodes)], op, 2))) > 0                then floor(useIndex*100) else "0" end_if), "%");
        
        //fprint(Unquoted, 0, "  Passes  : ", _plus(op(map([op(passes)], op, 2))), " (~ ", stringlib::formatf(float(_plus(op(map([op(passes)], op, 2)))/(_plus(op(map([op(nodes)], op, 2))) /*- _plus(op(map([op(unused)], op, 2)))*/)), 2), " passes per all nodes)");

        fprint(Unquoted, 0, "");

        if OPT[hold(Summary)] = TRUE then
          return()
        end_if;
        
        for lib in sort_libs do
          if OPT[hold(Filter)] <> "" and select(map(sort_files[lib],o->filename[o]),strmatch,OPT[hold(Filter)])=[] then
            //print("Skipped statistics for library '".lib."'");
            next;
          end_if;
          
          // compute useIndex of library
          nd := 0;
          ud := 0;
          for file in sort_files[lib] do
            nd := nd + _plus(op(nodes[file]));
            ud := ud + _plus(op(unused[file]));
          end_for;
          
          useIndex := float((nd - ud)/nd);

          if useIndex > 0.0 or OPT[Unused] = TRUE then
            
            fprint(Unquoted, 0, "Folder: ", (if lib = "" then "/" else lib end_if), "  Node Coverage: ", floor(useIndex*100), "%");
            
            for file in sort_files[lib] do
              
              useIndex := float((_plus(op(nodes[file])) - _plus(op(unused[file]))) / _plus(op(nodes[file])));
              
              if contains(cnt, file) and cnt[file] <> table() then
                if passes[file] > 0 then
                  if OPT[hold(Filter)] = "" or strmatch(filename[file], OPT[hold(Filter)]) then
                    fprint(Unquoted, 0, "  File: ", filename[file], "   Coverage: ", floor(useIndex*100), "%");
                    fprint(Unquoted, 0, "    Procs: ", nops(procedure[file]) - nops(select(procedure[file], X -> X[4] = 0)),
                           "/" , nops(procedure[file]), " (", nops(select(procedure[file], X -> X[4] = 0)), " hidden)");
                    fprint(Unquoted, 0, "    Nodes: ", nodes[file] - unused[file], "/" , nodes[file], " (", hidden[file], " hidden)" /*, "   Passes: ", passes[file]*/);
                    l := 0;
                    for line in sort([op(map([op(cnt[file])], op, 1))]) do
                      for c in sort([op(map([op(cnt[file][line])], op, 1))]) do
                        if (cnt[file][line][c][1] = 0 or OPT[Hidden] = TRUE) and 
                           ((cnt[file][line][c][2] > 0 and not OPT[hold(UnusedOnly)] = TRUE) or 
                            (cnt[file][line][c][2] = 0 and OPT[Unused] = TRUE)) or
                           OPT[All] = TRUE then
                          l := 1;
                          if OPT[hold(Lines)] = TRUE then
                            fprint(Unquoted, 0,
                                   "      Line", stringlib::format(expr2text(line).",".expr2text(c), 10, hold(Right)),
                                   ":", stringlib::format(expr2text(cnt[file][line][c][2]), 6, hold(Right)),
                                   " pass", _if(cnt[file][line][c][2] <> 1, "es", "  "),
                                   _if(cnt[file][line][c][1] > 0, " (hidden)", ""))
                          end_if
                        end_if
                      end_for
                    end_for;
                    if l = 0 then
                      fprint(Unquoted, 0, "      no lines passed")
                    end_if
                  end_if
                end_if
              end_if
            end_for
          end_if
        end_for;
      end_if;
    end_if;

    /////////////////////////////////////////////////////////////////////////////
    // A N N O T A T E
    /////////////////////////////////////////////////////////////////////////////
    
    if OPT[Annotate] then

      CLENGTH := 4;
      PLENGTH := 6;

      // open source files and files for output
      fprint(Unquoted, NoNL, 0, "Info: Export annotated files progress (per 10 files) .");
      fl := 0;
      for file in map([op(cnt)], op, 1) do
        fl := fl + 1;
        if fl mod 10 = 0 then
          fprint(Unquoted, NoNL, 0, ".")
        end_if;
        if OPT[hold(Filter)] <> "" and not strmatch(filename[file], OPT[hold(Filter)]) then
          //print("Skipped annotation for file '".filename[file]."'");
          next;
        end_if;
        // any line used??
        if passes[file] > 0 or OPT[All] = TRUE or OPT[Unused] = TRUE or OPT[Export] <> "" then // any lines used
          CLENGTH := length(nops(cnt[file])) + 1;
          PLENGTH := length(min(999999, max(passes[file])));
          
          if traperror((fd:= fopen(filename[file]))) <> 0 or fd = FAIL then
            warning("cannot open source file '".filename[file]."'");
            next
          end_if;
          //
          fname := chName(removeLibpath(filename[file]),
                          ".mu$" = (if OPT[Export] <> "" then ".tcov.html" else ".tcov" end_if));
          bname := baseName(chName(removeLibpath(filename[file]), ".mu$" = "", FALSE));
          if mkdirRec(fname) <> TRUE or
             traperror((OFILE := fopen(Text, fname, Write))) <> 0
            or OFILE = FAIL then
            warning("cannot open file '".fname."' for writing");
            next
          end_if;
          l:= 0;
          if OPT[Export] <> "" then

            allNodes    := _plus(op(nodes[file]));
            unusedNodes := _plus(op(unused[file]));
            allProcs    := nops(procedure[file]);
            unusedProcs := nops(select(procedure[file], X -> X[4] = 0));
            hiddenProcs := nops(select(procedure[file], X -> X[3] <> 0));

            if allProcs = 0 then useProcs:= 1.0;
            else                 useProcs:= float((allProcs - unusedProcs)/allProcs); end_if;
            if allNodes = 0 then useIndex:= 1.0;
            else                 useIndex:= float((allNodes - unusedNodes)/allNodes); end_if;

            fprint(Unquoted, OFILE,
                   // !! Put this page in strict mode for compatibility with IE !! "white-space:pre"
                   "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\" \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">", "\n",
                   "<HTML>", "\n",

                   "<HEAD>", "\n",
                   "  <TITLE>", filename[file], "</TITLE>", "\n",
                   "  <link rel=\"stylesheet\" type=\"text/css\" href=\"".path2css(removeLibpath(filename[file]))."\">", "\n",
                   op(prog::tcov_javascript),
                   "</HEAD>", "\n",
                   "<BODY class=\"tcov-summary\">", "\n",
                   "<H1>", "Generated by prog::tcov", "</H1>", "\n",
                   "<H2 class=\"tcov-summary-comment\">", OPT[Comment], "</H2>\n",
                   "<TABLE class=\"tcov-summary-table\">", "\n",
                   "  <TR><TD class=\"tcov-filename\" colspan=\"2\">", filename[file], "</TD></TR>", "\n",
                   "  <TR><TD>Procs:</TD> <TD>",
                   "<SPAN title=\"used procedures\">", allProcs - unusedProcs, 
                   "</SPAN> / <SPAN title=\"all procedures\">", allProcs, "</SPAN>",
                   "<SPAN title=\"procedures with option noDebug\"> (", hiddenProcs, " hidden)</SPAN>", "</TD></TR>", "\n",
                   "  <TR><TD>Nodes:</TD><TD>",
                   "<SPAN title=\"passed debug nodes\">", allNodes - unusedNodes, 
                   "</SPAN> / <SPAN title=\"all debug nodes\">", allNodes, "</SPAN>",
                   "<SPAN title=\"debug nodes in procedures with option noDebug\"> (", hiddenProcs, " hidden)</SPAN>", "\n",
                   "  <TR><TD>Proc Coverage:</TD><TD>", op(val2index(useProcs*100, "-proc")), "</TD></TR>", "\n",
                   "  <TR><TD>Node Coverage:</TD><TD>", op(val2index(useIndex*100, "")), "</TD></TR>", "\n",
                   "  <TR><TD>", 
                   "<A class=\"button-hide\" TITLE=\"toggle: hide or show ".(if OPT[hold(Highlight)] <> [] then "non-highlighted" else "passed and non-passable" end)." lines.\" HREF=\"javascript:setclass('src-table','l-table','l-thide');\">+/-</A>",
                   " ",
                   "<A class=\"button-back\" TITLE=\"toggle: back to previously selected line.\" HREF=\"javascript:go(lastf,lastl);\">&lt;&lt;</A>",
                   "</TD><TD>", 
                   (if OPT[hold(GraphicalInAnnotate)] then
                      op(graphIndex(filedata[file], fileline[file], TRUE))
                    else
                      "&nbsp;"
                    end_if),
                   "</TD></TR>", "\n",
                   "</TABLE>\n",
                   
                   "<P>", "\n",
                   "<TABLE id=\"src-table\" cellspacing=\"0\" cellpadding=\"0\" class=\"l-table\">", "\n"
                   );
          else
            fprint(Unquoted, OFILE, "// Generated by prog::tcov session", "\n",
                   "// *** ", OPT[Comment], " ***\n");
          end_if;

          procList := sort(procedure[file], (X,Y) -> X[1] < Y[1]);
          inProc := FALSE;              
          
          while (l:= l + 1; line:= ftextinput(fd)) <> null() do
            if contains(cnt, file) then

              procsInLine := select(procList, X -> X[1] = l);
              c := procsInLine;
              nextProcStart := -1;
              if c <> [] then
                c := c[1];
                nextProcStart := 0;
                if c[5] = l then // same line, save column
                  nextProcStart := c[2];
                end_if
              end_if;

              if contains(cnt[file], l) then // info about this line exists
                // all columns/debug nodes
                c := sort([op(map([op(cnt[file][l])], op, 1))]);
                
                p := 1; // start position

                if OPT[Export] <> "" then
                  if OPT[hold(Highlight)] <> [] then
                     tr_src_class:= td_src_class:= highlightedSourceCode(filename[file],l,TRUE);
                     if tr_src_class="" then tr_src_class:= " class=\"l-hide\"" else tr_src_class:= "" end_if;
                     if td_src_class="" then td_src_class:= " class=\"l-src\""  end_if;
                   else
                     tr_src_class:= (if max(cnt[file][l][dbg][2] $ dbg in c) > 0 then " class=\"l-hide\"" else "" end); // hide when passes > 0 
                     td_src_class:= " class=\"l-src\"";
                  end_if;  
                  fprint(Unquoted, NoNL, OFILE,
                         "<TR".tr_src_class.">",
                         "<TD class=\"l-up\">", (if l mod 20 = 1 then "<A TITLE=\"Up\" HREF=\"#\">&uarr;</A>" else "&nbsp;" end), "</TD>",
                         "<TD onClick=\"ugo('", bname, "','", l, "');\" class=\"l-num", // unhide+go
                         (//if select(procedure[file], X -> X[5] = l) <> [] then
                          if nextProcStart >= 0 then
                            inProc := TRUE;
                            if min(map(procsInLine, X->X[4])) = 0 then
                              "-proc-nouse"
                            else
                              "-proc"
                            end_if;
                          elif contains(endproc[file], l) then
                            inProc := FALSE;
                            "-endproc"
                          elif inProc = TRUE then
                            "-inproc"
                          else
                            ""
                          end_if),
                         "\">", "<A NAME=\"", "l", l, "\"/>", l, ":", "</TD>",
                         //"<TD class=\"l-idx\">", op(graphIndex(map(map([op(cnt[file][l])], op, 2), op, 2), [], 2)), "</TD>",
                         "<TD class=\"l-space\"></TD>",
                         "<TD".td_src_class.">");
                end_if;
                
                for i from 1 to nops(c) do
                  if i > 1 then
                    p := c[i] + 1 // column start for debug node - substring starts from 1!
                  end_if;
                  if OPT[Export] <> "" then

                    if i < nops(c) then // part of line
                      if c[i+1] > length(line) then
                        warning("debug node information points behind the end of the line: ".filename[file].":".expr2text(l).",".expr2text(c[i+1]));
                        if p <= length(line) then 
                          linepart := str2html(substring(line, p..length(line)))
                        else 
                          linepart := ""
                        end_if;  
                      else
                        linepart := str2html(substring(line, p..(c[i+1])))
                      end_if
                    elif p <= length(line) then // whole or rest of line
                      linepart := str2html(substring(line, p..length(line)))
                    else
                      userinfo(2, "debug node at end of the line: ".filename[file].":".expr2text(l).",".expr2text(p-1));
                    end_if;
                    
                    fprint(Unquoted, NoNL, OFILE,
                           "<SPAN class=\"", val2class(cnt[file][l][c[i]][2], "l-"), "\"",
                           " title=\"", cnt[file][l][c[i]][2], " pass(es)", "\"", ">",
                           // proc starts
                           (if nextProcStart = 0 or nextProcStart > 0 and nextProcStart > c[i] then
                              "<SPAN class=\"l-proc-def\">"
                            else
                              ""
                            end_if),
                           linepart,
                           (if nextProcStart = 0 or nextProcStart > 0 and nextProcStart > c[i] then
                              "</SPAN>"
                            else
                              ""
                            end_if),
                           "</SPAN>");
                  else
                    fprint(Unquoted, OFILE,
                           stringlib::format(expr2text(l), CLENGTH, Right), ":",
                           stringlib::format(expr2text(min(cnt[file][l][c[i]][2], max(9999, 10^PLENGTH-1))), PLENGTH, hold(Left)),
                           "",
                           (if i < nops(c) then // part of line
                              (" " $ p),  substring(line, p..c[i+1])
                            else // whole or rest of line
                              (" " $ p),  substring(line, p..(length(line)))
                            end_if));
                  end_if
                end_for;
                
                if OPT[Export] <> "" then
                  fprint(Unquoted, OFILE,
                         "</TD></TR>");
                end_if;              

              else // no info about this line
                if OPT[Export] <> "" then
                  if OPT[hold(Highlight)] <> [] then
                     tr_src_class:= td_src_class:= highlightedSourceCode(filename[file],l,TRUE);
                     if tr_src_class="" then tr_src_class:= " class=\"l-hide\"" else tr_src_class:= "" end_if;
                     if td_src_class="" then td_src_class:= " class=\"l-src\""  end_if;
                  else
                     tr_src_class:= " class=\"l-hide\"";
                     td_src_class:= " class=\"l-src\"";
                  end_if;  
                  fprint(Unquoted, OFILE,
                         "<TR".tr_src_class.">",
                         "<TD class=\"l-up\">", (if l mod 20 = 1 then "<A TITLE=\"Up\" HREF=\"#\">&uarr;</A>" else "&nbsp;" end), "</TD>",
                         "<TD onClick=\"ugo('", bname, "','", l, "');\" class=\"l-num", // unhide+go
                         (if nextProcStart >= 0 then
                            inProc := TRUE;
                            if min(map(procsInLine, X->X[4])) = 0 then
                              "-proc-nouse"
                            else
                              "-proc"
                            end_if;
                          elif contains(endproc[file], l) then
                            inProc := FALSE;
                            "-endproc"
                          elif inProc = TRUE then
                            "-inproc"
                          else
                            ""
                          end_if),
                         "\">", "<A NAME=\"", "l", l, "\"/>", l, ":", "</TD>",
                         //"<TD class=\"l-idx\">", /*op(graphIndex([0], [], 2)),*/ //"</TD>", // no graphical index for unpassed lines
                         "<TD class=\"l-space\"></TD>",
                         "<TD".td_src_class.">",
                         "<SPAN class=\"l-ndn\"", " title=\"no debug nodes\"", ">",
                         // proc starts
                         (if nextProcStart = 0 or nextProcStart > 0 and nextProcStart > c[i] then
                            "<SPAN class=\"l-proc-def\">"
                          else
                            ""
                          end_if),
                         (if line= "" then "&nbsp;" else str2html(line) end_if),
                         // proc starts
                         (if nextProcStart >= 0 then
                            "</SPAN>"
                          else
                            ""
                          end_if),
                         "</SPAN>",
                         "</TD></TR>");
                else
                  fprint(Unquoted, OFILE, stringlib::format(expr2text(l), CLENGTH, Right), ":", (" " $ PLENGTH), line)
                end_if
              end_if
            end_if
          end_while;
          if OPT[Export] <> "" then
            fprint(Unquoted, OFILE,
                   "</TABLE>", "\n",
                   
                   "<H2>End of File</H2>", "\n",
                   
                   "</BODY>", "\n",
                   "</HTML>", "\n");
          end_if;
          fclose(fd);
          fclose(OFILE)
        end_if
      end_for;
      fprint(Unquoted, 0, "done.");
    end_if;
    
    if OPT[hold(DataFrom)] <> "" and OPT[hold(DataFrom)] = finame then
      // do not delete data file!
    else
      stdlib::tcov(hold(Remove), finame); // remove temporary log file
    end_if;
    
  end_if;
  null()
end_proc:
