/* call-seq:
 *  property::gbasis(idents:DOM_SET) -> [basis:Type::ListOf(DOM_POLY), substitutions:Type::ListOf("_equal")]
 *
 * Retrieves the list of assumptions concerning the idents and returns
 * a groebner basis. To prevent umatched algebraic dependencies
 * terms which are not identifiers are substituted by identifiers.
 * Until now algebraic dependencies between these terms are ignored,
 * for example x+sqrt(x) is treated as x+X1.
 *
 * Example:
 *  property::gbasis({x}) assuming x+y=1
 *  -> [[poly(y + x - 1, [y, x])], []]
 *  property::gbasis({x}) assuming x+sqrt(y)=1
 *  -> [[poly(normalGroebner1 + x - 1, [normalGroebner1, x])], [y^(1/2) = normalGroebner1]]
 */
property::gbasis := proc(idents : DOM_SET, maxVariables = 4)
  local eqs, inds, basis, props,
    substitutions;
begin
  /* Get the properties. If we have nothing to do
   * i.e. no identifiers or no assumptions just return the empty list.*/
  if nops(idents)=0 then return([[],[]]); end_if;
  idents := property::_checkSanity(idents, TRUE);
  props := property::_showprops(idents);
  if nops(props)=0 then return([[],[]]); end_if;

  /* Select all appropiate assumptions. These are equalities and
   * element of-relations with one single element (x in {y}). */
  eqs := select(props, X->(_lazy_and(type(X)="_equal", not contains({type(op(X,1)), type(op(X,2))}, piecewise))));
  eqs := eqs.map(select(props, X->type(X)="_in" and type(op(X,2))=DOM_SET and nops(op(X,2))=1),
    X->op(X,1)=op(X,[2,1]));

  // we have to normalize since only Groebner bases of polynomial sets
  // (but not sets of rational functions) can be computed
  eqs :=  (()->(
     // work without assumptions
     unassume(eqs);
    select(map(eqs,
               proc(X)
               begin
                 numer(op(X,1)-op(X,2))
               end_proc
               ), _unequal, undefined);
  ))();

  // roots
  substitutions := [];

  /* Rationalize the equations and add minimal polynomials to it */
  eqs := map(eqs, proc(X)
    local v;
  begin
    v := rationalize(X, FindRelations=["_power"], MinimalPolynomials, Prefix=" normalGroebner");
    substitutions := substitutions.[op(map(v[2], X->(op(X,2)=op(X,1))))];
    v[1], op(v[3]);
  end_proc);
  idents := idents union freeIndets(eqs);

  /* Expand idents with all new identifiers. */
  idents := idents union select(indets(eqs, PolyExpr), X->type(X)=DOM_IDENT or 
    (not testtype(X, Type::Constant) and not contains({"_mult", "_plus"}, type(X)) and (type(X)<>"_power" or type(op(X,2))<>DOM_INT)));
  eqs := select(eqs, testtype, Type::PolyExpr([op(idents)],Type::Rational));

  /* we want  DOM_POLYs that where the varaibles are true DOM_IDENTs (to avoid breaking groebner algorithm) */
  idents := split(idents, testtype, DOM_IDENT);
  // idents is now a list [DOM_SET of DOM_IDENT, DOM_SET of expr, {}]
  assert(idents[3]={});
  substitutions := substitutions.map([op(idents[2])], X -> (X = genident("normalGroebner")));
  // substitutions is now a DOM_LIST of _equal(expr, DOM_IDENT)
  eqs := subs(eqs, substitutions);
  // eqs is now a DOM_LIST of polynomials
  idents  := idents[1] union map({op(substitutions)}, op, 2);
  // idents is now the set of all identifiers in eqs: DOM_SET of DOM_IDENT
  /* sort the indets*/
  inds:= sort([op(idents)]);
  // inds is now the ordered list of all identifiers in eqs: DOM_LIST of DOM_IDENT
  inds := select(inds, X->has(eqs, X));
  eqs := select(map(eqs, poly, inds), _unequal, FAIL);
  // eqs is now a DOM_LIST of DOM_POLY

  /* select only identifiers that reside in eqs and repolynimialize eqs */
  eqs := map(eqs, poly, inds);
  
  if maxVariables>0 and nops(inds)>maxVariables then
    return([eqs, substitutions]);
  end_if;

  if nops(eqs)=0 then
    substitutions:= [];
    basis := [];
  else
    /* substitute anything not being an identifier */
    basis := groebner::gbasis(eqs, DegInvLexOrder);
  end_if;
  [basis, substitutions];
end_proc: