/* rec::polysol(r,u,n,ord) tries to find polynomial solutions to the
   linear recurrence r for u(n) or order ord with polynomial coefficients.
   Reference: Calcul formel: mode d'emploi. Gomez, Salvy, Zimmermann, 1995, Masson,
      pages 175-176.

   Input: r is an equation in u(n), u(n+1),  ..., u(n+ord)
             u,n are indeterminates
             ord is a positive integer
   Output: a set of polynomial solutions, {} if no poly. solution
               or if recurrence not linear with poly. coeff.
   Examples: 
>> rec::polysol(n*u(n+2)-5*u(n+1)-(n+1)*u(n),u,n,2);

                         {    /        2      \ }
                         {    | n   3 n     3 | }
                         { a3 | - - ---- + n  | }
                         {    \ 2    2        / }

>> rec::polysol(r(n+2)-2*r(n+1)+r(n)-2,r,n,2);

                                           2
                             {a0 + n a1 + n }

>> rec::polysol(-n-u(n) + u(n + 1), u, n, 1);

                             {             2 }
                             {   n   a0   n  }
                             { - - + -- + -- }
                             {   2   2    2  }
*/
rec::polysol :=
proc(r,u,n,ord)
  local i,l,d,m,alpha,s,j,inho,t;
begin
  alpha:= genident("alpha"):

  inho:=r;
  for i from 0 to ord do
    if not testtype(inho,Type::PolyExpr(u(n+i))) then
      return({})
    end_if;
    if degree(inho,[u(n+i)])>=2 then
      return({})
    end_if;
    l[i]:=coeff(inho,[u(n+i)],1);
    if not testtype(l[i],Type::PolyExpr(n)) then
      return({})
    end_if;
    d[i]:=degree(l[i],[n]);
    inho:=coeff(inho,[u(n+i)],0);
  end_for;
  if inho <> 0 and not testtype(inho,Type::PolyExpr(n)) then
    return({})
  end_if;
  /* Now r = l[ord] u(n+ord) + ... + l[1] u(n+1) + l[0] u(n) + inho and
     l[ord],...,l[0],inho are polynomials in n */

  /* Compute indicial equation */
  m:=max(d[i] $ i=0..ord)+1;
  repeat
    m:=m-1;
    /* compute the coefficient of n^(alpha+m) if deg(u(n),n)=alpha */
    s:=0;
    for i from 0 to ord do
      /* [n^(alpha+m)] l[i]*(n+i)^alpha */
      for j from max(0,m) to d[i] do
        s:=s+coeff(l[i],[n],j)*binomial(alpha,j-m)*i^(j-m);
      end_for
    end_for;
    s := expand(s);
  until s <> 0 end_repeat;
  /* s is the indicial equation of r, i.e., the degrees of all polynomial
     solutions of the homogeneous equation are roots of s */

  t := -1;
  if has(s, alpha) then
    s := polylib::primpart(s, [alpha]);
    /* we cannot correctly handle parametric or algebraic coefficients */
    if testtype(s, Type::PolyExpr([alpha], Type::Rational)) then
      s := solvelib::iroots(s);

      /* t := maximal degree of a polynomial solution of the homogeneous
         equation */
      t := max(-1, op(s));
    end_if;
  end_if;

  /* t := maximal degree of a polynomial solution of the inhomogenenous
     equation */
  t := max(t, degree(inho, [n]) - m);

  /* compute general solution of the inhomogeneous equation */
  rec::polysol2(t, r, u, n, ord);
end_proc:


/* rec::polysol2(k,r,u,n,ord) returns a set with the general polynomial
   of degree at most k solving r, if it exists, or {} otherwise.
   the output may contain free parameters
 */
rec::polysol2 :=
proc(k,r,u,n,ord)
  local p,a,i,b;
begin
    if k < 0 then
      return({})
    end_if;

    /* p := generic polynomial of degree k with undetermined coeffs
       a[0],..,a[k] */
    a := [genident("C") $ i = 0..k];
    p := _plus(a[i+1]*n^i $ i = 0..k);

    /* plug p into r */
    r := expand(subs(r, u(n+i) = subs(p, n = n+i) $ i = 0..ord));
    if r = 0 then
      return({p})
    end_if;

    /* solve the linear system of equations in the unknowns a[0],...,a[k]
       that is equivalent to r = 0 */
    b := {coeff(r, [n])}, {a[i+1] $ i = 0..k};
    b := linsolve(b);
    /* linsolve makes implicit assumptions about pivots being nonzero,
       so the answer may be wrong for parametric problems */
    if b = FAIL then
      {}
    else
      {subs(p, b, EvalChanges)}
    end_if
end_proc:

