/*
 * series slot of heaviside - computes series(heaviside(f), x, ord, dir)
 *
 * Computes l := limit(f, x = 0, dir), or l := f(0) if dir = Undirected
 * returns symbolic "series" call if l cannot be determined
 *
 * If l is not real (so heaviside(l) = undefined) ==> returns error
 *
 * otherwise, if l is real:
 * If l = 0 (expansion around 0) ==> heaviside goes into the constant
 *                                   coefficient
 * If l <> 0 is real (so heaviside(l) = 0 or 1) and dir <> Undirected
 *   ==> returns heaviside(l)
 * Otherwise, if l <> o is real and dir = Undirected ==> symbolic "series" call
 *
 * The code is completely analogous to dirac::series
 */

heaviside::series := proc(f, x, ord, dir, opt)
  local l, h;
begin

  if dir = Undirected then
    // undirected expansion, compute f(0)
    if traperror((l := subs(f, x = 0, EvalChanges))) <> 0 then
      userinfo(1, "could not compute limit of first argument of heaviside");
      return(FAIL);
    end_if
  else
    // directional expansion, compute limit(f, x = 0)
    l := limit(f, x = 0, dir);
    if type(l) = "limit" or l = FAIL then
      userinfo(1, "could not compute limit of first argument of heaviside");
      return(FAIL);
    end_if;
  end_if;
 
  h := heaviside(l);
  if l <> 0 and h = undefined then
    Series::error("heaviside is not defined for non-real expansion points");
  elif l = 0 then // expansion around zero
    // heaviside(..) -> constant coefficient
    return(Series::Puiseux::const(heaviside(f), x, ord, dir))
  elif (h = 0 or h = 1) and dir <> Undirected then
    // directional expansion around real l <> 0
    return(Series::Puiseux::const(h, x, ord, dir)) // h + O(x^ord)
  else
    return(FAIL)
  end_if

end_proc:
