/*
 * series slot of lambertW - computes series(lambertW(f), x, n, dir)
 *
 * computes undirected expansions around x0 for -exp(-1) <= x0 < infinity
 * returns a symbolic Taylor expansion for all other values of x0
 */

lambertW::series :=
proc(branch, f, x, n, dir, opt)
  local s, t, k, mu, alpha, j, l, nextbranch, fcat;
begin

  // recursively expand the argument
  t := Series::series(f, x, n, dir, opt);

  //==================================================
  // fcat = utility for computing _fconcat of a series 
  // s1 with an expression s2 (where either s1 or the
  // expansion of s2 is/would be a gseries and the 
  // standard overloading of _fconcat would not work)
  //==================================================
  fcat:= proc(s1, s2, x, n, dir, opt)
         begin
           if contains({Series::Puiseux, Series::gseries},domtype(s1)) then
              if s2 = x then
                 return(s1)
              else
                 return(Series::series(subs(expr(s1), x = s2, EvalChanges), x, n, dir, opt));
              end_if:
           else
              return(Series::series(subs(s1, x = s2,EvalChanges), x, n, dir, opt));
           end_if:
         end_proc;

  //======================================================
  // Special case allowing to expand lambertW(k, exp(f)) 
  // with f given by a Puiseux series. This is not covered
  // by the code further down below, because expansions of
  // exp(x) and the like around x = infinity are gseries!
  //======================================================
  if domtype(t) = Series::gseries then
     if type(f) = "exp" then
        f:= op(f):
     else
        f:= ln(f):
     end_if:
     t := Series::series(f, x, n, dir, opt);
     if domtype(t) <> Series::Puiseux then
        return(FAIL);
     end_if:
     l:= limit(lmonomial(t), x, dir):
     if l = infinity then // expansion around +infinity
       s:= Series::gen["lambertWexp"](branch, n-2, f):
       return(Series::series(s, x, n, dir, opt)):
     elif l = -infinity then // expansion of lambertW(k, exp(x)) around x = -infinity
       if branch = 0 then 
          s:= Series::gen["lambertW/0/0"](n);
          s:= Series::Puiseux::create(1, 1, n+1, s, x, 0, dir);
          f:= exp(f);
          s:= fcat(s, f, x, n, dir, opt):
          if domtype(s) = Series::gseries and
             iszero(op(s, 2)) then 
             // there is no order term. Insert one:
             s:= subsop(s, 2 = lterm(series(f, x, 2, dir, opt))^(n+1)):
          end_if:
          return(s);
       else // branch <> 0
          s:= Series::gen["lambertWexp"](branch, n-2, f):
          return(Series::series(s, x, n, dir, opt));
       end_if:
     end_if:
     return(FAIL);
  end_if:

  if domtype(t) = Series::Puiseux then
    k := Series::Puiseux::ldegree(t);
    if k = FAIL then // t = O(..)
      k:= Series::Puiseux::order(t);
      if k > 0 then
        return(t)
      end_if;
      Series::error("order too small")
    end_if;
    if k > 0 then // expansion around 0
      if branch = 0 then
        s := Series::Puiseux::create(
                1, 1, n+1, Series::gen["lambertW/0/0"](n), x, 0, dir);
        if f = x then
          return(s)
        else
          // No need for fcat: both s and t are of type
          // Series::Puiseux, so @ should work fine
          return(s @ t)
        end_if
      else
        s := Series::gseries::create(Series::gen["lambertW/k/asympt"](branch, n-1, x),
                                     (1/(ln(x)+branch*I*2*PI))^(n), x=0);
        return(fcat(s, f, x, n, dir, opt));
      end_if
    elif k < 0 then // expansion around +/- infinity
      if dir <> Undirected then
        l := limit(lmonomial(t), x, dir);
        if l = infinity or l = -infinity then
          if dir=Left then
            f := -f;
            dir:= Right;
            if is(x < 0) = TRUE then 
               assume(x > 0): 
            elif is(x > 0) = TRUE then 
               assume(x < 0): 
            end_if:
          end_if;
          if dir=Right then
            // Maple 6 simply returns lambertW(x)
            // Asymptotic expansion of de Bruijn/Comtet,
            // formulation of Corless, Jeffrey, Knuth 1997
            s := Series::gseries::create(Series::gen["lambertW/k/asympt"](branch, n-1, 1/x),
                                         (1/(ln(1/x)+branch*I*2*PI))^(n), x=infinity);
            return(fcat(s, 1/f, x, n, dir, opt));
          end_if
        end_if
      end_if;
      // we can't do it
      return(FAIL)
    else // k = 0, expansion around a finite point x0 <> 0
      t := lmonomial(t, Rem);
      if branch=0 then
        if combine(t[1], exp) = -exp(-1) then
          // This is from the ISSAC'97 article of Corless, Jeffrey & Knuth,
          // section 3
          mu[0] := -1;
          mu[1] := 1;
          alpha[0] := 2;
          alpha[1] := -1;
          for k from 2 to n - 1 do
            alpha[k] := _plus(mu[j]*mu[k + 1 - j] $ j = 2..(k - 1));
            mu[k] := (k - 1)/(k + 1)*(mu[k - 2]/2 + alpha[k - 2]/4)
                     - alpha[k]/2 - mu[k - 1]/(k + 1);
          end_for;
          s := Series::Puiseux::create(2, 0, n,
                 [mu[k]*2^(k/2)*exp(1)^(k/2) $ k = 0..(n-1)], x, 0, dir);
          return(s @ t[2]);
        end_if;
      end_if;
      if branch=-1 then
        if combine(t[1], exp) = -exp(-1) then
          // This is from the ISSAC'97 article of Corless, Jeffrey & Knuth,
          // section 3.
          mu[0] := -1;
          mu[1] := 1;
          alpha[0] := 2;
          alpha[1] := -1;
          for k from 2 to n - 1 do
            alpha[k] := _plus(mu[j]*mu[k + 1 - j] $ j = 2..(k - 1));
            mu[k] := (k - 1)/(k + 1)*(mu[k - 2]/2 + alpha[k - 2]/4)
                     - alpha[k]/2 - mu[k - 1]/(k + 1);
          end_for;
          s := Series::Puiseux::create(2, 0, n,
                                       [(-1)^k*mu[k]*2^(k/2)*exp(1)^(k/2)
                                        $ k = 0..(n-1)], x, 0, dir);
          return(s @ t[2]);
        end_if
      end_if
    end_if;
    if (branch = 0 and is(t[1] < -exp(-1))=TRUE)
      or (branch <> 0 and is(t[1] < 0)=TRUE) then
      nextbranch := branch-1;
      if branch = 1 and is(t[1] > -exp(-1))=TRUE then
        nextbranch := -1;
      end_if;
      return((signIm(f)+1)/2 * Series::unknown(lambertW(branch, f), x, n, dir)
           - (signIm(f)-1)/2 * Series::unknown(lambertW(nextbranch, f), x, n, dir));
    end_if
  end_if;

  Series::unknown(lambertW(branch, f), x, n, dir)
end_proc:

// ensure that the domain Series is loaded
eval(Series):

//=================================================================
// expansion of lambertW(0, x) around x = 0:
//=================================================================
Series::gen["lambertW/0/0"] :=
proc(n)
  local i;
begin
  [(-i)^(i - 1)/fact(i) $ i = 1..n]
end_proc:
//=================================================================
// asymptotic expansion of lambertW(k, z) for z -> infinity
// The returned list is suitable input for Series::gseries::create
//=================================================================
Series::gen["lambertW/k/asympt"] :=
proc(branch, order, z)
  local ln_k, ln_ln_k, n, m;
begin
  ln_k := ln(z) + I*branch*2*PI;
  ln_ln_k := ln(ln_k);
  if order <= 0 then
    [[ln_k-ln_ln_k, 1]]
  else
    [[ln_k-ln_ln_k, 1],
     [(-1)^(n+1)
      *_plus((combinat::stirling1(n, n-m+1)*ln_ln_k^m/fact(m))$m=1..n), 
      ln_k^(-n)
     ] $ n=1..order];
  end_if:
end_proc:
//================================================================
// asymptotic expansion of lambertW(k, exp(z)) for z -> infinity
// The returned expressions needs to be expanded by Series::series
//================================================================
Series::gen["lambertWexp"] :=
proc(branch, order, z)
  local ln_k, ln_ln_k, n, m;
begin
  ln_k := z + I*branch*2*PI;
  ln_ln_k:= ln(ln_k);
  if order < 0 then
     z
  elif order = 0 then
     z +  I*branch*2*PI - ln(ln_k)
  else
     z +  I*branch*2*PI - ln(ln_k) + 
     _plus((-1)^(n+1)
           *_plus((combinat::stirling1(n,n-m+1)*ln_ln_k^m/fact(m)) $ m=1..n)
       /ln_k^n $ n=1..order);
  end_if:
end_proc:
