
// data type Series::Puiseux for truncated Puiseux series ---------------------

Series::Puiseux := newDomain("Series::Puiseux"):
Series::Puiseux::Name := Series::Puiseux::key:
Series::Puiseux::create_dom := hold(Series::Puiseux):
Series::Puiseux::info := "domain for truncated Puiseux series expansions":
Series::Puiseux::interface := {}:
Series::Puiseux::exported := {}:


// alias definitions ----------------------------------------------------------

alias(Puiseux=Series::Puiseux):
alias(gseries=Series::gseries):

//
// new alias Flag = extop(s,1)
// Flag=0 : no x in coeffs
// Flag=1 : x in coeffs
// (13.04.00, TL)
//
alias(Flag(s)=extop(s,1)):
//
// aliases for other extops
// (13.04.00, TL)
//
alias(BranchOrder(s)=extop(s,2)):
alias(Valuation(s)=extop(s,3)): // Valuation
alias(Err(s)=extop(s,4)): // order of the error term
alias(ListCoeffs(s)=extop(s,5)):
alias(Varpoint(s)=extop(s,6)):
alias(Variable(s)=op(extop(s,6),1)): // series variable
alias(Point(s)=op(extop(s,6),2)): // expansion point
alias(Direction(s)=extop(s,7)): // direction of expansion:
// Left, Right, Real or Undirected

// CHANGED (jngerhar) 23.02.01: new alias:
// Base(s, e) = (x - x0)^e if expansion point x0 is finite
//            = x^(-e)     if expansion point x0 = complexInfinity
//
alias(Base(s, e) =
  if Point(s) = complexInfinity then
    Variable(s)^(-e)
  else
    (Variable(s) - Point(s))^e
  end_if):


// construction and conversion ------------------------------------------------


//
// Puiseux::create is to be used from outside to create
// Puiseux expansions efficiently without knowing their
// internal structure. No argument checking is performed.
// if l = [], then v is set to n as well
//
// always sets Flag = 0
// expansion point defaults to 0
//
// (05.07.00, TL)
//
Puiseux::create := proc(bo, v, n, l, x, x0 = 0, dir = Undirected)
begin
  if nops(l) = 0 then
     new(Puiseux, 0, bo, n, n, l, x = x0, dir);
  else
     new(Puiseux, 0, bo, v, n, l, x = x0, dir);
  end_if
end_proc:


//
// Puiseux::zero(x = x0, n): create O((x - x0)^n)
//
// x: identifier
// x0: arithmetical expression not containing x
// n: rational number
//
// always sets Flag = 0
// expansion point is always 0
// no argument checking
//
// CHANGED (jngerhar) 25.02.01: new procedure
//
Puiseux::zero := proc(x, n, dir = Undirected)
begin
   if not type(x) = "_equal" then
     new(Puiseux, 0, denom(n), numer(n), numer(n), [], x = 0, dir)
   else
     new(Puiseux, 0, denom(n), numer(n), numer(n), [], x, dir)
   end_if
end_proc:


//
// Puiseux::one(x = x0, n): create 1 + O((x - x0)^n)
//
// x: identifier
// x0 (optional): arithmetical expression not containing x
// n: nonnegative rational number
//
// always sets Flag = 0
// no argument checking
// if n = 0 returns O(1)
//
// (13.04.00, TL)
// CHANGED (jngerhar) 25.02.01:
// - new calling syntax
// - n = 0 now works correctly
//
Puiseux::one := proc(x, n, dir = Undirected)
begin
   if not type(x) = "_equal" then
     x := (x = 0)
   end_if;
   if n = 0 then
     new(Puiseux, 0, 1, 0, 0, [], x, dir)
   else
     new(Puiseux, 0, denom(n), 0, numer(n), [1], x, dir)
   end_if
end_proc:


//
// Puiseux::const(f, x = x0, n): create f + O((x - x0)^n)
//
// f: arithmetical expression
// x: identifier
// x0 (optional): arithmetical expression not containing x
// n: nonnegative rational number
//
// always sets Flag = 0
// expansion point is always 0
// no argument checking
// if n = 0 returns O(1)
//
Puiseux::const := proc(f, x, n, dir = Undirected)
begin
   if not type(x) = "_equal" then
     x := (x = 0)
   end_if;
   if n = 0 then
     new(Puiseux, 0, 1, 0, 0, [], x, dir)
   else
     new(Puiseux, 0, denom(n), 0, numer(n), [f], x, dir)
   end_if
end_proc:


//
// NEW
// converts series with Flag=0 to series with Flag=1
// (13.04.00, TL)
// CHANGED (jngerhar) 23.02.01: now also works correctly
//   for expansion point <> 0
//
Puiseux::convert01 := proc(s)
   local l, i;
begin
   if Flag(s) = 0 then
      //
      // if Flag=0 then set ith coefficient l[i]:
      // l[i] <- l[i]*(Variable(s)-Point(s))^((Valuation+i-1)/BranchOrder)
      //
      l := ListCoeffs(s);
      l := [l[i]*Base(s, (Valuation(s) + i - 1)/BranchOrder(s))
            $ i = 1..nops(l)];
      extsubsop(s, 1 = 1, 5 = l);
   else
      s
   end_if;
end_proc:


//
// NEW
// tries to convert series with Flag=1 to series with Flag=0
// (12.07.00, TL)
//
// QUESTION: Is it too costly to use it after each procedure
//           (mainly "_mult", "_power") ???
// STATUS: - used in procedure "main" in "../STDLIB/series.mu"
//
// PROPOSAL: - power: a^s => use it if bo(a)|s
//           - mult : a*b => use it if bo(a)=bo(b)
//           - plus : a+b => use it if max(nops(a),nops(b))>nops(a+b)
//
// CHANGED (jngerhar) 26.02.01:
// - should now work correctly for Point <> 0 as well
// - added error term
//
Puiseux::convert10 := proc(s)
   local n;
begin
   n := Err(s);

   if Flag(s) = 1 then

      if Puiseux::iszero(s) then
        extsubsop(s, 1 = 0)
      else
        // convert s into an expression and
        // reconvert the expression into a series expansion via Puiseux::new
        Puiseux::new(Puiseux::expr(s), Varpoint(s), n - Valuation(s), Direction(s))

        // add error term O(Base(s, 1)^ord) to ensure that the order
        // of the result is not too high
        + Puiseux::create(BranchOrder(s), n, n, [], Variable(s), Point(s), Direction(s))
      end_if
   else
      s
   end_if;
end_proc:


//
// Puiseux::multbo(s, n)
//
// multiply branching order by the positive integer n and adjust
// the representation accordingly
// mathematically, the result represents the same object as s
//
Puiseux::multbo := proc(s, n)
local l, z, i;
begin
// CHANGED (jngerhar) 24.02.01: added if-statement
   if n = 1 then
     s
   else
     l := ListCoeffs(s);
     if l <> [] then
        z := 0 $ (n - 1);
        l := [(l[i], z) $ i = 1..nops(l) - 1, l[nops(l)]]
     end_if;
     extsubsop(s, 2 = BranchOrder(s)*n, 3 = Valuation(s)*n, 4 = Err(s)*n,
                  5 = l)
   end_if
end_proc:


//
// Puiseux::clean(s)
//
// rectify the data structure of s if Variable(s) occurs with integer
// exponents in the coefficients of s
// This works by converting s into an expression and applying "series"
// to the result
//
Puiseux::clean := proc(s)
  local res;
begin
  if has(ListCoeffs(s), Variable(s)) then
    res := FAIL;
    traperror((res := series(Puiseux::expr(s), Varpoint(s),
                             Err(s) - Valuation(s), Direction(s), NoWarning)));
    if contains({Puiseux, gseries}, domtype(res)) then
      // add the original error term to ensure that the error term of the
      // result is not too high
      res + extsubsop(s, 1 = 0, 3 = Err(s), 5 = [])
    else
      res
    end_if
  else
    s
  end_if
end_proc:


//
// Puiseux::condense(s)
//
// try to lower the branching order of s to 1 and
// do a quick check whether s can be converted to Flag = 0
//
Puiseux::condense := proc(s)
  local bo, v, e, l, l1, i;
begin
  bo := BranchOrder(s);
  v := Valuation(s);
  e := Err(s);

  if bo > 1 and domtype(v/bo) = DOM_INT and domtype(e/bo) = DOM_INT then
    l := ListCoeffs(s);
    if nops(l) > 0 then
      l1 := [0 $ nops(l)];
      (l1[1 + i*bo] := l[1 + i*bo]) $ i = 0..(ceil(nops(l)/bo - 1));
    else
      l1 := l;
    end_if;
    if l1 = l then
      v := v/bo;
      e := e/bo;
      s := extsubsop(s, 2 = 1, 3 = v, 4 = e,
                        5 = [l[1 + bo*i] $ i = 0..(ceil(nops(l)/bo) - 1)]);
      bo := 1;
    end_if;
  end_if;

  if Flag(s) = 1 then
    l := ListCoeffs(s);
    (l[i] := l[i]/Base(s, (v + i - 1)/bo)) $ i = 1..nops(l);
    if not has(l, Variable(s)) then
      s := extsubsop(s, 1 = 0, 5 = l)
    end_if;
  end_if;

  s
end_proc:


//
// Puiseux::new(f, p, n, dir): convert the expression f into an element of
//   domain Series::Puiseux
//
// This function is intended only for expressions that are mathematically
// truncated Puiseux series, i.e., polynomials in powers of x with
// rational, possibly negative, exponents. In fact, it handles also
// rational expressions in fractional powers of x, but special mathematical
// functions such as ln, exp, sin etc. are not further expanded; they
// simply go into the coefficients. O-terms are recognized as well.
// An error occurs if an O-term cannot be converted.
// FAIL is returned if some power with non-rational exponent occurs in f.
//
// p: x or (x = x0), where x is an identifier (the series variable) and
//    x0 is an arithmetical expression specifying a finite expansion point
//    or infinity, -infinity, complexInfinity.
//    If x0 is omitted, x = 0 is assumed.
// f: arithmetical expression in x
// n: (optional) desired number of terms of the expansion
// dir: (optional) direction of the expansion, one of
//      Undirected (default), Real, Left or Right
//
// Puiseux::new is the interface procedure, which does some argument
// checking and a change of variable to transform the expansion point to
// the origin. The main work is done by Puiseux::new_rec, which has the
// same calling syntax as Puiseux::new, but always computes expansions
// around 0. It recursively descends into the structure of the expression.
//
// CHANGED (jngerhar) 25.02.01: rewritten
//
Puiseux::new :=
proc(f: Type::Union(Type::Arithmetical, DOM_POLY), p, n = FAIL,
     dir = Undirected)
  local x, x0, x1, yofx, xofy, tmp, s, e, fac, dummy;
begin

   if args(0) < 2 then
     Series::error("wrong number of arguments");
   end_if;

   if args(0) = 3 and contains({Left, Right, Real, Undirected}, n) then
     // direction specified, but no order
     dir := n;
     n := ORDER;
   end_if;

   if n = FAIL then
     // no order specified; use default value
     n := ORDER;
   elif not testtype(n, Type::NonNegInt) then
     Series::error("nonnegative integer expected as third argument")
   end_if;

   if not contains({Left, Right, Real, Undirected}, dir) then
     Series::error("invalid direction");
   end_if;

   // check that p = x or p = (x - x0) specifices a valid series variable
   // and expansion point
   [x, x1] := Series::check_point(p);
   if has(x1, infinity) and not contains({infinity, -infinity}, x1) then
      // rewrite series(f(x), x = I*infinity) as series(f(I*x), x = infinity)
      fac:= 1:
      if testtype(x1, "_mult") then
         [x1, fac, dummy]:= split(x1, has, infinity);
         f:= subs(f, x = fac*x);
      end_if;
      if fac <> 1 then
         s:= Puiseux::new(f, x = x1, n, dir);
         if s = FAIL then
            return(FAIL)
         elif traperror((tmp:= s::dom::subs(s, x = x/fac))) = 0 then
            return(tmp);
         else
            return(FAIL);
         end_if;
      end_if;



   end_if;
   x0 := x1;

   // Prepare change of variable: x will be replaced by y(x),
   // such that after the substitution x is to tend to 0.
   // Moreover, modify the properties of x accordingly

   save x; // save property of x

   // the following piece of code is identical to the corresponding
   // one in Series::main
   if domtype(x0) = stdlib::Infinity then
         if Series::sign(dir) = sign(x0) then
           // specified Left for -infinity or Right for infinity
           Series::error("inconsistent direction")
         end_if;
         yofx := sign(x0)/x;
         xofy := yofx;
         tmp := getprop(yofx);
         dir := Right;
         assume(x > 0);
         if tmp <> yofx and tmp <> Type::Complex then
           assume(x, tmp, _and);
         end_if;
         x0 := complexInfinity;
   elif x0 = complexInfinity then
         yofx := 1/x;
         tmp := getprop(yofx);
         case dir
         of Left do // infinity
           dir := Right;
           assume(x > 0);
           x1 := infinity;
           break;
         of Right do // -infinity
           yofx := -1/x;
           tmp := getprop(yofx);
           dir := Right;
           assume(x > 0);
           x1 := -infinity;
           break;
         of Real do
           assume(x, Type::Real)
         end_case;
         if tmp <> yofx and tmp <> Type::Complex then
          /* if prop = x then
             assume(x, tmp)
           else
          */
           assume(x, tmp, _and)
          // end_if
         end_if;
         xofy := yofx;
   else
         yofx := x + x0;
         xofy := x - x0;
         tmp := getprop(x - x0);
         case dir
         of Right do
           assume(x > 0);
           break;
         of Left do
           assume(x < 0);
           break;
         of Real do
           assume(x, Type::Real);
           break;
         end_case;
         if tmp <> x - x0 and tmp <> Type::Complex then
/*           if prop = x then
             assume(x, tmp)
else
*/
           assume(x, tmp, _and)
  //         end_if
         end_if;
   end_if;

   // perform change of variable
   s := FAIL;
   traperror((s := subs(f, x = yofx)));

   if s <> FAIL then
     e := traperror((s := Puiseux::new_rec(s, x, n, dir)));
   end_if;


   if s <> FAIL and e <> 0 then
     lasterror()
   end_if;

   if domtype(s) = Puiseux then
     // reverse change of variables and return the result
     Puiseux::set_var(s, x, x, x1);
   else
     userinfo(2, "cannot create a Puiseux series; try 'series'");
     return(FAIL);
   end_if
end_proc:

//
// CHANGED (jngerhar) 25.02.01: new procedure
// see Puiseux::new above for a description
//
Puiseux::new_rec := proc(f, x, n, dir)
  local Opoints, s;
begin

  // trivial case: f does not depend on x
  if iszero(f) then // O(x^n)
    return(Puiseux::zero(x, n, dir))
  elif not has(f, x) then // f + O(x^n)
    return(Puiseux::const(f, x, n, dir));
  end_if;

  // check if f is a polynomial in x
  if testtype(f, Type::PolyExpr(x, Type::AnyType)) then
    return(Puiseux::frompoly(poly(f, [x]), x, n, dir))
  end_if;

  // recursively descend into expressions; this even
  // handles rational expressions
  case type(f)
  of "_plus" do
  of "_mult" do
    return(map(f, Puiseux::new_rec, x, n, dir));
  of "_power" do
    if contains({DOM_INT, DOM_RAT}, domtype(op(f, 2))) then
      return(Puiseux::_power(Puiseux::new_rec(op(f, 1), x, n, dir), op(f, 2)))
    else // not a rational exponent
      return(FAIL);
    end_if;
  of O do // this code is analogous to the corresponding code in Series::series

    // check whether the expansion of the O-term is around x = 0
     Opoints := O::points(f);
     if not has(Opoints, x) then
       // series variable does not occur in the O-term
       // ==> make it a coefficient
       return(Puiseux::const(f, x, n, dir));
     elif not has(Opoints, x = 0) then
       Series::error("O-term has a different expansion point");
     end_if;

     // convert the O-term into a Puiseux series
     s := Puiseux::new_rec(op(f, 1), x, n, dir);
     if domtype(s) <> Puiseux then
       Series::error("cannot convert O-term into a Puiseux series");
     end_if;

     if nops(Opoints) = 1 then
       // univariate O-term around x = 0
       // ==> just take lowest order term
       if Puiseux::iszero(s) then
         return(s)
       else
         return(Puiseux::truncate(s, Puiseux::ldegree(s)))
       end_if
     else
       // multivariate O-term ==>
       // convert coeffs into O-terms again
       return(map(s, O, op(Opoints)));
     end_if
  end_case;

  // last resort: consider f as a coefficient
  Puiseux::const(f, x, n, dir);
end_proc:


//
// Puiseux::series: convert a Series::Puiseux into a Series::Puiseux.
//   maybe with respect to a different variable
//   (different expansion point makes no sense!)
//   Just converts the argument into an expression via Puiseux::expr
//   + O-term and applies "series" to the result.
//
Puiseux::series := proc(s)
begin
  series(Puiseux::expr(s)
           + O(Base(s, Err(s)/BranchOrder(s)), Varpoint(s)),
         args(2..args(0)))
end_proc:


//
// Puiseux::frompoly(f, x, n): convert a polynomial into a Taylor series
//   around x = 0
//
// f: DOM_POLY in [...,x,...]
// x: identifier
// n: nonnegative integer specifying the number of terms of the series
//
// this method is for internal use only and does not perform any
// argument checking
//
Puiseux::frompoly:= proc(f, x, n, dir = Undirected)
   local v, e, i;
begin
   // to avoid coefficients such as 0.0
   f := mapcoeffs(f,
     proc() begin
       if iszero(args(1)) then 0 else args(1) end_if
     end_proc);

   if iszero(f) then
      return(Puiseux::zero(x, n, dir)) // O(x^n)
   end_if;

   // ignore coefficient rinf of f
   f := poly(f, Expr);

   v := ldegree(f);
   e := v + n;
   while e < degree(f, x) and iszero(coeff(f, x, e)) do
     e := e + 1;
   end_while;
   Puiseux::normal(Puiseux::create(1, v, e,
                   [coeff(f, x, i) $ i = v..min(degree(f, x), v + n - 1)],
                   x, 0, dir), 0) // quick normalization
end_proc:


//
// Puiseux::convert(f) - try to convert f into an element of Puiseux
// or return FAIL
//
Puiseux::convert := proc(f)
  local x, p, d, g, h, s, i, n;
begin
  case domtype(f)

  of Puiseux do
    return(f);

  of DOM_POLY do
    x := op(op(f, 2), 1); // first indeterminate
    // error term of the result has degree deg(f) + ORDER
    n := ORDER + degree(f, x) - ldegree(f, x);
    return(Puiseux::frompoly(f, x, n))

  of DOM_EXPR do
  of DOM_IDENT do
    x := (indets(f) minus Type::ConstantIdents);
    if nops(x) = 1 then
      if traperror((s := Puiseux::new(f, op(x)))) = 0 then
        return(s)
      end_if
    end_if;
    return(FAIL)

  of O do
    x := O::points(f);
    if nops(x) = 1 then
      s := FAIL;
      traperror((s := Puiseux::new(op(f, 1), op(x))));
      if domtype(s) = Puiseux then
        if nops(ListCoeffs(s)) = 0 then // s = O(..)
          return(s)
        else
          return(Puiseux::truncate(s, Puiseux::ldegree(s)));
        end_if
      end_if;
    end_if;
    return(FAIL);

  of contfrac do
    if testtype(extop(f, 2), Type::Rational) then
      return(FAIL)
    else
      return(extop(f, 2))
    end_if;

  of gseries do
    g := extop(f, 1);
    x := gseries::indet(f);
    p := gseries::point(f);
    d := Right;
    if p = infinity then
      p := complexInfinity;
      d := Left;
    elif p = -infinity then
      p := complexInfinity;
    end_if;

    s := Puiseux::zero(x = p, ORDER, d);
    for i from 1 to nops(g) do
      h := FAIL;
      traperror((h := Puiseux::new(_mult(op(op(g, i))), x = p, ORDER, d)));
      if domtype(h) <> Puiseux then
        return(FAIL);
      elif BranchOrder(s) <> BranchOrder(h) then
        n := lcm(BranchOrder(s), BranchOrder(h));
        s := Puiseux::multbo(s, n/BranchOrder(s));
        h := Puiseux::multbo(h, n/BranchOrder(h));
      end_if;
      s := Puiseux::plus(extsubsop(s, 4 = Err(h)), h);
    end_for;
    if extop(f, 2) <> 0 then
      // add error term
      h := FAIL;
      traperror((h := Puiseux::new(extop(f, 2), x = p, ORDER, d)));
      if domtype(h) <> Puiseux then
        return(FAIL);
      elif BranchOrder(s) <> BranchOrder(h) then
        n := lcm(BranchOrder(s), BranchOrder(h));
        s := Puiseux::multbo(s, n/BranchOrder(s));
        h := Puiseux::multbo(s, n/BranchOrder(h));
      end_if;
      s := extsubsop(s, 4 = Valuation(h));
    end_if;
    return(s);

  otherwise
    // convert f into an element of a kernel domain and try the conversion
    // again
    g := expr(f);
    if f = g then
      return(FAIL);
    else
      return(Puiseux::convert(g));
    end_if

  end_case;
end_proc:


//
// Puiseux::convert_to(s, T) - try to convert s into an element of the domain
// T (if T is a domain) or of the domain of T (if T is not a domain)
// or return FAIL
//
Puiseux::convert_to := proc(s, T)
  local l, i;
begin
  if domtype(T) <> DOM_DOMAIN then
    T := domtype(T)
  end_if;

  case T

  of Puiseux do
    return(s);

  of gseries do
    return(gseries::convert(s));

  of contfrac do
    return(Puiseux::contfrac(s));

  of DOM_POLY do
    if BranchOrder(s) = 1 then
      s := Puiseux::convert10(s);
      l := ListCoeffs(s);
      if Point(s) <> complexInfinity and Valuation(s) >= 0 then
        return(poly([[l[i],   Valuation(s) + i - 1 ] $ i = 1..nops(l)],
                    [Variable(s)]))
      elif Point(s) = complexInfinity and Valuation(s) + nops(l) - 1 <= 0 then
        return(poly([[l[i], -(Valuation(s) + i - 1)] $ i = 1..nops(l)],
                    [Variable(s)]))
      end_if;
    end_if;
    return(FAIL);

  of DOM_EXPR do
    return(Puiseux::expr(s));

  of O do
    if Point(s) <> complexInfinity then
      // return O(lterm(s))
      l := ldegree(s);
      if l = FAIL then
        l := Series::Puiseux::order(s);
      end_if;
      return(O((Variable(s) - Point(s))^l, Varpoint(s)));
    else
      return(FAIL);
    end_if;

  otherwise
    return(FAIL);

  end_case;
end_proc:


// access methods -------------------------------------------------------------

Puiseux::indet := proc(s) begin Variable(s) end_proc:

Puiseux::point := proc(s) begin Point(s) end_proc:

Puiseux::direction := proc(s) begin Direction(s) end_proc:

//
// Puiseux::order(s): return the order of the error term of s
//    - returns Err/BranchOrder
//
Puiseux::order := proc(s) begin Err(s)/BranchOrder(s) end_proc:

Puiseux::valuation := s -> extop(s, 3):
Puiseux::branchorder := s -> extop(s, 2):



//
// ldegree(s): compute the degree of the "smallest" term of s
//    - nonzero : degree/BranchOrder
//    - zero    : FAIL
// CAUTION: FAIL is needed in several procedure
//          if Err/BranchOrder is needed use procedure "order"
// (24.05.00, TL)
//
Puiseux::ldegree := proc(s)
   local l, v;
begin
   l := ListCoeffs(s);
   v := Valuation(s);
   while l <> [] do
      if Puiseux::iszero(l[1]) then
         delete l[1];
         v := v + 1
      else
         return(v/BranchOrder(s))
      end_if
   end_while;
   FAIL
end_proc:


//
// lcoeff(s): return the coefficient of the term of lowest order in s
// returns FAIL if s = O(...)
//
// Remark: coeff of (a*x)^(1/2)*(b*x)^(1/2) is (a*x)^(1/2)*(b*x)^(1/2)/x
// (05.07.00, T.L., with Juergen Gerhard)
// CHANGED (jngerhar) 23.02.01: now also works correctly for Point <> 0
//
Puiseux::lcoeff := proc(s)
   local l, i;
begin
   l := ListCoeffs(s);
   for i from 1 to nops(l) do
      if not Puiseux::iszero(l[i]) then
         if Flag(s) = 0 then
            return(l[i]);
         else
            return(l[i]/Base(s, (Valuation(s) + i - 1)/BranchOrder(s)));
         end_if;
      end_if
   end_for;
   FAIL
end_proc:


//
// lterm(s): return the term of lowest degree in s
// returns FAIL if s = O(...)
//
// (05.07.00, T.L., with Juergen Gerhard)
// CHANGED (jngerhar) 23.02.01: now also works correctly for Point <> 0
//
Puiseux::lterm := proc(s)
   local l, i;
begin
   l := ListCoeffs(s);
   for i from 1 to nops(l) do
      if not Puiseux::iszero(l[i]) then
         return(Base(s, (Valuation(s) + i - 1)/BranchOrder(s)))
      end_if;
   end_for;
   FAIL
end_proc:


//
// lmonomial(s): return the monomial u of lowest degree in s
// lmonomial(s, Rem) returns [u, s - u]
// both return FAIL if s = O(...)
//
// (05.07.00, T.L., with Juergen Gerhard)
// CHANGED (jngerhar) 23.02.01: now also works correctly for Point <> 0
//
Puiseux::lmonomial := proc(s, opt = FAIL)
   local v, l, i, u;
begin
   v := Valuation(s);
   l := ListCoeffs(s);

   i := 0;
   while l <> [] do
      if Puiseux::iszero(l[1]) then
         delete l[1]
      else
         if Flag(s) = 0 then
            u := l[1]*Base(s, (v + i)/BranchOrder(s));
         else
            u := l[1];
         end_if;
         if opt = hold(Rem) then
            return([u, Puiseux::normal(
              extsubsop(s, 3 = v + i + 1, 5 = subsop(l, 1 = null())),
              1)]) // normalize with Puiseux::iszero as zero test
         else
            return(u)
         end_if;
      end_if;
      i := i + 1;
   end_while;
   FAIL
end_proc:



//
// Puiseux::nthterm(s, n): return the nth nonzero term, counted from the
// lowest term on
//
// (05.07.00, T.L., with Juergen Gerhard)
// CHANGED (jngerhar) 25.02.01: removed simplify
//
Puiseux::nthterm := proc(s, n)
  local l, t, i;
begin
   l := ListCoeffs(s);
   if domtype(n) <> DOM_INT or n <= 0 or n > nops(l) then
     FAIL
   else
     i := 0;
     while i < nops(l) and n > 0 do
       i := i + 1;
//       t := simplify(l[i]);
       t := l[i];
       if not Puiseux::iszero(t) then
         n := n - 1;
       end_if;
     end_while;
     if n > 0 then
       return(FAIL);
     end_if;
     Base(s, (Valuation(s) + i - 1)/BranchOrder(s))
   end_if
end_proc:


//
// Puiseux::nthmonomial(s, n): return the nth nonzero monomial, counted from
// the lowest term on
//
// (05.07.00, T.L., with Juergen Gerhard)
// CHANGED (jngerhar) 25.02.01: removed simplify
//
Puiseux::nthmonomial := proc(s, n)
  local l, t, i;
begin
   l := ListCoeffs(s);
   if domtype(n) <> DOM_INT or n <= 0 or n > nops(l) then
      FAIL
   else
      i := 0;
      while i < nops(l) and n > 0 do
         i := i + 1;
//         t := simplify(l[i]);
         t := l[i];
         if not Puiseux::iszero(t) then
            n := n - 1;
         end_if;
      end_while;
      if n > 0 then
         return(FAIL);
      end_if;
      if Flag(s) = 0 then
         t*Base(s, (Valuation(s) + i - 1)/BranchOrder(s));
      else
         t;
      end_if;
   end_if
end_proc:


//
// Puiseux::nthcoeff(s, n): return the nth nonzero coefficient, counted from
// the lowest term on
//
// Remark: coeff of (a*x)^(1/2)*(b*x)^(1/2) is (a*x)^(1/2)*(b*x)^(1/2)/x
// (05.07.00, T.L., with Juergen Gerhard)
// CHANGED (jngerhar) 23.02.01: now also works correctly for Point <> 0
// CHANGED (jngerhar) 25.02.01: removed simplify
//
Puiseux::nthcoeff := proc(s, n)
  local l, t, i;
begin
   l := ListCoeffs(s);
   if domtype(n) <> DOM_INT or n <= 0 or n > nops(l) then
     FAIL
   else
     i := 0;
     while i < nops(l) and n > 0 do
       i := i + 1;
//       t := simplify(l[i]);
       t := l[i];
       if not Puiseux::iszero(t) then
         n := n - 1;
       end_if;
     end_while;
     if n > 0 then
       return(FAIL)
     end_if;
     if Flag(s) = 0 then
        t;
     else
        t/Base(s, (Valuation(s) + i - 1)/BranchOrder(s));
     end_if;
   end_if
end_proc:


//
// Puiseux::coeff(s) returns a list of all coefficients of the Puiseux
//   series s
//
// Puiseux::coeff(s, n) or
// Puiseux::coeff(s, x, n) returns the nth coefficient of the Puiseux
//   series s
// n: rational number
// If specified, x must equal the series variable
//
// If Point(s) = complexInfinity, then the coefficient of x^n, not
// of x^(-n) is returned!
//
// Flag = 0:  coefficient = list element
// Flag = 1:  coefficient = (list element)/Base^n
//
// Example: coeff of (a*x)^(1/2)*(b*x)^(1/2) is (a*x)^(1/2)*(b*x)^(1/2)/x
// (05.07.00, T.L., with Juergen Gerhard)
// CHANGED (jngerhar) 24.02.01: now also works correctly if Point <> 0
//
Puiseux::coeff:= proc(s, n)
   local fl, bo, v, l, sgn, i;
begin

   fl := Flag(s);
   bo := BranchOrder(s);
   v := Valuation(s);
   l := ListCoeffs(s);

   case args(0)
   of 0 do
      Series::error("missing argument")
   of 1 do
      //
      // return all coefficients
      // if Flag = 0 : op(ListCoeffs(s))
      // if Flag = 1 : list elements / power of Base
      //
      if fl = 0 then
         return(op(l))
      else
         return(l[i]/Base(s, (v + i - 1)/bo) $ i = 1 .. nops(l));
      end_if;
   of 3 do
      if n <> Variable(s) then
         Series::error("invalid variable")
      else
         n := args(3)
      end_if;
      // fall through
   of 2 do
      if n = Variable(s) or n = All then
        return(Puiseux::coeff(s))
      end_if;
      sgn := if Point(s) = complexInfinity then -1 else 1 end_if;
      if not testtype(n, Type::Rational) then
         Series::error("rational number expected")
      elif sgn*n < v/bo then
         return(0)
      elif sgn*n >= Err(s)/bo then
         return(FAIL)
      else
         //
         // x^(sgn*v/bo) corresponds to l[1],
         // thus x^n corresponds to l[1+sgn*n*bo-v]
         //
         if domtype(n*bo) <> DOM_INT or 1 + sgn*n*bo - v > nops(l) then
            return(0)
         elif fl = 0 then
            return(l[1 + sgn*n*bo - v])
         else // fl = 1
            return(l[1 + sgn*n*bo - v]/Base(s, sgn*n))
         end_if;
      end_if;
   otherwise
      Series::error("wrong number of arguments")
   end_case
end_proc:


//
// zero test
//
// both called for expressions and for Series::Puiseux!
//
Puiseux::iszero:=
proc(e)
begin
  if type(e) = Series::Puiseux then
    e := expr(e)
  end_if;
  if hastype(e, O) then
    return(FALSE)
  end_if;  
 
  bool(testeq(e, 0, 
    Steps = 0,                  // steps in Simplify
    Seconds = infinity,         // time limit in Simplify
    hold(NumberOfRandomRatTests) = 4, // insertions identifier <-> rational numbers
    NumberOfRandomTests = 15,    // different insertions identifier <-> numbers
    IgnoreProperties = FALSE,    // relevant for random tests + Simplify
    hold(ZeroTestMode) = TRUE         // (in the random tests, use only numbers consistent 
                                //  with the properties of the identifiers?)
  ) <> FALSE)

end_proc:
Puiseux::iszero:=prog::remember(Puiseux::iszero,
                                () -> [property::depends(args()), DIGITS]):


//
// checks if s is of type Base(s, 1) + O(...)
// (05.07.00, TL)
//
// Caution: works only for normalized series
//          ([0,...] is not allowed)
//          - only with procedure create possible,
//            otherwise the series is "normalized"
// CHANGED (jngerhar) 24.02.01, now also works for Point <> 0
//
Puiseux::equalx := proc(s)
begin
   if Valuation(s) = BranchOrder(s) and nops(ListCoeffs(s)) >= 1 then
      if Flag(s) = 0 then
         if ListCoeffs(s)[1]=1 then
            TRUE;
         else
            FALSE;
         end_if;
      else
         if ListCoeffs(s)[1] = Base(s, 1) then
            TRUE;
         else
            FALSE;
         end_if;
      end_if;
   else
      FALSE
   end_if;
end_proc:


// technical methods ----------------------------------------------------------

//
// Puiseux::truncate(s, n)
//
// truncate the Puiseux series s to order O(x^n), i.e., discard all terms of
// order >= n
//
// n: rational number
//
// if n is omitted, just discards zero terms at the end of the
// coefficient list
// assumes that the first term in the coefficient list, if non-empty,
// is nonzero
// does nothing if n >= e/bo. Otherwise, the result is guaranteed to be
// normalized, i.e., it has neither leading nor trailing zeroes in the
// coefficient list.
//
// CHANGED (jngerhar) 25.02.01: new method
//
Puiseux::truncate := proc(s, n)
  local nn, l, bo, v, e, i, j;
begin
  bo := BranchOrder(s);
  v := Valuation(s);
  e := Err(s);

  if args(0) < 2 then
    nn := (e - 1/2)/bo;
  else
    nn := n;
  end_if;

  if testargs() then
    // argument checking
    if not contains({DOM_INT, DOM_RAT}, domtype(nn)) then
      Series::error("second argument must be rational");
    end_if;
  end_if;

  if nn >= e/bo then
    s
  elif nn <= v/bo then
    // return O(x^ldegree(s))
    extsubsop(s, 1 = 0, 4 = v, 5 = [])
  else
    l := ListCoeffs(s);

    // determine the nonzero term of highest order < n
    for i from min(ceil(nn*bo) - v, nops(l)) downto 1 do
      if not Puiseux::iszero(l[i]) then
        break;
      end_if;
    end_for;

    // determine the nonzero term of least order >= n
    for j from ceil(nn*bo) - v + 1 to nops(l) do
      if not Puiseux::iszero(l[j]) then
        break;
      end_if;
    end_for;

    // discard zero terms and terms of order < n
    l := [op(l, 1..i)];

    if j > nops(ListCoeffs(s)) then
      // error term remains unchanged
      extsubsop(s, 5 = l)
    else
      extsubsop(s, 4 = v + j - 1, 5 = l)
    end_if
  end_if
end_proc:


//
// Puiseux::normal(s, lev): bring s into normal form by
// removing zero coefficients at the beginning and the end of the
// coefficient list
// Moreover, if the coefficient list is empty (i.e., s = O(..)),
// then set Valuation(s) = Err(s)
//
// s: Puiseux
//
// lev (optional): 0, 1 or 2 (default)
//   specifies the level of normalization:
//   0: quick normalization: now uses Puiseux::iszero
//      (this is used internally by most Puiseux attributes)
//   1: use Puiseux::iszero instead of iszero as a zero test
//      (this is used by Puiseux::lmonomial and Puiseux::_power;
//       it is necessary since higher level "series" users such
//       as limit and asympt replace Puiseux::iszero by their
//       own, more powerful routine and rely on lmonomial and _power
//       calling this routine)
//   2: apply normal to all coefficients first
//      (this is the intended behavior when normal is called from
//       outside series via overloading; default)
//
//   currently level 0 and 1 are equivalent
//
// CHANGED (jngerhar) 24.02.01, 03.03.01
//
Puiseux::normal:=
proc(s, lev = 2)
   local l, v, myiszero;
begin
  if contains({args()}, List) then
     error("Option List not allowed for Puiseux series")
  end_if;

   l := ListCoeffs(s);

  if lev = 2 then
    l:= map(l, normal)
  end_if;

  /*
   myiszero := iszero;
   if lev = 1 then
     myiszero := Puiseux::iszero;
   elif lev = 2 then
     l := map(l, normal);
   end_if;
 */
  myiszero:= Puiseux::iszero;

   if nops(l) = 0 then // s = O(..)
     return(extsubsop(s, 1 = 0, 3 = Err(s)))
   end_if;

   // remove leading zeroes
   v := Valuation(s);
   while l <> [] and myiszero(l[1]) do
      delete l[1];
      v := v + 1
   end_while;

   // remove trailing zeroes
   while l <> [] and myiszero(l[nops(l)]) do
      delete l[nops(l)];
   end_while;

   if nops(l) = 0 then // return O(..)
      return(extsubsop(s, 1 = 0, 3 = Err(s), 5 = []))
   end_if;

   extsubsop(s, 3 = v, 5 = l)
end_proc:


//
// change of variable: transform the series s around X = 0
// to a series around x = x0
// More precisely: substitute (x - x0) for X if x0 is finite,
// 1/x if x0 = complexInfinity or infinity,
// and -1/x if x0 = -infinity
//
Puiseux::set_var:=
proc(s, X, x, x0)
  local ba, l, s0, ln_simplify, dir, i;
begin
  dir := Direction(s);

  if x0 = complexInfinity and not contains({Left, Right}, dir)
    and BranchOrder(s) > 1 then
    // convert to Flag = 1
    s := Puiseux::convert01(s)
  end_if;

  // The coeffs may contain trancendental functions such as
  // s = (bo,v,Err,[exp(X), ln(X),..])
  //   -> exp(X)*X^(v/bo) + ln(X)*X^((v+1)/bo) +...

  // First, substitute the arguments of the trancendental
  // functions in the list.
  save x;
  case x0
    of -infinity  do
      ba:= -1/x;
      assume(x < -100);
      break
    of infinity do
      ba:= 1/x;
      assume(x > 100);
      break
    of complexInfinity do
      ba:= 1/x;
      unassume(x);
      break
    otherwise
      ba:= x - x0;
      if dir = Left then
        assume(x, Dom::Interval(x0 - 1/1000, x0))
      elif dir = Right then
        assume(x, Dom::Interval(x0, x0 + 1/1000));
      elif dir = Real then
        assume(x, Dom::Interval(x0 - 1/1000, x0 + 1/1000))
      else
        unassume(x)
      end_if
  end_case;

  l := eval(subs(ListCoeffs(s), X = ba));

  if domtype(x0) = stdlib::Infinity or
    (x0 = complexInfinity and contains({Left, Right}, dir)) then
    // directed expansion around infinity
    if domtype(x0) = stdlib::Infinity then
      s0 := sign(x0);
      dir := Left;
      if s0 = -1 then
        dir := Right;
      end_if
    else
      s0 := Series::sign(dir);
      dir := Series::invert(dir);
    end_if;
    // now:
    // * dir = Left <-> infinity, dir = Right <-> -infinity
    // * s0*x is positive

    // transform ln(1/x) -> -ln(x) etc. in the coefficients
    ln_simplify :=
    proc(e)
      local s, b, k, t;
    begin
      if e = -s0*x then
        return(I*PI + hold(ln)(-e))
      end_if;
      // check whether e = (-1)^s*(+-s0*x^t)^k,
      // with s=0,1 and t=1,-1 and rational k
      s := 0;
      b := e;
      if type(-e) = "_power" then
        s := 1;
        b := -e;
      end_if;
      if type(b) = "_power" then
        [b, k] := [op(b)];
        t := 1;
        if contains({x, -x}, 1/b) then
          t := -1;
          b := 1/b;
        end_if;
        // now e = (-1)^s * (b^t)^k
        if contains({DOM_INT, DOM_RAT}, domtype(k)) then
          if b = s0*x then
            return(s*I*PI + t*k*hold(ln)(b))
          elif b = -s0*x then
            // e = (-1)^s * (-(-b)^t)^k = (-1)^(s+k) * (-b)^(t*k)
            // reduce s + k modulo 2
            return((1 - 2*frac((1 - s - k)/2))*I*PI
                   + t*k*hold(ln)(-b))
          end_if
        end_if
      end_if;
      hold(ln)(e)
    end_proc:
    l := eval(subs(l, hold(ln) = ln_simplify));
  end_if;

  // For x0 = complexInfinity and dir = Right (<-> -infinity),
  // we also need to replace
  //  l[i] (--> l[i]*X^((v+i-1)/bo) = l[i]*(1/x)^((v+i-1)/bo))
  // with v = Valuation(s), bo = BranchOrder(s) by
  //  (-1)^(2*(v+i-1)/bo)*l[i] --> (-1)^(2*(v+i-1)/bo)*l[i]*x^(-(v+i-1)/bo):
  if x0 = complexInfinity and dir = Right then
    if Flag(s) = 0 then
      l := [(-1)^(2*(Valuation(s)+i-1)/BranchOrder(s))*l[i]
            $ i=1..nops(l)];
    end_if;
  elif x0 = -infinity then
    // Similarly, for x0 = -infinity, we also need to replace
    //  l[i] (--> l[i]*X^((v+i-1)/bo) = l[i]*(-1/x)^((v+i-1)/bo)) by
    //  (-1)^((v+i-1)/bo)*l[i] --> (-1)^(2*(v+i-1)/bo)*l[i]*x^(-(v+i-1)/bo):
    if Flag(s) = 0 then
      l := [(-1)^((Valuation(s)+i-1)/BranchOrder(s))*l[i]
            $ i=1..nops(l)];
    end_if;
  end_if:

  if domtype(x0) = stdlib::Infinity then
    extsubsop(s, 5 = l, 6 = (x = complexInfinity), 7 = dir)
  else
    extsubsop(s, 5 = l, 6 = (x = x0), 7 = dir)
  end_if
end_proc:


//
// change of variable: transform the series s around X = complexInfinity
// to a series around x = 0, i.e., substitute 1/x if for X
//
Puiseux::set_var2:= proc(s, X, x)
    local l, s0, ln_simplify, dir, i;
begin
    dir := Direction(s);

    if not contains({Left, Right}, dir) and BranchOrder(s) > 1 then
      // convert to Flag = 1
      s := Puiseux::convert01(s)
    end_if;

    // The coeffs may contain trancendental functions such as
    // s = (bo,v,Err,[exp(X), ln(X),..])
    //   -> exp(X)*X^(v/bo) + ln(X)*X^((v+1)/bo) +...

    // First, substitute the arguments of the trancendental
    // functions in the list.
    l := eval(subs(ListCoeffs(s), X = 1/x));

    if contains({Left, Right}, dir) then
      // directed expansion
      dir := Series::invert(dir);
      s0 := Series::sign(dir);

      // transform ln(1/x) -> -ln(x) etc. in the coefficients
      ln_simplify := proc(e)
        local s, b, k, t;
      begin
        if e = -s0*x then
          return(I*PI + hold(ln)(-e))
        end_if;
        // check whether e = (-1)^s*(+-s0*x^t)^k,
        // with s=0,1 and t=1,-1 and rational k
        s := 0;
        b := e;
        if type(-e) = "_power" then
          s := 1;
          b := -e;
        end_if;
        if type(b) = "_power" then
          [b, k] := [op(b)];
          t := 1;
          if contains({x, -x}, 1/b) then
             t := -1;
             b := 1/b;
          end_if;
          // now e = (-1)^s * (b^t)^k
          if contains({DOM_INT, DOM_RAT}, domtype(k)) then
            if b = s0*x then
              return(s*I*PI + t*k*hold(ln)(b))
            elif b = -s0*x then
              // e = (-1)^s * (-(-b)^t)^k = (-1)^(s+k) * (-b)^(t*k)
              // reduce s + k modulo 2
              return((1 - 2*frac((1 - s - k)/2))*I*PI
                     + t*k*hold(ln)(-b))
            end_if
          end_if
        end_if;
        hold(ln)(e)
      end_proc:
      l := eval(subs(l, hold(ln) = ln_simplify));
    end_if;

    // For dir = Left (<-> -infinity),
    // we also need to replace
    //  l[i] (-> l[i]*X^(-(v+i-1)/bo) = l[i]*(1/x)^(-(v+i-1)/bo))
    // with v = Valuation(s), bo = BranchOrder(s) by
    //  (-1)^(-2*(v+i-1)/bo)*l[i] -> (-1)^(-2*(v+i-1)/bo)*l[i]*x^((v+i-1)/bo)):
    if dir = Left then
       if Flag(s) = 0 then
          l := [(-1)^(-2*(Valuation(s)+i-1)/BranchOrder(s))*l[i]
                $ i=1..nops(l)];
       end_if;
    end_if:

    extsubsop(s, 5 = l, 6 = (x = 0), 7 = dir)
end_proc:


//
// converts series into expression
// (28.04.00, TL)
//
Puiseux::expr := proc(s)
   local fl, l, bo, v, i;
begin
   //
   // Flag inserted
   // (28.04.00, TL)
   //
   fl := Flag(s);
   bo := BranchOrder(s);
   v := Valuation(s);
   l := ListCoeffs(s);

   //
   // check whether Flag
   // = 0    -> l[i]*x^((v+i-1)/bo)
   // = 1    -> l[i]
   // (28.04.00, TL)
   //
   if fl = 0 then
      _plus(l[i]*Base(s, (v + i - 1)/bo) $ i = 1..nops(l));
   else
      _plus(op(l));
   end_if;
end_proc:


//
// prints a series
// (13.04.00, TL)
// CHANGED (jngerhar) 26.02.01
//
Puiseux::print:= proc(s)
   local fl, l, e, bo, v, mymult, i;
begin
   fl := Flag(s);
   bo := BranchOrder(s);
   v := Valuation(s);
   l := ListCoeffs(s);

   //
   // The following procedure serves to avoid the effects
   // - O(1)*x  --> O(x)
   // - 2*(x-1) --> 2*x-2
   // in the output
   //
   // a is a coefficient of s and b is a power of x
   //
   mymult := proc(a, b) begin
     if iszero(a) or b = 1 then
       a
     elif domtype(a) = O then
       hold(_mult)(a, b)
     elif type(b) = "_plus" and testtype(a, Type::Numeric) then
       if a = 1 then
         b
       else
         hold(_mult)(b, a)
       end_if
     else
       a * b
     end_if
   end_proc:

   //
   // if Flag =
   // 0 :  multiply with power of x
   // 1 :  nothing
   //
   if fl = 0 then
      l := [mymult(l[i], Base(s, (v + i - 1)/bo)) $ i = 1..nops(l)];
   end_if;

   // remove zeroes from l
   l := select(l, _unequal, 0);

   if Series::printO() then
     e := hold(O)(generate::sortSums(Base(s, Err(s)/bo)));
   else
     e := 0;
   end_if;

   if l = [] then
      e
   else
     l := map(l, generate::sortSums);
     if Series::printO() then
       hold(_plus)(op(l), e)
     elif nops(l)=1 then
       l[1]
     else
       hold(_plus)(op(l));
     end_if;
   end_if
end_proc:


Puiseux::expr2text :=
x -> "Series::Puiseux::create(".expr2text(extop(x, 2..5)).", ".
    expr2text(op(extop(x, 6), 1)).", ".expr2text(op(extop(x, 6), 2)).", ".
    expr2text(extop(x, 7)). ")":
  
// for generate::TeX
Puiseux::TeX := x -> generate::TeX(Puiseux::print(x)):


//
// expansion of procedure "testtype" for "Series::Puiseux"
//
Puiseux::testtype:= 
proc(s, T)
begin
   // domtype(s) = T checked by the kernel
   if T = dom or T = Type::Set then
      FALSE
   elif T = Type::Arithmetical then
      TRUE
   else
      FAIL
   end_if:
end_proc:


// map expand to the coefficients and normalize the result
// overloads expand
Puiseux::expand := proc(s)
begin
  Puiseux::normal(
    extsubsop(s, 5 = map(ListCoeffs(s), expand, args(2..args(0)))),
    0) // quick normalization
end_proc:


// map combine to the coefficients and quick normalize the result
// overloads combine
Puiseux::combine := proc(s)
begin
  Puiseux::normal(
    extsubsop(s, 5 = map(ListCoeffs(s), combine, args(2..args(0)))),
    0) // quick normalization
end_proc:


//
// Puiseux::subs(s, f = g): syntactical substitution
// Puiseux::subs(s, [f1 = g1, f2 = g2, ...])
//
// s: Series::Puiseux
// f, g, f1, g1, f2, g2, ... : arbitrary objects
//
// if the second argument is a list, then the series variable x
// must not occur in one of f1, f2, ...
//
// arbitrary substitutions are allowed in the coefficients
//
// if the series variable x occurs in f, then only the following
// substitutions are allowed:
//
// - x = a*(b*y-b*c)^k + Point(s)         if Point(s) is finite,
// - x = a*(b*y-b*c)^k                    if Point(s) = complexInfinity,
//
// where:
//
// - k is a nonzero rational number,
//   and an odd integer if s(a*x) is only valid for x < 0
// - y is a variable (may equal x)
// - a, b, c are constant expressions and a, b are nonzero,
//   c = 0 if p <> complexInfinity and k < 0 or
//            p = complexInfinity and k > 0
//   if Direction(s) <> Undirected, then a, b must be real
//
Puiseux::subs:= proc(s, sub)
    local fl, bo, v, l, n, x, p, d, i,
          f, g, ng, y, a, b, c, k, xIsPositive;
begin

    if domtype(sub) = DOM_LIST then
      f := map(sub, op, 1);
      g := map(sub, op, 2);
    else
      [f, g] := [op(sub)];
    end_if;

    x := Variable(s);

    if not has(f, x) then
       // l.h.s. of the substitution does not involve x
       if has(g, x) then
         // x occurs on the r.h.s. of the substitution
         // (e.g., subs(series(1/(x + y), x), y = x))
         Series::error("illegal substitution: rhs contains series variable");
       end_if;
       // map "subs" to the coefficients and the expansion point
       p := evalAt(Varpoint(s), sub);
       return(Puiseux::map(extsubsop(s, 6 = p), evalAt, sub)):

    elif f <> x then
      Series::error("illegal substitution")

    else // substitution for the series variable x

      p := Point(s);
      l := ListCoeffs(s);

      // check that g - p is of the form a*(b*y - b*c)^k
      if p <> complexInfinity then
        g := g - p;
      end_if;
      ng := normal(g); // only useful for k = 1
      y := op(indets(ng) minus Type::ConstantIdents);
      if nops(y) <> 1 then
        Series::error("illegal substitution: exactly one indeterminate expected");
      elif x <> y and has(l, y) then
        // y occurs in the coefficients
        // (e.g., subs(series(1/(x + y), x), x = y))
        Series::error("illegal substitution: coefficients contain the new variable");
      end_if;
      a := 1;
      if type(g) = "_mult" then
        g := split(g, has, y);
        a := g[2]*g[3];
        g := g[1];
      elif type(ng) = "_mult" then
        g := split(ng, has, y);
        a := g[2]*g[3];
        g := g[1];
      end_if;
      ng := normal(g); // only useful for k = 1
      k := 1;
      if type(g) = "_power" then
        [g, k] := [op(g)];
        g := normal(g);
      elif type(ng) = "_power" then
        [g, k] := [op(ng)];
      end_if;
      if not testtype(k, Type::Rational) then
        Series::error("illegal substitution: rational exponent expected")
      end_if;
      if not testtype(g, Type::PolyExpr([y])) then
        Series::error("illegal substitution")
      end_if;
      g := poly(g, [y]);
      if g = FAIL or degree(g) > 1 then
        Series::error("illegal substitution")
      end_if;
      b := coeff(g, 1);
      c := -coeff(g, 0)/b;
      if (p <> complexInfinity and k < 0) or
         (p = complexInfinity and k > 0) then
        if c <> 0 then
          Series::error("illegal substitution")
        end_if;
        c := complexInfinity;
      end_if;
      // c is the new expansion point

      fl := Flag(s);
      bo := BranchOrder(s);
      v := Valuation(s);
      n := Err(s);
      d := Direction(s);

/* // Walter (11.11.05): diesen Check rausgenommen, da sonst z.B.
   // series(arctan(x), x = I*infinity =
   // subs(series(arctan(I*x), x = infinity), x = x/I)
   // nicht funktioniert
      if d <> Undirected and
        is(a, Type::Real) <> TRUE or is(b, Type::Real) <> TRUE then
         // e.g., I*x is to be substituted for x in a real expansion
         Series::error("illegal substitution for a directed series expansion")
      end_if;
*/

      xIsPositive := UNKNOWN;
      if d = Right then
        xIsPositive := bool(p <> complexInfinity)
      elif d = Left then
        xIsPositive := bool(p = complexInfinity)
      end_if;

      /*
       * 1) perform the substitution x -> a*x
       */

      if d <> Undirected and is(a < 0) = TRUE then
        // change direction of expansion
        d := Series::invert(d);
        xIsPositive := not xIsPositive;
      end_if;

      if fl = 0 and nops(l) > 0 and a <> 1 then
        // multiply coefficients by powers of a
        if bo = 1 or (is(a >= 0) or xIsPositive) = TRUE then
          if p <> complexInfinity then
            l := [a^( (v + i - 1)/bo)*l[i] $ i = 1..nops(l)];
          else
            l := [a^(-(v + i - 1)/bo)*l[i] $ i = 1..nops(l)];
          end_if
        elif xIsPositive = FALSE then // x < 0 and a < 0
          if p <> complexInfinity then
            l := [(-1)^(-(v+i-1)/bo)*(-a)^( (v+i-1)/bo)*l[i] $ i = 1..nops(l)];
          else
            l := [(-1)^( (v+i-1)/bo)*(-a)^(-(v+i-1)/bo)*l[i] $ i = 1..nops(l)];
          end_if
        else
          // e.g., -x is to be substituted in x^(1/2) --> convert to Flag = 1
          return(Puiseux::subs(Puiseux::convert01(s), sub));
        end_if
      end_if;

      /*
       * 2) perform the substitution x -> x^k
       */

      if xIsPositive = FALSE /* x < 0 */ and domtype((k - 1)/2) <> DOM_INT then
         // e.g., x^(1/2) or x^2 is to be substituted for (x when x < 0)
         Series::error("illegal substitution: odd exponent expected")
      end_if;

      if domtype(k) <> DOM_INT and d = Real then
        // change direction to positive
        if p <> complexInfinity then
          d := Right
        else
          d := Left
        end_if;
        xIsPositive := TRUE;
      elif domtype(k/2) = DOM_INT and d <> Undirected then
        // change direction to real
        d := Real;
        xIsPositive := UNKNOWN;
      end_if;
      if k < 0 then
        // invert direction
        d := Series::invert(d);
      end_if;

      if fl = 0 and bo > 1 and (k <= -1 or k > 1) then
        // multiply coefficients by fractional powers of -1 if
        // necessary and possible
        case xIsPositive
        of UNKNOWN do
          // e.g., x^2 or 1/x is to be substituted in x^(1/2) in
          // an undirected expansion --> convert to Flag = 1
          return(Puiseux::subs(Puiseux::convert01(s), sub));
        of FALSE do // x < 0
          if p <> complexInfinity and k > 0 then
            // l[i] *                          (x^k)^(   (v+i-1)/bo)) =
            // l[i] * ((-1)^k)^( (v+i-1)/bo)
            //      * (-1)^(-k*(v+i-1)/bo)   *   x  ^( k*(v+i-1)/bo))
            l := [l[i] * ((-1)^k)^( (v+i-1)/bo)
                       * (-1)^(-k*(v+i-1)/bo)   $ i=1..nops(l)];
          else // p = complexInfinity
            // l[i] *                          (x^k)^(-  (v+i-1)/bo)) =
            // l[i] * ((-1)^k)^(-(v+i-1)/bo)
            //      * (-1)^( k*(v+i-1)/bo)   *   x  ^(-k*(v+i-1)/bo))
            l := [l[i] * ((-1)^k)^(-(v+i-1)/bo)
                       * (-1)^( k*(v+i-1)/bo)   $ i=1..nops(l)];
          end_if;
          break;
        // nothing needs to be done if x > 0, since then
        // (x^k)^((v+i-1)/bo)) = x^(k*(v+i-1)/bo))
        end_case
      end_if;

      if abs(numer(k)) > 1 and nops(l) > 0 then
        // insert zeroes into the coefficient list
        l := [l[1], ((0 $ abs(numer(k)) - 1), l[i]) $ i = 2..nops(l)];
      end_if;

      if domtype(k) <> DOM_INT then
        // change branching order if k is not an integer
        bo := bo * op(k, 2);
        k := op(k, 1);
      end_if;

      v := v*abs(k);
      n := n*abs(k);

      /*
       * 3) perform the substitution x -> b*x
       *    (code is completely analogous to 1) above)
       */

      if d <> Undirected and is(b < 0) = TRUE then
        // change direction of expansion
        d := Series::invert(d);
        xIsPositive := not xIsPositive;
      end_if;

      if fl = 0 and nops(l) > 0 and b <> 1 then
        if bo = 1 or (is(b >= 0) or xIsPositive) = TRUE then
          // multiply coefficients by powers of b
          if c <> complexInfinity then
            l := [b^( (v + i - 1)/bo)*l[i] $ i = 1..nops(l)];
          else
            l := [b^(-(v + i - 1)/bo)*l[i] $ i = 1..nops(l)];
          end_if;
        elif xIsPositive = FALSE then // x < 0 and a < 0
          if c <> complexInfinity then
            l := [(-1)^(-(v+i-1)/bo)*(-b)^( (v+i-1)/bo)*l[i] $ i = 1..nops(l)];
          else
            l := [(-1)^( (v+i-1)/bo)*(-b)^(-(v+i-1)/bo)*l[i] $ i = 1..nops(l)];
          end_if
        else
          // e.g., -x is to be substituted in x^(1/2) --> convert to Flag = 1
          return(Puiseux::subs(Puiseux::convert01(s), sub));
        end_if
      end_if;

      // y is the new series variable and c is the new expansion point
      s := extsubsop(s, 2 = bo, 3 = v, 4 = n, 5 = l, 6 = (y = c), 7 = d);

      if has(l, x) then
        // old series variable occurs in the coefficients
        // this includes the case Flag = 1
        // ==> substitute also in the coefficients and normalize the result
        s := Puiseux::normal(extsubsop(s, 5 = map(l, evalAt, sub)),
                             0) // quick normalization
      end_if;

      return(Puiseux::condense(s))

    end_if;
end_proc:


//
// Puiseux::map(s, f, ...) - apply f(  , ...) to all nonzero coefficients
//
// s: Puiseux
// f: arbitrary object representing a function
//
// Remark:
// - coeff of (a*x)^(1/2)*(b*x)^(1/2) is (a*x)^(1/2)*(b*x)^(1/2)/x
//   (05.07.00, T.L., with Juergen Gerhard)
// - if we have to apply map to the coefficients of a series expansion
//   with Flag=1, we proceed as follows:
//   - apply function to the coefficients
//     CAUTION: Be aware that the coefficients and the entries in the
//              list of coefficients differ in a certain power of x
//              EXAMPLE: entry      : (a*x)^(1/2)
//                       coefficient: (a*x)^(1/2)/x^(1/2)
//   - multiply the result with certain power of x
//              EXAMPLE: result     : sin((a*x)^(1/2)/x^(1/2))*x^(1/2)
//   (12.07.00, T.L.)
//
Puiseux::map := proc(s, f)
   local j, bo, v, l, optionalArguments;
begin
   bo := BranchOrder(s);
   v := Valuation(s);
   optionalArguments := args(3..args(0));

   //
   // if Flag=0 : apply function to the entries of the list
   // if Flag=1 : - apply function to the result of coeff()
   //             - multiply the entries of the result (list)
   //               with a certain power of x
   //             - the resulting list is the new list of the series expansion
   //
   if Flag(s) = 0 then
      Puiseux::normal(extsubsop(s, 5 = map(ListCoeffs(s),
                        z -> if iszero(z) then
                               z
                             else
                               f(z, optionalArguments)
                             end_if)),
                      0) // quick normalization
   else
      l := map([coeff(s)],
               z -> if iszero(z) then
                      z
                    else
                      f(z, optionalArguments)
                    end_if);
      l := [l[j]*Base(s, (v + j - 1)/bo) $ j = 1..nops(l)];
      Puiseux::normal(extsubsop(s, 5 = l), 0) // quick normalization
   end_if;
end_proc:


//
// procedure "has" for series expansions
//
Puiseux::has := proc(s, f)
begin
  has(ListCoeffs(s).[Variable(s), Point(s), Direction(s)], f)
end_proc:


//
// returns series with float coefficients
//
Puiseux::float := proc(s)
begin
  Puiseux::map(s, proc() begin
    (if args(1) = 0 then 0 else float(args(1)) end_if)
  end_proc)
end_proc:



// mathematical methods -------------------------------------------------------

//
// manages the addition of arbitrarily many terms
//
Puiseux::_plus :=
proc()
  local i;
  save MAXEFFORT;
begin
    case args(0)
    of 1 do return(args(1))
    of 2 do return(Puiseux::plus(args(1), args(2)))
    of 3 do
      if domtype(args(1)) = Puiseux then
        return(Puiseux::plus(Puiseux::plus(args(1), args(2)), args(3)))
      else
        return(Puiseux::plus(args(1), Puiseux::plus(args(2), args(3))))
      end_if
    otherwise // binary splitting
      MAXEFFORT:= MAXEFFORT/2;    
      _plus(_plus(args(i) $ i = 1..(args(0) div 2)),
            _plus(args(i) $ i = (1 + (args(0) div 2))..args(0)))
  end_case
end_proc:


//
// adds two series
// (13.04.00, TL)
//
Puiseux::plus :=
proc(a, b)
  local aa, bb, fl, i, va, vb, qa, qb, la, lb, dif, da, db, lc, boa, bob, dir;
begin
  if domtype(a) <> Puiseux then
      // domtype(b) = Puiseux
    return(Puiseux::plus(b, a))
  end_if;
  qa := Err(a);

  if domtype(b) <> Puiseux then
    assert(domtype(a) = Puiseux);
    if iszero(b) then
      return(a)
    end_if;
    if domtype(b) = DOM_POLY then
       b:= expr(b);
    end_if:
    if contains({DOM_EXPR, DOM_IDENT, DOM_INT,
                 DOM_RAT, DOM_FLOAT, DOM_COMPLEX, unit}, domtype(b)) and
      not has(b, Variable(a)) then
      // addition of a constant
      if qa <= 0 then
        return(a)
      else
        return(Puiseux::plus(a, extsubsop(a, 3 = 0, 5 = [b])))
      end_if
    end_if;
    // is b in a more general domain ?
    aa := b::dom::convert(a);
    if aa <> FAIL then
      return(aa + b)
    end_if;
    bb := FAIL;
    traperror((bb := Puiseux::new(b, Varpoint(a), qa - min(0, Valuation(a)),
                 Direction(a))));
    if bb <> FAIL then
      b := bb
    else
      traperror((bb := series(b, Varpoint(a), qa - min(0, Valuation(a)),
                              Direction(a), NoWarning)));
      if domtype(bb) = Puiseux then
        b := bb
      elif domtype(bb) = gseries then
        return(gseries::plus(bb, gseries::convert(a)))
      else
        Series::error("cannot convert argument");
      end_if;
    end_if:
  elif Varpoint(a) <> Varpoint(b) then
    Series::error("different series variables or expansion points")
  end_if;
  // now both a and b are of type Puiseux and have the same expansion point
  assert(domtype(a) = Puiseux);
  assert(domtype(b) = Puiseux);
  assert(Varpoint(a) = Varpoint(b));
  qb := Err(b);

  dir := Series::min(Direction(a), Direction(b));
  if dir = FAIL then
    Series::error("inconsistent directions");
  end_if;

  boa := BranchOrder(a);
  bob := BranchOrder(b);
  if boa <> bob then
    i := ilcm(boa, bob);
    return(Puiseux::plus(Puiseux::multbo(a, i/boa),
                         Puiseux::multbo(b, i/bob)));
  end_if;

  va := Valuation(a);
  vb := Valuation(b);

   //
   // set Flags (13.04.00, TL)
   //
   if vb >= qa then
      a
   elif va >= qb then
      b
   elif va > vb then
      Puiseux::plus(b, a)
   else // va <= vb then
      dif := vb - va; // dif >= 0
     
      //
      // check Flags
      // Flag(a) = Flag(b) => nothing changes
      // Flag(a) <> Flag(b) => convert and add
      // caution: coeffs and Flags change!
      // (13.04.00, TL)
      //
      if Flag(a) <> Flag(b) then
         aa := Puiseux::convert01(a);
         bb := Puiseux::convert01(b);
         la := ListCoeffs(aa);
         lb := ListCoeffs(bb);
         fl := 1;
      else
         la := ListCoeffs(a);
         lb := ListCoeffs(b);
         fl := Flag(a); // = Flag(b)
      end_if;
     
      da := nops(la);
      db := nops(lb) + dif;
      if da > db then // always overlapping
         lc := [la[i]               $ i = 1..dif,
                la[i] + lb[i - dif] $ i = (dif + 1)..db,
                la[i]               $ i = (db + 1)..min(da, qb - va)];
      elif dif <= da then // da <= db and overlapping
          lc := [la[i]               $ i = 1..dif,
                 la[i] + lb[i - dif] $ i = (dif + 1)..da,
                 lb[i - dif]         $ i = (da + 1)..min(db, qa - va)];
      else // da <= db and no overlapping
          lc := [la[i]       $ i = 1..da,
                 0           $ (dif - da),
                 lb[i - dif] $ i = (dif + 1)..min(db, qa - va)];
      end_if;
      Puiseux::normal(extsubsop(a, 1 = fl, 3 = va, 4 = min(qa, qb), 5 = lc,
                                7 = dir),
                      0); // quick normalization
   end_if;
end_proc:


//
// negates a series by changing the sign of the coefficients
//
Puiseux::_negate := proc(s)
begin
   extsubsop(s, 5 = map(ListCoeffs(s), _negate))
end_proc:


//
// subtracts y from x by adding -y
//
Puiseux::_subtract := (x, y) -> x + (-y):


//
// Puiseux::scalmult(s, a, k): computes (a*(x-x0)^k)*s
//
// s: Series::Puiseux with series variable x and expansion point x0
// a: arithmetical expression, assumed to be nonzero!
// k: rational number (optional; default: 0)
//
// CHANGED (jngerhar) 23.02.01: now also works correctly for Flag = 1
//   and for rational k
//
Puiseux::scalmult := proc(s, a, k = 0)
   local bo, bn, l, i;
begin
   bo := BranchOrder(s);
   if Flag(s) = 0 or k = 0 then
     l := map(ListCoeffs(s), _mult, a);
   else
     l := map(ListCoeffs(s), _mult, a*Base(s, k));
   end_if;
   if k = 0 then
      extsubsop(s, 5 = l)
   elif domtype(k*bo) = DOM_INT then
      extsubsop(s, 5 = l, 3 = Valuation(s) + k*bo, 4 = Err(s) + k*bo)
   elif domtype(k) = DOM_RAT then
      // new branch order
      bn := ilcm(bo, denom(k));
      if nops(l) > 0 and bn > bo then
	 // branch order has increased, so insert zeroes into
         // the coefficient list
         l := [l[1], ((0 $ bn/bo - 1), l[i]) $ i = 2..nops(l)];
      end_if;
      extsubsop(s, 5 = l, 2 = bn, 3 = Valuation(s)*bn/bo + k*bn,
                   4 = Err(s)*bn/bo + k*bn)
   else
      Series::error("third argument must be rational");
   end_if
end_proc:


//
// multiplies two series
// (13.04.00, TL)
//
Puiseux::mult:=
proc(a, b)
  local aa, bb, fl, va, vb, qa, qb, da, db, la, lb, i, j, i1, i2,
  l1, l2, l3, imax, boa, bob, dir;
begin
  if domtype(a) <> Puiseux then
    // domtype(b) = Puiseux
    return(Puiseux::mult(b, a))
  elif domtype(b) <> Puiseux then
    // domtype(a) = Puiseux
    case b
      of -1 do
        return(-a)
      of 0 do
        return(extsubsop(a, 1 = 0, 3 = Err(a), 5 = [])); // O(..)
      of 1 do
        return(a)
    end_case;
    if contains({DOM_EXPR, DOM_IDENT, DOM_INT,
                 DOM_RAT, DOM_FLOAT, DOM_COMPLEX}, domtype(b)) then
      if not has(b, Variable(a)) then
        // b does not contain the series variable ==> scalar multiplication
        return(Puiseux::scalmult(a, b))
      end_if;
         //
         // Try to convert via new. This only works, if b is an expression
         // that is a Puiseux series (mathematically)
         //
      bb := FAIL;
      traperror((bb := Puiseux::new(b, Varpoint(a), Err(a) - Valuation(a),
                                    Direction(a))));
      if bb <> FAIL then
        b := bb
      else
        traperror((bb := series(b, Varpoint(a), Err(a) - Valuation(a), Direction(a), NoWarning)));
        if domtype(bb) = Puiseux then
          b := bb;
        elif domtype(bb) = gseries then
          gseries::mult(bb, gseries::convert(a))
        end_if
      end_if:
    else // b may be of a more general domain (Series::gseries)
      aa := b::dom::convert(a);
      if aa <> FAIL then
        return(b::dom::_mult(aa, b))
      end_if;
    end_if;
    if domtype(b) <> Puiseux then
      Series::error("cannot convert argument");
    end_if:
  end_if;
  // now both a and b are of type Puiseux
  assert(domtype(a) = Puiseux);
  assert(domtype(b) = Puiseux);
  
  if Varpoint(a) <> Varpoint(b) then
    Series::error("different series variables or expansion points")
  end_if;

  dir := Series::min(Direction(a), Direction(b));
  if dir = FAIL then
    Series::error("inconsistent directions");
  end_if;

  boa := BranchOrder(a);
  bob := BranchOrder(b);
  if boa <> bob then
    i := ilcm(boa, bob);
    return(Puiseux::mult(Puiseux::multbo(a, i/boa),
                         Puiseux::multbo(b, i/bob)))
  end_if;

  va := Valuation(a);
  vb := Valuation(b);
  qa := Err(a);
  qb := Err(b);

  //
  // check Flags
  // Flag(a) = Flag(b) => nothing changes
   // Flag(a) <> Flag(b) => convert and multiply
  // caution: coeffs and Flags change! => aa, bb,
  // (13.04.00, TL)
   //
  if Flag(a) <> Flag(b) then
    aa := Puiseux::convert01(a);
    bb := Puiseux::convert01(b);
    la := ListCoeffs(aa);
    lb := ListCoeffs(bb);
    fl := 1;
   else
      la := ListCoeffs(a);
      lb := ListCoeffs(b);
      fl := Flag(a); // = Flag(b)
   end_if;

   da := nops(la);
   db := nops(lb);
   imax := min(qa - va, qb - vb); // maximal number of terms in the product
   if da <= db then
      i1 := da;
      i2 := db
   else
      i1 := db;
      i2 := da
   end_if;

   l1 := _plus((la[j]*lb[i + 1 - j]) $ j = 1..i) $ i = 1..i1;
   if da <= db then
      l2 := _plus((la[j]*lb[i + 1 - j]) $ j = 1..da)
            $ i = (da + 1)..min(db, imax);
   else
      l2 := _plus((la[i + 1 - j]*lb[j]) $ j = 1..db)
            $ i = (db + 1)..min(da, imax);
   end_if;
   l3 := _plus((la[j]*lb[i + 1 - j]) $ j = (i + 1 - db)..da)
         $ i = (i2 + 1)..min(da + db - 1, imax);
//      extsubsop(a,1=fl,3=va+vb,4=min(va+qb,vb+qa),5=map([l1,l2,l3],expand));
// CHANGED (jngerhar) 17.11.00: removed expand
   Puiseux::normal(extsubsop(a, 1 = fl, 3 = va + vb,
                             4 = min(va + qb, vb + qa), 5 = [l1, l2, l3],
                             7 = dir),
                   0) // quick normalization
end_proc:


//
// Puiseux::homog_mult: auxilliary procedure used only by Puiseux::_mult.
// Takes arbitrarily many arguments and recursively multiplies each two
// together via Puiseux::mult.
//
Puiseux::homog_mult := misc::genassop(Puiseux::mult, 1):


//
// mangages the multiplication of arbitrarily many series
//
Puiseux::_mult := proc()
  local s, others, dummy, ss;
begin
  [s, others, dummy] := split([args()], x -> domtype(x) = Puiseux);
  s:= Puiseux::homog_mult(op(s));
  others:= _mult(op(others)):
  if domtype(others) = DOM_POLY then
     others:= op(others, 1):
     ss:= FAIL:
     traperror((
        ss:= Puiseux::new(others, Varpoint(s), Err(s) - Valuation(s), Direction(s))
     ));
     if ss = FAIL then 
        ss:= series(others, Varpoint(s), Err(s) - Valuation(s), Direction(s), NoWarning);
     end_if; 
     others:= ss;
  end_if;
  Puiseux::mult(s, others)
end_proc:


//
// computes a^2
// (not used currently)
//
Puiseux::square := proc(s)
   local v, n, d, l, l1, l3, imax, i, j;
begin
   v := Valuation(s);
   n := Err(s);
   l := ListCoeffs(s);
   d := nops(l);
   imax := n - v; // maximal number of terms in the product
   l1 := _plus(l[j]*l[i + 1 - j] $ j = 1..i) $ i = 1..d;
   l3 := _plus(l[j]*l[i + 1 - j] $ j = i + 1 - d..d)
         $ i = (d + 1)..min(2*d - 1, imax);
   Puiseux::normal(extsubsop(s, 3 = 2*v, 4 = v + n, 5 = [l1, l3]),
                   0) // quick normalization
end_proc:


//
// divides a by b
//
// Idea: invert b and multiply
//
Puiseux::_divide := (a, b) -> a * _invert(b):


//
// Puiseux::_invert(s): invert the series
// (is about twice faster than Puiseux::power(s, -1))
//
// s: Series::Puiseux
//
Puiseux::_invert := proc(s)
   local v, n, d, l, i, a1, b, j;
begin
   s := Puiseux::normal(s, 1); // normalize using Puiseux::isero as zero test

   l := ListCoeffs(s);
   d := nops(l);
   n := Err(s);
   if d = 0 then // 1/O(x^n) -> division by zero
      Series::error("order too small")
   else
      //
      // 1/(a1*x^v + ... + ad*x^(v+d-1) + O(x^n))
      // = x^(-v)/(a1+ ... +ad*x^(d-1) + O(x^(n-v)))
      //
      a1 := l[1];
      v := Valuation(s);
      b := table();

      // inversion formula
      b[1] := 1/a1;
      for i from 3 to d + 1 do
        b[i - 1] := -_plus(l[j]*b[i - j] $ j = 2..i - 1)/a1
      end_for;
      for i from d + 2 to n - v + 1 do
        b[i - 1] := -_plus(l[j]*b[i - j] $ j = 2..d)/a1
      end_for;
      Puiseux::normal(extsubsop(s, 3 = -v, 4 = n - 2*v,
                                5 = [b[i] $ i = 1..n - v]),
                      0) // quick normalization
   end_if;
end_proc:


//
// Puiseux::_power(s, n): computes s^n
//
// s: Puiseux
// n: rational in general,
//    arbitrary arithmetical expression not containing the series variable
//    if s = const + O(..)
//
// optional arguments
// f: If present, s is the series expansion of f
//
Puiseux::_power := proc(s, n: Type::Arithmetical, f = FAIL)
begin
   if s = FAIL then
      return(s)
   end_if;

   if has(n, Variable(s)) then
     Series::error("exponent must not contain the series variable");
   end_if;

   // strip leading and trailing zeroes, using Puiseux::iszero as zero test
   s := Puiseux::normal(s, 1);

   if domtype(n) <> DOM_INT then
      if domtype(n) = DOM_RAT or
         (Valuation(s) = 0 and nops(ListCoeffs(s)) > 0
          and not has(op(ListCoeffs(s), 1), Variable(s))) then
        Puiseux::power(s, n, f)
      else
        Series::error("exponent must be a rational number");
      end_if;
   elif n < 0 then
      if n = -1 then
         Puiseux::_invert(s)
      else
         Puiseux::_invert(Puiseux::_power(s, -n))
      end_if
   elif n = 0 then // return 1
      Puiseux::one(Varpoint(s), Err(s) - Valuation(s), Direction(s))
   elif n = 1 then
      s
   //
   // changed (03.03.00, TL)
   // repeated squaring leads to coefficients such as
   // (a*x)(a*x)^(1/2)
   // -> use power(s,n)
   //
   //elif n mod 2 = 0 then Puiseux::_power(Puiseux::square(s),n div 2)
   //else Puiseux::mult(s,Puiseux::_power(Puiseux::square(s),n div 2))
   else Puiseux::power(s, n);
   end_if;
end_proc:


//
// Puiseux::power(s, n): computes s^n
//
// s: Puiseux
// n: rational in general,
//    arbitrary arithmetical expression not containing the series variable
//    if s = const + O(..)
//
// optional arguments
// f: If present, s is the series expansion of f
//
// from Zippel, page 165
//
// Idea: write s = a*x^(v/bo) * (1 + ...). Then s^n = B * C,
//       where B = (a*x^(v/bo))^n and C = (1 + ...)^n
// B has branching order denom(n*v/bo) and C has branching order bo
// If v = 0 and is(a < 0) = TRUE and n is not an integer
// (expansion on the branch cut):
//       s^n = (-a)^n * C * (-1)^(n*signIm(s))
//          or (-a)^n * C * (-1)^(n*signIm(f)) if f is present
//
// (08.05.00, TL)
// CHANGED (jngerhar) 23.02.01, 02.03.01:
// - restructured and cleaned up
//
Puiseux::power := proc(s, n, f = FAIL)
   local fl, bo, v, l, x, ll, k, q, d, l0, A, B, C, bn, a, aneg, res, nmod2,
         dir, j;
begin
   fl := Flag(s);
   bo := BranchOrder(s);
   v := Valuation(s);
   l := ListCoeffs(s);
   x := Base(s, 1);
   dir := Direction(s);

   //
   // test if s is a zero-series
   // (15.08.00, T.L.)
   //
   if Puiseux::iszero(s) then // O(x^(n*Err(s)/bo))
      if testtype(n, Type::Rational) then
        extsubsop(s, 1 = 0, 2 = bo*denom(n), 3 = Err(s)*numer(n),
                     4 = Err(s)*numer(n), 5 = []);
      else
        Series::error("exponent must be a rational number");
      end_if
   else

      //
      // if Valuation is not zero,
      // then we have to use the formula above
      //
      if v <> 0 then // implies n rational

         //
         // if Flag=0 then
         //    - compute C
         //    - get the power of x in B
         //
         if fl = 0 then
            //
            // first case: n is an integer
            //
            if domtype(n) = DOM_INT then
               // C := (s / lterm(s))^n
               C := Puiseux::power(extsubsop(s, 3 = 0, 4 = Err(s) - v), n);
               // return C * lterm(s)^n
               Puiseux::scalmult(C, 1, v/bo*n)
            //
            // second case: n is rational
            //    - set B := lmonomial(s)^n
            //    - compute C from formula via power
            //    - multiply B and C
            //        lterm(s)^n divides B: Flag=0
            //        otherwise           : Flag=1
            //
            else // domtype(n) = DOM_RAT
               // C := (s / lmonomial(s))^n
               C := Puiseux::power(extsubsop(s, 3 = 0, 4 = Err(s) - v,
                                             5 = map(l, _mult, 1/l[1])),
                                   n);
//               B := simplify((l[1]*x^(v/bo))^n);
               B := ((l[1]*Base(s, v/bo))^n);
               A := 1/combine(l[1]^n*Base(s, v/bo*n)/B);
               if not has(A, Variable(s)) then
                   // B = A * l[1]^n * x^(v/bo*n), use Flag = 0
                   // return C * A * l[1]^n * x^(v/bo*n)
                   Puiseux::scalmult(C, A*l[1]^n, v/bo*n)
               elif (dir = Right and Point(s) <> complexInfinity) or
                    (dir = Left and Point(s) = complexInfinity) or
                    is(x >= 0) = TRUE or
                    (is(l[1] >= 0) = TRUE and -1 < v/bo and v/bo <= 1) then
                   Puiseux::scalmult(C, l[1]^n, v/bo*n)
               elif dir = Right and Point(s) = complexInfinity then
                   Puiseux::scalmult(C,
                     ((-1)^(-v/bo)*l[1])^n*(-1)^( v/bo*n),
                     v/bo*n)
               elif (dir = Left or is(x < 0) = TRUE) and
                    Point(s) <> complexInfinity then
                   Puiseux::scalmult(C,
                     ((-1)^( v/bo)*l[1])^n*(-1)^(-v/bo*n),
                     v/bo*n)
               else
                   // B <> l[1]^n * x^(v/bo*n), use Flag = 1
                   // new branching order
                   bn := ilcm(BranchOrder(C), op(n, 2));
                   // convert to Flag = 1 and to the new branching order
                   C := Puiseux::convert01(Puiseux::multbo(C, bn/bo));
                   // return C * B
                   extsubsop(C, 2 = bo*extop(C, 2),
                             3 = v*n*bn, 4 = bo*Err(C) + v*n*bn,
                             5 = map(ListCoeffs(C), _mult, B));
               end_if;
            end_if;

         //
         // if Flag=1 then
         //    - compute power B of leading monomial l[1]
         //    - compute C := (s/l[1])^n
         //    - multiply every coefficient of C by B
         //
         else // fl = 1

            // B := lmonomial(s)^n
            B := l[1]^n;

            // C := (s/lmonomial(s))^n
            C := Puiseux::power(extsubsop(s, 3 = 0, 4 = Err(s) - v,
                                          5 = map(l, _mult, 1/l[1])),
                                n);

            // convert C to new branching order if necessary
            bn := BranchOrder(C);
            if domtype(v/bo*n*bn) <> DOM_INT then
              //--------------------------------------------------------------
              // Walter: bug fix 24.6.03, following jgerhar's advice via email
              // Fixes:  series(((-x^7)^(1/2) - x)^(1/3),x);
              // bn := op(v/bo*n, 2);
              bn := ilcm(op(v/bo*n, 2), bn);
              //--------------------------------------------------------------
              C := Puiseux::multbo(C, bn/bo);
            end_if;

            // return C * B
            extsubsop(C, 3 = v/bo*n*bn, 4 = Err(C) + v/bo*n*bn,
                         5 = map(ListCoeffs(C), _mult, B));
         end_if;

      else // v = 0 and s <> 0
         a := coeff(s, 0);
         aneg := bool(domtype(n) = DOM_RAT and is(a < 0) = TRUE);
         if aneg then
           l := map(l, _negate);
         end_if;

         q := Err(s); // error term is O(x^(q/bo))
         d := nops(l);
         //
         // use binomial theorem to compute the power
         //
         l0 := l[1]; // = +-a
         ll[1] := l0^n;
         for k from 1 to q - 1 do
            ll[k + 1] := _plus((j*n - k + j)*l[j + 1]*ll[k + 1 - j]
                               $ j = 1..min(k, d - 1)) /l0/k;
         end_for;
         res := Puiseux::normal(extsubsop(s, 5 = [ll[k] $ k = 1..q]),
                                0); // quick normalization
         nmod2 := 2*frac((n+1)/2) - 1; // n modulo 2
         if not aneg then
           res
         elif f <> FAIL then
           Puiseux::scalmult(res, (-1)^(nmod2*signIm(f)))
         else
           Puiseux::scalmult(res, (-1)^(nmod2*signIm(s)))
         end_if
      end_if;
   end_if;
end_proc:


//
// Puiseux::_fconcat(P, Q): compute the functional composition
// of the two Puiseux series P and Q. Q must tend to Point(P)
// for the composition to be valid.
//
// (28.04.00, TL)
// Order: OK
// (12.07.00, T.L.)
// Use old implementation (_fconcat0) to deal with Flag=0
// => more efficient
// (01.08.00, T.L.)
//
// CHANGED (jngerhar) 25.02.01:
// - corrected checks for invalid compositions
// - composition with O() now should work correctly
// - Point(P) <> 0 should now work correctly
//
Puiseux::_fconcat := proc(P, Q)
   local bP, bQ, vP, vQ, errP, errQ, varP, varQ, P_expr, Q_expr, PQ_expr,
         errPQ, res, y, y0, msg;
begin
   case args(0)
   of 1 do
     return(P)
   of 2 do
     break
   otherwise
     return(_fconcat(P, _fconcat(args(2..args(0)))))
   end_case;

   msg := "invalid composition";

   if domtype(Q) <> Puiseux then
     // domtype(P) = Series::Puiseux
     // try to convert Q into a series around y = 0
     y := op(indets(Q) minus Type::ConstantIdents);
     if nops(y) <> 1 then
       Series::error("can't compute composition")
     end_if;
     traperror((Q := Puiseux::new(Q, y, Err(P) - Valuation(P))));
     msg := "can't compute composition";
   elif domtype(P) <> Puiseux then
     // domtype(Q) = Series::Puiseux
     // try to convert P into a series around y = Q(Point(Q))
     y := op(indets(P) minus Type::ConstantIdents);
     if nops(y) <> 1 then
       Series::error("can't compute composition")
     end_if;
     vQ := ldegree(Q);
     if vQ = FAIL then
       if Err(Q) <= 0 then
         Series::error("order too small");
       else
         vQ := Err(Q);
       end_if;
     end_if;
     if vQ > 0 then
       y0 := 0
     elif vQ = 0 then
       y0 := lcoeff(Q)
     else // vQ < 0
       y0 := complexInfinity
     end_if;
     traperror((P := Puiseux::new(P, y = y0, Err(Q) - Valuation(Q))));
   end_if;

   if {domtype(P), domtype(Q)} <> {Puiseux} then
     Series::error("can't compute composition");
   end_if;

   varP := Variable(P);
   varQ := Variable(Q);

   if Point(P) = complexInfinity then
     // P @ Q = P(1/x) @ 1/Q
     return(Puiseux::_fconcat(
       Puiseux::set_var2(P, varP, varP),
       Puiseux::_invert(Q)))
   end_if;
   // now Point(P) is finite

   // remove leading and trailing zeroes
   P := Puiseux::normal(P, 0); // quick normalization

   bP := BranchOrder(P);
   bQ := BranchOrder(Q);
   vP := Valuation(P);
   vQ := Valuation(Q);
   errP := Err(P);
   errQ := Err(Q);
   if Puiseux::iszero(P) then
     vP := errP;
   end_if;
   if Puiseux::iszero(Q) then
     vQ := errQ;
   end_if;

   //
   // test whether is _fconcat is a valid operation
   //
   if vQ < 0 or (vQ = 0 and Point(P) = 0) then
      Series::error(msg);
   elif Point(P) <> 0 then
      //
      // Point(P) = a <> 0, therefore Q must tend to a
      //
      Q := lmonomial(Q, Rem);
      if Q = FAIL or vQ <> 0 or Q[1] <> Point(P) then
         Series::error(msg);
      end_if;
      //
      // P @ Q = P(x + a) @ (Q - a)
      //
      Q := Q[2];
      vQ := Valuation(Q);
      if Puiseux::iszero(Q) then
        vQ := errQ;
      end_if;
      P := extsubsop(P, 5 = eval(subs(ListCoeffs(P), varP = varP + Point(P))),
                        6 = (varP = 0));
   end_if;
   // Now vQ > 0 and Point(P) = 0

   //
   // for the error term: test if P or Q are zero
   // - P=0 ,Q=0  : O(x^(errP*errQ/bP/bQ))
   // - P=0 ,Q<>0 : O(x^(errP*vQ/bP/bQ))
   // - P<>0,Q=0  : vP>0: O(x^(vP*errQ/bP/bQ))
   //               vP=0: lcoeff(P) + O(order(second term of P)*order(Q))
   //
   if vP < 0 and Puiseux::iszero(Q) then // division by zero
     Series::error("invalid composition")
   elif Puiseux::iszero(P) then
      extsubsop(Q, 1 = 0, 2 = bP*bQ, 3 = errP*vQ, 4 = errP*vQ, 5 = []);
   elif Puiseux::iszero(Q) then
      if vP > 0 or Flag(P) = 1 then
         extsubsop(Q, 1 = 0, 2 = bP*bQ, 3 = vP*errQ, 4 = vP*errQ, 5 = []);
      else // vP = 0 and Flag(P) = 0
         // return lcoeff(P) + O(..)
         P := lmonomial(P, Rem);
         if Puiseux::iszero(P[2]) then
           vP := Err(P[2]);
         else
           vP := Valuation(P[2]);
         end_if;
         extsubsop(Q, 1 = 0, 2 = bP*bQ, 3 = 0, 4 = vP*errQ, 5 = [P[1]]);
      end_if
   else // neither P nor Q are of the form O(..)
      //
      // Flag(P) = 0 and none of the coeffs contains x
      // => use old procedure _fconcat, now called _fconcat0
      //
      if Flag(P) = 0 and not has(ListCoeffs(P), varP) then
         Puiseux::_fconcat0(P, Q);

      // otherwise:
      // - convert series P,Q into expressions P_expr,Q_expr
      // - substitute Q_expr in P_expr:
      //   PQ_expr := P_expr(Q_expr)
      // - convert PQ_expr into series
      else
         // convert P,Q into expressions P_expr,Q_expr
         P_expr := Puiseux::expr(P);
         Q_expr := Puiseux::expr(Q);

         // compute PQ_expr := P_expr(Q_expr)
         PQ_expr := subs(P_expr, varP = Q_expr);

         // the result has branch order bP*bQ and Valuation vP*vQ
         // compute the order of the error term of the result
         errPQ := min((errQ - vQ)*bP + vP*vQ, errP*vQ);

         // expand PQ_expr into a series and add an appropriate error
         // term to ensure that the order or the error term is not
         // too high
         res := FAIL;
         traperror((res := series(PQ_expr, Varpoint(Q), errPQ - vP*vQ + 1,
                                  Direction(Q), NoWarning)));
         if contains({Puiseux, gseries}, domtype(res)) then
           res + Puiseux::create(bP*bQ, errPQ, errPQ, [], varQ, Point(Q),
                                 Direction(Q))
         else
           res
         end_if
      end_if;
   end_if;
end_proc:


//
// Idea: from Zippel, page 167
// - former _fconcat, used by the new _fconcat
// - does no argument checking, this is done by _fconcat, and assumes:
//   Q <> O(..), P <> O(..), Point(P) = 0, Valuation(Q) > 0,
//   Flag(P) = 0 and the coefficients of P do not depend on the series
//   variable of P,
//   P is normalized, i.e., has no leading or trailing zeroes
//
// CHANGED (jngerhar) 02.03.01:
// - removed obsolete checks
// - corrected precision management
//
Puiseux::_fconcat0 := proc(P, Q)
   local n, R, i, l, d, v, vQ, lR, boP, boQ, k;
begin
   //
   // Idea:
   // - replace P by P(x^boP) and Q by Q^(1/boP)
   // - use Horner's rule to compute
   //   P(Q)/Q^v = a[v] + a[v+1]*Q + ... + a[v+d-2]*Q^(d-2) + a[v+d-1]*Q^(d-1)
   //            = a[v] + Q*(a[v+1] + ... + Q*(a[v+d-2] + Q*a[v+d-1]) ... )
   // - multiply the result by Q^v
   //

   boP := BranchOrder(P);
   // P := P(x^boP), Q := Q^(1/boP)
   if boP <> 1 then
      P := extsubsop(P, 2 = 1);
      Q := Puiseux::power(Q, 1/boP)
   end_if;
   boQ := BranchOrder(Q);
   vQ := Valuation(Q);

   //
   // P = a[v]*x^v + ... + a[v+d-1]*x^(v+d-1) + O(x^n)
   //
   v := Valuation(P);
   l := ListCoeffs(P);
   d := nops(l);
   n := Err(P);

   //
   // initialize R with a[p] + O(y^((n - v - d + 1)*vQ/boQ))
   //
   R := extsubsop(Q, 1 = 0, 3 = 0, 4 = (n - v - d + 1)*vQ, 5 = [l[d]]);
   k := 1;
   for i from d - 1 downto 1 do
      if iszero(l[i]) then
        k := k + 1;
        next;
      end_if;

      // R := R * Q^k
      if k = 1 then
        R := Puiseux::mult(R, Q);
      else
        R := Puiseux::mult(R, Puiseux::power(Q, k));
      end_if;

      //
      // add l[i]*y^0 to R
      //
      lR := [l[i], 0 $ (vQ*k - 1), op(ListCoeffs(R))];
      R := extsubsop(R, 3 = 0, 5 = lR);
      k := 1;
   end_for;
   //
   // now R = a[v] + a[v+1]*Q + ...
   //
   if v = 0 then
      R
   else
      Puiseux::mult(R, Puiseux::_power(Q, v))
   end_if
end_proc:


//
// Puiseux::revert(s): compute the inverse of the Puiseux series s with
// respect to functional composition
//
// Only works if the series variable does not appear in the coefficients.
// In particular, does not work for Flag = 1.
//
// (18.07.00, T.L., with Juergen Gerhard)
//
// ***WARNING***: assuming Point(s) = 0 and Valuation(s) > 0 and <> 1,
// revert(s) in general is only a one-sided inverse of s!
// Examples:
// s = x^2 -> revert(s) = x^(1/2):
// s @ revert(s) = x, revert(s) @ s = (x^2)^(1/2)
// and the other way round for s = x^(1/2)
// Similar statements hold in the general case
//
Puiseux::revert := proc(s)
   local v, l, n, m, i, j, k, C, u, x;
begin
   //
   // Does Variable(s) appear in the coefficients?
   // This includes the case Flag = 1!
   // => error
   //
   x := Variable(s);
   if has(ListCoeffs(s), x) then
      Series::error("can't compute functional inverse");
   end_if;

   if Point(s) = complexInfinity then
     // revert(s) = 1/revert(s(1/x))
     Puiseux::_invert(Puiseux::revert(Puiseux::set_var2(s, x, x)))
   else // Point(s) is finite

      // normalize using Puiseux::iszero as zero test
      s := Puiseux::normal(s, 1);

      v := Valuation(s);
      n := Err(s) - v; // number of known terms of s
      m := ListCoeffs(s);

      if n = 0 then // s = O(..)
         if v <= 0 then
           // cannot determine expansion point of functional inverse
           Series::error("order too small")
         elif Point(s) = 0 then // O(x^(bo/v))
           extsubsop(s, 1 = 0, 2 = v, 3 = BranchOrder(s), 4 = BranchOrder(s),
                        5 = [])
         else // Point(s) + O(x^(bo/v))
           Puiseux::create(v, 0, BranchOrder(s), [Point(s)], x)
         end_if;

      elif v < 0 then // expansion point complexInfinity
         // revert(s) = revert(1/s) @ (1/x)
         Puiseux::set_var(Puiseux::revert(Puiseux::_invert(s)),
                          x, x, complexInfinity);

      elif v = 0 then // finite expansion point
         // revert(s) = revert(s - C) @ (x - C)
         [C, s] := Puiseux::lmonomial(s, Rem);
         extsubsop(Puiseux::revert(s), 6 = (x = C))

      elif BranchOrder(s) > 1 then
        // revert(s) = revert(s^bo) @ x^bo
        Puiseux::power(Puiseux::revert(extsubsop(s, 2 = 1)), BranchOrder(s))

      elif v > 1 then
        // revert(s) = revert(s^(1/v)) @ x^(1/v)
        //
        // s := (s/x^v)^(1/v)
        s := Puiseux::power(extsubsop(s, 3 = 0, 4 = Err(s) - v), 1/v):
        // s := x*s
        s := extsubsop(s, 3 = 1, 4 = Err(s) + 1):
        // return revert(s) @ x^(1/v)
        extsubsop(Puiseux::revert(s), 2 = v)

      // v = 1 ==> functional inverse has expansion point 0
      // BranchOrder(s) = 1, and Point(s) is finite
      else

         if nops(m) < n then // pad with zeroes
           m := m . [0 $ (n - nops(m))]
         end_if;

         // Lagrangian inversion; see Knuth (1998) vol. 2, Algorithm 4.7 L

         u[0] := 1/m[1];
         l[0] := Point(s); // constant coefficient of revert(s)
         l[1] := u[0]; // coefficient of x in revert(s)
         for i from 2 to n do
           // compute coefficient l[i] of x^i in revert(s)
           for k from 0 to i - 2 do
             u[k] := (u[k] - _plus(u[k - j]*m[j + 1] $ j = 1..k))/m[1];
           end_for;
           u[i - 1] := -_plus(j*u[i - j]*m[j] $ j = 2..i)/m[1];
           l[i] := u[i - 1]/i;
           // invariant: u = (x/s)^i
         end_for;

         Puiseux::normal(
           Puiseux::create(1, 0, 1 + n, [l[i] $ i = 0..n], x),
           0) // quick normalization
      end_if
   end_if
end_proc:


//
// Puiseux::func_call(s, p): computes s(p)
//
// s: Series::Puiseux
// p: arithmetical expression (not checked)
//
// converts s into an expression by discarding the O-term and
// then calls subs
//
Puiseux::func_call:= proc(s, p)
   local s_expr;
begin
   if args(0) <> 2 then
      Series::error("invalid function call")
   end_if;
   p := context(p);

   //
   // convert s into expressions s_expr
   //
   s_expr := Puiseux::expr(s);

   //
   // compute s_expr(p)
   //
   eval(subs(s_expr, Variable(s) = p));
end_proc:


//
// Puiseux::diff(s, x1, x2, ...): differentiate the series s with respect to
//   x1, then with respect to x2, etc.
//
// s: Series::Puiseux
// x1, x2, ...: valid indeterminates for diff
//
// (05.07.00, TL)
// CHANGED (jngerhar) 26.02.01:
// - inserted Puiseux::normal
// - diff(O(1), x) now yields correctly O(1)
//
Puiseux::diff := proc(s, x)
  local l, v, bo, n, p, res, dir, i;
begin
  case args(0)
  of 0 do Series::error("wrong number of arguments")
  of 1 do return(s)
  of 2 do break // normal case
  otherwise // more than two args; proceed recursively
    return(Puiseux::diff(Puiseux::diff(s, x), args(3..args(0))))
  end_case;

  bo := BranchOrder(s);
  v := Valuation(s);
  n := Err(s);
  l := ListCoeffs(s);
  p := Point(s);
  dir := Direction(s);

  if has(p, x) then
     // x occurs in the expansion point
     FAIL
  elif x <> Variable(s) then
     //
     // x is not the series variable and does not occur in the expansion point
     // ==> differentiate coefficients
     //
     Puiseux::map(s, diff, x)
  else // differentiate with respect to the series variable
     if has(l, x) then
        //
        // coeffs depend on x as well,
        // may contain specfuncs or series is of type Flag = 1
        // ==> convert into an expression, differentiate expression,
        //     and apply "series" to the result
        // add an appropriate error
        // term to ensure that the order or the result is not
        // too high
        //
        res := FAIL;
        traperror((res := series(diff(Puiseux::expr(s), x), Varpoint(s),
                                 n - v + 1, dir, NoWarning)));
        if contains({Puiseux, gseries}, domtype(res)) then
           res + if n = 0 then
                   Puiseux::create(bo, n, n, [], x, p, dir)
                 elif p = complexInfinity then
                   Puiseux::create(bo, n + bo, n + bo, [], x, p, dir)
                 else
                   Puiseux::create(bo, n - bo, n - bo, [], x, p, dir)
                 end_if
        else
           res
        end_if
     //
     // coeffs do not contain x
     //
     elif p = complexInfinity then
	if n = 0 then
          Puiseux::normal(extsubsop(s, 3 = v + bo,
                                    5 = [l[i]*(1-v-i)/bo $ i = 1..nops(l)]),
                          0) // quick normalization
        else
          Puiseux::normal(extsubsop(s, 3 = v + bo, 4 = n + bo,
                                    5 = [l[i]*(1-v-i)/bo $ i = 1..nops(l)]),
                          0) // quick normalization
        end_if
     else // finite expansion point
        if n = 0 then
          Puiseux::normal(extsubsop(s, 3 = v - bo,
                                    5 = [l[i]*(v+i-1)/bo $ i = 1..nops(l)]),
                          0) // quick normalization
        else
          Puiseux::normal(extsubsop(s, 3 = v - bo, 4 = n - bo,
                                    5 = [l[i]*(v+i-1)/bo $ i = 1..nops(l)]),
                          0) // quick normalization
        end_if
     end_if
  end_if
end_proc:


//
// Puiseux::int(s, x): definite or indefinite integration of the series s
//   with respect to x
//
// s: Series::Puiseux
// x: identifier or of the form identifier = range
// further args, if any, are passed to int
//
// Idea: if Flag=0 and variable does not occur in the coeffs:
//                 usual integration
//       otherwise : - convert s into expression
//                   - integrate via int()
//                   - reconvert the solution into a series expansion
// (05.07.00, T.L.)
// CHANGED (jngerhar) 26.02.01
//
Puiseux::int := proc(s, x)
  local bo, v, var, n, p, l, A, mydivide, res, dir, i;
begin
  bo := BranchOrder(s);
  v := Valuation(s);
  n := Err(s);
  var := Variable(s);
  p := Point(s);
  l := ListCoeffs(s);
  dir := Direction(s);

  if domtype(x) = DOM_IDENT then
     //
     // indefinite integration
     //
     if has(p, x) then
        // x occurs in the expansion point
        FAIL
     elif x <> var then
        //
        // x is not the series variable and does not occur in the
        // expansion point ==> apply int to all coeffs
        //
        Puiseux::map(s, int, x, args(3..args(0)));
     else // integrate with respect to the series variable
        if has(l, x) or
          (if p = complexInfinity then
             v <= bo and n > bo and Puiseux::coeff(s, -1) <> 0
           else
             v <= -bo and n > -bo and Puiseux::coeff(s, -1) <> 0
           end_if) then
          //
          // x occurs in the coeffs or Flag = 1 or coeff of x^-1
          // is nonzero
          // ==> convert s into an expression and apply int to it
          //
          A := int(Puiseux::expr(s), x, args(3..args(0)));
          if has(A, int) then
             // could not compute integral
             hold(int)(args());
          else
             //
             // apply "series" to the result and
             // add an appropriate error
             // term to ensure that the order or the result is not
             // too high
             //
             res := FAIL;
             traperror((res := series(A, Varpoint(s), n - v + 1, dir,
                                      NoWarning)));
             if contains({Puiseux, gseries}, domtype(res)) then
               res + if p = complexInfinity then
                       Puiseux::create(bo, n - bo, n - bo, [], var, p, dir);
                     else
                       Puiseux::create(bo, n + bo, n + bo, [], var, p, dir);
                     end_if
             else
               res
             end_if
          end_if
        else
          //
          // Flag = 0 and x does not occur in the coeffs and coeff
          // of x^(-1) is zero ==> standard integration
          //

          // crude hack to handle the coefficient of x^(-1),
          // which is zero, correctly
          mydivide := (a, b) -> (if a = 0 then 0 else a/b end_if);

          if p = complexInfinity then
            extsubsop(s, 3 = v - bo, 4 = n - bo,
                      5 = [mydivide(l[i], 1-v+bo-i)*bo $ i = 1..nops(l)])

          else // finite expansion point
            extsubsop(s, 3 = v + bo, 4 = n + bo,
                      5 = [mydivide(l[i], v+bo+i-1)*bo $ i = 1..nops(l)])
          end_if
        end_if
     end_if
  else
     //
     // definite integration
     // I do not like to do the same tests as in STDLIB/int.mu
     // thus errors can occur!!!
     //
     if domtype(op(x, 1)) = DOM_IDENT then
        if has(p, op(x, 1)) then
           // integration variable occurs in the expansion point
           return(FAIL)
        elif op(x, 1) <> var then
           //
           // x is not the series variable and does not occur in the
           // expansion point ==> apply int to all coeffs
           //
           return(Puiseux::map(s, int, x, args(3..args(0))));
        elif Point(s) <> complexInfinity then
           //
           // definite integration with respect to the series variable:
           // return an expression with a symbolic integral of a
           // symbolic error term
           //
           return(int(Puiseux::expr(s), x, args(3..args(0)))
                  + int(O(Base(s, n/bo), Varpoint(s)), x, args(3..args(0))));
        end_if;
     end_if;
     //
     // case something is wrong, should I return an error here?
     //
     hold(int)(args());
  end_if;
end_proc:


//
// Puiseux::laplace(a, t, s) - compute Laplace transform in the variable s
//   of the Puiseux series a w.r.t. the variable t
//
// If Variable(a) = t, then a must be a Puiseux expansion around 0
// with Valuation(a) >= 0 and s must be an identifier.
// The result is then a Puiseux expansion around s = +infinity.
//
// If Variable(a) <> t, then the coefficients of a are transformed.
// The special case where Variable(a) occurs in s is taken care of.
//
Puiseux::laplace := proc(a, t, s)
   local x, bo, v, n, l, res, i;
begin
   x := Variable(a);
   bo := BranchOrder(a);
   v := Valuation(a);
   n := Err(a);
   l := ListCoeffs(a);

   if x = t then // transform w.r.t. the series variable
      if Point(a) <> 0 or v < 0 or domtype(s) <> DOM_IDENT then
        // we cannot compute the Laplace transform and return FAIL
        FAIL
      elif has(l, x) then
        // series variable occurs in the coefficients. This includes
        // the case Flag = 1.
        // convert into an expression, transform the expression,
        // and reconvert the result into a series, adding an appropriate
        // error term
        res := FAIL;
        traperror((res := series(transform::laplace(Puiseux::expr(a), t, s),
                                 s = infinity, n - v, NoWarning)));
        if contains({Puiseux, gseries}, domtype(res)) then
          res + extsubsop(a, 3 = n + bo, 4 = n + bo, 5 = [],
                          6 = (s = complexInfinity), 7 = Left)
        else
          res
        end_if
      else
        res := extsubsop(a, 3 = v + bo, 4 = n + bo,
                         5 = [l[i]*gamma((v + i - 1)/bo + 1) $ i = 1..nops(l)],
                         6 = (s = complexInfinity), 7 = Left);
        // the data structure may be invalid if s appears in the coefficients
        // of a; therefore we convert to an expression and reconvert to a
        // series if necessary
        Puiseux::clean(res)
      end_if
   else
      a := map(a, transform::laplace, t, s);
      // the data structure may be invalid if s appears in the coefficients
      // of a; therefore we convert to an expression and reconvert to a
      // series if necessary
      Puiseux::clean(a)
   end_if
end_proc:


//
// Puiseux::invlaplace(a, s, t) - compute inverse Laplace transform in the
//   variable t of the Puiseux series a w.r.t. the variable s
//
// If Variable(a) = s, then a must be a Puiseux expansion around infinity
// and t must be an identifier.
// The result is then a Puiseux expansion around t = 0.
// If Valuation(a) < 0, then BranchOrder(a) must be equal to 1
//
// If Variable(a) <> t, then the coefficients of a are transformed.
// The special case where Variable(a) occurs in t is taken care of.
//
Puiseux::invlaplace := proc(a, s, t)
   local x, bo, v, n, l, res, b, tb, i;
begin
   x := Variable(a);
   bo := BranchOrder(a);
   v := Valuation(a);
   n := Err(a);
   l := ListCoeffs(a);

   if x = s then // transform w.r.t. the series variable
      if Point(a) <> complexInfinity or (v < 0 and bo > 1)
         or domtype(s) <> DOM_IDENT then
        // we cannot compute the Laplace transform and return FAIL
        FAIL
      elif has(l, x) then
        // series variable occurs in the coefficients. This includes
        // the case Flag = 1.
        // convert into an expression, transform the expression,
        // and reconvert the result into a series, adding an appropriate
        // error term
        res := FAIL;
        traperror((res := series(transform::invlaplace(Puiseux::expr(a), s, t),
                                 t, n - v, NoWarning)));
        if contains({Puiseux, gseries}, domtype(res)) then
          res := res + extsubsop(a, 3 = n - bo, 4 = n - bo, 5 = [],
                                 6 = (t = 0), 7 = Undirected)
        end_if;
        return(res);
      elif v > 0 then
        res := extsubsop(a, 3 = v - bo, 4 = n - bo,
                         5 = [l[i]/gamma((v + i - 1)/bo) $ i = 1..nops(l)],
                         6 = (t = 0), 7 = Undirected)
      else // v <= 0, and bo = 1 if v < 0
        // b := principal part -> inv. transform tb = linear combination
        // of derivatives of dirac's
        b := extsubsop(Puiseux::truncate(a, 1/bo), 4 = n);
        tb := _plus(l[i]*dirac(t, 1 - v - i) $ i = 1..min(1 - v, nops(l)));

        a := a - b;
        v := Valuation(a);
        n := Err(a);
        l := ListCoeffs(a);

        res := extsubsop(a, 3 = v - bo, 4 = n - bo,
                         5 = [l[i]/gamma((v + i - 1)/bo) $ i = 1..nops(l)],
                         6 = (t = 0), 7 = Undirected)
               + Puiseux::create(bo, 0, n - bo, [tb], t); // tb + O(t^(n/bo - 1))
      end_if;
      // the data structure may be invalid if t appears in the coefficients
      // of a; therefore we convert to an expression and reconvert to a
      // series if necessary
      Puiseux::clean(res)
   else
      a := map(a, transform::invlaplace, s, t);
      // the data structure may be invalid if t appears in the coefficients
      // of a; therefore we convert to an expression and reconvert to a
      // series if necessary
      Puiseux::clean(a)
   end_if
end_proc:


//
// Puiseux::contfrac(s), Puiseux::_contfrac(s):
// compute continued fraction expansion of s, of the form
//
// a0 + x^e0
//      ---------------------------
//      a1 + x^e1
//           ----------------------
//           a2 + x^e2
//                -----------------
//                a3 + ...
//                     ------------
//                     ar + O(x^er)
//
// where
// - x = Base(s, 1) = Variable(s) - Point(s) if Point(s) is finite and
//   x^(-e0), x^(-e1) for x = Variable(s) if Point(s) = complexInfinity
// - a0, a1, ..., ar contain no powers of x (but may contain logs etc.)
// - a1, ..., ar are nonzero
// - e0 is a nonzero rational number, positive if a0 <> 0
// - e1, ..., er are positive rational numbers
//
// returns FAIL for Flag(s) = 1
//
// Puiseux::contfrac is the interface method, returns an object of type
//   (stdlib::)contfrac
// Puiseux::_contfrac returns a pair with the list
//   [[a0, x^e0], [a1, x^e1], ..., [ar, x^er]]
//   and the rational function represented by this list
//
// Juergen Gerhard, 24.07.01, 17.11.01
//
Puiseux::contfrac := proc(s)
begin
   if Flag(s) = 1 then
     FAIL
   else
     contfrac::create(op(Puiseux::_contfrac(s)), Variable(s), Point(s));
   end_if
end_proc:

Puiseux::_contfrac := proc(s)
   local bo, v, n, l, t, e, r, x, Sign;
begin
   // strip leading and trailing zeroes, using Puiseux::iszero as zero test
   s := Puiseux::normal(s, 1);

   bo := BranchOrder(s);
   v := Valuation(s);
   n := Err(s);
   l := ListCoeffs(s);
   if Point(s) = complexInfinity then
     Sign := -1;
     x := Variable(s);
   else
     Sign := 1;
     x := Variable(s) - Point(s);
   end_if;

   if Puiseux::iszero(s) then // s = O(x^(n/bo))
     return([[[0, x^(Sign*n/bo)]], 0])
   elif nops(l) = 1 then // s = l[1]*x^(v/bo) + O(x^(n/bo))
     if v = 0 then
       return([[[l[1], x^(Sign*n/bo)]], l[1]])
     else
       return([[[0, x^(Sign*v/bo)], [1/l[1], x^(Sign*(n - v)/bo)]],
              l[1]*x^(Sign*v/bo)])
     end_if;
   elif v <> 0 then // return x^(v/bo) / (s*x^(-v/bo))^(-1)
     r := Puiseux::_contfrac(Puiseux::_invert(extsubsop(s, 3 = 0, 4 = n - v)));
     return([[[0, x^(Sign*v/bo)]] . r[1], contfrac::normal(x^(Sign*v/bo)/r[2])])
   end_if;
   // now v = 0 and s has at least two nonzero terms

   // t := s - l[1],
   t := lmonomial(s, Rem)[2];
   e := ldegree(t);

   // return l[1] + x^e / (t*x^(-e))^(-1)
   r := Puiseux::_contfrac(Puiseux::_invert(
     extsubsop(t, 3 = 0, 4 = Err(t) - e*BranchOrder(t))));
   [[[l[1], x^(Sign*e)]] . r[1], contfrac::normal(l[1] + x^(Sign*e)/r[2])]
end_proc:


// Puiseux::Re(s), Puiseux::Im(s), Puiseux::conjugate(s)
//
// - return s if s = O(..)
// - map to the coefficients if direction <> Undirected
//     or s = const + O(..)
// - return unevaluatedly otherwise

Puiseux::Re := proc(s)
begin
  if Puiseux::iszero(s) then
    s
  elif (Valuation(s) = 0 and nops(ListCoeffs(s)) = 1) // s = const + O(..)
       or Direction(s) <> Undirected then
    Puiseux::map(s, Re)
  else
    hold(Re)(s)
  end_if
end_proc:

Puiseux::Im := proc(s)
begin
  if Puiseux::iszero(s) then
    s
  elif (Valuation(s) = 0 and nops(ListCoeffs(s)) = 1) // s = const + O(..)
       or Direction(s) <> Undirected then
    Puiseux::map(s, Im)
  else
    hold(Im)(s)
  end_if
end_proc:

Puiseux::conjugate := proc(s)
begin
  if Puiseux::iszero(s) then
    s
  elif (Valuation(s) = 0 and nops(ListCoeffs(s)) = 1) // s = const + O(..)
       or Direction(s) <> Undirected then
    Puiseux::map(s, conjugate)
  else
    hold(conjugate)(s)
  end_if
end_proc:

Puiseux::indets:= proc(s)
begin
   if s::dom = Puiseux then
      return(indets(ListCoeffs(s)) union
             indets(Variable(s))):
   else
      return(indets(s));
   end_if;
end_proc:

Puiseux::freeIndets := proc(s)
begin
   if s::dom = Puiseux then
      return(freeIndets(ListCoeffs(s)) union
             freeIndets(Variable(s))):
   else
      return(freeIndets(s));
   end_if;
end_proc:

Puiseux::indet:= s -> Variable(s):
Puiseux::point:= s -> Point(s):

// ===========================================================
// test if a Puiseux series is 0:
// FALSE, if any coefficient is <> 0.
// UNKNOWN, if only O-term or if all coeffs are hidden zeroes.
// ===========================================================
Puiseux::testeq:= proc(ex1, rhs=0)
local ex, x;
begin
  ex:= ex1 - rhs;
  if domtype(ex) <> Series::Puiseux then
     return(testeq(expr(ex), 0, args(3..args(0))));
  end_if:
  if lcoeff(ex) = FAIL then
     // ex is just the order term
     return(UNKNOWN);
  end_if;
  ex:= [coeff(ex)];
  // The following yields FALSE or UNKNOWN (never TRUE) 
  // This is compatible with testeq(O(x^5), 0) = UNKNOWN
  _lazy_and(testeq(x, 0, args(3 .. args(0))) $ x in [coeff(ex)], 
            UNKNOWN);
end_proc:

// define some methods to avoid creation via make_slot
Puiseux::evaluate := FAIL:
Puiseux::posteval := FAIL:
Puiseux::undefinedEntries := FAIL:
Puiseux::allEntries := FAIL:
Puiseux::allAutoEntries := FAIL:
Puiseux::whichEntry := FAIL:
Puiseux::new_extelement := FAIL:
Puiseux::create_dom_elem := FAIL:
Puiseux::eval := FAIL:
Puiseux::Content := FAIL:
Puiseux::MMLContent := FAIL:
//
Puiseux::domtype := FAIL:
Puiseux::type := FAIL:
Puiseux::isNeg:= FAIL:
Puiseux::isInverted:= FAIL:
Puiseux::slot := FAIL:
Puiseux::length := FAIL:
Puiseux::expose := FAIL:
Puiseux::bool := FAIL:
//
// the "id"-slot is necessary to avoid infinite recursion in "series"
Puiseux::id := () -> args():
Puiseux::contains := FAIL:
Puiseux::select := FAIL:
Puiseux::split := FAIL:
Puiseux::zip := FAIL:
Puiseux::_concat := FAIL:
Puiseux::unapply := FAIL:
Puiseux::intmult := FAIL:
Puiseux::simplify := e -> map(e, simplify, args(2..args(0))):
Puiseux::Simplify := e -> map(e, Simplify, args(2..args(0))):
Puiseux::rewrite := FAIL:
Puiseux::partfrac := FAIL:
// the following methods might be implemented some day ...
Puiseux::op := FAIL:
Puiseux::nops := FAIL:
Puiseux::subsop := FAIL:
Puiseux::_index := FAIL:
Puiseux::set_index := FAIL:
Puiseux::hastype := FAIL:
Puiseux::maprec := FAIL:
Puiseux::rectform := FAIL:
Puiseux::factor := FAIL:
Puiseux::evaluateIndex := FAIL:
Puiseux::sortSums := FAIL:
// Puiseux::CF see GENERATE/CF.mu
Puiseux::CF:= loadproc(Puiseux::CF, pathname("GENERATE"),"CF"):

//
// Puiseux::make_slot - generic overloading of specfuncs
//
// If f(s) is called for a domain or a function
// environment f and an object s of type Puiseux
// and no slot Puiseux::f exists (e.g., f = exp or f = sin),
// then Puiseux::make_slot(Puiseux, f) is called.
// Puiseux::make_slot creates the slot Puiseux::f, provided
// that one of the slots f::series or f::diff exists.
// Otherwise, it sets Puiseux::f to FAIL, which means that
// f itself is to handle the call.
//
// See the domainref docu for a documentation of make_slot
//
// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
// For some strange reason, this must be the last entry defined in this file !
// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
//
Puiseux::make_slot := proc(T, f)
  option escape;
begin
  if domtype(f) <> DOM_STRING then
    return(FAIL);
  end_if;
  f := text2expr(f);
  // the eval is necessary since text2expr returns an identifier
  if domtype(eval(f)) <> DOM_DOMAIN and domtype(eval(f)) <> DOM_FUNC_ENV then
    FAIL
  elif slot(eval(f), "series") <> FAIL or slot(eval(f), "diff") <> FAIL then
    proc(s)
      local x, x0, t, v, r;
    begin
      x := genident("x");
      // determine expansion point x0 = limit point of s
      v := ldegree(s);
      if v = FAIL then
        if Err(s) <= 0 then
           Series::error("order too small");
        else
           v := Err(s);
        end_if;
      end_if;
      if v >= 0 then
        x0 := coeff(s, 0)
      else // v < 0
        x0 := complexInfinity
      end_if;
      t := FAIL;
      traperror((t := series(eval(f)(x, args(2..args(0))), x = x0,
                             Err(s) - Valuation(s), NoWarning)));
      if domtype(t) = Puiseux then
        if traperror((r := Puiseux::_fconcat(t, s))) = 0 then
          return(r);
        end_if
      end_if;
      if domtype(eval(f)) = DOM_FUNC_ENV then
        f(args())
      else // DOM_DOMAIN
        FAIL
      end_if
    end_proc
  else
    FAIL
  end_if
end_proc:


unalias(BranchOrder, Valuation, Err, ListCoeffs, Variable, Point, Varpoint,
        Flag, Base):

unalias(Puiseux):
