//    

/*--
	ln/Series -- the function attribut "series" for ln
--*/

ln::series :=
proc(f, x, n, dir, opt)
  local mydir, s, others, dummy, l, t, a, k;
begin

  mydir:= if contains({Left, Right, Real}, dir) then
             dir
          else
             null();
          end_if:

  //=========================================
  // handle series(ln(c*exp(x)), x = infinity)
  //=========================================
  if hastype(f, "exp") then
    if type(f) = "exp" and
      ( (l:= limit(op(f), x = 0, mydir)) = infinity
        or
        l = -infinity
        or
        iszero(l)
      ) then
        s:= Series::series(op(f), x, n, dir, opt);
        if s <> FAIL then
          return(s);
        end_if;
    end_if:
    if type(f) = "_mult" then
       // write ln(s*others) = ln(s) + ln(others) 
       // where s = exp(something) 
       [s, others, dummy]:= split(f, x -> type(x) = "exp");
       if type(s) = "_mult" then
          s:= combine(s, exp);
       end_if:
       if type(s) = "exp" then 
          if (l:= limit(op(s), x = 0, mydir)) = infinity or
              l = -infinity or
              iszero(l) then
            s:= Series::series(op(s) + ln(others), x, n, dir, opt);
            if s <> FAIL then
               return(s);
            end_if:
          end_if:
       end_if:
    end_if:
  end_if:

  //=========================================
  // Use the series attribute of lngamma to
  // compute expansions of ln(c*gamma(x)):
  // series(ln(c*gamma(x)), x = infinity) =
  // series(ln(c) + lngamma(x), x = infinity)
  //=========================================
  if hastype(f, "gamma") then
    if type(f) = "gamma" then
        // I guess that igamma(a, x) will become
        // gamma(a, x) in some future version of
        // MuPAD. Make sure that these lines will
        // be adapted: 
        if nops(f) > 1 then 
           error("expecting gamma with one argument");
        end_if:
        if limit(op(f), x = 0, mydir) = infinity
          then
          s:= lngamma::series(op(f), x, n, dir, opt);
          if s <> FAIL then
            return(s);
          end_if;
        end_if:
    end_if:
    if type(f) = "_mult" then
        // write ln(s*others) = ln(s) + ln(others) 
        // where s = gamma(something) implying 
        [s, others, dummy]:= split(f, x -> type(x) = "gamma");
        if type(s) = "gamma" then // s can be a product of gamma terms:
           if nops(s) > 1 then 
              error("expecting gamma with one argument");
           end_if:
           if limit(op(s), x = 0, mydir) = infinity then
             s:= Series::series(ln(others), x, n, dir, opt) +
                 lngamma::series(op(s), x, n, dir, opt);
             if s <> FAIL then
                return(s);
             end_if:
           end_if:
        end_if:
    end_if:
  end_if:

  //=========================================
  // generic series code
  //=========================================
  // s := series(ln(1 + x), x)
  s := Series::Puiseux::create(1, 1, n + 1, Series::gen["ln"](n), x, 0, dir);
  
  // handle special cases first
  if f = 1 + x then
    s
  elif f = 1 - x then
    Series::Puiseux::create(1, 1, n + 1,
                            map(Series::gen["ln"](n), _negate@abs), x, 0, dir)
  else

    // recursively expand the argument into a series
    t := Series::series(f, x, n + 1, dir, opt);
    if domtype(t) <> Series::Puiseux then
      return(FAIL)
    end_if;

    k := Series::Puiseux::ldegree(t);
    a := Series::Puiseux::lcoeff(t);
    // f = t = a*x^k + higher order terms
    
    if k = FAIL then // t = O(..)
      Series::error("order too small")
      
    elif k = 0 then // expansion around the finite point a <> 0
      //
      // f = a*(1 + higher order terms)
      //
      
      // t := series(ln(1 + higher order terms))
      t := Series::Puiseux::lmonomial(t, Rem)[2]; // t := t - a
      if a <> 1 then
        t := Series::Puiseux::scalmult(t, 1/a)    // t := t/a
      end_if;
      t := Series::Puiseux::_fconcat(s, t);

      if is(a < 0) = TRUE then
        //
        // expansion on the branch cut:
        // ln(f) = signIm(f)*I*PI + ln(-f)
        //       = signIm(f)*I*PI + ln(-a) + ln(1 + higher order terms)
        //
        
        // find out signIm by series expansion ?!
        
        Series::Puiseux::const(signIm(f)*I*PI + ln(-a), x,
                               Series::Puiseux::order(t), dir) + t
      else
        //
        // ln(f) = ln(a) + ln(t)
        //
// This may still be wrong if a contains x or a can be simplified to 0
// or a negative number!
        Series::Puiseux::const(ln(a), x, Series::Puiseux::order(t), dir) + t
      end_if

    else // expansion around 0 or +-infinity
         //
      // f = a*x^k*(1 + higher oder terms)
      // ln(f) = ln(a*x^k) + ln(1 + higher order terms)
      // This is valid only when a*x^k is not real negative!
      // Otherwise, we use
      // ln(f) = ln(-f) + I*PI*signIm(f)
      // and proceed with -f
      
      if n = 0 then
        Series::error("order too small")
      end_if;
         
      // t := series(ln(1 + higher order terms))
      t := Series::Puiseux::lmonomial(t, Rem)[2]; // t := t - a*x^k
      // we are still going to have problems if we do not recognize that a<0
      // only solution: return a piecewise
      if dir=Right and is(a<0, Goal = TRUE)
        or
        dir = Left and is((-1)^k*a < 0, Goal = TRUE) then

        a:= -a;
        t:= -t;
        // Walter, 19.3.2010: the following Series::Puiseux::order(t) may be
        // negative (e.g., see series(ln(-1/x^3), x = 0, 1, Right).
        // This will raise an error. Since signIm is treated like a constant
        // by series, use Series::Puiseux::const instead:
        //     s:= s + Series::Puiseux(I*PI*signIm(f), x, 
        //                             ceil(Series::Puiseux::order(t)),
        //                             dir);
        // In the call series(ln(1 - 1/x), x = 0, Right),
        // Series::Puiseux produces signIm(1 - 1/x) = +1, whereas
        // Series::Puiseux::const produces a symbolic signIm(1 - 1/x) that 
        // is messed up later. We need to make assumptions on x to enforce
        // evaluation of signIm:
        if dir = Right then
           assume(0 < x < 10^(-DIGITS))
        elif dir = Left then
           assume(-10^(-DIGITS) < x < 0)
        elif dir = Real then
           assume(x in R_);
        end_if:
        s:= s + Series::Puiseux::const(I*PI*signIm(f), x, n, dir);
      end_if;
      t := Series::Puiseux::scalmult(t, 1/a, -k); // t := t/(a*x^k)
      t := Series::Puiseux::_fconcat(s, t);

      if not has(a, x) then
        // try to split ln(a*x^k), if possible
        
        if dir = Right or is(x >= 0) = TRUE then
          return(t + Series::Puiseux::const(ln(a) + k*ln(x), x, n, dir))
        elif dir = Left or is(x <= 0) = TRUE then
          // we might pull out (-1)^k as well here
          return(t + Series::Puiseux::const(ln((-1)^k*a) + k*ln(-x), x, n, dir))
        elif is(a >= 0) = TRUE then
          if k > -1 and k < 1 then
            return(t + Series::Puiseux::const(ln(a) + k*ln(x), x, n, dir))
          else
            return(t + Series::Puiseux::const(ln(a) + ln(x^k), x, n, dir))
          end_if
        elif dir = Real or is(x, Type::Real) = TRUE then
          if domtype(k/2) = DOM_INT then // k is even
            return(t + Series::Puiseux::const(ln(a) + ln(x^k), x, n, dir))
          elif domtype((k - 1)/2) = DOM_INT then // k is odd
            return(t + Series::Puiseux::const(ln(-a) + ln(-x^k), x, n, dir))
          end_if;
        elif is(a <= 0) = TRUE then
          return(t + Series::Puiseux::const(ln(-a) + ln(-x^k), x, n, dir))
        end_if;
      end_if;

      // default: no splitting
      t + Series::Puiseux::const(ln(a*x^k), x, n, dir)
    end_if
  end_if
end_proc:

// ensure that domain Series is loaded
Series:

// expansion for ln(1 + x) around x = 0
Series::gen["ln"] :=
proc(n)
  local t, i;
begin
  t := -1;
  [((t := -t)/i $ i = 1..n)]
end_proc:

// end of file 
