/*
 * series slot of sign - computes series(sign(f), x, n, dir)
 *
 * If dir <> Left, Right, Real ==> simply returns sign(f) + O(x^n).
 * Otherwise expands f, writes it as f0 * x^d * (1 + f1), simplifies
 * sign(f0) * sign(x)^d by taking into account the direction,
 * and returns sign(f0) * sign(x)^d * series((1 + f1) / abs(1 + f1)).
 */

sign::series := proc(f, x, n, dir, opt)
  local t, d, f0, f1, s, tr, ti, prop;
begin 

  // handle special case sign(x) separately for efficiency reasons
  if f = x then
    case dir
    of Right do
      return(Series::Puiseux::one(x, n, dir));
    of Left do
      return(-Series::Puiseux::one(x, n, dir));
    otherwise
      return(Series::Puiseux::const(sign(x), x, n, dir));
    end_case
  end_if;

  if dir <> Undirected then // directional expansion

    // assign the properties implied by the direction to the series variable
    prop := getprop(x);
    case dir
    of Left do
      assume(x < 0);
      break
    of Right do
      assume(x > 0);
      break
    of Real do
      assume(x, Type::Real);
      break
    end_case;
    if prop <> x then
      assume(x, prop, _and);
    end_if;

    // recursively expand the argument
    t := Series::series(f, x, n, dir, opt);

    if domtype(t) = Series::Puiseux and (d := ldegree(t)) <> FAIL then
      
      // write t as f0 * x^d * (1 + f1), then sign(f) =
      // sign(f0) * sign(x)^d * (1 + f1) / abs(1 + f1)
      f0 := lcoeff(t);
      f1 := Series::Puiseux::scalmult(lmonomial(t, Rem)[2], 1/f0, -d);

      s := sign(f0) * sign(x)^d;

      // return s * (1 + f1) / sqrt((1 + Re(f1)^2) + Im(f1)^2)

      // some shortcuts
      if normal(f - f0*x^d) = 0 then // f = f0*x^d
        // attach previous property to series variable
        if prop = x then
          delete x
        else
          assume(x, prop)
        end_if;
        return(Series::Puiseux::const(s, x, n, dir));
      elif ldegree(f1) = FAIL then // f = f0 * x^d * (1 + O(..))
        // attach previous property to series variable
        if prop = x then
          delete x
        else
          assume(x, prop)
        end_if;
        return(Series::Puiseux::scalmult(Series::Puiseux::one(x, n, dir) + f1, s, 0));
      end_if;
    
      // split f1 into real and imaginary part tr + ti
      tr := map(f1, Re);
      ti := f1 - tr;

      // attach previous property to series variable
      if prop = x then
        delete x
      else
        assume(x, prop)
      end_if;

      if ldegree(ti) = FAIL then // ti = O(..)
        return(Series::Puiseux::scalmult(Series::Puiseux::one(x, n, dir) + ti, s, 0))
      else
        // uses Puiseux::_power
        return(Series::Puiseux::scalmult(
                 (Series::Puiseux::one(x, n, dir) + f1) /
                  ((Series::Puiseux::one(x, n, dir) + tr)^2 - ti^2)^(1/2), s, 0))
      end_if

    end_if;

    // attach previous property to series variable
    if prop = x then
      delete x
    else
      assume(x, prop)
    end_if;

  end_if;

  // default: move sign(f) into the 0th coefficient
  return(Series::Puiseux::const(hold(sign)(f), x, n, dir));

end_proc:
