/*
   solvelib::checkVectorSolutions(sol, sys, vars, options)

   given a set sol of solution vectors to equation equ that might contain too many
   solutions, select the correct ones

   output a warning if there might be still too many

*/


alias(MAXNUMBEROFPIECEWISES = 3):
alias(MAXCONDLENGTH = 15):

solvelib::checkVectorSolutions:=
proc(sol, sys: DOM_LIST, vars: DOM_LIST, options = solvelib::getOptions())
  local check: DOM_PROC, checkRecursive: DOM_PROC, warn: DOM_BOOL, pws, pw, pcw,
  spurious;
  
begin
  if options[IgnoreSpecialCases] then
    pcw:= solvelib::ignoreSpecialCases
  else
    pcw:= piecewise
  end_if;  
  
  if options[IgnoreAnalyticConstraints] then
    check:= 
    proc(v)
      local res, i;
    begin
      // check that the solution can be plugged in without producing an error, and
      // that isnonzero gives nowhere TRUE on the result
      _lazy_and(traperror((res:= 
      map(sys, evalAt, {vars[i] = v[i] $i=1..nops(vars)})
      )) = 0,
      contains(map(res, numeric::isnonzero), TRUE) = 0
      )
    end_proc
  else
    check:= 
    proc(v)
      local res, i;
    begin
      if traperror((res:= 
        map(sys, evalAt, {vars[i] = v[i] $i=1..nops(vars)})
        )) <> 0 then
        FALSE
      else
        _and(op(map(res, _equal, 0)))
      end_if  
    end_proc
  end_if;      
  
  checkRecursive:= 
  proc(sol)
    local i, cond, result;
    save MAXEFFORT;
  begin
    case type(sol)
    of piecewise do
      return(piecewise::extmap(sol, checkRecursive, sys, vars, options))
    of "_union" do
    of "_intersect" do
      MAXEFFORT:= MAXEFFORT/nops(sol);
      return(map(sol, checkRecursive, sys, vars, options))
    of "_minus" do
      return(checkRecursive(op(sol, 1), sys, vars, options)
      minus
      op(sol, 2)
      )
    of DOM_SET do
      if options[IgnoreAnalyticConstraints] then
        return(select(sol, check))
      else
        if nops(sol) > 0 then
          MAXEFFORT:= MAXEFFORT/nops(sol)
        end_if;
        result:= spurious:= {};
        pws:= []:
        for i from 1 to nops(sol) do
          cond:= check(op(sol, i));
          if length(cond) > MAXCONDLENGTH and freeIndets(cond) <> {} or 
            traperror((
          pw:= pcw([cond, {op(sol, i)}],
          [not cond, {}]
          )), MaxSteps = 1) <> 0 then
            warn:= TRUE;
            result:= result union {op(sol, i)}
          elif type(pw) <> piecewise then
            result:= result union pw
          else
            pws:= pws.[pw];
            spurious:= spurious union {op(sol, i)}
          end_if;
        end_for;
        // if there are too many piecewises, then do not do a case analysis
        // because this would be too slow
        if nops(pws) > MAXNUMBEROFPIECEWISES then
          result:= result union spurious;
          warn:= TRUE
        else
          result:= result union _union(op(pws))
        end_if;  
      end_if;
      return(result)
    end_case;
    
    // default 
    warn:= TRUE;
    sol
  end_proc;
  
  warn:= FALSE;
  sol:= checkRecursive(sol);
  if warn and not options[NoWarning] and not options[IgnoreAnalyticConstraints] then
    warning("possibly spurious solutions")
  end_if;
  sol
end_proc:

// end of file

