//   

// solvelib::makeSystem(l, varset)

// converts l into a system of equations that
// does not contain non-algebraic subexpressions in x
// by adding some additional equations

/*

   l - expression
   varset - set of identifiers


   returns:

   - a system of algebraic equations eqsystem
   - a set of remaining (non-algebraic) equations
   - a list of variables vars
   - a list of conditions

  such that the coordinate labelled by x=...
  of every solution of the eqsystem and the nonalg. eq.
  for vars whose i-th
  entry satisfies the i-th condition is also a solution to l 


*/

solvelib::makeSystem:=
proc(l: Type::Union(DOM_LIST, DOM_SET), vars: Type::ListOf(DOM_IDENT))
  local subslist, i, j, s, rsid, nthroot, n,
  eqsystem: DOM_SET,
  nonalgebraic: DOM_SET,
  conditions: DOM_LIST;
  
begin

  l:= subs(l, hold(exp) =
                proc(u)
                  begin
                    if stdlib::hasmsign(u) then
                      1/exp(-u)
                    else
                      exp(u)
                    end_if
                end_proc,
                EvalChanges 
                );
  
  // turn l into a rational equation, with many variables
  [eqsystem, subslist]:= [rationalize(l, StopOn = (subexpr-> bool(indets(subexpr) intersect {op(vars)} = {}) or
              contains({DOM_INT, DOM_RAT, DOM_IDENT}, type(subexpr))),
              FindRelations = ["_power", "exp"],
              Recursive)];
  
  // subslist is a list of equations new_variable = irrational expression

  nonalgebraic:={};
  conditions:=[TRUE];
  
  for i from nops(subslist) downto 1 do
    s:= subslist[i];  // i-th equation
    // substitute in former operands wherever possible
    for j from 1 to i-1 do
      subslist[j]:=subs(subslist[j], rhs(s)=lhs(s))
    end_for;
    rsid:=rhs(s);
    /* if indets(rsid) intersect {op(vars)}={} then
      next
    end_if; */ 
    case type(rsid)
      of "_power" do
        assert(type(op(rsid,2)) <> DOM_INT);
        if type(op(rsid, 2))=DOM_RAT then
          // exponent is a fraction
          
        // if var = a^(p/q), write this as var^q=a^p
        // this in not an equivalence; hence it must be
        // checked whether the resulting var can be a q-th root at all
        // do this only if p and q are not too large
          if max(map([op(op(rsid, 2))], specfunc::abs)) >= Pref::autoExpansionLimit() then
            userinfo(10, "rational power with too large numerator or denominator");
            nonalgebraic:=nonalgebraic union {s};
            vars:=append(vars, lhs(s));
            break
          end_if; 
          eqsystem:= eqsystem union
                     { lhs(s)^denom(op(rsid, 2)) = op(rsid,1)^numer(op(rsid,2))};
          vars:= append(vars, lhs(s));
          nthroot:=lhs(s);
          n:=denom(op(rsid, 2));
          if n=2 then
            conditions:= append(conditions,
                                Re(nthroot)>=0 and Im(nthroot)>=0
                                or Im(nthroot)>0)
          else
            conditions:= append(conditions,
                                arg(Re(nthroot), Im(nthroot)) >= 0 and
                                arg(Re(nthroot), Im(nthroot)) <
                                2*PI/denom(op(rsid,2)))
          end_if;
          break
        else
          userinfo(10, "not a rational power");
          nonalgebraic:=nonalgebraic union {s};
          vars:=append(vars, lhs(s));
        end_if;
        break
      otherwise
        userinfo(10, "Cannot handle right hand side ".
                 expr2text(rhs(s)));
        nonalgebraic:= nonalgebraic union {s};
        vars:=append(vars, lhs(s));
    end_case;
    
  end_for;

  eqsystem, nonalgebraic, vars, conditions 
end_proc:
