/*--
	solvelib::solve_sys
--*/

// Sys - set or list of equations
// Unk - set or list of unknowns


solvelib::solve_sys :=
proc(Sys: DOM_LIST, Unk: DOM_LIST)
  local
  sys:DOM_LIST,
  newsys, i, j, k, m, newsys2, eq, R, pw,
  unk: DOM_LIST,
  protectedvars,
  newunk: DOM_LIST, otherunk: DOM_LIST, dummy: DOM_LIST,
  newunkpos, otherunkpos,
  options: DOM_TABLE,
  opt: DOM_TABLE,
  eqtypes, ring,
  sortfunc: DOM_PROC,
  makeExpression: DOM_PROC,
  mapfloat: DOM_PROC,
  toSplit: DOM_LIST, // or-conditions where each operand leads to a separate
                     // system
  ineqs: DOM_LIST, // list of constraints that must *not* hold TRUE
  leqs: DOM_LIST,  // inequalities of type "_less" or "_leequal"
  newIndices: DOM_LIST,
  solutions, sol, sls,
  IntModFlag: DOM_BOOL, // is one coefficient ring equal to IntMod(..) ?
  maxeffort, cond, 
  linears1: DOM_LIST, linears2: DOM_LIST, linears: DOM_LIST,
  tried: DOM_LIST;
  
  save MAXEFFORT;
  
begin

/********************************
  local methods
********************************/
  
  // makeExpression - turns any legal input (equation, polynomial, ...)
  // into a (sequence of) expression(s) and adjusts the coefficient domain at
  // the same time
  makeExpression:=
  proc(equation)
    local ring;
  begin
    case type(equation)
      of "_unequal" do
        ineqs:= ineqs.[op(equation, 1) - op(equation, 2)];
        null();
        break
      of "_not" do
        ineqs:= ineqs.[op(equation, 1)];
        null();
        break
      of "_in" do
        equation:= expand(equation);
        if type(equation) <> "_in" then
          makeExpression(equation)
        elif op(equation, 2) = C_ then
          // true
          0
        else
          equation
        end_if;
        break
      of "_subset" do 
        if type(op(equation, 1)) = DOM_SET then
           return(makeExpression(op(equation, [1, i]) in op(equation, 2)) $i=1..nops(op(equation, 1)))
        else
           warning("Cannot solve systems containing expressions of type _subset");
           null();  
        end_if;
        break
      of "_less" do
      of "_leequal" do
        leqs:= leqs.[equation];
        null();
        break
      of "_equal" do
        return(makeExpression(op(equation,1)-op(equation,2)))
      of "_or" do
      // split the system into several systems, one for each operand
        equation:= select(equation, x -> is(x) <> FALSE);
        if type(equation) <> "_or" then
          return(makeExpression(equation))
        end_if;
        toSplit:= toSplit.[equation];
        null();
        break
      of "_mult" do
        equation:= select([op(equation)], x -> is(x=0) <> FALSE);
        if nops(equation) = 0 then
          return(1)
        elif nops(equation) = 1 then
          return(makeExpression(op(equation, 1)))
        end_if;
        // split the system into several systems, one for each operand
        toSplit:= toSplit.[_or(op(equation, i) = 0 $i=1..nops(equation))];
        null();
        break
      of Factored do 
        if nops(equation) = 3 then
          return(makeExpression(op(equation, 2)))
        elif nops(equation) = 1 then
          return(makeExpression(op(equation, 1)))
        else 
          return(makeExpression(hold(_mult)(op(equation, 2*i) $i=1..nops(equation) div 2)))
        end_if
      of stdlib::Undefined do  
      of "exp" do
        // unsatisfiable, since exp has no zeroes
        return(1)
      of "_and" do
        map(op(equation), makeExpression);
        break
      of DOM_BOOL do
        if equation = TRUE then
          0
        else
          1
        end_if;
        break
      of DOM_POLY do
        ring:= op(equation, 3);
        if has(ring, IntMod) then
          ring:= Dom::IntegerMod(op(ring,1));
          IntModFlag:= TRUE;
        end_if;
        if R=hold(Expr) then
          R:=ring
        else
          // try to understand equation
          // as a polynomial over the previously chosen ring
          equation:= poly(equation, R);
          if equation = FAIL then
            error("Coefficient ring of polynomial does not ".
                  "match previously chosen domain for solving".
                               " or other coefficient ring");
          end_if;
        end_if;
        expr(equation);
        break
      otherwise
        if equation::dom::hasProp(Dom::Matrix) = TRUE then
          // should we warn? E.g., warning("Matrix used instead of a scalar inside a system");
          return(op(map([op(equation)], makeExpression)))
        end_if;
        equation
    end_case
  end_proc;


  /*
     local method mapfloat(S)
     converts a symbolic solution S into a floating-point solution
  */

  mapfloat:=
  proc(S)
  begin
    case type(S)
      of "solve" do
        traperror((S:= float(S)));
        if type(S) <> "solve" then
          S:= solvelib::vectorForm(S, unk)
        end_if;
        return(S)
      of "_intersect" do
      of "_union" do
      of "_minus" do
        return(map(S, mapfloat))
    end_case;
    // default
    float(S)
  end_proc;
        
  
  
  //////////////////////////////////////  
  // m a i n 
  /////////////////////////////////////

  // argument checking and conversion 

  sys:= Sys;

  if {op(Unk)} intersect Type::ConstantIdents <> {} then
    error("Invalid variable (mathematical constant was given)")
  elif nops(Unk) = 0 then
    error("Need variable(s) to solve for")
  end_if;
  unk:=[op(Unk)];
  
  options:= solvelib::getOptions(args(3..args(0)));

  // The case VectorFormat = FALSE has been caught in the main program
  // of solve
  options[VectorFormat] := TRUE;

  
  // special case: only one unknown
  if nops(unk) = 1 then
    sol:= solve(sys, unk[1], options);
    if has(sol, hold(solve)) then
      return(hold(solve)(sys, Unk, solvelib::nonDefaultOptions(options)))
    else  
      return(solvelib::numberSetToVectorSet(sol))
    end_if
    
  end_if;
  
  

  // check for protected identifiers
  // some methods like normalGroebner behave differently for them,
  // so we eliminate them right here
  protectedvars:= select(freeIndets([sys, Unk]), X -> protected(X) <> None);
  if nops(protectedvars) > 0 then
    newunk:= [genident() $j=1..nops(protectedvars)];
    sys:= subs(sys, op(protectedvars, i) = newunk[i] $i=1..nops(newunk));
    Unk:= subs(Unk, op(protectedvars, i) = newunk[i] $i=1..nops(newunk));
    sol:= solve(sys, Unk, options);
    return(subs(sol, newunk[i] = op(protectedvars, i) $i=1..nops(newunk)))
  end_if;
  


  
  if options[IgnoreSpecialCases] then
    pw:= solvelib::ignoreSpecialCases
  else
    pw:= piecewise
  end_if;

  
  // handling of floats
  if stdlib::hasfloat(sys) then
    sys:=numeric::rationalize(sys);
    sol:= solvelib::solve_sys(sys, Unk, options);
    sol:= mapfloat(sol);
    if hastype(sol, "solve") then
      // return the original system
      return(hold(solve)(sys, Unk, solvelib::nonDefaultOptions(options)))
    else
      return(sol)
    end_if
  end_if;


  // assumptions are just added as additional equations
  if not options[IgnoreProperties] then
    sys:= sys.property::showprops(unk);
    options[IgnoreProperties]:= TRUE
  end_if;

  if options[Real] then
    map(unk, assume, Type::Real, _and)
  end_if;
  
  R:= options[Domain];
  // convert all equations and polys to expressions
  // as a side effect, separate inequalities
  ineqs:= [];
  leqs:= [];
  toSplit:= [];
  IntModFlag:= FALSE; // no IntMod(..) yet
  sys:=map(sys, makeExpression);
  options[Domain]:= R;
  
  newunk:= split(sys, testtype, DOM_IDENT);
  if nops(newunk[1]) > 0 and has(newunk[2], newunk[1]) then
    if traperror((sys:= evalAt(newunk[2], map(newunk[1], _equal, 0)))) <> 0 then
       return({})
    end_if;
    toSplit:= evalAt(toSplit, map(newunk[1], _equal, 0));
    sys:= select(sys, _unequal, 0);
    sys:= newunk[1].sys;
  end_if;

  if nops(toSplit) > 0 then
    /*
    eq:= toSplit[1];
    delete toSplit[1];
    */
    // DNF
    toSplit:= expand(_and(op(toSplit)));
    if type(toSplit) <> "_or" then
      toSplit:= [toSplit]
    end_if;  
    return(
           _union(solvelib::solve_sys(sys.ineqs.leqs.[toSplit[i]],
                                      unk, options)
                  $i=1..nops(toSplit))
           )
  end_if;




  // try overloading
  eqtypes:= map(sys, domtype);
  eqtypes:= select(eqtypes, c -> (slot(c, "solve_sys")<>FAIL));
  if nops(eqtypes) > 0 then
    userinfo(50, "System solver has been overloaded");
    if leqs <> [] then
      error("Cannot solve systems with inequalities over domains")
    end_if;
    ring:= eqtypes[1];
    sys:= map(sys, ring::convert);
    ineqs:= map(ineqs, ring::convert);
    if has([sys, ineqs], FAIL) then
      error("Could not convert all of the equations")
    end_if;
    if ineqs <> [] then
      return(ring::solve_sys(sys, Unk, options) minus
             ring::solve_sys(ineqs, Unk, options))
    else
      return(ring::solve_sys(sys, Unk, options))
    end_if;
  end_if;


  if R <> hold(Expr) then
    if leqs <> [] then
      error("Cannot solve systems with inequalities over domains")
    end_if;
    if ineqs <> [] then
      return(solvelib::domsolve(sys, Unk, options) minus
             solvelib::domsolve(ineqs, Unk, options))
    else
      sol:= solvelib::domsolve(sys, Unk, options);
      if IntModFlag then
        // the polynomial ring was IntMod(..), not IntegerMod
        // we have to convert all entries of the vectors
        // from Dom::IntegerMod into expressions
        assert(type(sol) = DOM_SET);
        sol:= expr(sol)
      end_if;
      return(sol)
    end_if
  end_if;

  // toDo: single objects overload solve - intersect solutions
  // with solution to the rest of the system  
  
  if options[Multiple]=TRUE then
    error("Option multiple is not allowed in the system solver")
  end_if;


  userinfo(3, "System solver called for ".expr2text(sys)." in unknowns ".
	expr2text(Unk));
  userinfo(10, "Options are ".expr2text(options));
    

  // unsatisfiable equations?
  if contains(map(sys, X -> if testtype(X, Type::Arithmetical) then is(X=0) end_if), FALSE) > 0 then
    return({})
  end_if;

  // handle inequalities
  if nops(ineqs) > 0 then
    // delete properties
    assume(unk[i] in C_) $i=1..nops(unk);
    if nops(sys) = 0 and nops(leqs) > 0 then
      if nops(ineqs) > 0 then
        MAXEFFORT:= MAXEFFORT/3;
        return(solvelib::solve_minus(solvelib::ineqsys(leqs, Unk, options),
                                     solve(ineqs, unk, options) ))
      else
        return(solvelib::ineqsys(leqs, Unk, options))
      end_if
    end_if;
    // we can only handle a few cases of mixed systems, unfortunately
    MAXEFFORT:= MAXEFFORT/2;
    solutions := solvelib::solve_sys(sys, unk, options);
    if solutions = {} then
       return({})
    end_if;
    // to improve: check solutions instead of solving exceptions ?!
    for cond in ineqs do
      solutions:= solvelib::solve_minus(solutions, solve(cond, unk, options));
      if solutions = {} then
        return({})
      end_if;
    end_for;
    if nops(leqs) > 0 then
      return(solvelib::solve_intersect(solvelib::ineqsys(leqs, Unk, options), solutions))
    else
      return(solutions)
    end_if
  end_if;
  // handle inequalities of type _leequal and _less
  if leqs <> [] then
    if nops(sys) = 0 then // only inequalities to solve
      return(solvelib::ineqsys(leqs, Unk, options))
    end_if;
    MAXEFFORT:= MAXEFFORT/2;
    sol:= solvelib::solve_sys(sys, unk, options)
    assuming _and(i in C_ $i in unk);
    if type(sol) = DOM_SET then
      sol:= [op(sol)];
      cond:=[_and(evalAt(op(leqs, j), zip(unk, [op(i)], _equal))
                  $j=1..nops(leqs))
             $i in sol];
      sol:= [pw([cond[i], {sol[i]}], [not cond[i], {}])
             $i=1..nops(sol)];
      return(_union(op(sol)))
    else
      solutions:= solvelib::ineqsys(leqs, Unk, options);
      if hastype(solutions, "solve") then
        return(hold(solve)(Sys, Unk, solvelib::nonDefaultOptions(options)))
      end_if;
      sol:= solvelib::solve_intersect(sol, solutions);
      return(sol)
    end_if
  end_if;

  
  // remove zero equations
  sys:= select(sys, _not@iszero);

  // no equations left?
  if nops(sys)=0 then
    return(solvelib::cartesianProduct(getprop(Unk[i]) $i=1..nops(Unk)))
  end_if;

   
  
  // substitute unknowns that are not identifiers
  // substitute all indexed identifiers, including free parameters
  eq:= select({op(unk)}, not testtype, DOM_IDENT) union
       select(solvelib::indets(sys), x -> type(x) <> DOM_IDENT);
  if nops(eq) > 0 then 
    if map(eq, testtype, Type::Arithmetical) <> {TRUE} then
      error("Illegal variable to solve for")
    end_if;
    eq:= [(op(eq,i) = genident()) $i=1..nops(eq)];
    newsys:= solvelib::solve_sys(subs(sys, eq), subs(Unk, eq), options);
    sol:= subs(newsys, (op(eq,[i,2])=op(eq,[i,1])) $i=1..nops(eq));
    return(sol)
  end_if;




  // at this point, our system sys contains no more objects that
  // overload solve, or inequalities, or objects of type _not
  // however, there may still be non-expandable objects of type _in
  if select(sys, proc(eq)
                   begin
                     testtype(eq, Type::Arithmetical) <> TRUE and
                     type(eq) <> "_in"
                 end_proc)
    <> [] then
    error("Systems contains equation of unknown type")
  end_if;

  // we have to do the following things:
  // - split products and powers
 //  - solve _in - expressions

  [ineqs, sys, leqs]:= split(sys, eq-> type(eq) = "_in");

  
  if nops(ineqs) > 0 then
    for i from 1 to nops(ineqs) do
      if has(op(ineqs[i], 2), {op(Unk)}) then
        return(hold(solve)(Sys, Unk, solvelib::nonDefaultOptions(options)))
      end_if;
      assume(ineqs[i], _and);
      options[IgnoreProperties] := FALSE
    end_for;
  end_if;
 
  


  // first, we expand sums (e.g. sum(a+x[i], i=...))
  sys:= misc::maprec(sys, {"sum"} = expand);

  
  // remove powers

  sys:= map(sys, proc(eq)
                 begin
                   if type(eq)<>"_power" then
                     eq
                   elif testtype(op(eq,2), Type::Constant)<>TRUE then
                     eq
                   else 
                     case is(op(eq, 2)>0)
                     of TRUE do
                       return(op(eq, 1))
                     of FALSE do  
                       // negative exponent: unsolvable
                       return(FAIL)
                     otherwise // UNKNOWN
                       eq
                     end_case  
                   end_if
                 end_proc
                 );
  if contains(sys, FAIL)>0 then
    return({})
  end_if;
  

  
  
  // handle linear systems
  // parameters are handled via ShowAssumptions, over Expr only


  if (newsys:= solvelib::solve_islinear(sys, unk, options)) <> FALSE and
    // avoid a problem in solvelib::solve_islinear
    indets(misc::subExpressions(newsys, {"Re", "Im"})) intersect {op(unk)} = {}
    then
    /* solve returns a set of solutions
       whereas linsolve always returns one solution */
    assert(R = hold(Expr));
    newsys:= rationalize(newsys, StopOnConstants);
    // it may still happen that the system is nonlinear, as diff(f(c), c) may also give zero 
    // if f is a function environment (int, limit, ..) that determines its derivative in a special way. 
    // however, then c occurs syntactically within the substituted irrational expressions
    if not has(op(newsys, 2), {op(unk)}) then      
      i:= numeric::linsolve(op(newsys, 1), unk, hold(Symbolic),
      hold(ShowAssumptions));
      // assume that all equations are defined, in particular,
      // that all denominators of the coefficients are nonzero
      map([coeff(poly(op(newsys, [1, i]), unk)) $i=1..nops(op(newsys, 1))],
      proc(u)
      begin
        context(hold(assumeAlso)(denom(u) <> 0));
      end_proc
      ):
      
      solutions:= solvelib::completeLinearSolution
      (i, op(newsys, 1), unk, options);
      
      return(subs(solutions, op(newsys, 2)))
      
    end_if;
  end_if;



  sol:= solvelib::systemAlgebraic(sys, unk, Unk, options);
  if sol <> FAIL then
    return(sol)
  end_if;  
  
  
 

  
if MAXEFFORT > 5000 then
  // handle triangular systems
  for i from 1 to nops(unk) do
    otherunk:= [op(unk, 1..i-1), op(unk, i+1..nops(unk))];
    newsys:= split(sys, has, unk[i]);
    case nops(newsys[1])
      of 0 do
        // system does not depend on unk[i]
        solutions:= solve(sys, otherunk, options);
        if hastype(solutions, "solve") then
          return(hold(solve)(sys, Unk, solvelib::nonDefaultOptions(options)))
        else
          solutions:= solvelib::avoidAliasProblem(solutions, {unk[i]});
          solutions:= solvelib::selectIndices
          (
           (solvelib::cartesianProduct
            ({unk[i]}, solutions)
            ),
           [i, $1..i-1, $i+1..nops(unk)]
           );
          if solutions = FAIL then
            break
          end_if;
          return(solvelib::Union
                 (solutions,
                  unk[i], getprop(unk[i]))
                 )
        end_if
      of 1 do
        // unk[i] shows up in exactly one equation
        MAXEFFORT:= MAXEFFORT/2;
        solutions:= solve(newsys[1][1], unk[i], options);
        if not has(solutions, hold(solve)) then  
          sol:= solvelib::solve_sys(newsys[2], otherunk, options);
          // solutions to sys are all vectors [a1, .., an] for which
          // [a1, ..., a_{i-1}, a_{i+1}, .., a_n] is in sol and
          // a_{i} is in solutions
          if not hastype(sol, "solve") then
            solutions:=
            solvelib::selectIndices
            (
            solvelib::cartesianProduct({otherunk}, solutions),
            [$1..i-1, nops(unk), $i..nops(unk)-1]
            );
            if solutions = FAIL then
              break
            end_if;
            return(solvelib::Union(solutions, otherunk, sol))
          end_if;
        elif freeIndets(newsys[1][1]) intersect {op(unk)} = {unk[i]} then
          // no other variables occur in the equation. Thus it cannot become simpler 
          // by plugging in the solution of some other equation into some other variable
          return(hold(solve)(sys, Unk, solvelib::nonDefaultOptions(options))) 
        end_if; 
        // reset effort to old value
        MAXEFFORT:= MAXEFFORT*2
    end_case;
  end_for;
end_if;

  // additional heuristic: move transcendental function to the right if possible
  if options[IgnoreAnalyticConstraints] then
    sys:= solvelib::applyInverses(sys, unk, options)
  end_if;  


  // try to make the system into a polynomial system
  // by additional substitutions

  
  newsys:= map(sys, numer);
  // toDo: check whether solutions are zeroes of the denominator
  newsys:= solvelib::makeSystem({op(newsys)}, unk);
  // now newsys is a sequence equations, non-algebraic, vars, conditions
  if nops(op(newsys, 2)) = 0 and
    // no non-algebraic equations
    not hastype((sol:= solve(op(newsys, 1), op(newsys, 3), options)),
                "solve") then
    
    // find out the positions of the old variables    
    newIndices:= [contains(op(newsys, 3), op(unk, i) ) $i=1..nops(unk)];
    assert(contains(newIndices, 0) = 0);
    sol:= solvelib::selectIndices(sol, newIndices);
    if sol <> FAIL then
      return(solvelib::checkVectorSolutions(sol, sys, unk, options))
    end_if
  end_if;


  sol:= solvelib::tryRationalize(sys, unk, options);
  if sol <> FAIL then
    return(sol)
  end_if;  
  
  
  
  if MAXEFFORT < 500 then return(hold(solve)(sys, Unk, solvelib::nonDefaultOptions(options))) end_if;
  
 // handle linear equations in more complicated systems
 // first case: equation is of the form a*x + b = 0, where a is constant
 // second case: both a and b may depend on the other varaiables to solve for
 // handle first case first; instances of the second case are stored to be handled later

 // initialize set of all tuples i, j, eq such that the i-th equation is eq[1]*unk[j] + eq[2] and this linear equation falls under the second case
 linears1:= linears2:= []; 

 for i from 1 to nops(sys) do
   for j from 1 to nops(unk) do
     if ((eq:= Type::Linear(op(sys, i), [unk[j]]))) <> FALSE and not iszero(eq[1]) then
       if indets(eq[1]) minus Type::ConstantIdents = {} then
          // first case: handle this immediately
          // undetected zero?
          if is(eq[1] = 0, Goal = TRUE) then 
            sys:= subsop(sys, i = eq[2]);
            // we might also want to do a complete startover with the new system here
            next
          end_if;
          linears1:= linears1.[[i, j, eq]]
       else
          // eq[1] is not a constant
          // second case, handle later on
          linears2:= linears2.[[i, j, eq]]
       end_if;
     end_if;
   end_for;
 end_for;

 // determine priority: we want to solve "simple" equations first
 linears1:= prog::sort(linears1, length);
 linears2:= prog::sort(linears2, length);
 linears:= linears1.linears2;
 
 // keep track for which variables we already tried to solve
 tried:= [FALSE $nops(unk)];

 
 
 maxeffort:= MAXEFFORT;  
 for k from 1 to nops(linears) do
     if k>=2 and nops(linears1) >= 1 then
       // give up
       return(hold(solve)(sys, Unk, solvelib::nonDefaultOptions(options)))
     end_if;
     [i, j, eq]:= linears[k];
     tried[j]:= TRUE;
     if k > nops(linears1) then
       MAXEFFORT:= MAXEFFORT*4/5
     end_if;   
     // sys[i] is eq[1]*unk[j] + eq[2] = 0
     newsys:= sys; delete newsys[i];
     newunk:= unk; delete newunk[j];
     // solve sys[i] for unk[j], and substitute into the remaining system
     // newsys, to be solved for the remaining unknowns newunk
     // solutions of the original system are:
     // all values of newunk that make both eq[1] and eq[2] equal zero,
     // together with any value of unk[j], as far as they also satisfy
     // the remaining system newsys
     // *and*
     // all values of newunk that make eq[1] nonzero, such that
     // unk[j] -> -eq[2]/eq[1] may be substituted, as far as they satisfy
     // the remaining system after that substitution
     
     
     // solve case eq[1] = eq[2] = 0 first
     // solutions:= solvelib::solve_sys([eq[1], eq[2]], newunk, options);
     
     
     if traperror((newsys2:= subs(newsys, unk[j] = -eq[2]/eq[1]))) <> 0
     then
       solutions:= {}
     else
       sol:= solvelib::solve_minus
       (
       solvelib::solve_sys
       (newsys2, newunk, options),
       solvelib::solve_sys
       ([eq[1]], newunk, options)
       );
       assert(contains({nops(newunk), -1}, solvelib::dimension(sol)));
       if hastype(sol, "solve", {Type::Set}) then
         next
       end_if;
       solutions:=
       solvelib::substituteBySet
       (
       matrix(subsop(unk, j= -eq[2]/eq[1])),
       newunk,
       sol
       );
       if solutions = FAIL then
         // we do no break here since there are examples where solving for another varaible helps
         next
       end_if
     end_if;
     
     
     if not hastype(solutions, "solve", {Type::Set}) then
       sol:= solvelib::solve_sys(newsys.eq, unk, options);
       if not hastype(sol, "solve") then
         return(solutions union sol)
       end_if;
     end_if;
   end_for;
   if nops(linears) > 0 then MAXEFFORT:= maxeffort/2 end_if;
 


  if MAXEFFORT < 10000 then
    return(hold(solve)(sys, unk, solvelib::nonDefaultOptions(options)))
  end_if;
  

  // test whether the system is polynomial in at least some of the
  // variables; if yes, solve at least for them
   [newunk, otherunk, dummy]:= split(unk,
                  proc(x)
                  begin
                    not has(indets(sys, PolyExpr) minus {x}, x)
                  end_proc
                  );
  assert(nops(dummy) = 0);
  // now newunk is a list consisting of all "good" unknowns
  // select those equations that contain only "good" unknowns,
  // and solve for them
  if nops(newunk) > 0 then
    newsys:= select(sys, equ -> indets(equ) intersect ({op(unk)}
                    minus {op(newunk)}) = {});
    maxeffort:= MAXEFFORT:= MAXEFFORT/2;                
    solutions:= solvelib::solve_sys(newsys, newunk, options);
    if type(solutions) = DOM_SET then
      if solutions = {} then
        return({})
      end_if;
      assert(solvelib::dimension(solutions) = nops(newunk));
      assert(map(solutions, type) = {matrix} );

      otherunkpos:= map(unk, x-> contains(otherunk, x));
      newunkpos:= map(unk, x-> contains(newunk, x));
        // for each solution (given as a list), append the
        // solutions for the non-polynomial vars
      MAXEFFORT:= MAXEFFORT/nops(solutions);
      assert(nops(newunk) + nops(otherunk) = nops(unk));
      solutions:=
      _union(op
             (map
              (solutions,
               proc(v)
                 local nsys, solnsys;
               begin
                 assert(nops(v) = nops(newunk));
                 nsys:= subs(sys, newunk[i] = v[i] $i=1..nops(newunk),EvalChanges);
                 solnsys:= solvelib::solve_sys(nsys, otherunk, options);
                 assert(solnsys = {} or solvelib::dimension(solnsys) = nops(otherunk));
                 solvelib::cartesianProduct({v}, solnsys)
               end_proc
               )));
      assert(solutions = {} or solvelib::dimension(solutions) = nops(unk));
      solutions:= solvelib::selectIndices
      (solutions, [(if newunkpos[i] <> 0 then
                      newunkpos[i]
                    else
                      otherunkpos[i] + nops(newunk)
                    end_if)
                   $i=1..nops(unk)
                   ]
       );
      if solutions <> FAIL then
        return(solutions)
      end_if
      
      // else fall through
    end_if;
  else // nops(newunk) = 0
    userinfo(50, "System is not polynomial in any of the unknowns")
  end_if;
  


  assert(nops(unk) > 1); // the case of one unknown has been handled above
  MAXEFFORT:= MAXEFFORT/5;
  
  // very last attempt. Use the equation solver for any equation with respect to any of the variables
  // we us a crude heuristic to sort all possible pairs of equation and variable
  
  sortfunc:= 
  proc(l)
    local t, vl, i, j;
  begin
  [i, j]:= l;
  if traperror((vl:= evalAt(op(sys, i), {unk[t] = 0  $t=1..j-1, unk[t]=0 $t=j+1..nops(unk)}))) = 0 then
    length(vl)
  elif traperror((vl:= evalAt(op(sys, i), {unk[t]= 42/29  $t=1..j-1, unk[t]=42/29 $t=j+1..nops(unk)}))) = 0 then 
    length(vl)
  else
    length(op(sys, i))
  end_if  
  end_proc;
  
  linears:= select([$1..nops(unk)], j-> not tried[j]);
  linears:= prog::sort([([i, j] $i=1..nops(sys)) $j in linears], sortfunc);
  
  // we now try to solve some equations for some of the variables, to some cutoff point
  if nops(linears) > 10 then
    linears:= [op(linears, 1..10)]
  end_if;
  
  opt:= options;
  // do not use IgnoreSpecialCases after turning variables into free parameters
  // reason: if e.g. x=0, y=0 is a solution, the recursive call would handle "x=0 if y=0" as 
  // special
  opt[IgnoreSpecialCases]:= FALSE;
  for k from 1 to nops(linears) do
    [i, j]:= linears[k];
    MAXEFFORT:= maxeffort:= MAXEFFORT * 4/5;
    solutions:= solve(sys[i], unk[j], opt);
    if type(solutions) = piecewise and options[IgnoreSpecialCases] then
      // handle special cases now
      MAXEFFORT:= min(100, maxeffort/10);
      solutions:= piecewise::mapConditions(solutions, solvelib::specialToFalse, 
      cond -> not has(cond, {op(unk)})
      );
      MAXEFFORT:= maxeffort* 3/4  
    end_if;  
    // do only handle the case of finite sets and piecewises of finite sets at the moment      
    case type(solutions) 
    of DOM_SET do
      newsys:= sys;
      newunk:= unk;
      delete newsys[i];
      delete newunk[j];
      sol:= [FAIL $nops(solutions)];
      for m from 1 to nops(solutions) do
        // solve the remaining system, with the j-th unknown replaced by the solution
        sol[m]:= solvelib::solve_sys(evalAt(newsys, unk[j] = op(solutions, m)), newunk, opt);
        if type(sol[m]) = piecewise and options[IgnoreSpecialCases] then
          sol[m]:= piecewise::disregardPoints(sol[m])
        end_if;
        sls:= subsop(unk, j = op(solutions, m));
        sol[m]:= solvelib::substituteBySet(matrix(sls), newunk, sol[m]);
        if sol[m] = FAIL then break end_if;
      end_for;  
      if contains(sol, FAIL) > 0 then break end_if;
      return(_union(op(sol)))
    end_case
  end_for;

  
  userinfo(1,"Could not solve the non-algebraic system");
  hold(solve)(sys, Unk, solvelib::nonDefaultOptions(options))
 
end_proc:


solvelib::systemAlgebraic:=
proc(sys, unk, Unk, options)
  local i, solutions, sol, newsys;
begin  
  if MAXEFFORT = RD_INF then
    i:= 100000
  else
    i:= ceil(MAXEFFORT/1000)
  end_if;
 
  case traperror((newsys:= solvelib::solve_isalgebraic({op(sys)}, unk)), MaxSteps = i) 
  of 1321 do
    return(hold(solve)(sys, Unk, solvelib::nonDefaultOptions(options)))
  of 0 do 
 
    if newsys <> FALSE then
      case traperror
        ((solutions:= solvelib::solve_algebraic([op(op(newsys, 1))], unk, options
        )), 
        if MAXEFFORT = RD_INF then
          MaxSteps = 100000
        else
          MaxSteps = ceil(MAXEFFORT/1000)
        end_if
        )
      of 0 do
      // select correct solutions, depending on op(newsys, 2)
        if options[IgnoreAnalyticConstraints] then
          return(solvelib::checkVectorSolutions(solutions, sys, unk, options))
        else  
          if solutions = {} then
            return(solutions)
          elif traperror((
            sol:= _union(
            solve((op(newsys, [2, i])), unk, options) 
            $i=1..nops(op(newsys, 2)) )), MaxSteps = 10) <> 0 and not options[NoWarning] then
            warning("Possibly spurious solutions");
            return(solutions)
          else
            MAXEFFORT:= min(MAXEFFORT/2, 2000);
            solutions:= solvelib::solve_minus(solutions, sol);
            if options[IgnoreSpecialCases] and type(solutions) = piecewise then
              solutions:= piecewise::disregardPoints(solutions)
            end_if;
            return(solutions)
          end_if
        end_if 
      of 1321 do
        return(hold(solve)(sys, Unk, solvelib::nonDefaultOptions(options)))
      otherwise
        lasterror()
      end_case;
    end_if;
  end_case;
  
  FAIL  
end_proc:
  
solvelib::tryRationalize:=
proc(sys: DOM_LIST, unk: DOM_LIST, options: DOM_TABLE): Type::Union(DOM_FAIL, Type::Set)
  local newsys, newunk, subst, variables, i, j, k, eq, inds, X, x, hx, sol, S, indexpos,
  solutionsAtSingularities, newsys2
  ;
begin

  sys:= solvelib::expandTrig(sys, unk);  
  
  // rationalize the system
  [newsys, subst]:= [rationalize(sys, StopOnConstants, FindRelations = ["_power", "exp", "sin"])];
 
  
  // if
  // 1) every variable x of the system occurs only once on a right hand side of a substitution X = g(x) and
  // 2) each variable that appears on a right hand side does not appear in newsys anymore
  // then
  // we can solve newsys for those variables that do not appear on a right hand side, and the newly generated ones
  // for which the right hand side contains a variable
  // in the end, we have to solve for the original variables again
  
     
  if {op(unk)} intersect indets(newsys) intersect indets(subst) <> {} then
    return(FAIL)
  end_if;

  subst:= [op(subst)];
  variables:= [FAIL $nops(subst)];
  S:= table();
  newunk:= unk;
  indexpos:= {};
  
  solutionsAtSingularities:= {};
  
  for i from 1 to nops(subst) do
    eq:= op(subst, i);
    assert(type(eq) = "_equal");
    inds:= freeIndets(rhs(eq)) intersect {op(unk)};
    if nops(inds) >= 2 then
      // we cannot do this
      return(FAIL)
    end_if;
    if nops(inds) = 1 then
      x:= op(inds, 1);
      X:= op(eq, 1);
      j:= contains(unk, x);
      if not contains(indexpos, j) then 
        // we have a substitution of the form X = h(x), for some function h
        // special case: tan. If the right hand side originally does not occur, then
        // we have introduced a pole that we have to remove now
        if type(op(eq, 2)) = "tan" and not has(sys, op(eq, 2)) then
          // want to solve X = tan(h(x))
          // this gives arctan(X) = h(x) + PI*Z_ because we do not have to care about the pole
          hx:= op(eq, [2, 1]);
          if options[IgnoreAnalyticConstraints] then
            S[X]:= solve(hx = arctan(X), x, options)
          else  
            S[X]:= solvelib::preImage(hx, x, arctan(X) + PI*Z_, options);
            // tan(h(x)) has a singularity at h(x) = PI/2 + k*PI, k in Z_
            // nevertheless, the original system may have a solution there
            // hx occurs in the form sin(2*h(x)) and cos(2*h(x)) in the original system;
            // by periodicity, we need only check h(x) = PI/2
            // if sin(2*h(x)) and cos(2*h(x)) appear in the denominator such that plugging in is not possible, 
            // then this does not lead to solutions, and we may omit this step 
            if traperror((newsys2:= subs(sys, [sin(2*hx) = 0, cos(2*hx) = -1, tan(2*hx) = 0], EvalChanges))) = 0 then
              // if this did not catch all substitutions, then we give up
              if has(newsys2, x) then 
                return(FAIL)
              else
                if not has(newsys2, #k) then
                  k:= #k
                else
                  k:= genident("k")
                end_if;  
                newsys2:= newsys2.[hx - PI/2 + #k*PI];
                solutionsAtSingularities:= solutionsAtSingularities union
                solvelib::Union(solvelib::solve_sys(newsys2, unk, options), k, Z_)
              end_if  
            end_if  
          end_if  
        else 
          S[X]:= solve(eq, x, options)
        end_if;  
        if hastype(S, "solve") then
          return(FAIL)
        end_if;  
        newunk[j]:= X;
        indexpos:= indexpos union {j};
        subst[i]:= FAIL
      else
        // there has been another substitution X = h(x).
        return(FAIL)
      end_if 
    end_if;    
  end_for;
  
  subst:= select(subst, _unequal, FAIL);
  
  sol:= solve(newsys, newunk, options); 
 
  // substitute coordinates
  sol:= subs(solvelib::substituteCoordinates(sol, [[i, newunk[i], S[newunk[i]]] $i in indexpos]), subst, EvalChanges);
  if sol = FAIL then 
    sol 
  else
    sol
    union
    solutionsAtSingularities
  end_if  
end_proc:

/*
solvelib::substituteCoordinates(VS, l)

VS: vector set 
l: list of lists, each of the form [i, X, S(X)] where
     i: DOM_INT
     X: identifier
    S(X): set of scalars, depending on the parameter X

returns the union over all sets {(a1, ..., a_{i-1}, X, a_{i+1}, ...) ; X in S(a_i)} 
(with exactly those a_i replaced for which i equals some l[k][1], 1<= k <= nops(l))
where (a_1, ..., a_n) runs through the vector set VS. 


*/

solvelib::substituteCoordinates:=
proc(VS, l: DOM_LIST)
  local substvector: DOM_PROC, substImageSet: DOM_PROC, v, vars, res;
begin
  vars:= map(l, op, 2);
  
  substvector:= 
  proc(v)
    local j, i, X, S, sets;
  begin
    sets:= [FAIL $nops(l)];
    for j from 1 to nops(l) do
      [i, X, S]:= l[j];
      if traperror((sets[j]:= evalAt(S, X = v[i]))) <> 0 then
        return({})
      end_if;  
      v[i]:= X
    end_for;  
    solvelib::VectorImageSet(v, vars, sets)
  end_proc;
    
  
  /* IS = {[f1(y1, .., yk), ..., fn(y1, ..., yk)]; yi \in T_i }
     Given entries X_i = h_i(x_i) in l and the solutions x_i \in S(X_i)
     We have to solve h_1(x_1) = f1(y1, ..., yk), .., h_n(x_n) = f_n(y1,..., yk)
     This gives x_1 \in S_1(f_1(y1,.., yk)), ...
  
  
  */
  substImageSet:=
  proc(IS: solvelib::VectorImageSet)
    local v, j, i, X, S, sets, solution;
  begin
    // we start by handling the y_i as free parameters
    v:= expr(IS);
    sets:= [{v[i]} $i=1..nops(v)];
    for j from 1 to nops(l) do
      [i, X, S]:= l[j];
      if traperror((sets[i]:= evalAt(S, X = v[i]))) <> 0 then
        return({})
      end_if  
    end_for;  
    // now sets[i] = {fi(y1, ..., yk)} if nothing has been substituted; and
    //     sets[i] = S(fi(y1,..., yk)) otherwise
    // thus x_i in sets[i] is the solution
    solution:= solvelib::cartesianProduct(op(sets));
    // we have to attach back the information that y_i in T_i
    solvelib::Union(solution, IS::dom::variables(IS), solvelib::cartesianProduct(op(IS::dom::sets(IS))))
  end_proc;
  
  
  case type(VS)
  of "_union" do 
  of "_intersect" do
  of "_minus" do
    res:= map([op(VS)], solvelib::substituteCoordinates, l);
    if contains(res, FAIL) > 0 then
      return(FAIL)
    else  
      return(eval(op(VS, 0))(op(res)))
    end_if  
  of DOM_SET do
    return(_union(substvector(v) $v in VS))
  of solvelib::VectorImageSet do
    return(substImageSet(VS))
  of piecewise do 
    // the conditions do not depend on the variables to solve for, we may just map
    res:= piecewise::extmap(VS, solvelib::substituteCoordinates, l);
    if has(res, FAIL) then
      return(FAIL)
    else
      return(res)
    end_if
  end_case;  
  
  FAIL
end_proc:


solvelib::applyInverses:= 
proc(sys: DOM_LIST, unk: DOM_LIST, options: DOM_TABLE)
  local i, l, r, dummy, const, F;
begin
  for i from 1 to nops(sys) do
    if type(sys[i]) = "_plus" then 
      [l, r, dummy]:= split(sys[i], has, unk)
    else
      l:= sys[i];
      r:= 0
    end_if;
    if type(l) = "_mult" then
      // divide off constant factors
      [l, const, dummy]:= split(l, has, unk);
      r:= r/const;
    end_if;
    
    // the i-th equation is l = -r
    // if l = F(x) for a function we can invert, then we do so:
    // F(x) = -r becomes x = inverse(F)(-r)
    
    if type((F:= eval(op(l, 0)))) = DOM_FUNC_ENV and F::inverse <> FAIL then
      sys[i]:= op(l, 1) - (F@@(-1))(-r);
    end_if
    
  end_for;
  
  sys
end_proc:


/* end of file */

