/*++

Description:

        Chi -- The Hyperbolic Cosine Integral

	Chi(x):= EULER + ln(x) + int((cosh(t)-1)/t, t=0..x)
	       = ( - Ei(1, x) - Ei(1, -x) - PI*I*csign(x*I) ) / 2
               = ( - Ei(1, x) - Ei(1, -x) + PI*I*signIm(x) ) / 2


	           /  1    if Re(x) > 0 or Re(x) = 0 and Im(x) > 0
        csign(x) = |
                   \ -1    if Re(x) < 0 or Re(x) = 0 and Im(x) < 0

Properties:

  	Chi(x) = Chi(-x)+I*PI 	if x real, x < 0

Parameters:

  	x - an arithmetical expression

Examples:

	Chi(0.0)	->   singularity
	Chi(1.0)	->   0.837866941 
	Chi(-1.0)	->   0.837866941 + 3.141592654*I
	Chi(1.0*I)	->   0.3374039229 + 1.570796327*I
	Chi(-1.0*I)	->   0.3374039229 - 1.570796327*I
	Chi(1.0*I+2.0)	->   2.030296393 + 1.722703482*I
	Chi(-1.0*I+2.0)	->   2.030296393 - 1.722703482*I
	Chi(1.0*I-2.0)	->   2.030296393 + 1.418889172*I
	Chi(-1.0*I-2.0)	->   2.030296393 - 1.418889172*I
	Chi(-2.0*I-8.0)	-> - 27.31661519 + 205.0308091*I

++*/

Chi:= 
proc(x)
  option noDebug;
begin

  if args(0) <> 1 then 
     error("expecting one argument")
  elif x::dom::Chi <> FAIL then 
     return(x::dom::Chi(args())) 
  end_if;

  case type(x)
    of DOM_SET do
    of "_union" do
      return(map(x, Chi))
  end_case;
  
  if iszero(x) then 
     error("singularity") 
  end_if;
      
  case domtype(x)
    of DOM_INT do
    of DOM_RAT do
      if x < 0 then return(Chi(-x)+I*PI) end_if; break
    of DOM_FLOAT do
      return(Chi::float(x));
      break;
    of DOM_COMPLEX do
      if domtype(op(x,1)) = DOM_FLOAT or
         domtype(op(x,2)) = DOM_FLOAT then
        return(Chi::float(x))
      end_if;
  end_case;

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      if type(x)=Dom::ImageSet then
        return(map(x, Chi));
      else
        return(Dom::ImageSet(Chi(#x), #x, x));
      end_if;
    end_if;
    error("argument must be of 'Type::Arithmetical'")
  end_if;

  procname(x)
end_proc:

Chi := prog::remember(Chi, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("Chi")]):

Chi(   infinity):= infinity:
Chi( - infinity):= infinity + I*PI: 
Chi( I*infinity):=  I*PI/2: 
Chi(-I*infinity):= -I*PI/2: 
Chi( I*RD_INF):=   I*float(PI)/2: 
Chi(-I*RD_INF):=  -I*float(PI)/2: 
Chi(-I*RD_NINF):= -I*float(PI)/2: 

Chi:= funcenv(Chi):
Chi::print:="Chi":
Chi::type:="Chi":
Chi::info:=
   "Chi(x) -- the hyperbolic cosine integral: EULER + ln(x) + int((cosh(t)-1)/t, t=0..x)":

Chi::float := proc(x) 
local fx, a, b, c, d, e;
save DIGITS;
begin
  fx := float(x);
  if contains({DOM_FLOAT, DOM_COMPLEX}, domtype(fx)) then
     d:= DIGITS;
     // In case of cancellation, increase DIGITS until
     // the precision suffices:
     while TRUE do
       a:= - Ei::float(1, x):
       b:= - Ei::float(1, -x) :
       c:= float(PI)*I*signIm(fx):
       e:= (a + b + c)/2;
       if specfunc::abs(e) >= max(specfunc::abs(a),
                                  specfunc::abs(b),
                                  specfunc::abs(c)) *10^(d - DIGITS - 2) then
          break;
       else
          DIGITS:= 2*DIGITS;
          fx:= float(x);
       end_if;
     end_while;
     if domtype(fx) = DOM_FLOAT and fx > 0 and
        domtype(e) = DOM_COMPLEX then
        e:= Re(e):
     end_if;
     return(e);
  else
     hold(Chi)(fx)
  end_if;
end:

Chi::diff:=  proc(x, y) // diff(Chi(op1), y)
              local op1;
            begin  // x = Chi(op1)
                op1 := op(x, 1);
                cosh(op1)/op1 * diff(op1, y)
              end_proc:

Chi::undefined:= {0}:
Chi::realDiscont:= {0}:
Chi::complexDiscont :=
     loadproc(Chi::complexDiscont, pathname("STDLIB","DISCONT"), "Chi"):

//-----------------------------------------------------------------------
// Chi(x) = real for real x > 0.
// Chi(x) = Chi(-x) + I*PI for real x < 0.
// conjugate(Chi(x)) = Chi(conjugate(x)) for x in C_ minus (-infinity, 0]
//-----------------------------------------------------------------------
Chi::conjugate:=
  proc(x)
  begin
    if is(x >= 0) = TRUE then
      return(hold(Chi)(x))
    elif is(x <= 0) = TRUE then
      return(hold(Chi)(x) - 2*I*PI)
    end_if:
    if is(x <= 0) = FALSE then
      return(Chi(conjugate(x)));
    end_if;
    return(hold(conjugate)(hold(Chi)(x)));
  end_proc:


Chi::series:= loadproc(Chi::series, pathname("SERIES"), "Chi"):

// end of file 
