/*++
        Ci -- the cosine integral

	Ci(x):= EULER + ln(x) + int((cos(t)-1)/t, t=0..x)
++*/

Ci:=
proc(x)
  option noDebug;
begin

  if args(0) <> 1 then 
     error("expecting one argument");
  elif x::dom::Ci <> FAIL then 
     return(x::dom::Ci(args())) 
  end_if;

  case type(x)
    of DOM_SET do
    of "_union" do
      return(map(x, Ci))
  end_case;
  
  if iszero(x) then 
     error("singularity");
  end_if;
      
  case domtype(x)
    of DOM_INT do
    of DOM_RAT do
      if x < 0 then return(Ci(-x)+I*PI) end_if; break
    of DOM_FLOAT do
      if x < 0 // workaround for bug in Ci::float
         then return(Ci::float(-x)+I*float(PI))
         else return(Ci::float(x))
      end_if;
      break;
    of DOM_COMPLEX do
      if domtype(op(x,1)) = DOM_FLOAT or
        domtype(op(x,2)) = DOM_FLOAT then
        return(Ci::float(x))
      end_if;
  end_case;

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      if type(x)=Dom::ImageSet then
        return(map(x, Ci));
      else
        return(Dom::ImageSet(Ci(#x), #x, x));
      end_if;
    end_if;
    error("argument must be of 'Type::Arithmetical'")
  end_if;

  procname(x)
end_proc:

Ci := prog::remember(Ci, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("Ci")]):

Ci(infinity):=0:
Ci(-infinity):= I*PI: // = limit(Ci(x), x=-infinity, Right)
                      // along the negative real axis
Ci( I*infinity):= PI:
Ci(-I*infinity):= PI:

Ci:= funcenv(Ci, op(specfunc::Ci, 2)):
Ci::print:="Ci":
Ci::type:="Ci":
Ci::info:=
   "Ci(x) -- the Cosine integral EULER + ln(x) + int((cos(t)-1)/t, t=0..x)":
Ci::float:= specfunc::Ci:

Ci::diff:=  proc(x, y)
              local op1;
            begin  // e=Ci(f)
                op1 := op(x, 1);
                cos(op1)/op1 * diff(op1, y)
              end_proc:

Ci::"transform::laplace":= loadproc(Ci::"transform::laplace",
                                    pathname("TRANS", "LAPLACE"), "L_Ci"):

Ci::undefined:= {0}:
Ci::realDiscont:= {0}:
Ci::complexDiscont :=
    loadproc(Ci::complexDiscont, pathname("STDLIB","DISCONT"), "Ci"):

Ci::rectform:= loadproc(Ci::rectform, pathname("STDLIB", "RECTFORM"), "Ci"):

//-----------------------------------------------------------------------
// Ci(x) = real for real x > 0.
// Ci(x) = Ci(-x) + I*PI for real x < 0.
// conjugate(Ci(x)) = Ci(conjugate(x)) for x in C_ minus (-infinity, 0]
//-----------------------------------------------------------------------
Ci::conjugate:=
  proc(x)
  begin
    if is(x >= 0) = TRUE then
      return(hold(Ci)(x))
    elif is(x <= 0) = TRUE then
      return(hold(Ci)(x) - 2*I*PI)
    end_if:
    if is(x <= 0) = FALSE then
      return(Ci(conjugate(x)));
    end_if;
    return(hold(conjugate)(hold(Ci)(x)));
  end_proc:

Ci::series:= loadproc(Ci::series, pathname("SERIES"), "Ci"):

// end of file 
