/*++
   Ei.mu

	Ei  --  The Exponential Integral
        Ei(1, x) = int(exp(t)/t,t=-x..infinity) 
++*/

Ei:=
proc(n, x)
  option noDebug;
local xx, nn, sn, sx, dummy, i, k;
begin
  if args(0) = 0 then
    error("no arguments given")
  end_if:

  if args(0) = 1 then
     x:= args(1);
  elif args(0) > 2 then
    error("wrong no of args")
  end_if;

  if args(0) = 2 and n::dom::Ei <> FAIL then
     return(n::dom::Ei(args()))
  end_if;
  if x::dom::Ei <> FAIL then
     return(x::dom::Ei(args()))
  end_if;

  if args(0) = 1 then
    case type(x)
      of DOM_SET do
      of "_union" do
        return(map(x, Ei))
    end_case;
  else // args(0) = 2
    // ----------  map to sets -------------------------
    // This is quite complicated, because both arguments have
    // to be handled
    case [type(n),type(x)]
      of [DOM_SET, DOM_SET] do
          return({Ei(sn, sx) $ sn in n $ sx in x});
      of [DOM_SET, "_union"] do
      of ["_union", DOM_SET] do
          return(_union((Ei(sn, sx) $ sn in n) $ sx in x));
      of ["_union", "_union"] do
  
          // Ei({a} union A, C union D)
          //  -> Ei({a},C) union Ei({a},D) union Ei(A,C) union Ei(A,D)
          // Make sure that A, C, D are interpreted as sets, i.e.,
          // Ei({a},C) --> Ei(a, C), not {Ei(a,C)} !
  
          [n, nn, dummy]:= split(n, testtype, DOM_SET):
          if type(nn) <> "_union" then nn:= [nn] end_if:
          [x, xx, dummy]:= split(x, testtype, DOM_SET):
          if type(xx) <> "_union" then xx:= [xx] end_if:
  
          return(_union({(Ei(sn, sx) $ sn in n ) $ sx in x},
                         (Ei(sn, sx) $ sn in nn) $ sx in x,
                         (Ei(sn, sx) $ sn in n ) $ sx in xx,
                         (Ei(sn, sx) $ sn in nn) $ sx in xx ));
    end_case;
  
    case type(n)
      of DOM_SET do
      of "_union" do
         // x cannot be a set, if n is a set
         return(map(n, Ei, x));
    end_case;
  
    case type(x)
      of DOM_SET do
      of "_union" do
         // n cannot be a set, if x is a set
         return(map(x, (x,n) -> Ei(n,x), n))
    end_case;
  end_if:
  // --------------------------------------------------

  if iszero(x) then
     if args(0) = 1 then
        error("singularity")
     else
        if iszero(n - 1) then
           error("singularity") ;
        elif (domtype(n) = DOM_INT and n <= 1) or
             (domtype(n) = DOM_RAT and n <= 1) or
             (domtype(n) = DOM_FLOAT and n <= 1) or
             (domtype(n) = DOM_COMPLEX and Re(n) <= 1) then
           error("singularity") ;
        elif (domtype(n) = DOM_INT and n > 1) or
             (domtype(n) = DOM_RAT and n > 1) or
             (domtype(n) = DOM_FLOAT and n > 1) or
             (domtype(n) = DOM_COMPLEX and Re(n) > 1) then
           if domtype(x) = DOM_FLOAT then
              return(float(1/(n-1)));
           else
              return(1/(n-1));
           end_if:
        end_if;
     end_if:
  end_if:

  if args(0) = 2 then
     if iszero(n) and not iszero(x) then
        return(exp(-x)/x);
     end_if:
     if x = infinity then 
        return(0);
     end_if:
     if x = -infinity then 
        return(-infinity);
     end_if:
  end_if:


  if args(0) = 1 then
    case domtype(x)
      of DOM_FLOAT do
        return(Ei::float(x));
      of DOM_COMPLEX do
        if domtype(op(x, 1)) = DOM_FLOAT or 
           domtype(op(x, 2)) = DOM_FLOAT then
          return(Ei::float(x));
      end_if
    end_case;
  end_if:
  if args(0) = 2 then
    case domtype(x)
      of DOM_FLOAT do
        return(Ei::float(n, x));
      of DOM_COMPLEX do
        if domtype(op(x, 1)) = DOM_FLOAT or 
           domtype(op(x, 2)) = DOM_FLOAT then
          return(Ei::float(n, x));
      end_if
    end_case;
    case domtype(n)
      of DOM_FLOAT do
        return(Ei::float(n, x));
      of DOM_COMPLEX do
        if domtype(op(n, 1)) = DOM_FLOAT or 
           domtype(op(n, 2)) = DOM_FLOAT then
          return(Ei::float(n, x));
      end_if
    end_case;
  end_if:

  if args(0) = 2 and not testtype(n,Type::Arithmetical) then
    if testtype(n, Type::Set) then
      if testtype(x, Type::Set) and not testtype(x, Type::Arithmetical) then
        return(Dom::ImageSet(Ei(#n, #x), [#n, #x], [n, x]));
      else
        return(Dom::ImageSet(Ei(#n, x), [#n], [n]));
      end_if;
    end_if;
    error("the 1st argument must be of 'Type::Arithmetical'")
  end_if;

  if not testtype(x,Type::Arithmetical) then
    if testtype(x, Type::Set) then
      if args(0)=1 then
        return(Dom::ImageSet(Ei(#x), [#x], [x]));
      else
        return(Dom::ImageSet(Ei(n, #x), [#x], [x]));
      end_if;
    end_if;
    error("expecting arguments of type 'Type::Arithmetical'")
  end_if;

  // For negative integers n, Ei(n, x) ist exp(-x)*polynomial(1/x):
  if args(0) = 2 then 
    if iszero(frac(n)) and
      domtype(float(n)) = DOM_FLOAT and //i.e., n is real
      n < 0 and
      n >= -Pref::autoExpansionLimit()
    then
      return(exp(-x)*_plus((-1)^k*x^(-k-1)*_mult(n + i $ i = 0..k-1) $ k = 0..-n))
    end_if;
    if n=1 then
      // we have that Ei(1, x) = -Ei(-x) for x > 0  and 
      //              Ei(1, x) = -Ei(-x) - PI*I for x < 0
      if numeric::isless(x, 0) = TRUE then
        return(-Ei(-x) - PI*I)
      elif numeric::isless(0, x) = TRUE then
        return(-Ei(-x))
      // else
      // For complex x: if Im(x) <> 0, then Ei(1, x) = -Ei(-x) - sign(Im(x))* PI*I
      // there is no formula with signIm that would be correct for real x, too       
      // therefore, we express this in terms of ln
      // deactivated, gives problems everywhere in the library
      //   return(-Ei(-x) + 1/2*ln(-x) - 1/2*ln(-1/x) - ln(x))
      end_if;  
                                    
    end_if;  
  end_if:

  if args(0) = 1 then
     return(procname(x));
  else
     return(procname(n, x));
  end_if:
end_proc:

Ei := prog::remember(Ei,
       () -> [property::depends(args()), DIGITS, Pref::autoExpansionLimit(),
              slotAssignCounter("Ei")]):


Ei(infinity):= infinity:
Ei(-infinity):= 0:
Ei( I*infinity):= PI*I:
Ei(-I*infinity):=-PI*I:
Ei:= funcenv(Ei, op(specfunc::Ei, 2)):
Ei::info:=
    "Ei -- the exponential integral [try ?Ei for details]":
Ei::print:= "Ei":
Ei::type:= "Ei":
Ei::float:= proc(n, x)
local i;
begin
   if args(0) = 1 then
      x:= float(n);
      case x 
      of RD_INF do 
         return(RD_INF);
      of RD_NINF do 
         return(float(0));
      of RD_INF*I do 
         return(I*float(PI));
      of RD_NINF*I do 
         return(-I*float(PI));
      of RD_NAN do
         return(RD_NAN):
      end_case;
      if contains({DOM_FLOAT, DOM_COMPLEX}, domtype(x)) then
         if domtype(x) = DOM_FLOAT then
           if x < 0 then
             return(-Ei::float(1, -x));
           else
             return(-Re(Ei::float(1, -x)));
           end_if:
         else  // domtype(x) = DOM_COMPLEX
           return(-Ei::float(1, -x) + sign(Im(x))*I*float(PI));
         end_if:
      else // c contains symbols
         return(hold(Ei)(x))
      end_if;
   else
      x:= float(x):
      if x = RD_INF then
         return(float(0));
      elif x = RD_NINF then
         return(RD_NINF);
      elif x = RD_NAN then
         return(RD_NAN);
      end_if:
      if contains({DOM_FLOAT, DOM_COMPLEX}, domtype(x)) and
         contains({DOM_FLOAT, DOM_COMPLEX}, domtype(float(n))) then
         if iszero(n - 1) then
            // specfunc::Ei is rather slow for some arguments.
            // Implement the asymptotic expansion here for 
            // arguments that are sufficiently large:
            if DIGITS <= 16 and specfunc::abs(x) >= 10^4 then
              return(exp(-x)/x*_plus((-1)^i*i!/x^i $ i = 0..5));
            end_if:
            return(specfunc::Ei(x));
         end_if:
         return(x^float(n-1)*igamma(1 - n, x))
      else
         return(hold(Ei)(n, x))
      end_if;
   end_if;
end_proc:

// specfunc::Ei:

Ei::diff:= loadproc(Ei::diff, pathname("STDLIB", "DIFF"), "Ei"):

Ei::series:= loadproc(Ei::series, pathname("SERIES"), "Ei"):

Ei::rectform:= loadproc(Ei::rectform, pathname("STDLIB", "RECTFORM"), "Ei"):

Ei::Re:= 
proc(n, x)
begin
  if args(0) = 1 then
    if is(n in R_, Goal =TRUE) then 
      return(hold(Ei)(n))
    end_if;
    hold(Re)(hold(Ei)(n))
  else
    if is(n in R_ and x>0, Goal = TRUE) then
       return(hold(Ei)(n, x))
    end_if;
    hold(Re)(hold(Ei)(n, x))
  end_if
end_proc:

Ei::Im:= 
proc(n, x)
begin
  if args(0) = 1 then
    if is(n in R_, Goal =TRUE) then 
      return(0)
    end_if;
    hold(Im)(hold(Ei)(n))
  else
    if n=1 and is(x<0, Goal =TRUE) then
       return(-PI)
    end_if;
    if is(n in R_ and x>0, Goal = TRUE) then
       return(0)
    end_if;
    hold(Im)(hold(Ei)(n, x))
  end_if
end_proc:


Ei::conjugate:=
  proc(n, x)
  begin
    if args(0) = 1 then
      x:= n:
      if is(x, Type::Real) = TRUE then
         return(hold(Ei)(x)); // Ei(x) is real
      end_if:
      return(hold(Ei)(conjugate(x)));
    end_if:
    if args(0) = 2 then
      if domtype(n) = DOM_INT and n <= 0 then 
         if is(x, Type::Real) = TRUE then
            return(hold(Ei)(n, x));  // Ei(n, x) is real 
         else
            return(hold(Ei)(n, conjugate(x)));
         end_if;
      end_if:
      if is(n, Type::Real) = TRUE then 
        if is(x < 0) = FALSE then
          return(hold(Ei)(n, conjugate(x)));
        end_if;
      end_if:
      return(hold(conjugate)(hold(Ei)(n, x)));
    end_if:
  end_proc:

Ei::"transform::laplace":= loadproc(Ei::"transform::laplace",
                                    pathname("TRANS", "LAPLACE"), "L_Ei"):

Ei::expand:= loadproc(Ei::expand, pathname("STDLIB", "EXPAND"), "Ei"):

// end of file 
