/* Definite Integration using the Generalized Hypergeometric Functions
 * Ioannis Dimitrios Avgoustis (1974)
 * (3.1) Lemma 4
 *   pFq( a_p; b_q; z ), wenn ein a_p negative ganze Zahl ist, reduziert das hypergeom zu einem Polynom
 * (3.1) Lemma 5
 *   pFq( a_p; b_q; z ), wenn a_i-b_j natrliche Zahl, lsst sich das hypergeom als Summe aus p-1Fq-1 umschreiben
 */

alias(mysqrt = specfunc::Wurzelbehandlung):


specfunc::Hypergeom::generalReduction := proc(hyp)
  local l1, l2, z, i, j, k, n, a1, m, o, p, q;
begin
  if type(hyp)<>"hypergeom" then return(hyp); end_if;
  [l1, l2, z]:= [op(hyp)];
  p := nops(l1);
  q := nops(l2);
  //===================================================
  // Case 1: the first list contains a negative integer 
  //       --> hypergeom is a polynomial
  //===================================================
  for i from 1 to nops(l1) do
    if type(l1[i])=DOM_INT and l1[i]<0 then
      n := -l1[i];
      return( 1+_plus( ( _mult(op(map(l1,pochhammer,i)))
                        /_mult(op(map(l2,pochhammer,i)))
                        *z^i/i! )$ i = 1..n ) );
    end_if;
  end_for;
  //===================================================
  // Case 2: the two lists contain elements that differ by 
  // an integer --> pFq reduces to a sum of (p-1)F(q-1) terms:
  //===================================================
  for i from 1 to nops(l1) do
    for j from 1 to nops(l2) do
      if type(l1[i]-l2[j])=DOM_INT and l1[i]-l2[j]>0 then
        k := l1[i]-l2[j];
        a1 := l2[j];
        l1 := subsop(l1, i=null() );
        l2 := subsop(l2, j=null() );
        return( _plus( (  binomial(k,n)
                        * _mult(l1[m]+o$m=1..(p-1)$o=0..(n-1))
                        / _mult(l2[m]+o$m=1..(q-1)$o=0..(n-1))
                        / _mult(a1+o$o=0..(n-1))
/*
                        * _mult(pochhammer(l1[m], n) $m=1..(p-1))
                        / _mult(pochhammer(l2[m], n) $m=1..(q-1))
                        / pochhammer(a1, n)
*/
                        * z^n
                        * hypergeom(map(l1,X->X+n), map(l2,X->X+n), z) ) $n=0..k ) );
      end_if;
    end_for;
  end_for;
  return(hyp);
end_proc:

//=========== 0F1 ==================
/* Definite Integration using the Generalized Hypergeometric Functions
 * (3.2.1) Lemma 8
 * converts
 * a) 0F1( ;v+1;-z^2/4) to (z/2)^-v gamma(v+1) besselJ(v, z)
 * b) 0F1( ;v+1; z^2/4) to (z/2)^-v gamma(v+1) besselI(v, z)
 */
specfunc::Hypergeom::alg0F1toBessel := proc(hyp)
  local l1, l2, z, v;
begin
  if type(hyp)<>"hypergeom" then return(hyp); end_if;
  [l1, l2, z]:= [op(hyp)];
  if nops(l1)=0 and nops(l2)=1 then
    v := l2[1]-1;
    if stdlib::hasmsign( z ) then
      if domtype(v) = DOM_INT then 
         return( mysqrt((-z)^(-v))*gamma(v+1)*besselJ(v, 2*mysqrt(-z)) );
      else
         return( (-z)^(-v/2)*gamma(v+1)*besselJ(v, 2*sqrt(-z)) );
      end_if;
    else
      if domtype(v) = DOM_INT then 
         return( mysqrt(( z)^(-v))*gamma(v+1)*besselI(v, 2*mysqrt( z)) );
      else
         return( ( z)^(-v/2)*gamma(v+1)*besselI(v, 2*sqrt( z)) );
      end_if:
    end_if;
  end_if;
  hyp;
end_proc:

//=========== 1F1 ==================
/* Definite Integration using the Generalized Hypergeometric Functions
 * Algorithm 1F1-Red, page 63
 */
specfunc::Hypergeom::alg1F1Reduction := proc(hyp)
  local l1, l2, z, a, c, factor,
    reduceToErrorFunction, reduceToPolynomial, kummerTransformation;
begin

  // ===========================================
  /* This reduces a hypergeometric function to error-functions as in
     Corollary 10.1 */
  reduceToErrorFunction := proc(a,c)
    local m, prod, i;
  begin
    // This routine is only called, when c - a is a positive integer.
    // Since this may not be the case syntactically, we better
    // normalize the input:
    [a, c]:= normal([a, c]);
    m := round(normal(c-a));
    if domtype(m) <> DOM_INT then
      return(hypergeom([a], [c], z));
    end_if;
    // If a is 'simpler' than 'c', then express c as 'a + integer'.
    // Else, express a complicated 'a' in terms of the simpler 'c':
    if length(a) <= length(c) then
       c:= a + m;
    else
       a:= c - m;
    end_if;
    if m=1 then
      if c=1 then return(1);end_if;
      if a=1/2 and c=3/2 then
        return( erf(mysqrt(-z))/2*sqrt(PI)/mysqrt(-z) );
      else
        return( (gamma(a)-igamma(a,-z))*a*(-z)^(-a) );
      end_if;
    end_if;
    prod := _mult((a+i) $ i=0..(m-1))/(m-1)!;
    expand(_plus( (-1)^i*binomial(m-1,i)*prod/(a+i)*reduceToErrorFunction(a+i,a+i+1) $i=0..(m-1)));
  end_proc;

  // ===========================================
  reduceToPolynomial := proc()
  begin
    if c=1/2 then
      return((-a)!/(-2*a)!*(-1)^a*expr(orthpoly::hermite(-2*a,mysqrt(z))));
    end_if;
    if c=3/2 then
      return((-a)!/(-2*a+1)!*(-1)^a/2/mysqrt(z)*expr(orthpoly::hermite(-2*a+1,mysqrt(z))));
    end_if;
    (-a)!/pochhammer(c, -a)*expr(orthpoly::laguerre(-a,c-1,z));
  end_proc:

  // ===========================================
  kummerTransformation := proc()
  begin
    factor := factor*exp(z); 
    a := c-a; z := -z;
  end_proc;

  [l1, l2, z]:= [op(hyp)];
  if nops(l1)<>1 or nops(l2)<>1 then return(hyp); end_if;
  a := op(l1); c := op(l2);
  factor := 1;

  /* 2a=c => Kummer-Transformation und 0F1-Fall */
  if 2*a=c then
    return( exp(z/2)*specfunc::Hypergeom::alg0F1toBessel( hypergeom([], [a+1/2], z^2/16) ) );
  end_if;

  /* a ist negative Integer => Polynom */
  if is(-a in N_, Goal=TRUE) then
    return( reduceToPolynomial() );
  end_if;

  if not is(a-c in Z_, Goal=TRUE) then
    if is(a in Z_, Goal=TRUE) then
      kummerTransformation();
    else
      /* Lemma 16, Seite 63 */
      /* kappa = 1/2+mu-a = c/2-a */
      /* mu = (c-1)/2 */
      return( factor*whittakerM( c/2-a, (c-1)/2, z )/z^(c/2)*exp(z/2) );
    end_if;
  end_if;

  /* c-a ist negativer Integer => Kummertransformation und Polynom */
  if is(a-c in N_, Goal=TRUE) then
    /* Kummer-Transformation */
    kummerTransformation();
    return( factor*reduceToPolynomial() );
  end_if;

  /* Keine unbekannten, a positiv => error-Funktion */
  if indets( [a,c] ) minus Type::ConstantIdents={} and not is(-a in N_, Goal=TRUE) then
    return( factor*reduceToErrorFunction(a,c) );
  end_if;

  /* Ab zu Whittaker */
  return( factor*whittakerM( c/2-a, (c-1)/2, z )/z^(c/2)*exp(z/2) );
end_proc:

//=========== 2F1 ==================
specfunc::Hypergeom::alg2F1Reduction := proc(hyp)
  local l1, l2, z, a, b, c, X, n, m, l;
begin
  if type(hyp)<>"hypergeom" then return( hyp ); end_if;
  [l1, l2, z]:= [op(hyp)];
  if nops(l1)<>2 or nops(l2)<>1 then return(hyp); end_if;
  [a, b] := sort(l1); 
  c := op(l2);

  // ===================================================
  // Special values at z = 1
  // ===================================================
  if iszero(z - 1) then
    if is(Re(c - a - b) > 0) <> FALSE then
      if is(c - a in Z_ and c - a < 0) = TRUE or
         is(c - b in Z_ and c - b < 0) = TRUE then
          return(0)
	else
          return(gamma(c)*gamma(c - a - b)/gamma(c - a)/gamma(c - b))
      end_if;
    end_if;
  end_if:

  // ===================================================
  // Special values at z = 1/2
  // ===================================================
  if iszero(z - 1/2) then
     n:= a + b - 2*c;
     if domtype(n) = DOM_INT then
       if n > 0 then
         return( 2^(b-1)*gamma(c)/gamma(b)*
                _plus(binomial(n+1,m)*gamma((b+m)/2)/gamma((a+m-n)/2) $ m = 0 .. n+1));
       elif iszero(n) then
         return(sqrt(PI)*gamma((a+b)/2)*
                (1/gamma((a+1)/2)/gamma(b/2) + 1/gamma((b+1)/2)/gamma(a/2)));
       elif iszero(n+1) then
         return(sqrt(PI)*gamma(c)/gamma((a+1)/2)/gamma((b+1)/2));
       elif iszero(n+2) then
         return(2*sqrt(PI)*gamma(c)/(a-b)*
                (1/gamma(a/2)/gamma((b+1)/2) - 1/gamma(b/2)/gamma((a+1)/2)));
       end_if;
     end_if;
  end_if:

  // ===================================================
  // Special values at z = 0
  // ===================================================
  if iszero(z) then
     return(1);
  end_if;

  // ===================================================
  // Special values at z = -1
  // ===================================================
  if iszero(z+1) then
    if iszero(b-1) then [a, b]:= [b, a]; end_if;

    n:= -(c+b);
    if iszero(a-1) and domtype(n) = DOM_INT then
      if n >= -1 then
        return(2^(-n-2*b-2)*gamma(1-b)*gamma(-n-b)/gamma(-n-2*b) + 
               1/2 * _plus((-1)^m*pochhammer(b, m)/pochhammer(-b-n, m) $ m = 0..n+1)
              );
      elif n < -1 then
        return(2^(-n-2*b-2)*gamma(1-b)*gamma(-n-b)/gamma(-n-2*b) - 
  	     1/2 * _plus((-1)^m*pochhammer(b+n+1, m)/pochhammer(1-b, m) $ m = 1..-n-2)
  	    );
      end_if;
    end_if;

    n:= c-b;
    if iszero(a - 1) and domtype(n) = DOM_INT then
       case n 
       of -1 do return(1/2 - 1/4/(b-1));
       of  0 do return(1/2);
       of  1 do return(pochhammer(b, n)/2*(psi(b/2+1/2) - psi(b/2))  );
       of  2 do return(pochhammer(b, n)*(psi(b/2+1/2) - psi(b/2)) - (b+1)  );
       of  3 do return(pochhammer(b, n)*(psi(b/2+1/2) - psi(b/2)) - (b+2)*(b + 3/2)  );
       of  4 do return(pochhammer(b, n)*2/3*(psi(b/2+1/2) - psi(b/2)) - (b+3)*(2*b^2 + 7*b + 7)/3 );
       end_case;
    end_if:

 // The following is covered by the general reduction
 // to a polynomial (for arbitrary z) 
    if is(a in Z_) = TRUE and is(a < 0) = TRUE then
       return((-a)!*(-2)^(-a)/pochhammer(c, -a)*expr(orthpoly::jacobi(-a, -b + a, c-1, 0)));
    elif is(b in Z_) = TRUE and is(b < 0) = TRUE then
       return((-b)!*(-2)^(-b)/pochhammer(c, -b)*expr(orthpoly::jacobi(-b, -a + b, c-1, 0)));
    end_if:
 //

    // return to standard sorting
    [a, b]:= sort([a, b]):
  end_if;

  // ===================================================
  // Results for general z:
  // ===================================================
  case [a, b, c]
  of [1/2, 1/2, 3/2] do 
     X:= mysqrt(z);
     return(arcsin(X)/X);
  of [1/2, 1, 3/2] do
     X:= mysqrt(-z);
     return(arctan(X)/X);
  end_case;

  /* 3.2.3.2. Logarithmic and other algebraic cases */
  if type(a)=DOM_INT and type(b)=DOM_INT and type(c)=DOM_INT and c>0 then
    /* Ein negativer Integer => Polynom */
    if 0>a or 0>b then
      return( specfunc::Hypergeom::generalReduction(hyp) );
    end_if;
    /* Gauss-Euler-Transformation */
    if a>c or b>c then
      return((1-z)^(c-a-b)
             *specfunc::Hypergeom::generalReduction(
                    hypergeom([c-a,c-b],[c],z)));
    end_if;
    if a<c and b<c then
      m := abs(a-b);
      n := (a+b-m-2)/2; // = max(a, b) -1
      l := c-a-b+n; // = c - min(a, b) - 1
      X := genident();
      return( -(-1)^m*(n+m+l+1)!/n!/l!/(n+m)!/(m+l)! * 
               diff ( (1-X)^l*diff( (1-X)^m*ln(1-X)/X, X$(l+m) ), X$n ) | X=z );
    end_if;
  end_if;
  return(hyp);
end_proc:

//=====================================================
// Walter, 1.10.2008: very, very rudimentary version of
// 1F2 reductions:
//=====================================================
specfunc::Hypergeom::alg1F2Reduction := proc(hyp)
local l1, l2, z, a, b, c, sqrtz, n;
begin
  if type(hyp)<>"hypergeom" then return( hyp ); end_if;
  [l1, l2, z]:= [op(hyp)];
  if nops(l1)<>1 or nops(l2)<>2 then return(hyp); end_if;
  l2:= sort(l2): // make sure that l2 is in ascending order
  // The following formulas are not covered by generalReduction
  // and are special to 1F2:
  if l1 = [-1/2] and l2 = [1/2, 1/2] then
       if stdlib::hasmsign(z) then 
         return(cos(2*mysqrt(-z)) + 2*mysqrt(-z)*Si(2*mysqrt(-z)));
       else
         return(cosh(2*mysqrt(z)) - 2*mysqrt(z)*Shi(2*mysqrt(z)));
       end_if:
  elif l1 = [-1/2] and l2 = [1/2, 3/2] then
       if stdlib::hasmsign(z) then 
         return(cos(2*mysqrt(-z))/2 + sin(2*mysqrt(-z))/4/mysqrt(-z) + mysqrt(-z)*Si(2*mysqrt(-z)));
       else
         return(cosh(2*mysqrt(z))/2 + sinh(2*mysqrt(z))/4/mysqrt(z) - mysqrt(z)*Shi(2*mysqrt(z)));
       end_if:
  elif l1 = [1/2] and l2 = [3/2, 3/2] then
       if stdlib::hasmsign(z) then
         return(Si(2*mysqrt(-z))/2/mysqrt(-z));
       else
         return(Shi(2*mysqrt(z))/2/mysqrt(z));
       end_if;
  end_if:

  // 2 special cases of the besselI patterns further down below.
  // The return values here have a simpler representation than 
  // the generic return value.
  if l1 = [1] and l2 = [3/2, 2] then
       if stdlib::hasmsign(z) then
          return( (cos(2*mysqrt(-z)) - 1)/2/z );
       else
          return( (cosh(2*mysqrt(z)) - 1)/2/z );
       end_if;
  end_if;
  if l1 = [1] and l2 = [2, 5/2] then
       if stdlib::hasmsign(z) then
          return( (sin(2*mysqrt(-z))/mysqrt(-z) - 2)*3/4/z );
       else
          return( (sinh(2*mysqrt(z))/mysqrt(z) - 2)*3/4/z );
       end_if;
  end_if;

  //==========================================================
  // (see functions.wolfram.com -> Hypergeometric Functions 
  //  -> HypergeometricPFQ[{a1}, {b1, b2}, z] -> Specific values
  //  -> Specialized values -> For fixed b2, z (3 formulas)
  // The 2 patterns above are the special cases b=3/2 and b=5/2.
  if l1 = [1] and contains(l2, 2) > 0 then
     if l2[1] = 2 then
        b:= l2[2]
     else
        b:= l2[1]
     end_if:
     // hypergeom([1], [2, b], z) 
     sqrtz:= sqrt(z);
     return(gamma(b)*besselI(b - 2, 2*sqrtz)/z^(b/2) + (1-b)/z);
  end_if:

  //==========================================================
  // (see functions.wolfram.com -> Hypergeometric Functions 
  //  -> HypergeometricPFQ[{a1}, {b1, b2}, z] -> Specific values
  //  -> Specialized values -> For fixed a1, z (10 formulas)
  a:= op(l1,1):
  n:= -a:
  if is(n in N_ union {0}) = TRUE and contains(l2, 1/2 -n) > 0 then
        if l2[1] = 1/2 - n then
           b:= l2[2];
        else
           b:= l2[1];
        end_if:
        sqrtz:= sqrt(z);
        case b 
        of -2*n do // hypergeom([-n], [1/2-n, -2*n], z) 
           c:= 2^(2*n - 1)*PI*(n!)^2/((2*n)!)^2*z^(n + 1/2):
           return(c*besselI(-n - 1/2, sqrtz)^2 
                - c*besselI( n + 1/2, sqrtz)^2);
        end_case; 
  end_if:

  //==========================================================
  // (see functions.wolfram.com -> Hypergeometric Functions 
  //  -> HypergeometricPFQ[{a1}, {b1, b2}, z] -> Specific values
  //  -> Specialized values -> For fixed a1, z (10 formulas)
  a:= op(l1,1):
  if contains(l2, a-1/2) > 0 then
     if l2[1] = a-1/2 then
        b:= l2[2]
     else
        b:= l2[1]
     end_if:
     sqrtz:= sqrt(z); 
     case b 
     of 2*a - 3 do // hypergeom([a], [a-1/2,2*a-3], z)
        c:= (2*a-3)/(a-1)*2^(2*a-6)*z^(3/2-a)*gamma(a-3/2)^2:
        return( c*(6+z-10*a+4*a^2)*besselI(a-3/2, sqrtz)^2 
              + c*(4*a-5)*sqrtz*besselI(a-3/2, sqrtz)*besselI(a-1/2, sqrtz)
              + c*z*besselI(a-1/2, sqrtz)^2);
     of 2*a - 2 do // hypergeom([a], [a-1/2,2*a-2], z)
        c:= 2^(2*a-3)/(a-1)*gamma(a-1/2)^2*z^(3/2-a): 
        return(c*sqrtz*besselI(a-3/2, sqrtz)*besselI(a-1/2, sqrtz) 
              +c*(a-1)*besselI(a-3/2, sqrtz)^2);
     of 2*a - 1 do // hypergeom([a], [a-1/2,2*a-1], z)
        c:= 2^(2*a-3)*z^(3/2-a)*gamma(a - 1/2)^2:
        return(c*besselI(a - 3/2, sqrtz)^2 
              +c*besselI(a - 1/2, sqrtz)^2);
     of 2*a do // hypergeom([a], [a-1/2, 2*a], z) 
        c:= 2^(2*a-3)*gamma(a - 1/2)^2*(2*a-1)*z^(1/2 - a):
        return( c*2*sqrtz*besselI(a-1/2, sqrtz)*besselI(a-3/2, sqrtz) 
               +c*(1-2*a)*besselI(a-1/2, sqrtz)^2);
     of 2*a + 1 do // hypergeom([a], [a - 1/2, 2*a + 1], z) 
        c:= 2^(2*a-3)*(2*a-1)*z^(1/2-a)*gamma(a-1/2)^2:
        return(c*(2*a - 1)*besselI(a - 1/2, sqrtz)^2 
              +c*(2*a + 1)*besselI(a + 1/2, sqrtz)^2);
     end_case:
  end_if:
  if contains(l2, a+1/2) > 0 then
     if l2[1] = a+1/2 then
        b:= l2[2]
     else
        b:= l2[1]
     end_if:
     sqrtz:= sqrt(z);
     case b 
     of 2*a - 1 do // hypergeom([a], [a+1/2,2*a-1], z)
        c:= 2^(2*a-3)*(2*a-1)*z^(1-a)*gamma(a - 1/2)^2:
        return(c*besselI(a-3/2, sqrtz)*besselI(a-1/2, sqrtz));
     of 2*a do     // hypergeom([a], [a+1/2, 2*a], z) 
        c:= 2^(2*a-1)*z^(1/2 - a)*gamma(a+1/2)^2:
        return(c*besselI(a-1/2, sqrtz)^2);
     of 2*a + 1 do // hypergeom([a], [a+1/2, 2*a + 1], z) 
        c:= 2^(2*a-1)*z^(1/2-a)*gamma(a+1/2)^2:
        return(c*besselI(a - 1/2, sqrtz)^2 
              -c*besselI(a + 1/2, sqrtz)^2);
     end_case:
  end_if:
  if contains(l2, a+3/2) > 0 then
     if l2[1] = a+3/2 then
        b:= l2[2]
     else
        b:= l2[1]
     end_if:
     sqrtz:= sqrt(z);
     case b 
     of 2*a do     // hypergeom([a], [a+3/2, 2*a], z) 
        c:= 4^(a-1)*(2*a+1)*z^(1/2 - a)*gamma(a+1/2)^2:
        return(
             c*besselI(a-1/2, sqrtz)^2
            -c*besselI(a+1/2, sqrtz)*besselI(a-3/2, sqrtz)
              );
     of 2*a + 1 do // hypergeom([a], [a+3/2, 2*a + 1], z) 
        c:= 2^(2*a-1)*(2*a+1)*z^(-a)*gamma(a+1/2)^2:
        return(c*sqrtz*besselI(a - 1/2, sqrtz)^2 
              -c*sqrtz*besselI(a + 1/2, sqrtz)^2
              -c*2*a*besselI(a-1/2, sqrtz)*besselI(a+1/2, sqrtz)
             );
     end_case:
  end_if:

  //==========================================================
  // (see functions.wolfram.com -> Hypergeometric Functions 
  //  -> HypergeometricPFQ[{a1}, {b1, b2}, z] -> Specific values
  //  -> Specialized values -> For fixed b1, z (9 formulas)
  if l1 = [1/2] then
     b:= l2[1]: 
     sqrtz:= sqrt(z);
     case l2[1] + l2[2] 
     of 0 do // hypergeom([1/2], [b, -b], z)
         c:= -PI/sin(PI*b)/4/b*z:
         return(c*  besselI(-b-1,sqrtz)*besselI(b-1, sqrtz)
               +c*2*besselI( -b ,sqrtz)*besselI( b , sqrtz)
               +c*  besselI(1-b ,sqrtz)*besselI(b+1, sqrtz));
     of 1 do // hypergeom([1/2], [b, 1-b], z)
         c:=  PI/sin(PI*b)/2*sqrtz:
         return(c*besselI(b-1,sqrtz)*besselI(-b, sqrtz)
               +c*besselI(1-b,sqrtz)*besselI( b, sqrtz));
     of 2 do // hypergeom([1/2], [b, 2-b], z)
         c:=  PI*(1-b)/sin(PI*b):
         return(c*besselI(1-b,sqrtz)*besselI(b-1, sqrtz)):
     of 3 do // hypergeom([1/2], [b, 3-b], z)
         c:= PI/sin(PI*b)*(b-1)*(b-2)/(2*b-3):
         return(c*besselI(2-b,sqrtz)*besselI(-2+b, sqrtz)
               -c*besselI(1-b,sqrtz)*besselI(-1+b, sqrtz));
     end_case;
  end_if:
  if l1 = [3/2] then
     b:= l2[1]: 
     sqrtz:= sqrt(z);
     case l2[1] + l2[2] 
     of 2 do // hypergeom([3/2], [b, 2-b], z)
         c:= PI/sin(PI*b)*(1-b):
         return(c*(2*b-1)*besselI(1-b, sqrtz)*besselI(b-1, sqrtz)
               +c*sqrtz  *besselI( -b, sqrtz)*besselI(b-1, sqrtz)
               +c*sqrtz  *besselI(1-b, sqrtz)*besselI( b , sqrtz));
     of 3 do // hypergeom([3/2], [b,3-b], z)
         c:= PI/sin(PI*b)*(b-1)*(b-2):
         return(c*besselI(2-b, sqrtz)*besselI(b-2, sqrtz)
               +c*besselI(1-b, sqrtz)*besselI(b-1, sqrtz));
     end_case;
  end_if:
  return(hyp);
end_proc:

// ----------------------------------------------------
// Voldemort never finished the complete simplify code.
// This is a very rudimentary version only doing
// the reduction to the polynomial case and reducing
// 0F1, 1F1, 2F1, 1F2.
// ----------------------------------------------------
specfunc::Hypergeom::simplify:= proc(f)
local l1, l2, z, a0, b0, ak, bk, t, s, k,
  hyp, res;
begin
  if type(f)<>"hypergeom" then return(f); end_if;
  [l1, l2, z]:= [op(f)];
  l1:= map(l1, simplify);
  l2:= map(l2, simplify);
  z:= simplify(z);
  a0:=specfunc::Hypergeom::sel(l1):
  b0:=specfunc::Hypergeom::sel(l2):
  if a0 <> -infinity and a0 > b0 then
     t:=1: s:=t:
     for k from 1 to round(-a0) do
       ak:=_mult(op(map(l1,((x,k)->x+k-1),k))):
       bk:=_mult(op(map(l2,((x,k)->x+k-1),k))):
       t:=t*ak*z/bk/k:
       s:=s+t:
       if t=0 then break: end_if:
     end_for:
     return(s): 
  end_if:
  hyp := hold(hypergeom)(l1, l2, z);
  case [nops(l1),nops(l2)]
    of [0,1] do return( specfunc::Hypergeom::alg0F1toBessel(hyp) );
    of [1,1] do return( specfunc::Hypergeom::alg1F1Reduction(hyp) );
    of [1,2] do 
      res:= specfunc::Hypergeom::alg1F2Reduction(hyp);
      if type(res)="hypergeom" then
        res := specfunc::Hypergeom::generalReduction(res);
      end_if;
      break;
    of [2,1] do
      res := specfunc::Hypergeom::alg2F1Reduction(hyp);
      if type(res)="hypergeom" then
        res := specfunc::Hypergeom::generalReduction(res);
      end_if;
      break;
    otherwise
      res := specfunc::Hypergeom::generalReduction(hyp);
  end_case;
  if type(res)<>"hypergeom" then
    res := misc::maprec(res, {"hypergeom"}=specfunc::Hypergeom::simplify);
  end_if;
  return(res);
end_proc:
// -----------------------------------------------
// more formulas:
// Ci(z) = -z^2*hypergeom([1,1], [2,2,3/2],-z^2/4) + log(z) + EULER; 
// Chi(z) = z^2*hypergeom([1,1], [2,2,3/2], z^2/4) + log(z) + EULER; 
// Li(z) = log(z)*hypergeom([1, 1], [2,2], log(z)) + 1/2*(log(log(z))-log(1/log(z))) + EULER
// orthpoly::hermite(n, z) = (2*z)^n*hypergeom([-n/2, (1-n)/2], [], -1/z^2);

