/* integrates int( xpr, range ), where range is of kind Ident=left..right */
specfunc::MeijerG::convolutionInt := proc( xpr, range )
  local var, left, right, res;
begin
  /* Parameter prfen */
  if type(range)<>"_equal" then
    if type(range)=DOM_IDENT then
      warning("No range given, assuming 0..infinity");
      left := 0;
      right := infinity;
      var := range;
    else
      error("No variable/range given");
      return;
    end_if;
  else
    if type(op(range,1))<>DOM_IDENT or type(op(range,2))<>"_range" then
      error( "Range should be Ident=left..right" );
      return;
    end_if;
    var := op(range,1);
    left := op(range,[2,1]);
    right := op(range,[2,2]);
  end_if;

  /* Simple Substitutionen der Grenzen */
  /* linke Grenze = rechte Grenze => integral = 0 */
  if is(left=right, Goal=TRUE) then return( 0 ); end_if;

  /* rechte Grenze <= 0 => Integration drehen*/
  if is( right<=0, Goal=TRUE ) then
    /* substituiere: var := -var */
    xpr := subs( xpr, var=-var );
    [ left, right ] := [ -right, -left ];
  end_if;

  if right = 0 then
    return(FAIL);
  end_if;

  /* linke Grenze = -infinity and rechte Grenze != infintiy => drehen */
  if left=-infinity and right<>infinity then
    /* substituiere: var := -var */
    xpr := subs( xpr, var=-var );
    left := -right;
    right := infinity;
  end_if;

  res := specfunc::MeijerG::matchInt( xpr, var, left, right );
  if has(res, undefined) then return(FAIL); end_if;
//if hastype( res, "meijerG" ) then return( FAIL ); end_if;
  return( res );
end_proc:

specfunc::MeijerG::convolutionInt := funcenv( specfunc::MeijerG::convolutionInt ):
autoload( specfunc::MeijerG::convolutionInt::int_2_24_2_2 ): /* Integrationsformel Prudnikov, Integrals and Series, Vol 3. 2.24.2.2*/

/*
  * Analysiert inwieweit der Parameter von G eine durch folgende Funktion in
  * abhngigkeit von x dargestellt werden kann:
  * a*(x + b)^(c/d)
  */
specfunc::MeijerG::convolutionInt::analyzeSubst := proc( G, ident )
  local var, a, b, c, d, s;
begin
  var := op(G,3);
  a := 1; b := 0; c := 1; d := 1;
  if type(var) = "_plus" then
    s := split( var, X->not contains( indets(X), ident ) );
    b := s[1]; var := s[2];
  end_if;
  if type(var) = "_mult" then
    s := split( var, X->not contains( indets(X), ident ) );
    a := s[1]; var := s[2];
  end_if;
  if type(var) = "_power" then
    if type(op(var,2))=DOM_INT then
      c := op(var,2);
      d := 1;
      var := op(var,1);
    elif type(op(var,2))=DOM_RAT then
      c := op(var,[2,1]);
      d := op(var,[2,2]);
      var := op(var,1);
    else
      return( FAIL );
    end_if;
  end_if;
  if var=ident then return( [a,b,c,d] ); end_if;
  return( FAIL );
end_proc:

/* Wrapper: liefert fr a*G(...) die Liste [a,G(...)] zurck
  */
specfunc::MeijerG::convolutionInt::extractFactor := proc( res )
  local s;
begin
  if type(res)="_mult" then
    s := split( res, X->type(X)<>"meijerG" );
    return( [s[1],s[2]] );
  end_if;
  return( [1,res] );
end_proc;

/* mymu
  * Berechnet mu wie in "Integrals and Series, Vol 3, p345
  */
specfunc::MeijerG::convolutionInt::mymu := proc( G1 )
begin
  _plus( op(op(G1,[2,1])) ) + _plus( op(op(G1,[2,2])) ) - _plus( op(op(G1,[1,1])) ) - _plus( op(op(G1,[1,2])) ) +
    ( nops(op(G1,[1,1])) + nops(op(G1,[1,2])) - nops(op(G1,[2,1])) - nops(op(G1,[2,2])) )/2 + 1;
end_proc;

specfunc::MeijerG::convolutionInt::delta := proc( k, a )
  local i;
begin
  (a+i)/k $ i=0..(k-1);
end_proc;

specfunc::MeijerG::matchInt := proc( xpr, var, left, right, substitute=TRUE )
  local targets, /* Liste aller mglichen Integralformeln */
    factors, /* Liste aller Faktoren des Integranden */
    constFact, /* Konstanter Vorfaktor */

    alpha, /* 1-Exponent bei vorkommen von var (i.e. var^(alpha-1) ist Faktor) */
    beta, /* Exponent bei (a-x)^(beta-1), fr Formeln 2.24.2-3 */
    GList, /* Liste der G-Funktionen im Integranden */
    remains, /* Liste aller brigen Faktoren */

    int_zero_to_infinity, /* proc: die Integrationen von 0..infinity durchgehen */
    int_zero_to_const, /* proc: die Integration von 0..right (right positiv) */
    int_const_to_infinity, /* proc: die Integration von left..infinity (left positiv) */

    getG, /* proc: wandelt einen Faktor in einen G-Term um */
    getGListTry,

    i, a, b, tmp, s, factor, Int, I1, I2, G; /* Temporre Variablen */
begin
  getG := proc( term )
    local G, a, b, i;
  begin
    G := meijerG::convertTo( factor, var, All );
    if G=FAIL then return( FAIL ); end_if;
    if substitute then
      for i in G do
        [ a, b ] := specfunc::MeijerG::convolutionInt::extractFactor( G[2] );
        if type(b)="meijerG" then
          a := Type::Linear( op(b,3), [var] );
          if a<>FALSE and a[2]<>0 then
            b := specfunc::MeijerG::matchInt( subs(xpr, var=var-a[2]), var, left+a[2], right+a[2], FALSE );
            if b<>FAIL then return( b ); end_if;
          end_if;
        end_if;
      end_for;
    end_if;
    substitute := FALSE;
    GList := GList.[ G ];
    return( TRUE );
  end_proc:

  int_zero_to_infinity := proc( GList )
    local formula, /* Die betrachtete Formel */
      l, i, j, R, RSum, glist, cond, solutions;
  begin
    solutions := [];
    for l in GList do
      glist := map(l,op,2);
      cond := _and(op(map(l, op, 1)));
      for i from 1 to nops(glist) do
        if type(op(glist,[i]))="_plus" then
          RSum := 0;
          for j in [op(op(glist,[i]))] do
            R := int_zero_to_infinity( [subsop(l, i=[op(l,[i,1]),j])] );
            if R=FAIL then return(FAIL); end_if;
            RSum := RSum+R;
          end_for;
          return( RSum );
        end_if;
      end_for;
      for formula in targets do
        case formula
          of `#2.24.1.1` do
            if nops( glist )=2 then
              i := meijerG::twoG( alpha, glist[1], glist[2], var );
              if i<>FAIL then
                if cond=TRUE then
                  return(i);
                else
                  solutions := solutions.[[cond, i]];
                end_if;
              end_if;
            end_if;
            break;
          of `#2.24.2.1` do
            if nops( glist )=1 then
              i := meijerG::oneG( alpha, glist[1], var );
              if i<>FAIL then
                if cond=TRUE then
                  return(i);
                else
                  solutions := solutions.[[cond, i]];
                end_if;
              end_if;
            end_if;
        end_case;
      end_for;
    end_for;
    if solutions<>[] then
      solutions := select(map(solutions, X->[simplify::simplifyCondition(X[1]), X[2]]), X->(X[1]<>FALSE and not has(X[1], var)));
      if solutions=[] then return(FAIL); end_if;
      return(piecewise(op(solutions)));
    end_if;
    return( FAIL );
  end_proc;

  int_zero_to_const := proc( bound )
    local formula, i, j, glist, cond, solutions;
  begin
    if bound=0 then return( 0 ); end_if;
    solutions := [];
    for j in GList do
      glist := map(j,op,2);
      cond := _and(op(map(j, op, 1)));
      for formula in targets do
        case formula
          of `#2.24.2.2` do
            if nops( glist )=1 then
              i := specfunc::MeijerG::convolutionInt::int_2_24_2_2( bound, alpha, beta, glist, var );
              if i<>FAIL then
                if cond=TRUE then
                  return(i);
                else
                  solutions := solutions.[[cond, i]];
                end_if;
              end_if;
            end_if;
        end_case;
      end_for;
      i := ( int_zero_to_infinity( [[op(j), [ TRUE, meijerG( [[],[1]], [[0],[]], var/bound) ]]] /* heaviside(right-var) */ ) );
      if i<>FAIL then
        if type(i)=piecewise then
          solutions := solutions.[op(i)];
        else
          return(i);
        end_if;
      end_if;
    end_for;
    if solutions<>[] then
      solutions := select(map(solutions, X->[simplify::simplifyCondition(X[1]), X[2]]), X->(X[1]<>FALSE and not has(X[1], var)));
      if solutions=[] then return(FAIL); end_if;
      return(piecewise(op(solutions)));
    end_if;
    FAIL;
  end_proc;

  int_const_to_infinity := proc( bound )
    local glist, j;
  begin
    for j in GList do
      glist := map(j,op,2);
      if bound=0 then return( int_zero_to_infinity( GList ) ); end_if;
      i := ( int_zero_to_infinity( [[op(j), [ TRUE, meijerG( [[1],[]], [[],[0]], var/bound) ]]] /* heaviside(var-left) */ ) );
      if i<>FAIL then return( i ); end_if;
    end_for;
    FAIL;
  end_proc;

  if not is( (left=-infinity or left in R_) and (right=infinity or right in R_), Goal=TRUE ) then return( FAIL ); end_if;

  /* Vorbereitung */
  targets := { `#2.24.1.1`, `#2.24.2.1`, `#2.24.2.2` };
  alpha := 1;
  beta := 1;
  GList := [];
  remains := [];
  assume( var>left and var>0 );

  /* Integranden zerlegen */
  if type(xpr)="_mult" then
    factors := [ op(xpr) ];
  else
    factors := [ xpr ];
  end_if;

  /* Konstante Faktoren suchen */
  s := split( factors, X->has( X, var ) );
  factors := s[1];
  assert( s[3]=[] );
  constFact := _mult( op( s[2] ) );

  /* Konstantes Integral? */
  if factors=[] then return( (right-left)*constFact ); end_if;

  /* Faktoren durchgehen */
  for factor in factors do
    case type(factor)
      of DOM_IDENT do
        assert( factor=var );
        targets := targets intersect { `#2.24.1.1`, `#2.24.2.1`, `#2.24.2.2` };
        alpha := alpha+1;
        break;
      of "_power" do
        // if the integration variable appears in the exponent
        // (e.g. like x^x) then it should not be added to alpha
        if has(op(factor, 2), var) then
          remains := remains.[ factor ];
          break;
        end_if;
        if op(factor,1)=var then
          alpha := alpha+op(factor,2);
          targets := targets intersect { `#2.24.1.1`, `#2.24.2.1`, `#2.24.2.2` };
        else
          /* Testen, ob wir (a-x)^(beta-1) haben */
          a := Type::Linear( op(factor,1), [var] );
          if a<>FALSE and (a[1]=1 or a[1]=-1) then
            if left=0 and a[2]/a[1]=-right then
              targets := targets intersect { `#2.24.2.2` };
              constFact := constFact*(-a[1])^op(factor,2);
              beta := op(factor,2)+1;
            else
              remains := remains.[ factor ];
            end_if;
          else
            remains := remains.[ factor ];
          end_if;
        end_if;
        break;
      otherwise
        remains := remains.[ factor ];
    end_case;

    /* Sind Mglichkeiten brig geblieben? */
    if targets = {} then return( FAIL ); end_if;
  end_for;

  /* Restliche Faktoren untersuchen */
  getGListTry := proc( remains, factors=1 )
    local i, G, res;
  begin
    for i from 1 to nops(remains) do
      res := getGListTry( subsop( remains, i=null() ), remains[i]*factors );
      if res<>FAIL then return( res ); end_if;
    end_for;
    for i from 1 to nops(remains) do
      G := meijerG::convertTo( factors*remains[i], var, All );
      if G<>FAIL and G<>[] then
        return( [ subsop( remains, i=null() ), G ] );
      end_if;
    end_for;
    return( FAIL );
  end_proc:

  while nops(remains)>0 do
    tmp := getGListTry( subsop( remains, 1=null() ), remains[1] );
    if tmp=FAIL then
      G := meijerG::convertTo( remains[1], var, All );
      if G<>FAIL then
        tmp := [ subsop( remains, 1=null() ), G ];
      else
        /* Wir konnten keine G-Reprsentation finden. Bei Summe mappen */
        if type( remains[1] )="_plus" then
          i := contains( factors, remains[1] );
          if i>0 then
            tmp := map( remains[1], X->specfunc::MeijerG::matchInt( X*_mult(op(subsop(factors,i=null()))), var, left, right ) );
            if tmp<>FAIL then return( tmp*constFact ); end_if;
          end_if;
        end_if;
        return( FAIL );
      end_if;
    end_if;
    /* Lineare Substitution prfen */
    if substitute then
      [ a, b ] := specfunc::MeijerG::convolutionInt::extractFactor( tmp[2] );
      if type(b)="meijerG" then
        a := Type::Linear( op(b,3), [var] );
        if a<>FALSE and a[2]<>0 then
          b := specfunc::MeijerG::matchInt( subs(xpr, var=var-a[2]), var, left+a[2], right+a[2], FALSE );
          if b<>FAIL then return( constFact*b ); end_if;
        end_if;
      end_if;
    end_if;
    substitute := FALSE;
    remains := tmp[1];
    GList := GList.[ tmp[2] ];
  end_while;

  Int := FAIL;
  if nops(GList)=0 then
    GList := [];
  else
    GList := combinat::cartesianProduct( op(GList) );
  end_if;

  /* Grenzen untersuchen */
  /* rechte Grenze = infinity */
  if right=infinity then
    if left=0 then
      Int := int_zero_to_infinity( GList );
    elif left=-infinity then
      I1 := int_zero_to_infinity( GList );
      I2 := specfunc::MeijerG::matchInt ( subs(xpr,var=-var)/constFact, var, 0, infinity, substitute );
      Int := Simplify( I1+I2 );
    elif is(left<0,Goal=TRUE) then
      I1 := int_zero_to_infinity( GList );
      I2 := specfunc::MeijerG::matchInt ( subs(xpr,var=-var)/constFact, var, 0, -left, substitute );
      Int := Simplify( I1+I2 );
    else
      Int := int_const_to_infinity( left );
    end_if;
    if Int<>FAIL then
      Int := Int*constFact;
      userinfo( 2, "Integral of ".expr2text(xpr)." is ".expr2text(Int) );
    end_if;
    return( Int );
  end_if;

  assert( right <> infinity );

  /* Wir wissen, dass die rechte Grenze positiv, aber nicht infinity ist */

  if left=0 then
    /* Integral 0..right */
    Int := int_zero_to_const( right );
  elif is( left<=0, Goal=TRUE ) then
    /* I1 := Integral ber 0..right */
    I1 := int_zero_to_const( right );
    /* I2 := Integral ber left..0 */
    I2 := specfunc::MeijerG::matchInt ( subs(xpr,var=-var)/constFact, var, 0, -left, substitute );
    Int := Simplify( I1+I2 );
  elif is(left>=0, Goal=TRUE) then
    /* Erster Versuch: gemeinsame Grenze 0 */
    /* I1 := Integral ber 0..right */
    I1 := int_zero_to_const( right );
    if I1<>FAIL then
      /* I2 := Integral ber 0..left */
      I2 := int_zero_to_const( left );
      if I2<>FAIL then Int := Simplify( I1-I2 ); end_if;
    end_if;

    /* Zweiter Versuch: gemeinsame Grenze infinity */
    if Int=FAIL then
      /* I1 := Integral ber left..infinity */
      I1 := int_const_to_infinity( left );
      if I1<>FAIL then
        /* I2 := Integral ber right..infinity */
        I2 := int_const_to_infinity( right );
        if I2<>FAIL then Int := Simplify( I1-I2 ); end_if;
      end_if;
    end_if;
  end_if;
  if Int<>FAIL then
    Int := Int*constFact;
    userinfo( 2, "Integral of ".expr2text(xpr)." is ".expr2text(Int) );
  end_if;
  return( Int );
end_proc:

/* conv1
* Berechnet das Integralprodukt zweier G-Funktionen nach Prudnikov, 2.24.1/1.
*/
meijerG::conv1 :=
proc( alpha, G1, G2, subst1, subst2, var )
  local mu, ro, n, m, mult, k, a, b, c, d, X, l, as, bs, i, t, u, v, omega, sigma,
  conditions, checkConditions, cond,
  bstar, cstar, p, q, r, s, res;
begin
  checkConditions :=
  proc()
    local i, j, i1, i2,
    val, subcond, subconds, subcondsval, allCond;
  begin
    subcondsval := table();
    subconds :=
    [
     TRUE, /* 1 */
     proc()
       local j, h;
     begin
       simplify::simplifyCondition( _and( (Re(alpha+d[h]+l/k*b[j])>0 $ h=1..s)
                                         $j=1..m ) );
     end_proc, /* 2 */
     proc()
       local i, g;
     begin
       for i from 1 to n do
         for g from 1 to t do
           if not is(Re(alpha+c[g]+l/k*a[i])<l/k, Goal=TRUE) then
             return(FALSE)
           end_if;
         end_for;
       end_for;
       return(TRUE);
     end_proc, /* 3 */
     proc()
       local g;
     begin
       for g from 1 to t do
         if is( (p-q)*Re(alpha+c[g]-1)-r*Re(mu)>-3*r/2, Goal=TRUE )=FALSE then
           return( FALSE )
         end_if
       end_for;
       return(TRUE);
     end_proc, /* 4 */
     proc()
       local h;
     begin
       for h from 1 to s do
         if is( (p-q)*Re(alpha+d[h])-r*Re(mu)>-3*r/2, Goal=TRUE )=FALSE then
           return( FALSE )
         end_if;
       end_for;
       return(TRUE)
     end_proc, /* 5 */
     proc()
       local i;
     begin
       for i from 1 to n do
         if is( (u-v)*Re(alpha+r*a[i]-r)-Re(ro)>-3/2, Goal=TRUE )=FALSE then
           return( FALSE )
         end_if;
       end_for;
       return(TRUE);
     end_proc, /* 6 */
     proc()
       local j;
     begin
       for j from 1 to m do
         if is( (u-v)*Re(alpha+r*b[j])-Re(ro)>-3/2, Goal=TRUE )=FALSE then
           return( FALSE )
         end_if;
       end_for;
       return(TRUE);
     end_proc, /* 7 */
     ()->simplify::simplifyCondition
     ( abs(q-p-r*(v-u)) + 2*Re( (q-p)*(v-u)*alpha+r*(v-u)*(mu-1)+(q-p)*(p-1) )
      >0 ), /* 8 */
     ()->simplify::simplifyCondition
     ( abs(q-p-r*(v-u)) - 2*Re( (q-p)*(v-u)*alpha+r*(v-u)*(mu-1)+(q-p)*(p-1) )
      >0 ), /* 9 */
     ()->simplify::simplifyCondition( abs(arg(sigma))<bstar*PI ), /* 10 */
     ()->simplify::simplifyCondition( abs(arg(sigma))=bstar*PI ), /* 11 */
     ()->simplify::simplifyCondition( abs(arg(omega))<cstar*PI ), /* 12 */
     ()->simplify::simplifyCondition( abs(arg(omega))=cstar*PI ) /* 13 */
    ];
    subcond :=
    proc(num)
    begin
      if not contains( subcondsval, num ) then
        if num>nops(subconds) then
          return( FALSE )
        end_if;
        subcondsval[num] := subconds[num]();
      end_if;
      return( subcondsval[num] );
    end_proc;
    conditions :=
    [
     [m*n*s*t<>0, bstar>0, cstar>0, [1, 2, 3,10,12] ], /* 1 */
     [u=v, bstar=0, cstar>0, sigma>0, Re(ro)<1, [1,2,3,12] ],
     [p=q, cstar=0, bstar>0, omega>0, Re(mu)<1, [1,2,3,10] ], /* 3 */
     [p=q, u=v, bstar=0, 0=cstar, sigma>0, omega>0, Re(mu)<1, Re(ro)<1,
      sigma^l<>omega^k, [1,2,3] ],
     [p=q, u=v, bstar=0, 0=cstar, sigma>0, omega>0, Re(mu+ro)<1,
      sigma^l=omega^k, [1,2,3] ], /* 5 */
     [p>q, s>0, bstar>0, cstar>=0, [1,2,3,5,10,13] ],
     [p<q, t>0, bstar>0, cstar>=0, [1,2,3,4,10,13] ], /* 7 */
     [u>v, m>0, cstar>0, bstar>=0, [1,2,3,7,11,12] ],
     [u<v, n>0, cstar>0, bstar>=0, [1,2,3,6,11,12] ], /* 9 */
     [p>q, u=v, bstar=0, cstar>=0, sigma>0, Re(ro)<1, [1,2,3,5,13] ],
     [p<q, u=v, bstar=0, cstar>=0, sigma>0, Re(ro)<1, [1,2,3,4,13] ], /* 11 */
     [p=q, u>v, bstar>=0, cstar=0, omega>0, Re(mu)<1, [1,2,3,7,11] ],
     [p=q, u<v, bstar>=0, cstar=0, omega>0, Re(mu)<1, [1,2,3,6,11] ], /* 13 */
     [p<q, u>v, bstar>=0, cstar>=0, [1,2,3,4,7,11,13] ],
     [p>q, u<v, bstar>=0, cstar>=0, [1,2,3,5,6,11,13] ], /* 15 */
     [p>q, u>v, bstar>=0, cstar>=0, [1,2,3,5,7,8,11,13,14] ],
     [p<q, u<v, bstar>=0, cstar>=0, [1,2,3,4,6,9,11,13,14] ], /* 17 */
     [t=0, s>0, bstar>0, q-p-l/k*(v-u)>0, [1,2,10] ],
     [s=0, t>0, bstar>0, q-p-l/k*(v-u)<0, [1,3,10] ], /* 19 */
     [n=0, m>0, cstar>0, q-p-l/k*(v-u)<0, Re(sigma)>0, [1,2,12] ],
     // Note: The condition Re(sigma)>0 is not in the papers.
     // It is added because
     // meijerG::convolutionInt(x*exp(p*x)*exp(-4*x^2), x=0..infinity)
     // gave the same result for p := -p
     [m=0, n>0, cstar>0, q-p-l/k*(v-u)>0, [1,3,12] ], /* 21 */
     [s*t=0, bstar>0, cstar>0, [1,2,3,10,12] ],
     [m*n=0, bstar>0, cstar>0, [1,2,3,10,12] ], /* 23 */
     [m+n>p, t=q-p-l/k*(v-u)=0, s>0, bstar>0, cstar<0,
      abs( arg(omega) )<(m+n-p+1)*PI, [1,2,10,14,15] ],
     [m+n>q, s=q-p-l/k*(v-u)=0, t>0, bstar>0, cstar<0,
      abs( arg(omega) )<(m+n-q+1)*PI, [1,3,10,14,15] ], /* 25 */
     [p=q-1, t=q-p-r*(v-u), q-p-r*(v-u)=0, s>0, bstar>0, cstar >= 0,
      cstar*PI<abs(arg(omega)),abs(arg(omega))<(cstar+1)*PI, [1,2,10,14,15] ],
     [p=q+1, s=q-p-r*(v-u), q-p-r*(v-u)=0, t>0, bstar>0, cstar >= 0,
      cstar*PI<abs(arg(omega)), abs(arg(omega))<(cstar+1)*PI, [1,3,10,14,15] ], /* 27 */
     [p<q-1, t=q-p-r*(v-u), q-p-r*(v-u)=0, s>0, bstar>0, cstar >= 0,
      cstar*PI<abs(arg(omega)), abs(arg(omega))<(m+n-p+1)*PI, [1,2,10,14,15] ],
     [p>q+1, s=q-p-r*(v-u), q-p-r*(v-u)=0, t>0, bstar>0, cstar >= 0,
      cstar*PI<abs(arg(omega)), abs(arg(omega))<(m+n-q+1)*PI, [1,3,10,14,15] ], /* 29 */
     [n=q-p-r*(v-u), q-p-r*(v-u)=0, s+t>u, m>0, cstar>0, bstar<0,
      abs(arg(sigma))<(s+t-u+1)*PI, [1,2,12,14,15] ],
     [m=q-p-r*(v-u), q-p-r*(v-u)=0, s+t>v, n>0, cstar>0, bstar<0,
      abs(arg(sigma))<(s+t-v+1)*PI, [1,3,12,14,15] ], /* 31 */
     [n=q-p-r*(v-u), q-p-r*(v-u)=0, u=v-1, m>0, cstar>0, bstar>=0,
      bstar*PI<abs(arg(sigma)), abs(arg(sigma))<(bstar+1)*PI, [1,2,12,14,15] ],
     [m=q-p-r*(v-u), q-p-r*(v-u)=0, u=v+1, n>0, cstar>0, bstar>=0,
      bstar*PI<abs(arg(sigma)), abs(arg(sigma))<(bstar+1)*PI, [1,3,12,14,15] ], /* 33 */
     [n=q-p-r*(v-u), q-p-r*(v-u)=0, u<v-1, m>0, cstar>0, bstar>=0,
      bstar*PI<abs(arg(sigma)), abs(arg(sigma))<(s+t-u+1)*PI, [1,2,12,14,15] ],
     [m=q-p-r*(v-u), q-p-r*(v-u)=0, u>v+1, n>0, cstar>0, bstar>=0,
      bstar*PI<abs(arg(sigma)), abs(arg(sigma))<(s+t-v+1)*PI, [1,3,12,14,15] ] /* 35 */
     ];

    allCond := FALSE;
    for i in conditions do
      val := TRUE;
      for j in i do
        if type(j)=DOM_LIST then
          for i1 in j do
            i2 := subcond(i1);
            if i2=FALSE then val := FALSE; break; end_if;
            val := val and i2;
          end_for;
        else
          i2 := simplify::simplifyCondition(j);
          if i2=FALSE then val := FALSE; break; end_if;
          val := val and i2;
        end_if;
      end_for;
      if val=TRUE then return(TRUE); end_if;
      if val<>FALSE then allCond := allCond or val; end_if;
    end_for;
    return( allCond );
  end_proc;

  s := nops(op(G1,[2,1]));
  t := nops(op(G1,[1,1]));
  u := nops(op(G1,[1,1])) + nops(op(G1,[1,2]));
  v := nops(op(G1,[2,1])) + nops(op(G1,[2,2]));
  m := nops(op(G2,[2,1]));
  n := nops(op(G2,[1,1]));
  p := nops(op(G2,[1,1])) + nops(op(G2,[1,2]));
  q := nops(op(G2,[2,1])) + nops(op(G2,[2,2]));
  a := op(G2,[1,1]).op(G2,[1,2]);
  b := op(G2,[2,1]).op(G2,[2,2]);
  c := op(G1,[1,1]).op(G1,[1,2]);
  d := op(G1,[2,1]).op(G1,[2,2]);
  ro := specfunc::MeijerG::convolutionInt::mymu( G1 );
  mu := specfunc::MeijerG::convolutionInt::mymu( G2 );
  l := subst2[3];
  k := subst2[4];
  if l<=0 or k<=0 then return( FAIL ); end_if;
  r := l/k;
  sigma := subst1[1];
  omega := subst2[1];
  bstar := s+t-(u+v)/2;
  cstar := m+n-(p+q)/2;
  cond := sigma<>0 and omega<>0 and checkConditions();
  if cond=FALSE then return( FAIL ); end_if;

  as := [specfunc::MeijerG::convolutionInt::delta( k, a[i] ) $ i=1..n,
         specfunc::MeijerG::convolutionInt::delta( l, 1-alpha-d[i] ) $ i=1..s],
        [specfunc::MeijerG::convolutionInt::delta( l, 1-alpha-d[i] ) $ i=(s+1)..v,
         specfunc::MeijerG::convolutionInt::delta( k, a[i] )$i =(n+1)..p ];
  bs := [specfunc::MeijerG::convolutionInt::delta( k, b[i] ) $ i=1..m,
         specfunc::MeijerG::convolutionInt::delta( l, 1-alpha-c[i] ) $ i=1..t],
        [specfunc::MeijerG::convolutionInt::delta( l, 1-alpha-c[i] ) $ i=(t+1)..u,
         specfunc::MeijerG::convolutionInt::delta( k, b[i] )$i =(m+1)..q ];

  X := omega^k*k^(k*(p-q))/( sigma^l*l^(l*(u-v)) );
  mult := (
           ( k^mu * l^(ro+alpha*(v-u)-1)*sigma^(-alpha) ) /
           ((2*PI)^(bstar*(l-1)+cstar*(k-1)))
           ) ;
  res := mult*meijerG( [as], [bs], X );
  if cond<>TRUE then
    res := piecewise( [cond,res] )
  end_if;
  return( res )
end_proc;

meijerG::twoG :=
proc( alpha, G1, G2, var )
  local m1, m2, m3, m4, subst1, subst2, i;
begin
  [m1,G1] := specfunc::MeijerG::convolutionInt::extractFactor( G1, var );
  if G1=FAIL then return(FAIL); end_if;
  [m2,G2] := specfunc::MeijerG::convolutionInt::extractFactor( G2, var );
  if G2=FAIL then return(FAIL); end_if;
  subst1 := specfunc::MeijerG::convolutionInt::analyzeSubst( G1, var );
  subst2 := specfunc::MeijerG::convolutionInt::analyzeSubst( G2, var );
  if subst1=FAIL or subst2=FAIL then return( FAIL ); end_if;
  if subst1[2]=0 and subst2[2]=0 then
    if subst2[3]=1 and subst2[4]=1 then
      [subst1,subst2] := [subst2,subst1];
      [G1,G2] := [G2,G1];
    end_if;
    if subst1[3]<>1 or subst1[4]<>1 then
      if is( subst1[3]/subst1[4]>0, Goal=TRUE ) then
        /* x^s substituieren */
        alpha := ( alpha-1 )*(subst1[4]/subst1[3]) +
                 (subst1[4]/subst1[3] - 1) + 1;
        m1 := m1 * subst1[4]/subst1[3];
        i := subst2[3]/subst2[4] * subst1[4]/subst1[3];
        if type(i)=DOM_INT then
          subst2[3] := i;
          subst2[4] := 1;
        elif type(i)=DOM_RAT then
          subst2[3] := op(i,1);
          subst2[4] := op(i,2);
        else
          return( FAIL );
        end_if;
        subst1[3] := 1;
        subst1[4] := 1;
        G1 := subsop( G1, 3=subst1[1]*var );
      else
        return( FAIL );
      end_if;
    end_if;
    m3 := meijerG::conv1( alpha, G1, G2, subst1, subst2, var );
    [m3, m4] := specfunc::MeijerG::convolutionInt::extractFactor( m3, var );
    if type(m4)="meijerG" then
      m4 := meijerG::Simplify(m4)
    end_if;
    m3 := m3*m4;
  else
    return( FAIL );
  end_if;
  userinfo( 10, "Integral calculated to be ".expr2text(m1*m2*m3) );
  return( simplify(m1*m2*m3) );
end_proc:

/* conv2
  * Berechnet das Integral einer G-Funktionen nach Prudnikov, 2.24.2/1.
  */
meijerG::conv2 := proc( alpha, G1, subst1, var )
  local omega, a1, a2, b1, b2, cstar, m, n, p, q, cond, i, res, mu, subcond;
begin
  omega := subst1[1];
  cond := omega<>0;
  m := nops(op(G1,[2,1]));
  n := nops(op(G1,[1,1]));
  p := nops(op(G1,[1,1])) + nops(op(G1,[1,2]));
  q := nops(op(G1,[2,1])) + nops(op(G1,[2,2]));
  cstar := m+n-(p+q)/2;
  [ a1,a2 ] := op(G1,[1]);
  [ b1,b2 ] := op(G1,[2]);
  if b1<>[] then
    cond := cond and simplify::simplifyCondition( -min( op( map( b1, Re ) ) )< Re( alpha ) );
    if cond=FALSE then return(FAIL); end_if;
  end_if;
  if a1<>[] then
    cond := cond and simplify::simplifyCondition( Re(alpha) < 1 - max( op( map( a1, Re ) ) ) );
    if cond=FALSE then return(FAIL); end_if;
  end_if;
  mu := specfunc::MeijerG::convolutionInt::mymu( G1 );
  subcond := ( m<>0 or n<>0 ) and abs(arg(omega))<cstar*PI;
  if p=q then
    subcond := subcond or cstar=0 and omega>0 and Re( _plus( (b1.b2)[i]-(a1.a2)[i] $i=1..q ) )<0;
  else
    subcond := subcond or cstar>=0 and abs(arg(omega))=cstar*PI and Re (mu +(q-p)*alpha)<3/2
  end;
  cond := cond and _and(op(map(b1.a2, X->not 1-X-alpha in N_)))
    and _and(op(map(b2.a1, X->not X+alpha in N_)));
  cond := simplify::simplifyCondition( cond and subcond );
  if cond=FALSE then return( FAIL ); end_if;
  res := ( omega^(-alpha) *
    _mult( op(map( b1, X->gamma(X+alpha) )) )*
    _mult( op(map( a1, X->gamma(1-X-alpha) )) ) /
    _mult( op(map( a2, X->gamma(X+alpha) )) ) /
    _mult( op(map( b2, X->gamma(1-X-alpha) )) ) );
  if cond=TRUE then
    return( res );
  else
    return( piecewise( [cond,res] ) );
  end_if;
end_proc;

meijerG::oneG := proc( alpha, G1, var )
  local m1, subst1;
begin
  [m1,G1] := specfunc::MeijerG::convolutionInt::extractFactor( G1, var );
  if G1=FAIL then return(FAIL); end_if;
  if type(G1)<>"meijerG" then return( FAIL ); end_if;
  subst1 := specfunc::MeijerG::convolutionInt::analyzeSubst( G1, var );
  if subst1=FAIL or subst1[2]<>0 then return( FAIL ); end_if;
  if subst1[3]/subst1[4]<>1 then
    if is( subst1[3]/subst1[4]>0, Goal=TRUE ) then
      /* x^s substituieren */
      alpha := ( alpha-1 )*(subst1[4]/subst1[3]) + (subst1[4]/subst1[3] - 1) + 1;
      m1 := m1 * subst1[4]/subst1[3];
      subst1[3] := 1;
      subst1[4] := 1;
      G1 := subsop( G1, 3=subst1[1]*var );
    else
      return( FAIL );
    end_if;
  end_if;
  return( Simplify( m1*meijerG::conv2( alpha, G1, subst1, var ) ) );
end_proc;
