/*++
        Si -- the Sine integral

	Si(x):= int(sin(t)/t,t=0..x)

	Reference: Abramowitz & Stegun, pages 231-233
++*/

Si:=
proc(x)
  option noDebug;
  local f;
begin
  if args(0) <> 1 then 
     error("expecting one argument");
  elif x::dom::Si <> FAIL then 
     return(x::dom::Si(args()))
  end_if;

  case type(x)
    of DOM_SET do
    of "_union" do
      return(map(x, Si))
    of DOM_INT do
    of DOM_RAT do
      if x < 0 then return(-Si(-x)) end_if; break
    of DOM_FLOAT do return(Si::float(x))
    of DOM_COMPLEX do
      if domtype(op(x,1)) = DOM_FLOAT or
         domtype(op(x,2)) = DOM_FLOAT
      then return(Si::float(x))
      end_if;
    //also normalize complex numbers ?
    //if op(x,1)<0 or (op(x,1)=0 and op(x,2)<0) then
    //   return(-Si(-x))
    //end_if;
  end_case;

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      if type(x)=Dom::ImageSet then
        return(map(x, Si));
      else
        return(Dom::ImageSet(Si(#x), #x, x));
      end_if;
    end_if;

    error("argument must be of 'Type::Arithmetical'")
  end_if;

// in analogy to sin, cos:
  if type(x)= "_mult" then
      f:= op(x, nops(x));
      if testtype(f, Type::Real) and f < 0
         then return(-Si(-x))
      end_if;

    //also normalize complex factors ?
    //if testtype(f, Type::Complex) and 
    //   (Re(f) < 0 or (Re(f)=0 and Im(f)<0))
    //   then return(-Si(-x))
    //end_if;
  end_if;

  procname(x)
end_proc:

Si := prog::remember(Si,  
  () -> [property::depends(args()), DIGITS, slotAssignCounter("Si")]):

Si(0):=0:
Si(infinity):=PI/2:
Si(-infinity):=-PI/2:
Si( I*infinity):= I*infinity:
Si(-I*infinity):=-I*infinity:

Si:= funcenv(Si, op(specfunc::Si, 2)):
Si::print:= "Si":
Si::type:= "Si":
Si::info:= "Si(x) -- the sine integral int(sin(t)/t,t=0..x)":
Si::float:= specfunc::Si:

Si::diff:= proc(x, y)
             local op1;
           begin  // e=Si(f)
             op1 := op(x,1);
             sin(op1)/op1*diff(op1, y)
           end_proc:

Si::complexDiscont:= {}:
Si::realDiscont:= {}:
Si::undefined:= {}:

Si::"transform::laplace":= loadproc(Si::"transform::laplace",
                                    pathname("TRANS", "LAPLACE"), "L_Si"):

Si::conjugate:= x -> Si(conjugate(x)):

Si::rectform:= loadproc(Si::rectform, pathname("STDLIB", "RECTFORM"), "Si"):

Si::series:= loadproc(Si::series, pathname("SERIES"), "Si"):

Si::isolate :=
proc(Lhs, Rhs, x, options, reclevel, known)
begin
  if iszero(Rhs) then
    // Si(x)=0 <=> x=0
    solvelib::isolate(op(Lhs), Rhs, x, options, reclevel+1, known);
  elif (options[Real]=TRUE or is(x in R_, Goal=TRUE)) and
    is(abs(Rhs)>2) then // TODO: we know max(Si(x), x in R_)<2, but there are smaller bounds
    {};
  else
    hold(solve)(Lhs=Rhs, x, solvelib::nonDefaultOptions(options));
  end_if;
end_proc:

// end of file 
