/*++
arccos -- the inverse cosine

arccos(x)

x - expression

NB:  The branch cuts are the real intervals (-infinity, -1) and
     (1, infinity).
     Note: for x >=1 :  arccos(x) =  limit(arccos(x+eps*I),eps=0, eps>0)
                                  = -limit(arccos(x-eps*I),eps=0, eps>0)
           for x <=-1:  arccos(x) =  limit(arccos(x-eps*I),eps=0, eps>0)

     I.e., on the left real line the values coincide with the limits from
     below, on the right real line with the limit from above. OK.

     Consistent with arccos(x)+arccos(-x) = PI. OK.
++*/

arccos:=
proc(x)
  local f, handleCos;
  option noDebug;
begin
  if args(0) = 0 then 
     error("no arguments given")
  elif x::dom::arccos <> FAIL then 
     return(x::dom::arccos(args()))
  elif args(0) <> 1 then 
     error("wrong no of args")
  end_if;

  case type(x)
    of DOM_INT do
    of DOM_RAT do
       if x < 0 then return(PI - arccos(-x)) end_if;
       break;

    of DOM_FLOAT do
       return(arccos::float(x));

    of DOM_COMPLEX do
       if domtype(op(x,1)) = DOM_FLOAT or
          domtype(op(x,2)) = DOM_FLOAT then
           return(arccos::float(x))
       end_if;
       if op(x,1) = 0 then return(PI/2 - arcsin(x)) end_if;
       break;

    of DOM_SET do
    of "_union" do
      return(map(x,arccos))
    of "_intersect" do
    of "_minus" do
       return(Dom::ImageSet(arccos(`#t`), `#t`, x))
  end_case;

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      return(Dom::ImageSet(arccos(#t), #t, x));
    end_if;

    error("argument must be of 'Type::Arithmetical'")
  end_if;

  //=====================================
  handleCos:= proc(x)  // = arccos(cos(x))
  local f, s;
  begin
    if (is(x>=0) and is(x<=PI))=TRUE then 
       return(x)
    end_if;
    f:=floor(Re(x)/PI):

    s:= signIm(PI/2 - x);
    if domtype(s) <> DOM_INT then
       return(FAIL);
    end_if;
    f:= s*ceil(-s*Re(PI/2 - x)/PI - 1/2);
    if type(f) = DOM_INT then
       case f mod 2
       of 0 do return(x-f*PI);
       of 1 do return((f+1)*PI-x);
       end_case;
    end_if:
    FAIL;
  end_proc:
  //=====================================

  case type(x)
    of "_mult" do
       f:= op(x, nops(x));
       if (testtype(f, Type::Real) and f < 0) or
          (testtype(f/I, Type::Real) and f/I < 0) then
             return(PI - arccos(-x));
       end_if;
       if type(-x) = "cosh" then //-cosh(y) = cos(PI/2 + y*I)
          f:= handleCos(PI/2 + I*op(-x));
          if f <> FAIL then
             return(f);
          end_if;
       end_if:
       if type(x/I) = "sinh" then //I*sinh(y) = cos(PI/2 - I*y)
          f:= handleCos(PI/2 - I*op(x/I));
          if f <> FAIL then
             return(f); 
          end_if;
       end_if:
       if type(-x/I) = "sinh" then //-I*sinh(y) = cos(PI/2 + I*y)
          f:= handleCos(PI/2 + I*op(-x/I));
          if f <> FAIL then
             return(f);
          end_if;
        end_if:
       break;
    of "cos" do // arccos : [-1..1] --> [PI..0]
       f:= handleCos(op(x));
       if f <> FAIL then
         return(f);
       end_if; 
       break;
    of "sin" do // sin(y) = cos(-y + PI/2);
       f:= handleCos(PI/2 - op(x));
       if f <> FAIL then
         return(f);
       end_if; 
       break;
    of "cosh" do // cosh(y) = cos(y*I)
       f:= handleCos(I*op(x));
       if f <> FAIL then
         return(f);
       end_if; 
       break;
  end_case;

/*// It is too dangerous to have **automatic** rewriting of
  // arccos(-x) into a sum of 2 terms! Let the user request
  // this explicitely via simplify/Simplify!
  //
  // It is useful to have a normal form such that expressions like
  // arccos(x-y) + arccos(y-x) are simplified automatically. 
  if stdlib::hasmsign(x) then
      PI - procname(-x)
  else
      procname(x)
  end_if
*/
      procname(x)
end_proc:

arccos := prog::remember(arccos,  
  () -> [property::depends(args()), DIGITS, slotAssignCounter("arccos")]):

arccos( 0):=  PI/2:
arccos( 1):= 0:
arccos(-1):= PI:
arccos( 1/2):=   PI/3:
arccos(-1/2):= 2*PI/3:
arccos( (3^(1/2))/2):=   PI/6:
arccos(-(3^(1/2))/2):= 5*PI/6:
arccos( (2^(1/2))/2):=   PI/4:
arccos(-(2^(1/2))/2):= 3*PI/4:
arccos( 1/4*2^(1/2)*(5+5^(1/2))^(1/2)):=   PI/10:
arccos(-1/4*2^(1/2)*(5+5^(1/2))^(1/2)):= 9*PI/10:
arccos( 1/4*2^(1/2)*(5-5^(1/2))^(1/2)):= 3*PI/10:
arccos(-1/4*2^(1/2)*(5-5^(1/2))^(1/2)):= 7*PI/10:
arccos( ((2-2^(1/2))^(1/2))/2):= 3*PI/8:
arccos(-((2-2^(1/2))^(1/2))/2):= 5*PI/8:
arccos( ((2+2^(1/2))^(1/2))/2):=   PI/8:
arccos(-((2+2^(1/2))^(1/2))/2):= 7*PI/8:
arccos( 1/4*(5^(1/2)+1)):=   PI/5:
arccos(-1/4*(5^(1/2)+1)):= 4*PI/5:
arccos( 1/4*(5^(1/2)-1)):= 2*PI/5:
arccos(-1/4*(5^(1/2)-1)):= 3*PI/5:
arccos( (6^(1/2)-2^(1/2))/4):= 5*PI/12:
arccos(-(6^(1/2)-2^(1/2))/4):= 7*PI/12:
arccos( (6^(1/2)+2^(1/2))/4):=   PI/12:
arccos(-(6^(1/2)+2^(1/2))/4):=11*PI/12:

arccos( infinity):=   -I*infinity:
arccos(-infinity):= PI+I*infinity:
arccos( I*infinity):= PI/2 - I*infinity:
arccos(-I*infinity):= PI/2 + I*infinity:

arccos:= funcenv(arccos, op(specfunc::arccos, 2)):
arccos::info := "arccos -- the inverse cosine":
arccos::print := "arccos":
arccos::type := "arccos":
arccos::float := specfunc::arccos:
arccos::hull := DOM_INTERVAL::arccos@hull:

arccos::inverse := "cos":

arccos::diff :=
  proc(f)
    local op1;
  begin
    op1 := op(f, 1);
    if iszero(1-op1^2) then
       0
    else
       -diff(op1, args(2..args(0))) / sqrt(1-op1^2)
    end_if;
  end_proc:

arccos::expand := x -> arccos(expand(op(x))):

arccos::simplify:= loadproc(arccos::simplify, pathname("STDLIB", "SIMPLIFY"),
                            "arccos"):



arccos::complexDiscont :=
   loadproc(arccos::complexDiscont, pathname("STDLIB", "DISCONT"), "arccos"):

arccos::realDiscont := {}:
arccos::undefined := {}:

arccos::series := loadproc(arccos::series, pathname("SERIES"), "arccos"):

arccos::rectform :=
    loadproc(arccos::rectform, pathname("STDLIB","RECTFORM"), "acos"):
arccos::getprop :=  proc(x:"arccos") local v; begin v := getprop(op(x)); if contains({Dom::Interval, solvelib::BasicSet}, type(v)) then return(arccos(v)); else return(C_); end_if; end_proc:

arccos::Content := stdlib::genOutFunc("Carccos", 1):

// end of file
