/*++
arccosh -- the inverse hyperbolic cosine

arccosh(x)

x - expression
++*/

arccosh:=
proc(x)
  option noDebug;
  local f, handleCosh, s;
begin
  if args(0) = 0 then error("no arguments given")
  elif x::dom::arccosh <> FAIL then return(x::dom::arccosh(args()))
  elif args(0) <> 1 then error("wrong no of args")
  end_if;

  case type(x)
  of DOM_FLOAT do
     return(arccosh::float(x));
  of DOM_COMPLEX do
     if domtype(op(x,1)) = DOM_FLOAT or
        domtype(op(x,2)) = DOM_FLOAT then
         return(arccosh::float(x))
     end_if;
     break;
  of DOM_SET do
  of "_union" do
     return(map(x,arccosh));
  of "_intersect" do
  of "_minus" do
     return(Dom::ImageSet(arccosh(`#z`), `#z`, x))
  end_case;

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      return(Dom::ImageSet(arccosh(#x), #x, x));
    end_if;
    error("argument must be of 'Type::Arithmetical'")
  end_if;

  //=============================================
  handleCosh:= proc(x) // compute arccosh(cosh(x))
  local f, fmod4, Rex, sRe;
  begin
    if testtype(x, Type::Real) or
       is(x in R_) = TRUE then
        return(abs(x));
    end_if;

    f:= Im(x)*2/PI:
    fmod4:= modp(floor(f), 4);
    if domtype(fmod4) <> DOM_INT then // x is symbolic
       return(FAIL);
    end_if;
    Rex:= Re(x):
    s:= 1;
    sRe:= sign(Rex):
    if domtype(sRe) <> DOM_INT then 
       return(FAIL);
    end_if:
    if domtype(f) = DOM_INT then
      if fmod4 = 0 or fmod4 = 2 then
         s:= -s;
      end_if;
      if sRe < 0 and fmod4 = 2 then 
         x:= -x;
      end_if:
      if sRe >= 0 and fmod4 = 0 then 
         x:= -x;
      end_if:
      if fmod4 = 3 then
         x:= -x;
      end_if;
    elif sRe = 0 and ((fmod4 = 0) or (fmod4 = 1)) then
      // do nothing
    elif sRe > 0 and ((fmod4 = 0) or (fmod4 = 3)) then
      // do nothing
    elif sRe < 0 and ((fmod4 = 1) or (fmod4 = 2)) then
      // do nothing
    else
      x:= -x;
    end_if:
    f:= round(Im(x)/2/PI);
    if domtype(f) = DOM_INT then
      return(s*(x - f*2*I*PI));
    end_if:
    return(FAIL);
  end_proc;
  //=============================================

  case type(x)  // use case instead of if so that we can use break
  of "_mult" do
     if type(-x) = "cos" then
        f:= handleCosh(PI*I + I*op(-x)); // -cos(x) = - cosh(I*x) = cosh(PI*I + I*x)
        if f <> FAIL then
           return(f);
        end_if;
     elif type(-x) = "cosh" then
        f:= handleCosh(PI*I + op(-x)); // -cosh(x) = cosh(PI*I + x)
        if f <> FAIL then
           return(f);
        end_if;
     end_if;
     if type(-x) = "sin" then  //-sin(x) = cosh(I*PI/2 + I*x)
        f:= handleCosh(I*PI/2 + I*op(-x));
        if f <> FAIL then
           return(f);
        end_if;
     end_if;
     if type(x/I) = "sinh" then
        f:= handleCosh(I*PI/2 + op(x/I)); // I*sinh(x) = cosh(I*PI/2 + x)
        if f <> FAIL then
           return(f);
        end_if;
     end_if;
     if type(-x/I) = "sinh" then
        f:= handleCosh(I*PI/2 - op(-x/I)); //-I*sinh(x) = cosh(I*PI/2 - x)
        if f <> FAIL then
           return(f);
        end_if;
     end_if;
     break;
  of "cos" do
     f:= handleCosh(I*op(x)); // cos(y) = cosh(I*y);
     if f <> FAIL then
        return(f);
     end_if;
     break;
  of "cosh" do // arccosh(cosh(y))=y when 0 <= Im(y) <= PI
     f:= handleCosh(op(x));
     if f <> FAIL then
       return(f);
     end_if;
     break;
  of "sin" do // sin(x) = cosh(I*PI/2 - I*x)
     f:= handleCosh(I*PI/2 - I*op(x)); 
     if f <> FAIL then
        return(f);
     end_if;
     break;
  end_case;

  procname(x)
end_proc:

arccosh := prog::remember(arccosh,  
  () -> [property::depends(args()), DIGITS, slotAssignCounter("arccosh")]):

arccosh(1):= 0:
arccosh(0):= I*PI/2:
arccosh(-1):=I*PI:
arccosh( 1/2):= PI*I/3:
arccosh(-1/2):= PI*I*2/3:
arccosh( 2^(1/2)/2):= PI*I/4:
arccosh(-2^(1/2)/2):= PI*I*3/4:
arccosh( 3^(1/2)/2):= PI*I/6:
arccosh(-3^(1/2)/2):= PI*I*5/6:
arccosh( infinity):= infinity:
arccosh(-infinity):=-infinity + PI*I:

arccosh:= funcenv(arccosh, op(specfunc::arccosh, 2)):
arccosh::print := "arccosh":
arccosh::info := "arccosh -- the inverse hyperbolic cosine":
arccosh::type := "arccosh":
arccosh::float := specfunc::arccosh:
arccosh::hull := DOM_INTERVAL::arccosh@hull:

// Folgendes complexDiscont ist richtig!  Fehler in  Abramowitz/Stegun!
arccosh::complexDiscont :=
    loadproc(arccosh::complexDiscont, pathname("STDLIB","DISCONT"), "arccosh"):
arccosh::realDiscont := {}:
arccosh::undefined := {}:

arccosh::inverse := "cosh":

arccosh::expand := x -> arccosh(expand(op(x))):

arccosh::simplify:= loadproc(arccosh::simplify, pathname("STDLIB", "SIMPLIFY"),
                             "arccosh"):

arccosh::diff :=
  proc(f)
    local op1;
  begin
    op1 := op(f, 1);
    if iszero(op1^2-1) then
       0
    else
       diff(op1, args(2..args(0))) / sqrt(op1^2-1)
    end_if;
  end_proc:

arccosh::series := loadproc(arccosh::series, pathname("SERIES"), "arccosh"):

arccosh::rectform:= loadproc(arccosh::rectform, pathname("STDLIB", "RECTFORM"),
                             "arccosh"):

arccosh::getprop :=  proc(x:"arccosh") local v; begin v := getprop(op(x)); if contains({Dom::Interval, solvelib::BasicSet}, type(v)) then return(arccosh(v)); else return(C_); end_if; end_proc:

arccosh::Content := stdlib::genOutFunc("Carccosh", 1):

// end of file
