/*++
arccot -- the inverse cotangens

arccot(x)

x - expression
++*/

arccot:=
proc(x)
  option noDebug;
  local f, handleCot;
begin
  if args(0) = 0 then
    error("no arguments given")
  elif x::dom::arccot <> FAIL then
    return(x::dom::arccot(args()))
  elif args(0) <> 1 then
    error("wrong no of args")
  end_if;

  case type(x)
    of DOM_INT do
    of DOM_RAT do
      if x < 0 then return(-arccot(-x)) end_if;
      break;

    of DOM_FLOAT do
      return(arccot::float(x));

    of DOM_COMPLEX do
      if x = I then
        error("singularity");
      elif domtype(op(x,1)) = DOM_FLOAT or
        domtype(op(x,2)) = DOM_FLOAT then
        return(arccot::float(x))
      elif op(x, 1) = 0 then
        return(-I*arccoth(-I*x));
      elif op(x, 1) < 0 and op(x, 2) < 0 then
        return(-arccot(-x));
      end_if;
      break;

    of DOM_SET do
    of "_union" do
       return(map(x, arccot))
    of "_intersect" do
    of "_minus" do
       return(Dom::ImageSet(arccot(`#t`), `#t`, x))
  end_case;

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      return(Dom::ImageSet(arccot(#t), #t, x));
    end_if;
    error("argument must be of 'Type::Arithmetical'")
  end_if;

  // ==========================================
  handleCot:= proc(x) // compute arccot(cot(x))
  local f, s;
  begin
      if (is(x>-PI/2) and is(x<=PI/2)) = TRUE then 
         return(x);
      end:
      s:= -signIm(x):
      if domtype(s) <> DOM_INT then
         return(FAIL);
      end_if:
      f:= s*round(s*Re(x)/PI);
      if type(f) = DOM_INT then 
         if x - f*PI = -PI/2 then
            return(PI/2)
	 else 
           return(x - f*PI)
	 end_if:
      end_if;
      return(FAIL);
   end_proc;
  // ==========================================

  case type(x)
    of "_mult" do
      f:= op(x, nops(x));
      if (testtype(f, Type::Real) and f < 0) or
         (testtype(f/I, Type::Real) and f/I < 0) then
          return(-arccot(-x));
      end_if;
      if type(-x) = "cot" then // -cot(y) = cot(-y)
         f:= handleCot(-op(-x));
         if f <> FAIL then
            return(f);
         end_if;
      end_if:
      if type(-1/x) = "tan" then // -1/tan(y) = cot(-y)
         f:= handleCot(-op(-1/x));
         if f <> FAIL then
            return(f);
         end_if;
      end_if:
      if type(I/x) = "tanh" then // I/tanh(y) = cot(-y*I)
         f:= handleCot(-I*op(I/x));
         if f <> FAIL then
            return(f);
         end_if;
      end_if:
      if type(-I/x) = "tanh" then //-I/tanh(y) = cot(y*I)
         f:= handleCot(I*op(-I/x));
         if f <> FAIL then
            return(f);
         end_if;
      end_if:
      if type(x/I) = "coth" then //I*coth(y) = cot(-y*I)
         f:= handleCot(-I*op(x/I));
         if f <> FAIL then
            return(f);
         end_if;
      end_if:
      if type(-x/I) = "coth" then //-I*coth(y) = cot(-y*I)
         f:= handleCot(I*op(-x/I));
         if f <> FAIL then
            return(f);
         end_if;
      end_if:
      break;
    of "_power" do 
      if type(1/x) = "tan" then // 1/tan(y) = cot(y)
         f:= handleCot(op(1/x));
         if f <> FAIL then
            return(f);
         end_if;
      end_if:
      break;
    of "cot" do // the real part of arccot is in ]-PI/2,PI/2]
       f:= handleCot(op(x));
       if f <> FAIL then
         return(f);
       end_if:
       break;
    end_case;

    procname(x)
end_proc:

arccot := prog::remember(arccot,  
  () -> [property::depends(args()), DIGITS, slotAssignCounter("arccot")]):

arccot( infinity) := 0:
arccot(-infinity) := 0:

arccot(0) := PI/2:
arccot( (3^(1/2))/3 ):= PI/3:
arccot(-(3^(1/2))/3 ):=-PI/3:
arccot( 1):=  PI/4:
arccot(-1):= -PI/4:
arccot( 1/5*5^(1/2)*(5 + 2*5^(1/2))^(1/2) ):=   PI/5:
arccot( cos(PI/5)/sin( PI/5) )             :=   PI/5:
arccot(-cos(PI/5)/sin(-PI/5) )             :=   PI/5:
arccot( 1/tan( PI/5))                      :=   PI/5:
arccot(-1/5*5^(1/2)*(5 + 2*5^(1/2))^(1/2) ):= - PI/5:
arccot(-cos(PI/5)/sin(PI/5) )              := - PI/5:
arccot( cos(PI/5)/sin(-PI/5) )             := - PI/5:
arccot( cos(4*PI/5)/sin(4*PI/5) )          := - PI/5:
arccot(-1/tan( PI/5))                      := - PI/5:
arccot( 1/5*5^(1/2)*(5 - 2*5^(1/2))^(1/2) ):= 2*PI/5:
arccot( cos(2*PI/5)/sin( 2*PI/5) )         := 2*PI/5:
arccot(-cos(2*PI/5)/sin(-2*PI/5) )         := 2*PI/5:
arccot(-cos(3*PI/5)/sin( 3*PI/5) )         := 2*PI/5:
arccot( 1/tan(2*PI/5))                     := 2*PI/5:
arccot(-1/5*5^(1/2)*(5 - 2*5^(1/2))^(1/2) ):=-2*PI/5:
arccot(-cos(2*PI/5)/sin(2*PI/5) )          :=-2*PI/5:
arccot( cos(2*PI/5)/sin(-2*PI/5) )         :=-2*PI/5:
arccot( cos(3*PI/5)/sin( 3*PI/5) )         :=-2*PI/5:
arccot(-1/tan(2*PI/5))                     :=-2*PI/5:
arccot( (3^(1/2)) ):= PI/6:
arccot(-(3^(1/2)) ):=-PI/6:
arccot( 2^(1/2)+1 )              :=   PI/8:
arccot( cos(PI/8)/sin( PI/8))    :=   PI/8:
arccot( 1/tan( PI/8))            :=   PI/8:
arccot(-1/tan(-PI/8))            :=   PI/8:
arccot(-cos(PI/8)/sin(-PI/8))    :=   PI/8:
arccot(-2^(1/2)-1 )              := - PI/8:
arccot(-cos(PI/8)/sin( PI/8))    := - PI/8:
arccot( cos(PI/8)/sin(-PI/8))    := - PI/8:
arccot(-1/tan( PI/8))            := - PI/8:
arccot( 1/tan(-PI/8))            := - PI/8:
arccot( 2^(1/2)-1 )              := 3*PI/8:
arccot( cos(3*PI/8)/sin(3*PI/8)) := 3*PI/8:
arccot(-cos(3*PI/8)/sin(-3*PI/8)):= 3*PI/8:
arccot( 1/tan( 3*PI/8))          := 3*PI/8:
arccot(-1/tan(-3*PI/8))          := 3*PI/8:
arccot(-2^(1/2)+1 )              :=-3*PI/8:
arccot(-cos(3*PI/8)/sin(3*PI/8)) :=-3*PI/8:
arccot( cos(3*PI/8)/sin(-3*PI/8)):=-3*PI/8:
arccot(-1/tan( 3*PI/8))          :=-3*PI/8:
arccot( 1/tan(-3*PI/8))          :=-3*PI/8:
arccot( (5+2*5^(1/2))^(1/2) )      :=   PI/10:
arccot( cos(PI/10)/sin( PI/10) )   :=   PI/10:
arccot(-cos(PI/10)/sin(-PI/10) )   :=   PI/10:
arccot( 1/tan(   PI/10))           :=   PI/10:
arccot(-(5+2*5^(1/2))^(1/2) )      := - PI/10:
arccot(-cos(PI/10)/sin( PI/10) )   := - PI/10:
arccot( cos(PI/10)/sin(-PI/10) )   := - PI/10:
arccot(-1/tan(   PI/10))           := - PI/10:
arccot( (5-2*5^(1/2))^(1/2) )      := 3*PI/10:
arccot( cos(3*PI/10)/sin( 3*PI/10)):= 3*PI/10:
arccot(-cos(3*PI/10)/sin(-3*PI/10)):= 3*PI/10:
arccot( 1/tan( 3*PI/10))           := 3*PI/10:
arccot(-(5-2*5^(1/2))^(1/2) )      :=-3*PI/10:
arccot(-cos(3*PI/10)/sin( 3*PI/10)):=-3*PI/10:
arccot( cos(3*PI/10)/sin(-3*PI/10)):=-3*PI/10:
arccot(-1/tan( 3*PI/10))           :=-3*PI/10:
arccot( 2+3^(1/2) )                  :=   PI/12:
arccot( cos(PI/12)/sin( PI/12))      :=   PI/12:
arccot(-cos(PI/12)/sin(-PI/12))      :=   PI/12:
arccot(-cos(11*PI/12)/sin( 11*PI/12)):=   PI/12:
arccot( cos(11*PI/12)/sin(-11*PI/12)):=   PI/12:
arccot( 1/tan( PI/12))               :=   PI/12:
arccot(-1/tan(-PI/12))               :=   PI/12:
arccot(-2-3^(1/2) )                  := - PI/12:
arccot(-cos(PI/12)/sin(PI/12))       := - PI/12:
arccot( cos(PI/12)/sin(-PI/12))      := - PI/12:
arccot( cos(11*PI/12)/sin( 11*PI/12)):= - PI/12:
arccot(-cos(11*PI/12)/sin(-11*PI/12)):= - PI/12:
arccot(-1/tan( PI/12))               := - PI/12:
arccot( 1/tan(-PI/12))               := - PI/12:
arccot( 2-3^(1/2) )                  := 5*PI/12:
arccot( cos(5*PI/12)/sin( 5*PI/12))  := 5*PI/12:
arccot(-cos(5*PI/12)/sin(-5*PI/12))  := 5*PI/12:
arccot(-cos(7*PI/12)/sin( 7*PI/12))  := 5*PI/12:
arccot( cos(7*PI/12)/sin(-7*PI/12))  := 5*PI/12:
arccot( 1/tan( 5*PI/12))             := 5*PI/12:
arccot(-1/tan(-5*PI/12))             := 5*PI/12:
arccot(-2+3^(1/2) )                  :=-5*PI/12:
arccot(-cos(5*PI/12)/sin( 5*PI/12))  :=-5*PI/12:
arccot( cos(5*PI/12)/sin(-5*PI/12))  :=-5*PI/12:
arccot( cos(7*PI/12)/sin( 7*PI/12))  :=-5*PI/12:
arccot(-cos(7*PI/12)/sin(-7*PI/12))  :=-5*PI/12:
arccot(-1/tan( 5*PI/12))             :=-5*PI/12:
arccot( 1/tan(-5*PI/12))             :=-5*PI/12:

arccot:= funcenv(arccot):
arccot::type := "arccot":
arccot::print := "arccot":
arccot::info := "arccot -- the inverse cotangent":

arccot::inverse := "cot":


arccot::undefined := {}:
arccot::realDiscont := {0}:
arccot::complexDiscont :=
  loadproc(arccot::complexDiscont, pathname("STDLIB","DISCONT"), "arccot"):

arccot::expand := x -> arccot(expand(op(x))):

arccot::simplify:= 
  loadproc(arccot::simplify, pathname("STDLIB", "SIMPLIFY"), "arccot"):

arccot::diff :=
  proc()
  local op1;
  begin
    op1:= op(args(1),1);
    if iszero(1+op1^2) then
       0
    else
       -diff(op1, args(2..args(0))) / (1+op1^2)
    end_if;
  end_proc:

arccot::float := e ->
                 if iszero(e) then
                   float(PI/2)
                 elif indets(float(e)) <> {} then
                   // e is symbolic
                   hold(arccot)(float(e))
                 else
                   arctan::float(1/e)
                 end_if:

arccot ::hull := DOM_INTERVAL::arccot@hull:

arccot::rectform :=
    loadproc(arccot::rectform, pathname("STDLIB","RECTFORM"), "acot"):

arccot::series := loadproc(arccot::series, pathname("SERIES"), "arccot"):
arccot::getprop :=  proc(x:"arccot")
  local v;
  begin
    v := getprop(op(x));
    if type(v)=Dom::Interval then
      return(arccot(v));
    else
      return(C_);
    end_if;
  end_proc:

arccot::Content := stdlib::genOutFunc("Carccot", 1):

// end of file
