/*++
arccoth -- the inverse hyperbolic cotangens

arccoth(x)

x - expression
++*/

arccoth:=
proc(x)
  option noDebug;
  local f, handleCoth;
begin
  if args(0) < 1 then error("no arguments given")
  elif x::dom::arccoth <> FAIL then return(x::dom::arccoth(args()))
  elif args(0) <> 1 then error("wrong no of args")
  end_if;

  case type(x)
    of DOM_INT do
    of DOM_RAT do
       if x = 1 then
          error("singularity")
       elif x < 0 then
          return(-arccoth(-x))
       end_if;
       break;
    of DOM_FLOAT do
       return(arccoth::float(x));
    of DOM_COMPLEX do
       if domtype(op(x,1)) = DOM_FLOAT or
          domtype(op(x,2)) = DOM_FLOAT then
           return(arccoth::float(x))
       end_if;
       break;
    of DOM_SET do
    of "_union" do
       return(map(x, arccoth))
    of "_intersect" do
    of "_minus" do
       return(Dom::ImageSet(arccoth(`#z`), `#z`, x))
  end_case;

  if not testtype(x,Type::Arithmetical) then
     /* generic handling of sets */
     if testtype(x, Type::Set) then
       return(Dom::ImageSet(arccoth(#t), #t, x));
     end_if;
     error("argument must be of 'Type::Arithmetical'")
  end_if;

  //==============================================
  handleCoth:= proc(x) // compute arccoth(coth(x))
  local f, s;
  begin
    if testtype(x, Type::Real) or
       is(x in R_) = TRUE then
        return(x);
    end_if;
    s:= signIm(I*x);
    if domtype(s) = DOM_INT then
       f:= s*round(s*Im(x)/PI);
       if domtype(f) = DOM_INT then
          return(x - f*I*PI):
       end_if:
    end_if;
    return(FAIL);
  end_proc;
  //==============================================

  case type(x)
    of "_mult" do // normalize using arccoth(-x) = -arccoth(x).
                  // Note that this parity holds because in
                  // MuPAD: arccoth(x) = ((ln(1+1/x)-ln(1-1/x))/2.
       f:= op(x, nops(x));
       if (testtype(f, Type::Real) and f < 0) or 
          (testtype(f/I, Type::Real) and f/I < 0) then 
          return(-arccoth(-x)) 
       end_if;
  end_case;

  if type(x/I) = "cot" then // I*cot(y) = coth(-y*I)
     f:= handleCoth(-I*op(x/I));
     if f <> FAIL then
       return(f);
     end_if;
  end_if:
  if type(-x/I) = "cot" then // -I*cot(y) = coth(y*I)
     f:= handleCoth(I*op(-x/I));
     if f <> FAIL then
        return(f);
     end_if;
  end_if:
  if type(x) = "coth" then // coth(x) = coth(x)
       f:= handleCoth(op(x));
       if f <> FAIL then
          return(f);
       end_if;
  end_if:
  if type(-x) = "coth" then // -coth(x) = coth(-x)
       f:= handleCoth(-op(-x));
       if f <> FAIL then
          return(f);
       end_if;
  end_if:
  if type(-1/x) = "tanh" then // -1/tanh(x) = coth(-x)
     f:= handleCoth(-op(-1/x));
     if f <> FAIL then
        return(f);
     end_if;
  end_if:
  if type(1/x) = "tanh" then // 1/tanh(x) = coth(x)
     f:= handleCoth(op(1/x));
     if f <> FAIL then
        return(f);
     end_if;
  end_if;
  if type(I/x) = "tan" then // I/tan(x) = coth(-x*I)
     f:= handleCoth(-I*op(I/x));
     if f <> FAIL then
        return(f);
     end_if;
  end_if:
  if type(-I/x) = "tan" then // -I/tan(x) = coth(x*I)
     f:= handleCoth(I*op(I/x));
     if f <> FAIL then
        return(f);
     end_if;
  end_if:
  procname(x)
end_proc:

arccoth := prog::remember(arccoth,  
  () -> [property::depends(args()), DIGITS, slotAssignCounter("arccoth")]):

arccoth( infinity) :=  0:
arccoth(-infinity) :=  0:
arccoth( I*infinity) :=  0:
arccoth(-I*infinity) :=  0:
arccoth(0) := -I*PI/2:
arccoth( I/3^(1/2)):= -I*PI/3:
arccoth(-I/3^(1/2)):=  I*PI/3:
arccoth( I):= -I*PI/4:
arccoth(-I):=  I*PI/4:
arccoth( I*3^(1/2)):= -I*PI/6:
arccoth(-I*3^(1/2)):=  I*PI/6:

arccoth:= funcenv(arccoth):
arccoth::type := "arccoth":
arccoth::print := "arccoth":
arccoth::info := "arccoth -- the inverse hyperbolic cotangent":
arccoth::inverse := "coth":

arccoth::undefined := {-1, 1}:
arccoth::realDiscont := {-1, 1}:
arccoth::complexDiscont :=
    loadproc(arccoth::complexDiscont, pathname("STDLIB","DISCONT"), "arccoth"):

arccoth::diff :=
  proc(f)
    local op1;
  begin
    op1 := op(f, 1);
    if iszero(op1^2-1) then
       0
    else
       -diff(op1, args(2..args(0))) /(op1^2-1)
    end_if;
  end_proc:

arccoth::float := e -> // arccoth(0) = -PI/2*I
                       // arccoth(1) = singularity
                       // arccoth(e) = arctanh(1/e) , e <> 0
                  if e =  1 or e = float( 1) or
                     e = -1 or e = float(-1) then
                    error("singularity");
                  elif iszero(e)
                    then float(arccoth(0))
                  elif indets(float(e)) <> {} then
                    // e is symbolic
                    hold(arccoth)(float(e))
                  else arctanh::float(1/e)
                  end_if:

arccoth::hull := DOM_INTERVAL::arccoth@hull:

arccoth::expand := x -> arccoth(expand(op(x))):

arccoth::simplify:= loadproc(arccoth::simplify, pathname("STDLIB", "SIMPLIFY"),
                             "arccoth"):

arccoth::series := loadproc(arccoth::series, pathname("SERIES"), "arccoth"):

arccoth::rectform :=
    loadproc(arccoth::rectform, pathname("STDLIB","RECTFORM"), "arccoth"):

arccoth::Re:= x -> ln(abs((x+1)/(1-x)))/2:
arccoth::Im:= x -> (arg(1+1/x)-arg(1-1/x))/2:

arccoth::getprop :=  proc(x:"arccoth") local v; begin v := getprop(op(x)); if contains({Dom::Interval, solvelib::BasicSet}, type(v)) then return(arccoth(v)); else return(C_); end_if; end_proc:

arccoth::Content := stdlib::genOutFunc("Carccoth", 1):

// end of file
