/*++
arctanh -- the inverse hyperbolic tangens

arctanh(x)

x - expression
++*/

arctanh:=
proc(x)
  option noDebug;
  local f, handleTanh;
begin
  if args(0) = 0 then error("no arguments given")
  elif x::dom::arctanh <> FAIL then return(x::dom::arctanh(args()))
  elif args(0) <> 1 then error("wrong no of args")
  end_if;

  case type(x)
    of DOM_INT do
    of DOM_RAT do
       if x = 1 then
          error("singularity");
       elif x < 0 then return(-arctanh(-x)) end_if;
       break;
    of DOM_FLOAT do
       return(arctanh::float(x));
    of DOM_COMPLEX do
       if domtype(op(x,1)) = DOM_FLOAT or
          domtype(op(x,2)) = DOM_FLOAT then
           return(arctanh::float(x))
       end_if;
       break;
    of DOM_SET do
    of "_union" do
       return(map(x, arctanh))
    of "_intersect" do
    of "_minus" do
        return(Dom::ImageSet(arctanh(`#z`), `#z`, x))
  end_case;

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      return(Dom::ImageSet(arctanh(#t), #t, x));
    end_if;
    error("argument must be of 'Type::Arithmetical'")
  end_if;

  //==============================================
  handleTanh:= proc(x) // compute arctanh(tanh(x))
  local f, s;
  begin
    if testtype(x, Type::Real) or
       is(x in R_) = TRUE then
       return(x);
    end_if;
    s:= signIm(I*x);
    if domtype(s) = DOM_INT then
       f:= s*round(s*Im(x)/PI);  
       if domtype(f) = DOM_INT then
          return(x - f*I*PI) 
       end_if:
    end_if;
    return(FAIL);
  end_proc;
  //==============================================

  case type(x)
    of "_mult" do // normalize using arctanh(-x) = -arctanh(x)
       f:= op(x, nops(x));
       if (testtype(f, Type::Real) and f < 0) or
          (testtype(f/I, Type::Real) and f/I < 0) then 
         return(-arctanh(-x));
       end_if;
       if type(x/I) = "tan" then // I*tan(y) = tanh(y*I)
          f:= handleTanh(I*op(x/I));
          if f <> FAIL then
            return(f);
          end_if;
       end_if:
       if type(-x/I) = "tan" then // -I*tan(y) = tanh(-y*I)
          f:= handleTanh(-I*op(-x/I));
          if f <> FAIL then
            return(f);
          end_if;
       end_if:
       if type(-1/x) = "coth" then // -1/coth(x) = tanh(-x)
          f:= handleTanh(-op(-1/x));
          if f <> FAIL then
            return(f);
          end_if;
       end_if:
       if type(I/x) = "cot" then // I/cot(x) = tanh(x*I)
          f:= handleTanh(I*op(I/x));
          if f <> FAIL then
            return(f);
          end_if;
       end_if:
       if type(-I/x) = "cot" then // -I/cot(x) = tanh(-x*I)
          f:= handleTanh(-I*op(I/x));
          if f <> FAIL then
            return(f);
          end_if;
       end_if:
       break;
    of "_power" do 
       if type(1/x) = "coth" then // 1/coth(x) = tanh(x)
          f:= handleTanh(op(1/x));
          if f <> FAIL then
            return(f);
          end_if;
       end_if:
       break;
    of "tanh" do // arctanh(tanh(x))=x if -PI/2 < Im(x) < PI/2
       f:= handleTanh(op(x));
       if f <> FAIL then 
          return(f);
       end_if;
       break;
  end_case;

  procname(x)
end_proc:

arctanh := prog::remember(arctanh,  
  () -> [property::depends(args()), DIGITS, slotAssignCounter("arctanh")]):

arctanh(0):= 0:
arctanh( infinity):= -I*PI/2:
arctanh(-infinity):=  I*PI/2:
arctanh( I*infinity):=  I*PI/2:
arctanh(-I*infinity):= -I*PI/2:
arctanh( 3^(1/2)*I):=  I*PI/3:
arctanh(-3^(1/2)*I):= -I*PI/3:
arctanh( I):=  I*PI/4:
arctanh(-I):= -I*PI/4:
arctanh( I/3^(1/2)):=  I*PI/6:
arctanh(-I/3^(1/2)):= -I*PI/6:

arctanh:= funcenv(arctanh, op(specfunc::arctanh, 2)):
arctanh::info := "arctanh - the inverse hyperbolic tangens":
arctanh::type := "arctanh":
arctanh::print := "arctanh":
arctanh::float := specfunc::arctanh:
arctanh::hull := DOM_INTERVAL::arctanh@hull:
arctanh::inverse := "tanh":
arctanh::undefined := {-1, 1}:
arctanh::realDiscont := {-1, 1}:
arctanh::complexDiscont :=
    loadproc(arctanh::complexDiscont, pathname("STDLIB","DISCONT"), "arctanh"):

arctanh::diff :=
  proc(f)
    local op1;
  begin
    op1 := op(f, 1);
    if iszero(1-op1^2) then
       0
    else
       diff(op1, args(2..args(0))) / (1-op1^2)
    end_if;
  end_proc:

arctanh::expand := x -> arctanh(expand(op(x))):

arctanh::simplify:= loadproc(arctanh::simplify, pathname("STDLIB", "SIMPLIFY"),
                             "arctanh"):
arctanh::rectform:= loadproc(arctanh::rectform, pathname("STDLIB", "RECTFORM"),
                             "arctanh"):
// see the comments in the file above for explanation
arctanh::Re:= x -> ln(abs((x+1)/(x-1)))/2:
arctanh::Im:= x -> (arg(1+x)-arg(1-x))/2:

arctanh::series := loadproc(arctanh::series, pathname("SERIES"), "arctanh"):
arctanh::getprop :=  proc(x:"arctanh") local v; begin v := getprop(op(x)); if contains({Dom::Interval, solvelib::BasicSet}, type(v)) then return(arctanh(v)); else return(C_); end_if; end_proc:

arctanh::Content := stdlib::genOutFunc("Carctanh", 1):

// end of file
