// (Complex) Floating-point evaluation of the Bessel and modified Bessel 
// functions of the 1st and 2nd kinds                                                   

specfunc::Bessel:=newDomain("specfunc::Bessel"):
specfunc::Bessel::create_dom:=hold(specfunc::Bessel):

alias(BesselJ=besselJ::float):
alias(BesselY=besselY::float):
alias(BesselI=besselI::float):
alias(BesselK=besselK::float):
//
alias(realJI=specfunc::Bessel::realJI):
alias(realYK=specfunc::Bessel::realYK):
alias(taylorJI=specfunc::Bessel::taylorJI):
alias(complexJI=specfunc::Bessel::complexJI):
alias(complexYK=specfunc::Bessel::complexYK):
alias(asymptI=specfunc::Bessel::asymptI):
alias(asymptK=specfunc::Bessel::asymptK):
alias(asymptJY=specfunc::Bessel::asymptJY):
alias(intorderYK=specfunc::Bessel::intorderYK):
alias(cmagdiff=specfunc::Bessel::cmagdiff):
alias(rawdigits=specfunc::Bessel::rawdigits):
alias(signc=specfunc::Bessel::signc):
alias(estimateJI=specfunc::Bessel::estimateJI):
alias(findtmaxJI=specfunc::Bessel::findtmaxJI):
//
alias(sabs=specfunc::abs):
alias(ssqrt=specfunc::sqrt):
alias(ssin=specfunc::sin):
alias(ssinh=specfunc::sinh):
alias(scos=specfunc::cos):
alias(scosh=specfunc::cosh):
alias(sexp=specfunc::exp):
alias(sgamma=specfunc::gamma):
alias(strunc=specfunc::trunc):
alias(sround=specfunc::round):
alias(sfrac=specfunc::frac):
alias(fln=specfunc::Bessel::fln):
//
alias(ln10=0.4342944819): // = 1/ln(10)
fln:= ln::float:

// =================================================
// estimateJI returns an approximation of 
// ln(|besselJ(v, zin)/(z/2)^v| for fn = -1 and of
// ln(|besselI(v, zin)/(z/2)^v| for fn =  1.
// Note: 
//   besselI(v, z) = z^v/(I*z)^v * besselJ(v,I*z)
// =================================================
estimateJI:=proc(v,zin,vs,zs,fn)
local ang,r,r2,z,res,sv,sz,s;
save DIGITS;
begin
  // The approximations are taken from lead terms of asymptotic expansions
  // of besselJ taken from Abramowitz and Stegun in various regimes and 
  // "liberally" applied to the complex plane. 
  // This routine does NOT work for indices v that are negative integers
  // (use reflection formula in this case).
  DIGITS:=10:
  // s is the scaling factor of the result caused by the
  // transformations v -> -v, z -> -z of the parameters
  if fn = 1 then
    s:= sabs(float(zin^v/(I*zin)^v)):
    sz:= -I:  // restore the original z by zin:= sz*z
    z:= I*zin:  
  else
    s:= 1: 
    sz:= 1;
    z:=zin:
  end_if:
  // note besselJ(-v, z) = (-1)^v*besselJ(v, z) if v in Z_
  sv:= 1;
  if domtype(float(v)) = DOM_FLOAT and 
     float(v) < 0 and 
     iszero(frac(v)) then
     v:= -v; 
     sv:= -1; // sv is needed to restore the original v
  end_if:
  // Note besselJ(v, -z) = (-z)^v/z^v * besselJ(v, z)
  // Use this flip to avoid Re(z) < 0
  if Re(float(z)) < 0 then
     s:= s * sabs((float(z))^v/float(-z)^v);
     z:= -z; 
     sz:= -sz; // restore the original z by z:= sz*z;
  end_if:
  //=====================================================
  // The following estimates are available:
  // 1) special case: v = z, |v| >> 1     (9.3.5  of I&S)
  // 2) |z|^2 < 4*vs and Re(v)>= -0.5     (Taylor)
  // 3) |z - v| < 0.1*|v|                 (9.3.23 of I&S)
  // 4) |z| < |v|/2                       (9.3.1  of I&S)
  // 5) |z| > |v|^2/2                     (9.2.5  of I&S)
  // 6) else generic upper estimate       (9.1.63 of I&S)
  //=====================================================

  //if not iszero(vs) and iszero(z-v) then
  if iszero(z-v) then
   // 9.3.5. approximated to 10 digits
   res:=0.4473073186/v^(1/3):
  elif zs^2 < 2*vs and Re(float(v)) >= -1/2 then
    // beginning of the Taylor series: 
    // besselJ(v,z) = (z/2)^v/gamma(v+1)* (1 - z^2/4/(v+1) + ...
    res:= (z/2)^v/sgamma(v+1)
  elif not iszero(vs) and sabs(float((z-v)/vs)) < float(1/10) then
    // Constructed from 9.3.23 with the Airy function approximated
    // for small arguments - transition region.
    res:=-0.8946146374e-1*float((1/v)^(1/3)*(z-6*v)/v):
  elif vs > 2*zs and (domtype(float(v)) = DOM_FLOAT and float(v) > 0) then
    // 9.3.1 i.e. approximation valid for large index v and
    // fixed arguments.
    res:=(((sexp(float(1))*z)/(2*v))^v)/ssqrt(float(2*PI*v)): 
  elif zs > vs^2/2 and not (domtype(float(z)) = DOM_FLOAT and float(z)<0)  then
    // lead term of 9.2.5 i.e. asymptotic expansion for large
    // arguments and fixed indices.
    ang:=float(z-float((PI/2)*(v+(1/2)))):
    res:=ssqrt(float(2/(PI*z)))*(scos(ang)-ssin(ang)*((4*v^2-1)/(8*z))):
  else // When all else fails use upper bound formula of 9.1.63.
    res:= RD_INF:
    if domtype(float(v)) = DOM_FLOAT and iszero(frac(v)) then 
      // v is an integer.
      // Note that besselJ(-v, z) = (-1)^v*besselJ(v, z),
      // so |besselJ(-v,z)| = |besselJ(v,z)| for integer v.
      // Use vs = |v| instead of v:
      r:=float(z/vs): 
      r2:=ssqrt(float(1-r^2)):
      res:=min(res, sabs((r^vs)*sexp(vs*r2)/((1+r2)^vs))):
    end_if:
    if Re(v) >= -1/2 then
       res:= min(res, sabs((z/2)^v*exp(sabs(Im(z)))/gamma(v+1)));  // (9.1.62 of I & S)
    end_if;
    // The following upper bounds are only valid along the real line
    // for positive v:
    if domtype(float(z)) = DOM_FLOAT and
       domtype(float(v)) = DOM_FLOAT then
       if float(v) >= 0 then
         res:= min(res, 1);           // (9.1.60 of I & S)
       end_if;
       if float(v) >= 1 then
         res:= min(res, 1/sqrt(2.0)); // (9.1.60 of I & S)
       end_if;
    end_if;
    if res = RD_INF then 
       return(FAIL);  // no estimate evailable.
    end_if;
  end_if:
  res:= s * res; // correct the result
  v:= sv*v;      // restore the original v
  z:= sz*z;      // restore the original z
  fln(sabs(float(res/((z/2)^v)))):
end_proc:

//============================================================
// Note that besselJ(v, x) 
//  = (z/2)^v * sum((-z^2/4)^k/k!/gamma(v+k+1), k=0..infinity)
// Find and return the maximal term of the sum (ignoring the
// factor (z/2)^v).
// 
// The summation indices k such with |t[k+1]/t[k]| = 1 are solutions
// of the following 4th order equation (with y = k + 1)
// k:= numeric::solve(y^2*(abs(v)^2+2*Re(v)*y+y^2)-(abs(z)/2)^4,y);
// k:= map(k, round@_subtract, 1):
// k:= select(k, iszero@Im):
// k:= {0} union select(k, x -> bool(x >= 0)):
// estimate =  max(map(k, f -> abs((-z^2/4)^k/k!/gamma(v + k + 1))));
//
// Below, you find Voldemort's original code:
//============================================================
findtmaxJI:=proc(v,z,z2_4,vs,zs)
  local half,i,k,ds,ks,t,t2,t3,term,term0,PIf;
  save DIGITS;
begin
  DIGITS:=10:
  if iszero(v+1) or 
     (type(float(v))=DOM_FLOAT and 
      iszero(sfrac(v+1)) and 
      float(v) < float(0)) then
    term0:=float(0):
  else
    term0:=float(1/sgamma(float(v+1))):
  end_if:
  term:=float(ssqrt(vs^2+sabs(4*z2_4))/2):
  k:={-(vs/2)+term,term,(vs/2)+term}:
  k:=map(k,sround):
  ds:=sabs(term0):
  PIf:=float(PI):
  half:=float(1/2);
  for i from 1 to nops(k) do
        ks:=op(k,i):
        t:=float(ks+1): t2:=float(t+v): t3:=t*t2:
        if ks = 0 then
          ds:=max(ds,sabs(term0*z2_4/float(v+1))):
        elif ks > 0 then
    //    term:=(z2_4^ks)/(sgamma(t)*sgamma(t2)):
          term:=float((z2_4^ks/2)*sexp(t2+t)/(PIf*ssqrt(t)*(t2^float(v+half))*(t3^ks))):
          ds:=max(ds,sabs(term),sabs(term*z2_4/t3),sabs(float(ks*(ks+v))*term/z2_4)):
        end_if:
   end_for:
   ds;
end_proc:

signc:=proc(x)
local r;
begin
    if type(x)=DOM_FLOAT then
     return(specfunc::sign(x)):
    else
     r:=op(x,1):
     if iszero(r) then
       specfunc::sign(op(x,2))
     else
       specfunc::sign(r);
     end_if:
    end_if:
end_proc:

fln:= ln::float:

//--------------------------------------------------
// This routine seems to count the number of leading
// digits that do not vanish, e.g.:
//  rawdigits(0.01234) --> 4
//  rawdigits(0.00001234567) --> 7
//  rawdigits(1230.06780000) --> 8
// What the hell can this be used for ????
//--------------------------------------------------
/*
rawdigits:= proc(f:DOM_FLOAT)
local s,i,j,t;
begin
   s:= expr2text( f ):
   i:= length(s)-1:
   j:=0:
   while i > 0 and s[i] = "0" and s[i-1] <> "." do i:= i-1 end_while:
   while j < i and s[j] = "0" or s[j] ="." do j:= j+1 end_while:
   t:=substring( s,j..i ):
   i:= length(t):
   if type(text2expr(t))=DOM_FLOAT then i:=i-1: end_if:
   i:
end_proc:
*/

//------------
// Quick hack:
//------------

rawdigits:= f -> DIGITS:

//-----------------------------------------------------

cmagdiff:=proc(z)
begin
    if iszero(z) then return(10*DIGITS) end_if;
    if type(float(z))=DOM_FLOAT then
         sround(ln10*fln(sabs(z)))
    elif iszero(Re(z)) then
         -sround(ln10*fln(sabs(Im(z))))
    else sround(ln10*fln(sabs(Re(z)/Im(z)))) 
    end_if:
end_proc:

// realJI                                                                      
// Controller for real Float evaluation of Bessel J & I                        
//                                                                             
// fn - function index: -1=J, 1=I                                              
// reflect - boolean flag to indicate whether or not to use reflection formula 
//                                                                             

realJI := proc(v,x,fn,reflect)
    local oldDIGITS,coff,try_asympt, s, abs_x, k_min,v2,RHS,LHS;
    save DIGITS;
    begin
    if reflect then
        // Note: signc(I*x) = -1, since x is real 
        if iszero(sfrac(v)) then
            (-1)^v * realJI(v,-x,fn,FALSE):
          elif iszero(sfrac(v-1/2)) then
            float(I*(-1)^(strunc(v-1/2))*realJI(v,-x,fn,FALSE)):
          else
            s:=realJI(v,-x,fn,FALSE):
            DIGITS:=DIGITS+1+max(0,sround(ln10*fln(sabs(v)))):
            coff:=sexp(float(I*PI*v)):
            coff*s:
        end_if:

      else
         oldDIGITS:=DIGITS:

/* // Walter, 21.4.09: 
   //   Do not use the special case for half integer indices, 
   //   since it is numerically unstable! E.g., this code
   //   produces
   //   besselJ(1000.5, 100.0) =  1.940570105e889 (wrong!)
   //   The correct value is:
   //   besselJ(1000.5, 100.0) =  4.258121398e-871

        // test for half-integer indices:
        //if domtype(v) <> DOM_INT and iszero(sfrac(2*v)) then
        if domtype(v) <> DOM_INT and 
             float(sfrac(2*v)) < float(10^(-DIGITS)) then
             v:=sround(2*v)/2:
             DIGITS:=10;
             abs_x := sabs(x):
             k_min:= v-(1/2):
             //-------------------------------------------
             // Bugfix (Walter): added the following line:
             // For halfintegers, there is an *exact* asymptotic
             // representation cos(x)*poly(1/x) + sin(x)*poly(1/x)
             // with the polys having roughly |ceil(v/2)| terms.
             // Make sure, that k_min is well above these number
             // of terms, because k_min is the stopping criterion
             // in asymptJY

             k_min:= sabs(k_min) + 10;

             //-------------------------------------------
             crit:=float(abs_x -(k_min-10));
             try_asympt:=not(bool( (abs_x<10) and (v>10) and (crit <= 0)));
             DIGITS:=oldDIGITS;
        else
*/
             DIGITS:=10;
             abs_x := sabs(x):
             k_min := strunc(ssqrt(float(abs_x^2+v^2))+abs_x) + 1:
             //-------------------------------------------
             // Bugfix (Walter): added the following line:

             k_min:= sabs(k_min);

             //-------------------------------------------
             v2:=float(v^2);
             if has(v2,I) then v2:=op(v2,1): end_if:
             // .18394 ~ 0.5*exp(-1.0) 
             coff:=float((ln(sabs(scos(PI*v)))*k_min+v2-1.693145659*k_min^2+
             k_min^2*ln(k_min)-k_min^2*ln(abs_x))/k_min);
             // coff:=float(scos(float(PI*v))*sexp(v^2/k_min)*
             //            (0.18394*k_min/abs_x)^k_min): 
             DIGITS:=oldDIGITS;
             // Change in digit setting 
             //RHS:=DIGITS + length(DIGITS+9)+3:
               RHS:= DIGITS:
             if coff < -1000.0 then
             //if iszero(coff) then
                LHS := infinity:
             else
                //LHS:=-sround(ln10*fln(sabs(coff))):
                LHS:=-sround(ln10*coff):
             end_if:
             try_asympt := bool(LHS > RHS):
//      end_if:
        s:=FAIL:
        if try_asympt then
            if x < 0 then
                return(realJI(v,x,fn,TRUE)):
            end_if:

            if fn = -1 then 
                s := float(asymptJY(v,x,-1,k_min)):
              else
                s := float(asymptI(v,x,k_min)):
            end_if:
        end_if:

        if not try_asympt or s = FAIL then
            s := float(taylorJI(v,x,fn)):
        end_if:

        if s = FAIL then
           warning("could not compute a float approximation");
           return(FAIL);
        end_if:

        //if x > 0 or iszero(sfrac(v)) then
        if float(x) > 0 or iszero(sfrac(v)) then
           Re(s):
        elif float(x) < 0 and 
            (not iszero(sfrac(v))) and iszero(sfrac(2*v)) then
            I*Im(s):
        else
            s:
        end_if:
    end_if:
end_proc:


// realYK                                                                
//                                                                       
// Controller for real Float evaluation of Bessel Y(v,x) & K(v,x), x>0   
//                                                                       
//   v - real                                                            
//   x - real, > 0                                                       
//   fn - function index: -1=K, 1=Y                                      

realYK := proc(v,x,fn)
    local oldDIGITS, coff,crit,try_asympt, s, abs_x, k_min, v2, predsize, d1, 
    t1, t2, try, lost;
    save DIGITS;
    begin

    oldDIGITS:=DIGITS:
    // test for half-integer indices:
    if type(float(v))=DOM_FLOAT and domtype(v) <> DOM_INT 
               and float(sfrac(2*v)) < float(10^(-DIGITS)) then
         v:=sround(2*v)/2:
         DIGITS:=10;
         abs_x := sabs(x):
         k_min:= v-(1/2):
         // -------------------------------------------
         // Bugfix (Walter): added the following line:
         // For halfintegers, there is an *exact* asymptotic
         // representation cos(x)*poly(1/x) + sin(x)*poly(1/x)
         // with the polys having roughly |ceil(v/2)| terms.
         // Make sure, that k_min is well above these number
         // of terms, because k_min is the stopping criterion
         // in asymptK


         k_min:= abs(k_min) + 10:
         // -------------------------------------------
         crit:=float(abs_x -(k_min-10));
         try_asympt:=not(bool( (abs_x<10) and (v>10) and (crit <= 0)));
         DIGITS:=oldDIGITS;
    else
         crit:=-1:
         DIGITS := 10:
         abs_x     := sabs(x):
         k_min     := strunc(ssqrt(float(abs_x^2+v^2))+abs_x) + 1:

         // .18394 ~ 0.5*exp(-1.0)  
         v2:=float(v^2);
         coff:=float((ln(sabs(scos(PI*v)))*k_min+v2-1.693145659*k_min^2+
         k_min^2*ln(k_min)-k_min^2*ln(abs_x))/k_min);
 
         //coff:=scos(float(PI*v))*sexp(v^2/k_min)* 
         //           (0.18394*k_min/abs_x)^k_min:
         DIGITS:=oldDIGITS:
         // Change in digit setting 
         //if iszero(coff) then
         if coff < -1000.0  then
           try_asympt := TRUE:
         else
          try_asympt := bool(-sround(ln10*coff) > DIGITS):
          //try_asympt := bool(-sround(ln10*fln(sabs(coff))) > DIGITS):
         end_if:
    end_if:
    s:=FAIL:
    if try_asympt then
        if x < 0 and crit < 0  then
            // Reflection formula must be handled by complex controller 
            return(float(complexYK(v,x,fn,TRUE))):
        end_if:

        if fn = -1 then 
            s := float(asymptK(v,x,k_min)):
          else
            s := float(asymptJY(v,x,1,k_min)):
        end_if:
    end_if:

    if not try_asympt or s = FAIL then
        if type(v)=DOM_INT then
            if fn = 1 then
                // Bessel Y:  0.79785 ~ sqrt(2/PI) 
                if abs_x < sabs(float(v)) then
                 coff:=sgamma(float(v+1)):
                 DIGITS:=10:
                 t1 := float((x/2)^v):
                 predsize:=(t1*scos(float(v*PI))-1/t1)/coff:
                    if iszero(predsize) then
                        predsize := 1/coff:
                    end_if:
                  else
                    predsize := float(0.79785/ssqrt(x)*(ssin(
                    x-(float(PI/2))*(v+1/2))+scos(x-float(PI/2)*(v+1/2))
                    *(4*v^2-1)*float(1/8)/x)):
                end_if:
              else
                // Bessel K: 0.39893 ~ 1/sqrt(2*PI) 
                if abs_x < sabs(float(v)) then
                    coff:=sgamma(float(v+1)):
                    DIGITS:=10:
                    t1 := float((x/2)^v):
                    predsize := float((t1-1/t1)/coff):
                    if iszero(predsize) then
                        predsize := coff:
                    end_if:
                  elif abs_x < 1 then
                    DIGITS:=10:
                    if iszero(v) then
                        predsize := fln(float(abs_x/2)):
                      else
                        predsize := float((1/2)/(abs_x/2)^v):
                    end_if:
                  else
                    DIGITS:=10:
                    predsize:=float(0.39893*sexp(-x)/ 
                                 ssqrt(x)*(1+(4*v^2-1)*float(1/8)/x)):
                end_if:
            end_if:
            DIGITS:=oldDIGITS:
            s := float(intorderYK(v,x,fn,sround(ln10*fln(sabs(predsize))))):
          else
            lost := 3:
            d1   := max(0, sround(ln10*fln(sabs(v)))):
            for try from 1 to 5 do
                DIGITS := DIGITS + lost:
                DIGITS := DIGITS + d1:

                if fn = 1 then
                    t1 := 1/ssin(float(v*PI)):
                    // Refinement (Walter, 12.3.09)
                    if testtype(v, Type::Real) and iszero(frac(2*v)) then
                       if round(2*v) mod 2 = 0 then // v is an integer
                          t2:= float((-1)^round(2*v));
                       else // v is a half integer such as 1/2, 3/2, ...
                          t2:= float(0);
                       end_if;
                    else
                       t2 := scos(float(v*PI)):
                    end_if:
                    // end of Refinement
                    DIGITS:=DIGITS - d1+ max(0,-sround(ln10*fln(sabs(x)))):
                    s  := float(t2*BesselJ(v,x) - BesselJ(-v,x)):
                  else
                    t1 := float((PI/2)/ssin(float(v*PI))):
                    DIGITS:=DIGITS-d1+max(0,-sround(ln10*fln(sabs(v))),
                                     strunc(0.87*sabs(x))):
                    s  := float(BesselI(-v,x)-BesselI(v,x)):
                end_if:

                if iszero(s) then
                    // The 0's of Y and K are irrational, so this should never 
                    // happen unless complete cancellation has occurred. 
                    lost := 20:
                  else
                    lost := oldDIGITS - rawdigits(Re(s)):
                end_if:

                if lost <= 0 then
                    break:
                  else
                    lost := lost + 2:
                end_if:
            end_for:

            s := s * t1:

        end_if:
    end_if:
    if float(x) < 0 and
      (not iszero(sfrac(v))) and iszero(sfrac(2*v)) then
            I*Im(s):
    else 
        s:
    end_if:
end_proc:

// taylorJI
//
// (Complex) Float evaluation of Bessel J and I functions via Taylor series
//   (A and S 9.1.10 and 9.6.10)
//
// Assumes the order (v) is not a negative integer.
//
// fn - function index: -1=J, 1=I
//

taylorJI := proc(v,z,fn)
local k,tol,try,fv,fz,t,tmax,z2_4,s,abs_s,abs_t,vs,zs,lostdigits, oldDIGITS;
save DIGITS;
begin
    tol:= float(10^(-DIGITS)): // DIGITS set outside Bessel
    oldDIGITS := DIGITS:
    DIGITS:=10:
    vs:=sabs(float(v)):
    zs:=sabs(float(z)):
    z2_4:=float(z^2/4):
    if float(zs^2/4) > vs then
       // estimate lost digits
       if type(z)=DOM_FLOAT and zs <= 2000 
         and type(float(v))=DOM_FLOAT and vs <= 2000 then
         lostdigits:=max(0,sround(1.05*(-0.16*vs+0.42*zs)));  
       else 
         tmax:=findtmaxJI(v,z,z2_4,vs,zs):
         abs_s:=estimateJI(v,z,vs,zs,fn):
         if abs_s = FAIL then
            if iszero(tmax) then 
               abs_s:= 1;
            else
               abs_s:= fln(tmax):
            end_if:
         end_if:
         lostdigits := sround(sabs(ln10*(abs_s-fln(tmax)))): 
         //account for lost digits and add some guard digits
       end_if:
       DIGITS := oldDIGITS + lostdigits + 20:
    else
       DIGITS := oldDIGITS:
       lostdigits := 0;
    end_if:

    for try from 1 to 10 do  // try with more and more DIGITS
        fz    := float(z):  // assuming z is exact on input
        fv    := float(v):  // assuming v is an integer
        t     := float(1)/sgamma(float(v+1)):
        s     := t:
        tmax  := sabs(t):
        abs_s := tmax:
        z2_4  := fn*fz^2/4:  
        for k from 1 to RD_INF do
            t     := t*z2_4/(k*(fv+k)):
            abs_t := sabs(t):
            tmax  := max(abs_t,tmax):
            s     := s + t:
            abs_s := sabs(s):
            if abs_t <= tol*abs_s then 
               break
            end_if
        end_for:
        // assuming abs_s <> 0 :
        abs_s:=fln(abs_s):
        lostdigits := sround(sabs(ln10*(abs_s-fln(tmax)))): 
        if oldDIGITS > DIGITS - lostdigits then 
           // do it again with more DIGITS
           DIGITS:= oldDIGITS + lostdigits + 30:
         else // we are done
           break:
        end_if:
    end_for:
    s := s * (fz/2)^v:
end_proc:

// complexJI                                                                  
//                                                                            
// Controller for complex Float evaluation of the Bessel J & I functions    
//                                                                            
// fn - function index: -1=J, 1=I                                             
// reflect - boolean flag to indicate whether or not to use reflection formla 

complexJI := proc(v,z,fn,reflect)
    local coff,crit,d,s,oldDIGITS, predsize, predDIGITS, resDIGITS, lost, 
      k_min, asy_bdry, try_asympt, abs_z, abs_v, z_mags, v_r, v_i, t, u, try;
    save DIGITS;
    begin
    oldDIGITS := DIGITS:
    if reflect then
        if type(v)=DOM_INT then
            (-1)^v * complexJI(v,-z,fn,FALSE):
          elif iszero(sfrac(Re(v))) then
                DIGITS:=DIGITS+1+max(0,sround(ln10*fln(sabs(Im(v))))):
                coff:=sexp(float(Im(v)*PI*signc(I*z))):
                DIGITS:=oldDIGITS:
                coff*(-1)^strunc(Re(v))*complexJI(v,-z,fn,FALSE):
          elif iszero(sfrac(Re(v)-1/2)) then
                DIGITS:=DIGITS+1+max(0,sround(ln10*fln(sabs(Im(v))))):
                coff:=sexp(float(Im(v)*PI*signc(I*z))):
                DIGITS:=oldDIGITS:
                -coff*signc(I*z)*I*(-1)^(strunc(Re(v)-1/2))*complexJI(v,-z,fn,FALSE):
          else
                DIGITS:=DIGITS+1+max(0,sround(ln10*fln(sabs(v)))):
                coff:=sexp(float(-I*PI*v*signc(I*z))):
                DIGITS:=oldDIGITS:
                coff*complexJI(v,-z,fn,FALSE):
        end_if:
    else
        coff      := sgamma(float(v+1)):
        DIGITS    := 10:
        abs_z     := sabs(z):
        abs_v     := sabs(float(v)):
        v_r       := float(Re(v)):
        v_i       := sabs(float(Im(v))):
        z_mags    := cmagdiff(z):
        // This boundary needs work  
        if abs_v > 1 and abs_z < abs_v then 
            predsize := (z/2)^v*(1+float(1/4)*fn*z^2/float(v+1)) /coff:
        elif fn = 1 then
            // Bessel I 
            t := (4*v^2-1)*float(1/8)/z:
            u := ssqrt(float(2*PI*z)):
            predsize := sexp(z)/u*(1-t):
            if iszero(v_r) or (sabs(Im(z)) > 6*Re(z)) then
                // exponentially subdominant term signiend_ifcant here 
                predsize := predsize + sexp(-PI*(v+1/2)*I-z)/u*(1+t):
            end_if:
        else
            // Bessel J:  0.63662 ~ 2/PI  
            predsize := ssqrt(0.63662/z)*(scos(z-(float(PI/2))*(v+1/2))
                               -ssin(z-(float(PI/2))*(v+1/2))*(4*v^2-1)*float(1/8)/z):
        end_if:
        if iszero(v_i) then
            predDIGITS := max(cmagdiff(predsize) - z_mags, 0) + 2:
          elif iszero(v_r) and fn = 1 then
            //error("no formula yet"): 
            predDIGITS := 2:
          else
            predDIGITS := max(cmagdiff(predsize),0) + 2:
        end_if:
        if abs_z > v_i and
           not(type(float(v))=DOM_FLOAT and domtype(v) <> DOM_INT
           and iszero(sfrac(2*v))) then
          // .18394 ~ 0.5*exp(-1.0) 
          k_min:=strunc(ssqrt(abs_z^2+v_r^2)+ssqrt(abs_z^2-v_i^2)) + 1:
          try:= sabs(scos(float(PI*v))* sexp(v_r^2/k_min)):
          if iszero(try) then
            asy_bdry:= -10^5*sabs(k_min*fln(0.18394*k_min/abs_z)):
          else
            asy_bdry:=-sround(ln10*(fln(try) + k_min*fln(0.18394*k_min/abs_z))):
          end_if:
        else
          asy_bdry :=float(0):
        end_if:
        for try from 1 to 5 do
           DIGITS := oldDIGITS + predDIGITS: 
         
          // test for half-integer indices:
          if type(float(v))=DOM_FLOAT and domtype(v) <> DOM_INT 
             and float(sfrac(2*v)) < float(10^(-DIGITS)) then
              v:=sround(2*v)/2:
              d:=DIGITS:
              DIGITS:=10;
              abs_z := sabs(z):
              k_min:= abs(v)-(1/2):
              crit:=float(abs_z -(k_min-10));
              try_asympt:=not(bool( (abs_z<10) and (abs(v)>10) and (crit <= 0)));
              DIGITS:=d;
          else
              crit:=-1;
              // Change in digit setting 
              try_asympt := bool(asy_bdry > DIGITS):
          end_if:
            s := FAIL:
            if try_asympt then
                if signc(z) = -1 and crit < 0 then
                    DIGITS := oldDIGITS:
                    return(float(complexJI(v,z,fn,TRUE))):
                end_if:
                if fn = -1 then
                    s := float(asymptJY(v,z,-1,k_min)):
                  else
                    s := float(asymptI(v,z,k_min)):
                end_if:
            end_if:
            if not try_asympt or s = FAIL then
                s := float(taylorJI(v,z,fn)):
            end_if:

            if iszero(v_i) then
                resDIGITS := cmagdiff(s) - z_mags:
            elif iszero(v_r) and fn = 1 then
                resDIGITS:=0:
              else
                resDIGITS := cmagdiff(s):
            end_if:

            if resDIGITS <= predDIGITS then
                lost := 0:

                if not iszero(Re(s)) then
                    lost := oldDIGITS - rawdigits(Re(s)):
                  else
                    lost := 0:
                end_if:

                if not(iszero(Im(s))) then
            //lost := max(lost, oldDIGITS + 2 - length(op(Im(s),1))):
                    lost := max(lost, oldDIGITS -rawdigits(Im(s))):
                end_if:

                if lost <= 0 then
                    break:
                  else
                    predDIGITS := predDIGITS + lost + 2:
                end_if:

              else
                predDIGITS := resDIGITS + 2:
            end_if:
        end_for:
        s;
    end_if:
end_proc:

// complexYK                                                                  
//                                                                            
// Controller for complex Float evaluation of the Bessel Y & K functions      
//                                                                            
// fn - function index: -1=K, 1=Y                                             
// reflect - boolean flag to indicate whether or not to use reflection formla 

complexYK := proc(v,z,fn,reflect)
    local oldDIGITS, crit,coff, d,lost, s, abs_z, abs_v, v_r, v_i, v_mags, z_mags, 
        predsize, predDIGITS, k_min, asy_bdry, try_asympt, d1, t1, t2, 
        resDIGITS, z1, try;
    save DIGITS;
    begin
    oldDIGITS := DIGITS:
    // Bugfix (Walter): use the reflection formula only for fn = 1
    // (i.e., for besselY). I replace
    // if  reflect then
    // by
    // if fn = 1 and reflect then
    if fn = 1 and reflect then
        lost   := 2:
        DIGITS := DIGITS + 2:

        for try from  1 to 5 do
            DIGITS := DIGITS + lost:

            d := DIGITS:
            if type(v)=DOM_INT then
                s := (-1)^v * BesselY(v,-z):

              elif iszero(sfrac(Re(v))) then
                DIGITS:=DIGITS+1+max(0,sround(ln10*fln(sabs(Im(v))))):
                coff:=sexp(float(Im(v)*PI*signc(I*z))):
                DIGITS:=d:
                s := coff*(-1)^strunc(Re(v))*BesselY(v,-z):

              elif iszero(sfrac(Re(v)-1/2)) then
                if iszero(Im(v)) then
                   DIGITS:= DIGITS + 1;
                else
                   DIGITS:=DIGITS+1+max(0,sround(ln10*fln(sabs(Im(v))))):
                end_if;
                coff:=sexp(float(Im(v)*PI*signc(I*z))):
                DIGITS:=d:
                s := -coff*signc(I*z)*I*(-1)^(strunc(Re(v)-1/2))*BesselY(v,-z):
              else
                DIGITS:=DIGITS+1+max(0,sround(ln10*fln(sabs(v)))):
                coff:=sexp(float(-fn*I*PI*v*signc(I*z))):
                DIGITS:=d:
                s := coff*BesselY(v,-z):
            end_if:

            s := s - float(2*I*signc(I*z)*BesselJ(-fn*v,-z)):

            if not iszero(Re(s)) then
                lost := oldDIGITS - rawdigits(Re(s)):
              else
                lost := 0:
            end_if:

            if not(iszero(Im(s))) then
                lost := max(lost, oldDIGITS - rawdigits(Im(s))):
            end_if:

            if lost <= 0 then
                break:
            end_if:
        end_for:

      else
        DIGITS := 10:
        abs_z  := sabs(z):
        abs_v  := sabs(float(v)):
        v_r    := float(Re(v)):
        v_i    := Im(abs_v):
        if type(float(v))=DOM_FLOAT then
            v_mags := 0:
          else
            v_mags := max(0,cmagdiff(v)):
        end_if:

        if type(float(z))=DOM_FLOAT then
            z_mags := 0:
          else
            z_mags := cmagdiff(z):
        end_if:

        // This boundary needs work 
        if fn = 1 then
            // Bessel Y:  .79785 ~ sqrt(2/PI) 
            if abs_v > 1 and abs_z < abs_v then 
                t1 := (z/2)^v*(1-(z^2/4)/(v+1)):
                if type(v)=DOM_INT then
                    t2 := 1:
                  else
                    t2 := ssin(float(v*PI)):
                end_if:
                DIGITS:=oldDIGITS:

                //------------------------------
                // Bugfix (Walter). Inserted the 
                // if iszero ... branch to avoid
                // division by zero
                //------------------------------
                if iszero(t1) then t1:= 10.0^(-DIGITS); end_if;
                if iszero(t2) then t2:= 10.0^(-DIGITS); end_if;
                predsize:=(t1*scos(float(v*PI))-1/t1)/t2
                           /sgamma(float(v+1)):
                DIGITS:=10:
              else
                //if z is negative then 
                if type(float(z))=DOM_FLOAT and float(z) < 0 then
                    z1 := -z:
                    if type(v)=DOM_INT then
                        t1 := float(1):
                      else
                        t1 := sexp(float(fn*I*PI*v)):
                    end_if:
                  else
                    z1 := z:
                    t1 := 1:
                end_if:
                predsize := t1*0.79785/ssqrt(z1)*(ssin(z1-(float(PI/2))
                    *(v+1/2))+scos(z1-(float(PI/2))*(v+1/2))*(4*v^2-1)*float(1/8)/z1):
            end_if:
          else
            // Bessel K: 0.39893 ~ 1/sqrt(2*PI) 
            if abs_v > 1 and abs_z < abs_v then
                t1 := (z/2)^v*(1+(z^2/4)/(v+1)):
                if type(v)=DOM_INT then
                    t2 := 1:
                  else
                    t2 := ssin(float(v*PI)):
                end_if:
                predsize := (t1-1/t1)/t2:
                DIGITS:=oldDIGITS:
                predsize:=predsize/sgamma(float((v+1))):
                DIGITS:=10:
              else
                t1 := (4*v^2-1)*float(1/8)/z:
                predsize := 0.39893*sexp(-z)/ssqrt(z)*(1+t1):
            end_if:
        end_if:

        if (type(v)=DOM_INT and type(float(z))=DOM_FLOAT) then
            predDIGITS := 2:
          elif type(float(v))=DOM_FLOAT then
            predDIGITS := max(cmagdiff(predsize) - z_mags, 0) + 2:
          elif type(float(z))=DOM_FLOAT then
            predDIGITS := max(cmagdiff(predsize),0) + 2:
          else
            predDIGITS := max(cmagdiff(predsize),0) + v_mags + 2:
        end_if:

        if abs_z > v_i and 
           not(type(float(v))=DOM_FLOAT and 
           domtype(v) <> DOM_INT and
           iszero(sfrac(2*v))) then
            // .18394 ~ 0.5*exp(-1.0) 
            k_min := strunc(ssqrt(abs_z^2+v_r^2)+ssqrt(abs_z^2-v_i^2)) + 1:
            try:= sabs(scos(float(PI*v))* sexp(v_r^2/k_min)):
            if iszero(try) then
              asy_bdry:= -10.0^5 + fln(0.18394*k_min/abs_z)*k_min:
            else
              asy_bdry := -sround(ln10*(fln(try) + k_min*fln(0.18394*k_min/abs_z))):
            end_if:
          else
            asy_bdry := 0:
        end_if:
        lost := 3:
        for try from 1 to 5 do
            DIGITS     := oldDIGITS + predDIGITS:
        if type(float(v))=DOM_FLOAT and domtype(v) <> DOM_INT 
               and float(sfrac(2*v)) < float(10^(-DIGITS)) then
             v:=sround(2*v)/2:
             d:=DIGITS:
             DIGITS:=10;
             abs_z := sabs(z):
             k_min:= abs(v)-(1/2):
             crit:=float(abs_z -(k_min-10));
             try_asympt:=not(bool( (abs_z<10) and (v>10) and (crit <= 0)));
             DIGITS:=d:
        else
            crit := -1:
            // Change in digit setting 
            //try_asympt := bool(asy_bdry > DIGITS+length(DIGITS+9)+3):
            try_asympt := bool(asy_bdry > DIGITS): 
        end_if:
            s := FAIL:
            if try_asympt then
                if fn = 1 and signc(z) = -1 and crit < 0 then
                    // Don't use reflection formula for Bessel K since it 
                    // suffers from high cancellation, and the asymptotic 
                    // series converges on |arg(z)| < 3/2*PI anyway.      
                    DIGITS := oldDIGITS:
                    return(float(complexYK(v,z,fn,TRUE))):
                end_if:

                if fn = 1 then
                    s := float(asymptJY(v,z,1,k_min)):
                  else
                    s := float(asymptK(v,z,k_min)):
                end_if:
            end_if:

            if not try_asympt or s = FAIL then
                if type(v)=DOM_INT then
                    s := float(intorderYK(v,z,fn,sround(ln10*fln(sabs(predsize))))):

                  else
                  // The value of "lost" is initialized outside the main loop 
                    d1 := max(0, sround(ln10*fln(sabs(v)))):

                    for try from 1 to 5 do
                        DIGITS := DIGITS + lost:
                        DIGITS := DIGITS + d1:
                        if fn = 1 then
                            t1 := float(1/ssin(float(v*PI))):
                            DIGITS:=DIGITS-d1+max(0,-sround(ln10*fln(sabs(z)))): 
                            // Refinement (Walter, 12.3.09)
                            if testtype(v, Type::Real) and iszero(frac(2*v)) then
                               if round(2*v) mod 2 = 0 then // v is an integer
                                  t2:= float( (-1)^round(2*v));
                               else // v is a half integer such as 1/2, 3/2, ...
                                  t2:= float(0);
                               end_if;
                            else
                               t2 := scos(float(v*PI)):
                            end_if:
                            // end of refinement
                            s  := float(t2*BesselJ(v,z) - BesselJ(-v,z)): 
                          else
                            t1 := float((PI/2)/ssin(float(v*PI))):
                            DIGITS:=DIGITS-d1+max(0,-sround(ln10*fln(sabs(v))),
                                                        strunc(0.87*Re(z))):
                            s  := float(BesselI(-v,z)-BesselI(v,z)):
                        end_if:

                        if iszero(s) then
                            // The 0's of Y and K are irrational, so this 
                            // should never happen unless complete cancellation
                            // has occurred. 
                            lost := 20:
                          else
                            lost := oldDIGITS - rawdigits(Re(s)):
                        end_if:

                        if lost <= 0 then
                            break:
                          else
                            lost := lost + 2:
                        end_if:
                    end_for:

                    s := s * t1:
                end_if:
            end_if:

            if type(v)=DOM_INT and type(float(z))=DOM_FLOAT then
                resDIGITS := 0:
              elif type(float(v))=DOM_FLOAT then
                resDIGITS := cmagdiff(s) - z_mags:
              elif type(float(z))=DOM_FLOAT then
                resDIGITS := cmagdiff(s):
              else
                resDIGITS := cmagdiff(s) + v_mags:
            end_if:

            if resDIGITS <= predDIGITS then
                lost := 0:

                if Re(s) <> 0 then
                    lost := oldDIGITS - rawdigits(Re(s)):
                  else
                    lost := 0:
                end_if:

                if not(iszero(Im(s))) then
                    lost := max(lost, oldDIGITS - rawdigits(Im(s))):
                end_if:

                if lost <= 0 then
                    break:
                  else
                    predDIGITS := predDIGITS + lost + 2:
                end_if:

              else
                predDIGITS := resDIGITS + 2:
            end_if:
        end_for:
    end_if:

    s:
end_proc:

//--------------------------------------------------------------------------
// Routine: asymptJY                                                          
//                                                                            
// (Complex) Float   evaluation of Bessel J & Y functions                     
//  via asymptotic expansion     (Erdelyi, 7.13.1(3,4))                       
//                                                                            
// Note: This code is more complicated than most asymptotic series code       
// because it is possible for the terms of the series to initially increase   
// and yet eventually converge.  Also, since it is the difference of two      
// series, we have to check for catastrophic cancellation.  See comments below.
//                                                                            
//  fn - function index: -1=J, 1=Y                                            
//--------------------------------------------------------------------------

asymptJY := proc(v,z,fn,k_max)
    local oldDIGITS, d, d2,a, abs_z, abs_v2, z1, t, tt, s1, s2,  
        k, predsize, retry, lost, m, s, mag, tol, try, two_k, tmp;
    save DIGITS;
    begin
    oldDIGITS := DIGITS:

   // If |v|^2 > 2|z| then the terms of this asymptotic series will increase 
   // for a while before decreasing (and then ultimately increases again--   
   // we anticipate convergence before then).  The largest term is then roughly
   // exp(|v|^2/(2|z|)).  We only need convergence to the original precision, 
   // however.  Note: .21715 ~ log[10](e)/2 .                                 

    d      := DIGITS:
    DIGITS := 10;
    abs_z  := sabs(z):
    abs_v2 := sabs(float(v))^2:
    if abs_v2 < 2*k_max*abs_z then
        DIGITS := d + strunc(float(0.21715*abs_v2/abs_z)) + 1:
    else
        DIGITS := d:
    end_if:
    // We assume the final results for s1 and s2 will be roughly O(1), but 
    // in the case when |v|^2 > 2|z| they could be smaller.  We'll have to     
    // do a post mortem to catch this and see if recomputation will converge.  
    // We only try recomputation once.                                         
    predsize := -2:
    retry    := FALSE:
    // We also assume there will be no more than 2 digits cancellation in      
    // the final computation, which is a difference of two series: this    
    // margin is already included in d.                                        
    lost := 0:

    for try from 1 to 2 do
        a      := float(4)*v^2:   
        z1     := float(1/8)/z:   
        t      := float(1):
        s1     := t:
        s2     := float(0):
        tol    := float(10^(-d)):
        for k from 1 to k_max step 2 do 
            two_k  := 2*k;
            t      := t*(a-(two_k-1)^2)*z1/k:
            tt     := t:
            s2     := s2 + t:
            t      := -t*(a-(two_k+1)^2)*z1/(k+1):
            s1     := s1 + t:
            if sabs(t)  <= tol*sabs(s1)
            or sabs(tt) <= tol*sabs(s2)
               then break;
            end_if;
        end_for:
        if k > k_max then
            return(FAIL):
        end_if:
        // bugfix by walter, 19.5.03; added: if iszero(s2) then ...
        if iszero(s2) then
           mag := sround(ln10*fln(sabs(s1)));
        else
           mag := min(sround(ln10*fln(sabs(s1))),sround(ln10*fln(sabs(s2)))):
        end_if;
        // Now check for the possibility that the sum of the two series     
        // produces catastrophic cancellation.  Continue computing at       
        // the higher setting of DIGITS to minimize this risk.              
        t  := float(z - PI*((v/2)+(1/4))):
        if fn = -1 then
            // Bessel J 
            s1 := scos(t) * s1:
            s2 := ssin(t) * s2:
          else
            // Bessel Y 
            s1 := ssin(t) * s1:
            s2 := scos(t) * s2:
        end_if:

        s := s1 + fn*s2:
        if iszero(s1) or iszero(s) then
             // no cancellation problem. Set m:= lost + 1 to enforce stepping
             // into the 'insignificant cancellation' branch below.
             m := lost + 1;
        else m := sround(ln10*fln(sabs(s1/s))):
        end_if;
        if m < lost+2 then
            // Insignificant cancellation 
            // must check size versus expected size 
            if mag < predsize then
                if retry then
                    // 2nd attempt, still too small.  Give up. 
                    return(FAIL):
                  else
                    predsize := mag - 2:
                end_if:

            else
                // Bugfix (?) (Walter): Voldemort's line
                // s := s * ssqrt(float(2/PI/z)):
                // was replaced by the new line below.
                // I am not at all sure that this is correct.
                // However, besselY(3/2, -1) changes its sign
                // with the new line (fixing a bug);

                s := s * ssqrt(float(2/PI))/ssqrt(z):

                  break:
            end_if:
          
          else
            // sround(ln10*fln(sabs(s1))) ~ sround(ln10*fln(sabs(s2))) 
            if retry then
                // 2nd attempt, still too much cancellation.  Give up. 
                return(FAIL):
            else
                lost     := m:
                predsize := min(predsize,mag-2):
            end_if:
        end_if:

        DIGITS := DIGITS - predsize + lost:
        d := d + lost:

        // Check to see if we can expect convergence with this new value 
        // of DIGITS.                                                    
        d2 := DIGITS:
        DIGITS := 10:
        if not(type(float(v))=DOM_FLOAT and domtype(v) <> DOM_INT
              and iszero(sfrac(2*v))) then
            // Arrgr:  ln(exp(-2|z|) ??? Stupid idea: exp tends to produce over/underflows
            //tmp:= -sround(ln10*fln(sabs(sexp(-float(2)*abs_z))*     sabs(scos(float(PI*v)))) );
              tmp:= -sround(ln10*   (         (-float(2)*abs_z) + fln(sabs(scos(float(PI*v))))) ); 
             if type(v)<>DOM_INT and tmp <= d+mag then
                   // Not going to converge 
                    return(FAIL):
               else retry := TRUE: 
               end_if:
        else retry := TRUE:
        end_if: 
        DIGITS:=d2:
    end_for:

    s:
end_proc:


// Routine: asymptK                                                         
//                                                                          
// (Complex) Float evaluation of Bessel K function via asymptotic expansion 
//     (Erdelyi, 7.13.1(7))                                                 
//                                                                          
// Note: This code is more complicated than most asymptotic series code     
// because it is possible for the terms of the series to initially increase 
// and yet eventually converge.  See comments below.                        

asymptK := proc(v,z,k_max)
    local coff,oldDIGITS, d, d2, a, abs_z, abs_v2, z1, t, s, k, predsize,
        retry, mag, try, tol, two_k;
    save DIGITS;
    begin
    oldDIGITS := DIGITS:

   // If |v|^2 > 2|z| then the terms of this asymptotic series will increase  
   // for a while before decreasing (and then ultimately increases again--    
   // we anticipate convergence before then).  The largest term is then roughly
   // exp(|v|^2/(2|z|)).  We only need convergence to the original precision, 
   // however.  Note: .21715 ~ log[10](e)/2 .                                 

    d      := DIGITS:
    DIGITS := 10:
    abs_z  := sabs(z):
    abs_v2 := sabs(float(v))^2:
    if abs_v2 < 2*k_max*abs_z then
        DIGITS := d + strunc(float(0.21715*abs_v2/abs_z)) + 1:
    else
        DIGITS := d:
    end_if:
    // We assume the final result for s will be roughly O(1), but         
    // in the case when |v|^2 > 2|z| they could be smaller.  We'll have to    
    // do a post mortem to catch this and see if recomputation will converge. 
    // We only try recomputation once.                                        
    predsize := -2:
    retry    := FALSE:

    for try from  1 to 2 do
        a     := float(4*v^2):
        z1    := float(1/(8*z)):
        t     := float(1):
        s     := t:
        tol   := float(10^(-d)):
        for k from 1 to k_max do 
            two_k  := 2*k:
            t     := t * (a-(two_k-1)^2) * z1 / k:
            s     := s + t:
            if sabs(t) <= sabs(s)*tol then break end_if
        end_for:
        if k > k_max then
            return(FAIL):
        end_if:
    
        // Bugfix (Walter, August 08): added the following 'if iszero(s)'
        // to avoid the singularity in ln:
        if iszero(s) then 
           mag:= RD_INF;
        else
           mag := sround(ln10*fln(sabs(s))):
        end_if:

        if mag < predsize then
            if retry then
                // 2nd attempt, still too small.  Give up. 
                return(FAIL):
              else
                predsize := mag - 2:
            end_if:

        else
            // Note: A&S 9.7.2 has the wrong closure on the branch cut. Don't 
            // change this formula!                                           
            d2 := DIGITS:
            DIGITS := DIGITS+max(0,sround(ln10*fln(sabs(z)))):
            coff := sexp(-z):
            DIGITS := d2:
            // Bugfix (Walter): replaced the following line
            // s := coff*ssqrt(float(PI/(2*z)))*s:
            // by
            s := coff*ssqrt(float(PI/2))/sqrt(float(z))*s:
            break:
        end_if:

        DIGITS := DIGITS - predsize:

        //  Check to see if we can expect convergence with this new value     
        // of DIGITS.                                                         

         d2 := DIGITS:
         DIGITS := 10:
         if not(type(float(v))=DOM_FLOAT and domtype(v) <> DOM_INT
               and iszero(sfrac(2*v))) then
           if type(v)<> DOM_INT and -sround(ln10*fln(sabs(sexp(-float(2)*abs_z)*
              sabs(scos(float(PI*v)))))) <= d+mag then
             // Not going to converge 
             return(FAIL):
           else
             retry := TRUE:
           end_if:
         else
           retry := TRUE:
         end_if:
         DIGITS := d2:
    end_for:

    s:
end_proc:

// Routine: asymptI                                                           
// (Complex) Float evaluation of Bessel I function via asymptotic expansion 
//     (Erdelyi, 7.13.1(5): see also Whittaker & Watson, 17.7, for extension  
//     to larger argument)                                                    
//                                                                            
// Note: This code is more complicated than most asymptotic series code       
// because it is possible for the terms of the series to initially increase   
// and yet eventually converge.  Also, since it is the difference of two      
// series, we have to check for catastrophic cancellation.  See comments below.

asymptI := proc(v,z,k_max)
    local coff,coff1,coff2,oldDIGITS,d,d2,a,abs_z,abs_v2,z1,t,tt,s1,s2,
        k, predsize, retry, lost, m, mag, s,sgn, try, tol, two_k;
    save DIGITS;
    begin
    oldDIGITS := DIGITS:
    // Change in digit setting 
    //DIGITS    := DIGITS + length(DIGITS+9) + 3:

  // If |v|^2 > 2|z| then the terms of this asymptotic series will increase   
  // for a while before decreasing (and then ultimately increases again--     
  // we anticipate convergence before then).  The largest term is then roughly
  // exp(|v|^2/(2|z|)).  We only need convergence to the original precision,  
  // however.  Note: .21715 ~ log[10](e)/2 .                                  

    d      := DIGITS:
    DIGITS := 10:
    abs_z  := sabs(z):
    abs_v2 := sabs(float(v))^2:
    if abs_v2 < 2*k_max*abs_z then
        DIGITS := d + strunc(float(0.21715*abs_v2/abs_z)) + 1:
    else
        DIGITS := d:
    end_if:
  // We assume the final results for s1 & s2 will be roughly O(1), but    
  // in the case when |v|^2 > 2|z| they could be smaller.  We'll have to      
  // do a post mortem to catch this and see if recomputation will converge.   
  // We only try recomputation once.                                          
    predsize := -2:
    retry    := FALSE:

  // We also assume there will be no more than 2 digits cancellation in       
  // the final computation, which is a difference of two series: this     
  // margin is already included in d.                                         
    lost := 0:

    for try from 1 to 2 do
        a     := float(4*v^2):
        z1    := float(1/(8*z)):
        t     := float(1):
        s1     := t:
        tol   := float(10^(-d)):
        if not(iszero(sfrac(v)-1/2)) and (iszero(v) or not(iszero(Re(v)))) 
                     and Re(z)>0 and sabs(Im(z)) < 6*Re(z) then
            // Reasonably well into the right half plane: in this case, a 
            // simpler series converges and we won't need to worry about  
            // catastrophic cancellation                                  
            for k from 1 to k_max do 
                two_k  := 2*k:
                t      := -t * (a-(two_k-1)^2) * z1 / k:
                s1     := s1 + t:
                if sabs(t) <= sabs(s1)*tol then break end_if
            end_for:
            if k > k_max then
                return(FAIL):
            end_if:

            //-----------------------------------------
            // Bugfix (Walter): safeguard agains ln(0.0)

            if iszero(s1) then s1:= 10^(-DIGITS) end_if;
            //-----------------------------------------

            mag := sround(ln10*fln(sabs(s1))):

            if mag < predsize then
                if retry then
                    // 2nd attempt, still too small.  Give up.  
                    return(FAIL):
                  else
                    predsize := mag - 2:
                end_if:

              else
                d2 := DIGITS:
                DIGITS:=DIGITS+max(0,sround(ln10*fln(sabs(z)))):
                coff := sexp(z):
                DIGITS := d2:
                s := s1/ssqrt(float(2*PI*z))*coff:
                break:
            end_if:

          else
            s2     := t:
            sgn    := 1:
            tol    := float(10^(-d)):
            for k from 1 to k_max do 
              two_k  := 2*k;
              sgn    := -sgn:
              t      := t * (a-(two_k-1)^2) * z1 / k:
              s2     := s2 + t:
              tt     := sgn*t:
              s1     := s1 +tt:
              if sabs(tt)  <= tol*sabs(s1)
              or sabs(t) <= tol*sabs(s2)
                 then break;
              end_if;
            end_for:
            if k > k_max then
                return(FAIL):
            end_if:

            //-----------------------------------------
            // Bugfix (Walter): safeguard agains ln(0.0)
            if iszero(s1) then s1:= 10.0^(-DIGITS); end_if;
            if iszero(s2) then s2:= 10.0^(-DIGITS); end_if;
            //-----------------------------------------

            mag := min(sround(ln10*fln(sabs(s1))),sround(ln10*fln(sabs(s2)))):

            // Now check for the possibility that the sum of the two series  
            // produces catastrophic cancellation.  Continue computing at    
            // the higher setting of DIGITS to minimize this risk.           
            d2 := DIGITS:
            DIGITS := DIGITS+max(0,sround(ln10*fln(sabs(z)))):
            coff1 := sexp(z):
            coff2 := float(-signc(I*z)*I*sexp(-z-signc(I*z)*I*v*PI)): 
            DIGITS := d2:
            s1 := s1 * coff1:
            s2 := s2 * coff2:
            s  := s1 + s2:

            //-----------------------------------------
            // Bugfix (Walter): safeguard agains ln(0.0)
            if iszero(s1) then s1:= 10^(-DIGITS); end_if;
            if iszero(s)  then s:= 10^(-DIGITS); end_if;
            //-----------------------------------------
            m  := sround(ln10*fln(sabs(s1/s)));

            //-----------------------------------------
            // Bug (Walter): try 
            // DIGITS:= 100: float(PI):DIGITS:= 10:
            // besselI(5/2, 2/10.0^5)
            // You get into the 'm < lost 2' branch and
            // get a wrong result!
            //-----------------------------------------
            
            if m < lost+2 then
                 // Insignificant cancellation        
                // must check size versus expected size  
                if mag < predsize then
                    if retry then
                        // 2nd attempt, still too small.  Give up.  
                        return(FAIL):
                      else
                        predsize := mag - 2:
                    end_if:

                  else
                    s := s / ssqrt(float(2*PI*z)):
                    break:
                end_if:

              else
                // sround(ln10*fln(sabs(s1))) ~ sround(ln10*fln(sabs(s2))) 
                if retry then
                    // 2nd attempt, still too much cancellation.  Give up. 
                    return(FAIL):
                  else
                    lost     := m:
                    predsize := min(predsize,mag-2):
                end_if:
            end_if:
        end_if:
        DIGITS := DIGITS - predsize + lost:
        d      := d + lost:

        // Check to see if we can expect convergence with this new value  
        // of DIGITS.                                                     
         d2 := DIGITS:
         DIGITS := 10:

         m:= sabs(scos(float(PI*v)));
         if type(v)<>DOM_INT and
            not iszero(m) and
            -sround(ln10*fln(sabs(sexp(-float(2)*abs_z)* m))) <= d+mag then
            // Not going to converge 
            return(FAIL):
          else
            retry := TRUE:
         end_if:
         DIGITS := d2:
    end_for:

    s:
end_proc:


// Routine: intorderYK                                                      
//                                                                          
// (Complex) Float evaluation of Bessel Y & K functions for integer order 
//     (A&S 9.1.11, 9.6.11)                                                 
//                                                                          
//  fn - function index: -1=K, 1=Y                                          

intorderYK := proc(n,z,fn,predsize)
    local oldDIGITS, F, c, abs_z, z_half, abs_z2, n2, n_half, max_size,
        pred_s1, k, pred_s2, pred_s3, lost, redo_s1, redo_s2, redo_s3, s,
        s1, s2, s3, t, t1, t2, k_max, r, res, res_s1, res_s2, res_s3,
        pred0, u,try,term,tol;
    save DIGITS;
    begin
    oldDIGITS := DIGITS:
    DIGITS    := 10:

    if fn = -1 then
        // Bessel K 
        F        := BesselI:
        c        := 1/2:
      else
        // Bessel Y 
        F        := BesselJ:
        c        := 1/PI:
    end_if:

    abs_z      := sabs(z):
    z_half     := z/2:
    abs_z2     := abs_z^2:
    n2         := n^2:
    n_half     := n/2:

    if fn = 1 then
        // Estimate size of BesselJ(n,z) 
        if abs_z < n then
            pred0 := z_half^n/sgamma(float(n+1)):
          else
            // 0.63662 ~ 2/PI  
            pred0 := ssqrt(0.63662/z)*(scos(z-float(PI/2)*float(n+1/2))
                  -ssin(z-float(PI/2)*float(n+1/2))*(4*n^2-1)*float(1/8)/z):
        end_if:

      else
        // Estimate size of BesselI(n,z)  
        if abs_z < n then
            pred0 := z_half^n/sgamma(float(n+1)):
          else
            t := (4*n^2-1)*float(1/8)/z:
            u := ssqrt(float(PI*z)):
            pred0 := sexp(z)/u*(1-t):
            if sabs(Im(z)) > 6*Re(z) then
                // exponentially subdominant term signiend_ifcant here 
                pred0 := pred0+sexp(-float(PI*(n+1/2)*I-z))/u*(1+t):
            end_if:
        end_if:
    end_if:
    
    // Estimate size of 1st term in sum 
    term:=fln(z_half)*pred0:
    if iszero(term) then
     pred_s1 := -infinity:
    else
     pred_s1 := sround(ln10*fln(sabs(term))):
    end_if:

    // Estimate size of 2nd term in sum 
    if abs_z <= n then
        k       := min(n-1, sround(n_half+ssqrt(n2-abs_z2)/2)):
        term := z_half^(2*k-n)*(n-k-1)!/k!:
      else
        term := z_half^(n-2)*n!:
    end_if:
    if iszero(term) then pred_s2 := -infinity: else pred_s2:=sround(ln10*fln(sabs(term))):end_if:
    
    // Estimate size of 3rd term in sum 
    if n < 2 then
        term := pred0:
      else
        term := fln(float(n))*pred0:
    end_if:
    if iszero(term) then pred_s3 := -infinity: else pred_s3:=sround(ln10*fln(sabs(term))):
    end_if:
    
    // can now estimate number of digits which will be lost in the final 
    // sum: note that if this estimate is low, only the terms whose sizes 
    // were underestimated, and which are larger than the final sum, need 
    // to be recomputed 
    max_size := max(pred_s1,pred_s2,pred_s3):
    lost     := max(4, 4+max_size-predsize):
    redo_s1  := TRUE:
    redo_s2  := TRUE:
    redo_s3  := TRUE:
    s1       := float(0):
    s2       := float(0):
    s3       := float(0):

    for try from 1 to 5 do
        DIGITS := oldDIGITS + lost + 2:
        if iszero(z_half-1) then
            s1 := float(0):
          elif redo_s1 and max_size - pred_s1 < 3*DIGITS then
            s1 := 2*fn^(n+1)*fln(float(z/2))*F(n,z):
        end_if:

        if redo_s2 and max_size - pred_s2 < 3*DIGITS then
            if iszero(n) then
                s2 := float(0):

              else
                r  := float(fn * z^2/4):
                t  := sgamma(float(n)):
                s2 := t:

                for k from 1 to (n-1) do
                    t  := t * r / (k*(n-k)):
                    s2 := s2 + t:
                end_for:

                s2 := -fn * (z/2)^(-n) * s2:
            end_if:
        end_if:

        if redo_s3 and max_size - pred_s3 < 3*DIGITS then
            // estimate location and size of largest term in sum and add 
            // required guard digits                                     
            k_max  := sround(n_half+ssqrt(n2+abs_z2)/2):
            DIGITS := DIGITS+2+max(0,sround(ln10*fln(sabs(
            sabs(z_half)^(2*k_max+n)/k_max!/(n+k_max)!))) - pred_s3):
            r      := float(-fn*z^2/4):
            t1     := float(psi(1) + psi(n+1)):
            t2     := 1/sgamma(float(n+1)):
            s3     := t1 * t2:
            tol    := float(10^(-DIGITS)):
            for k from 1 to RD_INF do
                t1     := t1 + float(1/k + 1/(n+k)):
                t2     := t2 * r / (k*(n+k)):
                s3     := s3 + t1*t2:
                if sabs(t1*t2) <= sabs(s3)*tol then break end_if
            end_for:

            s3 := -fn^(n+1) * (z/2)^n * s3:
        end_if:

        s      := s1 + s2 + s3:
       if iszero(s)  then    res:=-infinity:else res:=sround(ln10*fln(sabs(s))):end_if:
       if iszero(s1) then res_s1:=-infinity:else res_s1:=sround(ln10*fln(sabs(s1))):end_if:
       if iszero(s2) then res_s2:=-infinity:else res_s2:=sround(ln10*fln(sabs(s2))):end_if:
       if iszero(s3) then res_s3:=-infinity:else res_s3:=sround(ln10*fln(sabs(s3))):end_if:

        if max(res_s1,res_s2,res_s3) - res < lost then
            break:

          else
            if res_s1 > pred_s1 then
                pred_s1 := res_s1:
                redo_s1 := TRUE:
            end_if:
            if res_s2 > pred_s2 then
                pred_s2 := res_s2:
                redo_s2 := TRUE:
            end_if:
            if res_s3 > pred_s3 then
                pred_s3 := res_s3:
                redo_s3 := TRUE:
            end_if:
            max_size := max(pred_s1,pred_s2,pred_s3):
            lost := max(res_s1,res_s2,res_s3) - res + 2:
        end_if:
    end_for:

    s := float(c*s):
end_proc:

unalias(realJI):
unalias(realYK):
unalias(taylorJI):
unalias(complexJI):
unalias(complexYK):
unalias(asymptI):
unalias(asymptK):
unalias(asymptJY):
unalias(intorderYK):
unalias(cmagdiff):
unalias(rawdigits):
unalias(signc):
unalias(estimateJI):
unalias(findtmaxJI):

unalias(ln10):
unalias(sabs):
unalias(ssqrt):
unalias(ssin):
unalias(scos):
unalias(sexp):
unalias(sgamma):
unalias(sround):
unalias(strunc):
unalias(sfrac):
unalias(fln):

specfunc::Bessel::MMLContent :=
(Out, data, fnsym) -> Out::Capply(Out::Ccsymbol(
  output::MMLPresentation::msub(output::MMLPresentation::mi(fnsym),
                              Out(op(data, 1)))),
  Out(op(data, 2))):

/* ===
// =================================================================================
// == Experimental implementation of the Steed algorithms (see Numerical Recipes) ==
// =================================================================================

// The present implementations works for v < x:
// Step1: Choose vv with vv > x
//        Compute f = J'[vv](x) / J[vv](x)
//        by a contfrac
// Step2: Compute J'[v]/J[v]
//        by a stable backward iteration 
// Step3: Compute Y[v]/J[v], Y'[v]/J[v]
//        by computing pq = (J'[v](x)+I*Y'[v](x))/(J[v](x)+I*Y[v](x))
//        via a contrac
// Step4: Solve Wronski(J[v], Y[v]) = 2/PI/x for J[v]
//
// For v > x we have to use the strategy:
// Step1: Compute f = J'[v](x) / J[v](x) by a contfrac
// Step2: Choose some vv with vv < x and compute
//        J'[vv]/J[vv] by a stable backward iteration
// Step3: Compute Y[vv]/J[vv], Y'[vv]/J[vv]
//        by computing pq = (J'[vv](x)+I*Y'[vv](x))/(J[vv](x)+I*Y[vv](x))
//        via a contrac
// Step4: Solve Wronski(J[vv], Y[vv]) = 2/PI/x for J[vv]
// Step5: scale J[v] from the scaling of J[vv]

specfunc::Bessel::Steed:= proc(v, x)  // for real v >= |x|
local vv, dv, p0, q0, p1, q1, bk, dbk, f, J, Jp, oldJ,
      ak, eps, g, k, p, pq, q, s, v2;
begin
   [v, x]:= float([v, x]);
   dv:= ceil(max(v, abs(x)) + 10 - v):
   vv:= v + dv: // This vv satisfies vv >= |x|

   //------------------------------------------------------------
   // compute the contfrac f = J[v]'/J[v] = v/x - J[v+1]/J[v] 
   // = v/x - 1/(2*(v+1)/x - 1/(2*(v+2)/x - 1/(2*(v+3)/x - ...)))
   // It converges fast for v = vv > |x|:
   //------------------------------------------------------------
   p0:= 1: q0:= 0:
   p1:= vv/x: q1:= 1;
   bk:= 2*vv/x: dbk:= 2/x:
   while TRUE do
     bk:= bk + dbk:
     [p1, p0]:= [bk*p1 - p0, p1];
     [q1, q0]:= [bk*q1 - q0, q1];
      if specfunc::abs(p1*q0 - p0*q1) <= 10^(-DIGITS)*specfunc::abs(p1*q0) then
         break;
     end_if;
   end_while:
   f:= p1/q1; 
   s:= sign(q0/q1);

   //------------------------------------------------------------
   // Choose some random tiny value for J[vv], initialize 
   // J'[vv] = f*J[vv], and do a stable backwards iteration 
   //    J[v-1] = v/x*J[v] + J'[v]
   //   J'[v-1] = (v-1)/x*J[v-1] - J[v]
   // towards smaller values of vv until vv = v is reached.
   // This way we compute the following J which is besselJ(v, x) 
   // up to an unknown factor eps/besselJ(vv, x).
   //------------------------------------------------------------
   eps:= 10.0^(-123456789);  
   J:= s*eps;  // This should be J = besselJ(vv, x);
   Jp:= f*J; // Make sure the ratio Jp/J is the correct f
   repeat
     oldJ:= J:
     J:= vv/x*J + Jp: 
     Jp:= (vv-1)/x*J - oldJ; 
     vv:= vv-1;
   until vv < v + 0.1 end_repeat;
   // The new ratio for v instead of vv:
   f:= Jp/J:
   s:= sign(J);

   //------------------------------------------------------------
   // compute the contfrac pq = (J'[v] + I*Y'[v])/(J[v] + I*Y[v]) 
   // = -1/2/x + I + I/x* (
   //  ((1/2)^2-v^2)/(2*(x+I) + ((3/2)^2-v^2)/(2*(x+2*I) + ...))
   //         )
   // This contfrac is valid for v, x > 0 (where both J and Y are real).
   // The contrfrac converges for v < |x|.
   //------------------------------------------------------------
   // Remark: Here is a version of the contfrac that holds off the 
   // real line, too. With H2(v, x) = besselJ(v, x) - I*besselY(v, x) 
   // one has
   // H2(v+1,z)/H2(v,z) = v/z*H2(v, z) - H2'(v,z)/H2(v,z) =  
   // (v+1/2)/z + I + 1/z * (
   //   (v^2-(1/2)^2)/(2*(I*z+1) 
   //   +(v^2-(3/2)^2)/(2*(I*z+2) + 
   //    +(v^2-(5/2)^2)/(2*(I*z+3) + ...))))
   // In other words:
   // H2'(v,z)/H2(v,z) =  (J'[v] - I*Y'[v])/(J[v] - I*Y[v])
   // = -1/2/z - I - 1/z * (
   //    ((1/2)^2-v^2)/(2*(-I*z-1) 
   //    +((3/2)^2-v^2)/(2*(-I*z-2) + 
   //     +((5/2)^2-v^2)/(2*(-I*z-3) + ...))))
   // = -1/2/z - I + I/z * (
   //    ((1/2)^2-v^2)/(2*(-z+I) 
   //    +((3/2)^2-v^2)/(2*(-z+2*I) + 
   //     +((5/2)^2-v^2)/(2*(-z+3*I) + ...))))
   // Taking the complex conjugate, replacing v by cv = conjugate(v),
   // z by cz = conjugate(z) and using conjugate(J(v,z)) = J(cv,cz) 
   // etc we arrive at the version stated in the Numerical Recipes:
   // (J'[v] + I*Y'[v])/(J[v] + I*Y[v]) =
   //  -1/2/z + I + I/z * (
   //   ((1/2)^2-v^2)/(2*(z+I) 
   //   +((3/2)^2-v^2)/(2*(z+2*I) + 
   //    +((5/2)^2-v^2)/(2*(z+3*I) + ...))))
   //------------------------------------------------------------
   // 
   p0:= 1: q0:= 0:
   p1:= 0: q1:= 1;
   k:= -1/2: 
   v2:= v^2:
   bk:= 2*x; dbk:= 2*I:
   while TRUE do
     k:= k + 1;
     ak:= k^2 - v2:
     bk:= bk + dbk:
     [p1, p0]:= [bk*p1 + ak*p0, p1];
     [q1, q0]:= [bk*q1 + ak*q0, q1];
      if specfunc::abs(p1*q0 - p0*q1) <= 10^(-DIGITS)*specfunc::abs(p1*q0) then
         break;
     end_if;
   end_while:
   pq:= -1/2/x + I + I/x * p1/q1;
   p:= Re(pq):
   q:= Im(pq):
   g:= (p - f)/q;

   //solve Wronski(J[v](x), Y[v](x)) = 2/PI/x for J = J[v](x):
   J:= s*(2/float(PI)/x/(q + g*(p-f)))^(1/2); 
end_proc:

==== */
