/*++
        besselK(v,z) is the modified bessel function of the second kind.
        It satisfes the modified bessel equation: 
               2                2    2
              x  y'' + x y' - (x  + v ) y = 0
++*/

besselK:=
proc(v,z)
  local fv,fz,i,n,res,t,twoz,sv,sz,vv,zz,dummy;
begin
  if args(0) <> 2 then
    error("expecting two arguments")
  elif z::dom::besselK <> FAIL then
    return( z::dom::besselK(args()) )
  elif {DOM_BOOL,DOM_FAIL,DOM_NULL,DOM_NIL} intersect
       {domtype(v), domtype(z)} <> {} then
    error("invalid operand")
  end_if;

  // ----------  map to sets -------------------------
  // This is quite complicated, because both arguments have
  // to be handled
  case [type(v),type(z)]
    of [DOM_SET, DOM_SET] do
        return({besselK(sv, sz) $ sv in v $ sz in z});
    of [DOM_SET, "_union"] do
    of ["_union", DOM_SET] do
        return(_union((besselK(sv, sz) $ sv in v) $ sz in z));
    of ["_union", "_union"] do

        // besselK({a} union A, C union D)
        //  -> besselK({a},C) union besselK({a},D) union besselK(A,C) union besselK(A,D)
        // Make sure that A, C, D are interpreted as sets, i.e.,
        // besselK({a},C) --> besselK(a, C), not {besselK(a,C)} !

        [v, vv, dummy]:= split(v, testtype, DOM_SET):
        if type(vv) <> "_union" then vv:= [vv] end_if:
        [z, zz, dummy]:= split(z, testtype, DOM_SET):
        if type(zz) <> "_union" then zz:= [zz] end_if:

        return(_union({(besselK(sv, sz) $ sv in v ) $ sz in z},
                       (besselK(sv, sz) $ sv in vv) $ sz in z,
                       (besselK(sv, sz) $ sv in v ) $ sz in zz,
                       (besselK(sv, sz) $ sv in vv) $ sz in zz ));
  end_case;

  case type(v)
    of DOM_SET do
    of "_union" do
       // z cannot be a set, if v is a set
       return(map(v, besselK, z));
  end_case;

  case type(z)
    of DOM_SET do
    of "_union" do
       // v cannot be a set, if z is a set
       return(map(z, (z,v) -> besselK(v,z), v))
  end_case;

  // --------------------------------------------------
  if testtype(z, Type::Set) and not testtype(z, Type::Arithmetical) then
    if testtype(v, Type::Set) and not testtype(v, Type::Arithmetical) then
      return(Dom::ImageSet(besselK(#v, #z), [#z, #v], [z, v]));
    elif testtype(v, Type::Arithmetical) then
      return(Dom::ImageSet(besselK(v, #z), [#z], [z]));
    end_if;
  elif testtype(v, Type::Set) and not testtype(v, Type::Arithmetical) then
      return(Dom::ImageSet(besselK(#v, z), [#v], [v]));
  end_if;
  if map([v, z], testtype, Type::Arithmetical) <> [TRUE, TRUE] then
    error("arguments must be of type 'Type::Arithmetical'")
  end_if:
  // --------------------------------------------------

  if z=infinity then
        return(0);
  end_if:

  fv:= domtype(float(v)):
  fz:= domtype(float(z)):
  if (fv=DOM_FLOAT or fv=DOM_COMPLEX) and (fz=DOM_FLOAT or fz=DOM_COMPLEX)
   and has([map((op(v),op(z)),type)],DOM_FLOAT) then
         return(besselK::float(v,z))
  end_if:

  if iszero(z) then
        error("singularity"):
  end_if:

  if (domtype(v)=DOM_INT or (domtype(v)=DOM_FLOAT and iszero(v-round(v)))) 
        and v < 0 then
        return(procname(-v,z));
  end_if:
  if iszero(v-1/2) or iszero(v+1/2) then
    return(1/2*2^(1/2)*PI^(1/2)/z^(1/2)*exp(-z));
  elif not(type(v)=DOM_INT or (type(v)=DOM_FLOAT and iszero(v-round(v))))
  and (type(2*v)=DOM_INT or (type(2*v)=DOM_FLOAT and iszero(2*v-round(2*v)))) then
    // might want to limit this expansion to say abs(v) <= 10
    // and let "expand" take care of cases > 10.
    // Worked out from 10.2.15 and 10.2.16 of A and S
    n := trunc(abs(v));
    twoz:= 2*z:
    t:= 1:     // i=0
    res:= (PI^(1/2)/(2^(1/2)*sqrt(z)))*exp(-z)*
             _plus(1, (t:= t*(n+i)*(n+1-i)/i/twoz) $ i=1..n);
    return(res)
  end_if:
  procname(v,z)
end_proc:

besselK := prog::remember(besselK, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("besselK")]):

besselK := funcenv(besselK):
besselK::type := "besselK":
besselK::print := "besselK":
besselK::info :=
  "besselK(v,z) -- modified bessel function of second kind (order v, argument z)":

besselK::float :=
    loadproc(besselK::float, pathname("STDLIB","FLOAT"), "besselK"):

besselK::series :=
    loadproc(besselK::series, pathname("SERIES"), "besselK"):

besselK::Content := stdlib::genOutFunc("CbesselK", 2):

besselK::MMLContent :=
(Out, data) -> specfunc::Bessel::MMLContent(Out, data, "K"):

//================================================================
// There are 2 formulas for diff of the bessels:
// (1) diff(besselK(v,x), x) =  besselK(v-1,x) - v/x*besselK(v,x)
// (2) diff(besselK(v,x), x) =  besselK(v+1,x) + v/x*besselK(v,x)
// We choose the 'downward' formula (1), if v >= 1 or
// v = something symbolic + number with number >= 1.
// Otherwise, the 'upward' formula (2) is used.
//================================================================
besselK::diff := proc(e, x)
local v, f, dfdx;
begin
   // We cannot differentiate w.r.t. the index (=op(e, 1)):
   if has(op(e,1),[args(2..args(0))]) then
      return(hold(diff)(args()))
   end_if:
   if args(0)>2 then
         return(diff(diff(e,x),args(3..args(0))));
    end_if:
    v:=op(e,1): // the index of besselK
    f:=op(e,2): // the argument of besselK
    dfdx:= diff(f, x):
    if iszero(dfdx) then
       return(0)
    end_if:
    // Switch between the 'upward' and the 'downward' formula to
    // avoid drifting towards high values of v when computing
    // high derivatives. For the decision, split v into
    //   v = symbolic/imaginary + Type::Real
    if besselJ::split_index(v)[2] > 0 then // downward
       return(-besselK(v-1,f)*dfdx - v/f*besselK(v, f)*dfdx);
     else // upward
       return(-besselK(v+1,f)*dfdx + v/f*besselK(v, f)*dfdx);
     end_if;
end_proc:

//==========================================================
// The simplify attribute of besselK(v, x) splits the index
// v=s+n, where s is symbolic/imginary and n is of Type::Real.
// The recursion besselK(v-1,x)-besselK(v+1,x)=-2*v/x*besselK(v,x)
// is used upwards or downwards to reduce n to a value n0 with
// -1 < n0 <= 1.
// Thus,
//   besselK(s+n,x) = c0(n,x)*besselK(s,x) + c1(n,x)*besselK(s+1, x)
// where c0(n, x) and c1(n, x) are rational expressions in x:
//==========================================================
besselK::simplify:= proc(f)
local v, x, vx, n, cn, maxn;
begin // f = besselK(v, x)
   if not type(f) = "besselK" then
      return(f);
   end_if:
   // First, simplify the arguments
   [v, x]:= map([op(f)], simplify, op(args(2..args(0))));
   // Check, if the simplified arguments yields an explicit result
   f:=  eval(subsop(f, 1 = v, 2 = x));
   // If there is an explicit result: run!
   if not contains({"besselJ", "besselI", "besselK", "besselY"}, type(f)) then
      return(f);
   end_if;
   vx:= [v, x]:
   [v, n]:= besselJ::split_index(v); // use the same strategy as in besselJ!
   f:= op(f, 0):
   cn:= ceil(n);
   v:= v - (cn - n);
   n:= cn;
   // if n is large, the reduction to besselK(v, x) and besselK(v+1,x)
   // will be quite costly. If arithmetic with s or x can cause
   // expression swell, we should restrict the shifts n. If s or x
   // are of Type::Numeric, there is no expression swell, but only
   // basic arithmetic with O(n) operations, i.e., n may be larger.
   if testtype(v, Type::Numeric) and
      testtype(x, Type::Numeric) then
      maxn:= 10^4: // hardcoded limit. Shouldn't we make if configurable?
   else
      maxn:= 20:   // hardcoded limit. Shouldn't we make if configurable?
   end_if:
   if specfunc::abs(n) > maxn then
      return(f(op(vx)));
   else
     return( besselK::recurrenceCoeff(v,n,x,[1,0])*f( v ,x)
            +besselK::recurrenceCoeff(v,n,x,[0,1])*f(v+1,x));
   end_if:
end_proc:

//==========================================================
besselK::Simplify:= besselK::simplify:

//==========================================================
besselK::expand :=
    loadproc(besselK::expand, pathname("STDLIB","EXPAND"), "besselK"):

//==========================================================
// utility for computing the coefficients c0(v,x), c1(v,x) in the
// identity
//  besselK(v+n,x) = c0(v,x)*besselK(v,x) + c1(v,x)*besselK(v+1,x),
//  c0(v,x) = recurrenceCoeff(v,n,x,[1,0])
//  c1(v,x) = recurrenceCoeff(v,n,x,[0,1])
//==========================================================
besselK::recurrenceCoeff:= proc(v, n : DOM_INT, x, c0c1)
local c0, c1, k;
begin
  [c0, c1]:= c0c1:
  if n > 1 then
    for k from 1 to n-1 do
      [c1, c0]:= [ 2*(v + k) * c1 + x*c0, x*c1];
    end_for:
    return(normal(c1/x^(n-1)));
  elif n = 1 then
    return(normal(c1));
  elif n = 0 then
    return(normal(c0));
  elif n < 0 then
    for k from 0 downto n+1 do
      [c0, c1]:= [-2*(v + k) * c0 + x*c1, x*c0];
    end_for:
    return(normal(c0*x^n));
  end_if;
  assert(FALSE); // should not arrive here
end_proc:

//==========================================================
besselK::TeX :=
proc(b, ex, prio)
  local s;
begin
  s := generate::tex(op(ex, 2), output::Priority::Fconcat);
  if length(s) < 7 or s[1..7] <> "\\left(" then
    s := "\\left(".s."\\right)";
  end_if;
  _concat("K_{",
          generate::TeX(op(ex,1)),
          "}",
          s);
end_proc:

besselK::conjugate:= proc(v, z)
local b;
begin
   b:= numeric::isnonzero(Im(besselK(v, z)));
   if b = FALSE then
      // y = besselK(v, z) is real
      return(hold(besselK)(v, z));
   elif b = TRUE then
      if numeric::isnonzero(Re(besselK(v, z))) = FALSE then
        // y = besselK(v, z) is on the imaginary axis
        return(-hold(besselK)(v, z));
      else
        // y = besselK(v, z) is somewhere in the complex plane
        return(hold(conjugate)(hold(besselK)(v, z)));
      end_if;
   end_if;

   if is(v in R_) = TRUE and
      is(z >= 0) = TRUE then // y is real
      return(hold(besselK)(v, z))
   end_if:

   // we cannot decide whether y is real or not
   return(hold(conjugate)(hold(besselK)(v, z)));
end_proc:


// end of file 
