/*++
  besselY -- bessel function of the second kind, order v and argument z
++*/

besselY:=
proc(v,z)
  local fv,fz,n,sv,sz,vv,zz,dummy;
begin
  if args(0) <> 2 then
    error("expecting two arguments")
  elif z::dom::besselY <> FAIL then
    return( z::dom::besselY(args()) )
  elif {DOM_BOOL,DOM_FAIL,DOM_NULL,DOM_NIL} intersect
       {domtype(v), domtype(z)} <> {} then
    error("invalid operand")
  end_if;

  // ----------  map to sets -------------------------
  // This is quite complicated, because both arguments have
  // to be handled
  case [type(v),type(z)]
    of [DOM_SET, DOM_SET] do
        return({besselY(sv, sz) $ sv in v $ sz in z});
    of [DOM_SET, "_union"] do
    of ["_union", DOM_SET] do
        return(_union((besselY(sv, sz) $ sv in v) $ sz in z));
    of ["_union", "_union"] do

        // besselY({a} union A, C union D)
        //  -> besselY({a},C) union besselY({a},D) union besselY(A,C) union besselY(A,D)
        // Make sure that A, C, D are interpreted as sets, i.e.,
        // besselY({a},C) --> besselY(a, C), not {besselY(a,C)} !

        [v, vv, dummy]:= split(v, testtype, DOM_SET):
        if type(vv) <> "_union" then vv:= [vv] end_if:
        [z, zz, dummy]:= split(z, testtype, DOM_SET):
        if type(zz) <> "_union" then zz:= [zz] end_if:

        return(_union({(besselY(sv, sz) $ sv in v ) $ sz in z},
                       (besselY(sv, sz) $ sv in vv) $ sz in z,
                       (besselY(sv, sz) $ sv in v ) $ sz in zz,
                       (besselY(sv, sz) $ sv in vv) $ sz in zz ));
  end_case;

  case type(v)
    of DOM_SET do
    of "_union" do
       // z cannot be a set, if v is a set
       return(map(v, besselY, z));
  end_case;

  case type(z)
    of DOM_SET do
    of "_union" do
       // v cannot be a set, if z is a set
       return(map(z, (z,v) -> besselY(v,z), v))
  end_case;
  // --------------------------------------------------
  if testtype(z, Type::Set) and not testtype(z, Type::Arithmetical) then
    if testtype(v, Type::Set) and not testtype(v, Type::Arithmetical) then
      return(Dom::ImageSet(besselY(#v, #z), [#z, #v], [z, v]));
    elif testtype(v, Type::Arithmetical) then
      return(Dom::ImageSet(besselY(v, #z), [#z], [z]));
    end_if;
  elif testtype(v, Type::Set) and not testtype(v, Type::Arithmetical) then
      return(Dom::ImageSet(besselY(#v, z), [#v], [v]));
  end_if;
  if map([v, z], testtype, Type::Arithmetical) <> [TRUE, TRUE] then
    error("arguments must be of type 'Type::Arithmetical'")
  end_if:
  // --------------------------------------------------

  if z=infinity then
        return(0);
  end_if:
  fv:= domtype(float(v)):
  fz:= domtype(float(z)):
  if (fv=DOM_FLOAT or fv=DOM_COMPLEX) and (fz=DOM_FLOAT or fz=DOM_COMPLEX)
   and has([map((op(v),op(z)),type)],DOM_FLOAT) then
         return(besselY::float(v,z))
  end_if:

  if iszero(z) then
        error("singularity"):
  end_if:

  if (domtype(v)=DOM_INT or 
      (domtype(v)=DOM_FLOAT and iszero(v-round(v)))) 
         and v < 0 then
        return((-1)^v*procname(-v,z));
  end_if:

  if not(type(v)=DOM_INT or
     (type(v) = DOM_FLOAT and iszero(v - round(v)))) 
     and (type(2*v)=DOM_INT or
                (type(2*v) = DOM_FLOAT and iszero(2*v - round(2*v)))) then
    // might want to limit this expansion to say abs(v) <= 10
    // and let "expand" take care of cases > 10.
    // Abramowitz and Stegun eq. 10.1.15
    n := v-1/2;
    return((-1)^(n+1) * besselJ(-v,z));
  end_if:
  procname(v,z)
end_proc:

besselY := prog::remember(besselY, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("besselY")]):

besselY := funcenv(besselY):
besselY::type := "besselY":
besselY::info :=
"besselY(v,z) -- bessel function of the second kind, order v and argument z":
besselY::print := "besselY":

besselY::float :=
    loadproc(besselY::float, pathname("STDLIB","FLOAT"), "besselY"):

besselY::series :=
    loadproc(besselY::series, pathname("SERIES"), "besselY"):

besselY::Content := stdlib::genOutFunc("CbesselY", 2):

besselY::MMLContent :=
(Out, data) -> specfunc::Bessel::MMLContent(Out, data, "Y"):

//================================================================
// There are 2 formulas for diff of the bessels:
// (1) diff(besselY(v,x), x) =  besselY(v-1,x) - v/x*besselY(v,x)
// (2) diff(besselY(v,x), x) = -besselY(v+1,x) + v/x*besselY(v,x)
// We choose the 'downward' formula (1), if v >= 1 or
// v = something symbolic + number with number >= 1.
// Otherwise, the 'upward' formula (2) is used.
//================================================================
besselY::diff := proc(e, x)
   local v, f, dfdx;
begin
   // We cannot differentiate w.r.t. the index (=op(e, 1)):
   if has(op(e,1),[args(2..args(0))]) then
      return(hold(diff)(args()))
   end_if:
   if args(0)>2 then
      return(diff(diff(e,x),args(3..args(0))));
   end_if:
   v:=op(e,1): // the index of besselY
   f:=op(e,2): // the argument of besselY
   dfdx:= diff(f, x):
   if iszero(dfdx) then
      return(0)
   end_if:
   // Switch between the 'upward' and the 'downward' formula to
   // avoid drifting towards high values of v when computing
   // high derivatives. For the decision, split v into
   //   v = symbolic/imaginary + Type::Real
   if besselJ::split_index(v)[2] > 0 then // downward
     return( besselY(v-1,f)*dfdx - v/f*besselY(v, f)*dfdx);
   else // upward
     return(-besselY(v+1,f)*dfdx + v/f*besselY(v, f)*dfdx);
   end_if;
end_proc:

besselY::simplify:= besselJ::simplify: // yes, besselJ::simplify!
besselY::Simplify:= besselY::simplify:
besselY::expand :=
    loadproc(besselY::expand, pathname("STDLIB","EXPAND"), "besselY"):

besselY::TeX :=
proc(b, ex, prio)
  local s;
begin
  s := generate::tex(op(ex, 2), output::Priority::Fconcat);
  if length(s) < 7 or s[1..7] <> "\\left(" then
    s := "\\left(".s."\\right)";
  end_if;
  _concat("Y_{",
       generate::TeX(op(ex,1)),
       "}",
       s);
end_proc:

besselY::conjugate:= proc(v, z)
local b;
begin
   b:= numeric::isnonzero(Im(besselY(v, z)));
   if b = FALSE then
      // y = besselY(v, z) is real
      return(hold(besselY)(v, z));
   elif b = TRUE then
      if numeric::isnonzero(Re(besselY(v, z))) = FALSE then
        // y = besselY(v, z) is on the imaginary axis
        return(-hold(besselY)(v, z));
      else
        // y = besselY(v, z) is somewhere in the complex plane
        return(hold(conjugate)(hold(besselY)(v, z)));
      end_if;
   end_if;

   if is(v in R_) = TRUE and
      is(z >= 0) = TRUE then // y is real
      return(hold(besselY)(v, z))
   end_if:

   // we cannot decide whether y is real or not
   return(hold(conjugate)(hold(besselY)(v, z)));
end_proc:


// end of file 
