/*++
        beta -- the Beta function

        beta( x, y ) := gamma( x ) * gamma( y ) / gamma( x + y )

++*/

beta :=
proc(x,y) 
  option noDebug;
  local isInteger, gx, gy, gxy, tx, ty, txy;
begin
  if args(0) <> 2 then
    error("expecting two arguments")
  elif x::dom::beta <> FAIL then
    return(x::dom::beta(args()))
  end_if;

  case type(x)
    of DOM_SET do
    of "_union" do
      return(map(x, beta, y))
  end_case;
  case type(y)
    of DOM_SET do
    of "_union" do
      return(map(y, beta, x))
  end_case;

  if not testtype(x, Type::Arithmetical) then
    if testtype(y, Type::Set) and not testtype(y, Type::Arithmetical) then
      return(Dom::ImageSet(beta(#x, #y), [#x, #y], [x, y]));
    else
      return(Dom::ImageSet(beta(#x, y), [#x], [x]));
    end_if;
    error("arguments must be of type 'Type::Arithmetical'")
  elif not testtype(y, Type::Arithmetical) then
    if testtype(y, Type::Set) then
      return(Dom::ImageSet(beta(x, #y), [#y], [y]));
    end_if;
    error("arguments must be of type 'Type::Arithmetical'")
  end_if;

  // If one argument is a float, then try to convert the
  // other one to a float as well:
  case domtype(x)
  of DOM_FLOAT do 
  of DOM_COMPLEX do
     if domtype(op(x,1))=DOM_FLOAT or
        domtype(op(x,2))=DOM_FLOAT
     then y:= float(y);
     end_if:
  end_case;
  case domtype(y)
  of DOM_FLOAT do 
  of DOM_COMPLEX do
     if domtype(op(y,1))=DOM_FLOAT or
        domtype(op(y,2))=DOM_FLOAT
     then x:= float(x);
     end_if:
  end_case;

  // return symbolic calls of beta with a sorted sequence
  // of arguments to guarantee beta(x,y)=beta(y,x):
  [x,y]:= sort([x,y]);

  isInteger:= proc(x) begin // x=integer or x=float(integer)?
    bool(domtype(x)=DOM_INT)    
    or (testtype(x, Type::Real) and iszero(frac(x)))
  end_proc:

  if iszero(x) or iszero(y) then error("singularity")
  elif iszero(x-1) then return(1/y)
  elif iszero(y-1) then return(1/x)
  elif isInteger(x) and   // x = integer or x = float(integer)
       isInteger(y) then  // y = integer or y = float(integer)
    if x + y <= 0
    then // can reflect negative argument to positive argument
       if x<0 and 0<y then return((-1)^y*beta(1-x-y,y)) end_if;
       if y<0 and 0<x then return((-1)^x*beta(x,1-x-y)) end_if;
       error("singularity");
    else 
       if (x<0 and 0<y) or (y<0 and 0<x) then 
          error("singularity")
       end_if:
    end_if:
  elif isInteger(x) and x <= 0 and testtype(y, Type::Numeric) then
       error("singularity")
  elif isInteger(y) and y <= 0 and testtype(x, Type::Numeric) then
       error("singularity")
  elif isInteger(x+y) and x+y <= 0 then
       return(0)
  end_if;

  // neither x nor y nor x+y is a negative integer:
  if testtype(x, Type::Numeric) and 
     testtype(y, Type::Numeric) then
     gx:= gamma(x): 
     tx:= type(gx);
     gy:= gamma(y): 
     ty:= type(gy);
     gxy:= gamma(x+y):
     txy:= type(gxy);
     if contains({tx, ty, txy}, DOM_INT) then
        if tx = "gamma" then
           gx:= expand(gx);
        end_if:
        if ty = "gamma" then
           gy:= expand(gy);
        end_if:
        if txy = "gamma" then
           gxy:= expand(gxy);
        end_if:
     end_if:
     return(gx*gy/gxy):
  else 
     return(procname(x, y))
  end_if:
end_proc:

beta := prog::remember(beta, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("beta")]):

beta := funcenv(beta):
beta::print:= "beta":
beta::info:=
    "beta -- the Beta function beta(x,y)=gamma(x)*gamma(y)/gamma(x+y)":
beta::type:= "beta":

beta::diff:=
  proc()
    local otherargs, op11, op12;
  begin 
    otherargs:= args(2..args(0));
    op12:= op(args(1), 2);
    op11:= op(args(1), 1);
    (diff(op11, otherargs)*psi(op11) + diff(op12, otherargs)*psi(op12) -
     diff(op11 + op12, otherargs) * psi(op11 + op12)) * beta(op11, op12)
  end_proc:

beta::expand:=
	proc(b)
     local x, y;
   begin
     [x, y]:= [op(b)];
     expand(gamma(x)*gamma(y)/gamma(x+y), args(2..args(0)))
   end_proc:

beta::float:= (a, b) -> beta(float(a), float(b)):

beta::hull := (x, y) -> DOM_INTERVAL::beta(hull(x), hull(y)):

beta::Content := stdlib::genOutFunc("Cbeta", 2):

// series(beta(f, g), x, n, dir, opt)
beta::series :=
  proc(f, g, x, n, dir, opt)
  begin
    combine(Series::series(gamma(f)*gamma(g)/gamma(f + g), x, n, dir, opt))
  end_proc:
