/*++
cosh -- the hyperbolic cosine

cosh(x)

x - expression
++*/

cosh:=
proc(x)
  option noDebug;
  local f, y;
begin
  if args(0) = 0 then error("no arguments given")
  elif x::dom::cosh <> FAIL then return(x::dom::cosh(args()))
  elif args(0) <> 1 then error("wrong no of args") 
  end_if;

  case type(x)
  of DOM_INT do
  of DOM_RAT do
     if x < 0 then return(cosh(-x)) end_if;
     break;
  of DOM_FLOAT do
     return(cosh::float(x));
  of DOM_COMPLEX do
     if domtype(op(x,1)) = DOM_FLOAT or
        domtype(op(x,2)) = DOM_FLOAT then
          return(cosh::float(x))
     end_if;
     break;
  of DOM_SET do
  of "_union" do
     return(map(x, cosh))
  of "_intersect" do
  of "_minus" do
      return(Dom::ImageSet(cosh(`#z`), `#z`, x))
  end_case;

  if not testtype(x,Type::Arithmetical) then
      /* generic handling of sets */
      if testtype(x, Type::Set) then
        return(Dom::ImageSet(cosh(#x), #x, x));
      end_if;
      error("argument must be of type 'Type::Arithmetical'")
  end_if;

  case type(x)
  of "_mult" do
     f:= op(x, nops(x));
     if testtype(f, Type::Real) then
       if f < 0 then return(cosh(-x)) end_if;
       elif type((f:=x/I/PI))=DOM_INT then return((-1)^f) // cosh(n*I*PI)= (-1)^n 
       elif type(2*f)=DOM_INT then return(0) // cosh( (n+1/2)*I*PI ) = 0 
     end_if;
     f:= x/PI/I;
     if contains({DOM_RAT, DOM_INT}, domtype(f)) then
        if f < 0 then return(cosh(-x)) end_if;
        if f < 1/2 then break end_if;
        if f < 1 then return(-cosh((1-f)*PI*I)) end_if;
        if f < 2 then return(cosh((2-f)*PI*I)) end_if;
        return(cosh((f-2*(floor(f) div 2))*PI*I));
     end_if;
     f:= x/I;
     if type(f) = "arg" then  // x = cosh(I*arg(y)) = I*Re(y)/abs(y)
        return(Re(op(f))/abs(op(f))):
     end_if:
     break;
  of "_plus" do
     // due to performance: move this code to cosh::simplify
     // if sysorder(-x,x)=TRUE then return(cosh(-x)) else break end_if

     // handle cosh(y + integer*PI) -> (-1)^integer*cosh(y)
     // We could also do cosh(y + PI/2) -> -sinh(y).
     // At the moment, we decide not to do it.
     if has(x, PI) then
        for y in x do
            f:= y/PI/I;
            if testtype(f, DOM_INT) and f<>0 then
               return((-1)^f*cosh(x-y))
            end_if;
        end_for:
     end_if:
     break;
  // cf Table 4.5.60 p. 85 in Abramowitz and Stegun 
  of "arccosh" do return(op(x))
  of "arcsinh" do return(sqrt(1+op(x)^2))
  of "arctanh" do return(1/(1-op(x))^(1/2)/(1+op(x))^(1/2))
  of "arccoth" do x:= 1/op(x):
     return(1/(1-x)^(1/2)/(1+x)^(1/2))
  end_case;

  procname(x)
end_proc:

cosh := prog::remember(cosh, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("cosh")]):

cosh(0):= 1:
cosh(I*PI):= -1:
cosh(I*PI/2):= 0:
cosh(I*PI/3):= 1/2:
cosh(I*PI/4):= (2^(1/2))/2:
cosh(I*PI/  5):= (5^(1/2)+1)/4:
cosh(I*PI*2/5):= (5^(1/2)-1)/4:
cosh(I*PI/6):= (3^(1/2))/2:
cosh(I*PI/  8):= ((2+2^(1/2))^(1/2))/2:
cosh(I*PI*3/8):= ((2-2^(1/2))^(1/2))/2:
cosh(I*PI  /10):= 2^(1/2)/4*(5+5^(1/2))^(1/2):
cosh(I*PI*3/10):= 2^(1/2)/4*(5-5^(1/2))^(1/2):
cosh(I*PI/  12):= (6^(1/2)+2^(1/2))/4: // simpler than ((2+3^(1/2))^(1/2))/2: 
cosh(I*PI*5/12):= (6^(1/2)-2^(1/2))/4: // = ((2-3^(1/2))^(1/2))/2:

cosh( infinity):= infinity:
cosh(-infinity):= infinity:

cosh:= funcenv(cosh, op(specfunc::cosh, 2)):
cosh::print := "cosh":
cosh::type := "cosh":
cosh::info := "cosh -- the hyperbolic cosine":
cosh::float := specfunc::cosh:

cosh::hull := DOM_INTERVAL::cosh@hull:

cosh::inverse := "arccosh":

cosh::diff :=
  proc(f)
    local op1;
  begin
    op1 := op(f, 1);
    diff(op1, args(2..args(0))) * sinh(op1)
  end_proc:

cosh::expand :=
    loadproc(cosh::expand, pathname("STDLIB","EXPAND"),"cosh"):

cosh::conjugate :=
    loadproc(cosh::conjugate, pathname("STDLIB","CONJ"),"cosh"):

cosh::rectform :=
    loadproc(cosh::rectform, pathname("STDLIB","RECTFORM"),"cosh"):

cosh::series := loadproc(cosh::series, pathname("SERIES"),"cosh"):

cosh::simplify :=
    loadproc(cosh::simplify, pathname("STDLIB","SIMPLIFY"),"cosh"):

cosh::Re := loadproc(cosh::Re, pathname("STDLIB","RE"),"cosh"):

cosh::Im := loadproc(cosh::Im, pathname("STDLIB","IM"),"cosh"):

cosh::"transform::laplace":=
    loadproc( cosh::"transform::laplace",
        pathname("TRANS","LAPLACE"), "L_cosh"):

cosh::Content := stdlib::genOutFunc("Ccosh", 1):

// end of file 
