/*

dawson(x): Dawson's integral

dawson(x) = exp(-x^2) * int(exp(t^2), t = 0..x)
          = 1/2 * sqrt(PI) * exp(-x^2) * erfi(x) <-- erfi nicht implementiert!
          = 1/2 * sqrt(PI) * exp(-x^2) * (-I) * erf(I*x)

dawson(-x) = -dawson(x)
conjugate(dawson(x)) = dawson(conjugate(x))

*/

dawson :=
proc(x)
  local domx;
begin
  if args(0) <> 1 then
    error("wrong number of arguments")
  elif x::dom::dawson <> FAIL then
    return(x::dom::dawson(args()))
  end_if:

  if not testtype(x, Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      if type(x)=Dom::ImageSet then
        return(map(x, dawson));
      else
        return(Dom::ImageSet(dawson(#x), #x, x));
      end_if;
    end_if;
    error("argument must be of 'Type::Arithmetical'")
  end_if:

  domx := domtype(x);
  if domx = DOM_FLOAT or
     (domx = DOM_COMPLEX and 
      (domtype(op(x,1)) = DOM_FLOAT or domtype(op(x,2)) = DOM_FLOAT)) then
    return(dawson::float(x))
  end_if;

  if stdlib::hasmsign(x) then
    -procname(-x)
  else
    procname(x)
  end_if;
end_proc:

dawson := prog::remember(dawson, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("dawson")]):

dawson(0) := 0:
dawson(infinity) := 0:
dawson(-infinity) := 0:
dawson(RD_INF) := float(0):
dawson(RD_NINF) := float(0):
dawson(RD_NAN) := RD_NAN:

dawson := funcenv(dawson):
dawson::print := "dawson":
dawson::info  := "dawson -- Dawson's integral":
dawson::type  := "dawson":

dawson::float :=
proc(x)
  local fx, tmp, c, y, s, m;
begin
  fx := float(x);
  if not contains({DOM_FLOAT, DOM_COMPLEX}, domtype(fx)) then
    return(hold(dawson)(fx))
  end_if;
  if fx = RD_INF or fx = RD_NINF then
    return(float(0));
  end_if;
  
  // Beware: if fx is real and large, erf(I*fx) is extremely 
  // slow. Then it will be multiplied by exp(..) = 0.0.
  // Stabilize, by using the asymptotic series
  // dawson(x) = 1/(2*x)
  //           + 1/(4*x^3) 
  //           + 3/8/x^5 
  //           + 15/16/x^7
  //           + O(1/x^10) 
  // for x = +- infinity:
  // In fact, this asymptotic expansion seems to hold for
  // |arg(x)| < PI/8:
  if specfunc::abs(Im(fx)) <= 0.99*specfunc::abs(Re(fx)) then
    if specfunc::abs(fx) > 0.7*10.0^(DIGITS/2) then
       return(1/2/fx);
    elif specfunc::abs(fx) > 0.9*10.0^(DIGITS/4) then
       return(1/2/fx + 1/4/fx^3);
    elif specfunc::abs(fx) > 1.1*10.0^(DIGITS/6) then
       return(1/2/fx + 1/4/fx^3 + 3/8/fx^5);
    else
       tmp:= exp(-fx^2):
       // If DIGITS is very large, we may need
       // many terms of the asymptotic expansion:
       if iszero(tmp) then
          c:= 1/2;
          y:= fx;
          s:= c/y;
          m:= 0;
          repeat
            m:= m + 1;
            c:= c*(2*m -1)/2;
            y:= y*fx^2;
            s:= s + c/y;
          until specfunc::abs(c/y) <= specfunc::abs(s)*10.0^(-DIGITS)
          end_repeat;
          return(s);
       else
          return(float(sqrt(PI)/2/I * tmp * erf(I*fx)));
       end_if:
    end_if:
  else // domtype(fx) = DOM_COMPLEX
     tmp:= exp(-fx^2):
     if iszero(tmp) then
       error("numerical exception (underflow/overflow)")
     else
       return(float(sqrt(PI)/2/I * tmp * erf(I*fx)));
     end_if:
  end_if:
end_proc:

dawson::diff :=
proc(f, x)
  local g;
begin
  g := op(f,1);
  return((1 - 2*g*dawson(g))*diff(g,x))
end_proc:



/*

Asymptotic expansion for +/-infinity is

 sum((2m)!/m!/(2*x)^(2m+1), m = 0..infinity)

 (derived from asymptotic expansion of erfc, 7.2.14 in M. Abramovitz and I.A. Stegun)

*/
dawson::series :=
proc(f, x, order, dir,opt)
  local t, d, s, c, m;
begin
  // recursively expand the argument
  t:=Series::series(f, x, order, dir,opt);

  if domtype(t) = Series::Puiseux then
    d := ldegree(t);
    if d < 0 then // f goes to infinity
      s := FAIL;
      if dir <> Undirected then
        s := limit(lmonomial(t), x, dir);
        if s = FAIL or type(s) = "limit" then
          s := FAIL
        elif domtype(s) = Dom::Interval then
          s := op(sign(s))
        else
          s := sign(s)
        end_if
      end_if;
      if s = 1 or s = -1 then // +/-infinity 
        c := 1/2;
/*
        s := [c];
        for m from 1 to (order-1) div 2 do
          c := c*(2*m-1)/2;
          s := s . [ 0, c ];
        end_for;
*/
        s:= [c, (c := c*(2*m-1)/2; 0, c) $ m = 1 .. ((order-1) div 2)]:
        s:=Series::Puiseux::create(1, 1, 2*((order-1) div 2)+3, s, x);
        s:=Series::Puiseux::_fconcat(s, 1/t);
        return(s)
      end_if;
    elif d > 0 then // f goes to zero
/*
      s := [1];
      for m from 1 to (order-1) div 2 do
        s := s . [ 0, (-1)^m*2^(2*m)*m!/(2*m+1)! ]
      end_for;
*/
      s:= [1, (0, (-1)^m*2^(2*m)*m!/(2*m+1)!) $ m = 1 .. ((order-1) div 2)]:
      s:=Series::Puiseux::create(1, 1, 2*((order-1) div 2)+3, s, x, 0, dir);
      s:=Series::Puiseux::_fconcat(s, t);
      return(s)
    end_if;
  end_if;

  // Produce a Taylor series if possible.
  Series::unknown(hold(dawson)(f), x, order,dir)
end_proc:


dawson::undefined := {}:
dawson::realDiscont := {}:
dawson::complexDiscont := {}:

dawson::conjugate :=
proc(x)
begin
  hold(dawson)(conjugate(x))
end_proc:

dawson::rectform :=
proc(xx)
  local x;
begin
  x := rectform::new(xx);
  if iszero(extop(x, 2)) and iszero(extop(x, 3)) then
    new(rectform, hold(dawson)(extop(x, 1)), 0, 0)
  elif iszero(extop(x, 1)) and iszero(extop(x, 3)) then
    new(rectform, 0, hold(dawson)(extop(x, 2)*I)/I, 0)
  else
    new(rectform, 0, 0, hold(dawson)(xx))
  end_if
end_proc:

// dawson() preserves sign along the real and imaginary axis,
// but is not sign-preserving in general
dawson::sign :=
proc(x)
  local s;
begin
  s := sign(op(x,1));
  if contains({0, 1, -1, I, -I}, s) then
    s;
  else
    hold(sign)(x);
  end_if;
end_proc:

