/* The dilogarithm function: dilog(x) = int(ln(t)/(1-t),t=1..x)

  Reference: Abramowitz & Stegun, Handbook of Mathematical Functions,
	section 27.7

related to to Li2(x) polylog(2, 1-x) = int(-log(1-t)/t,t=0..x)
Ref: L. Lewin, Dilogarithms and associated functions, Macdonald, London, 1958
     L. Lewin, Polylogarithms and associated functions, 1981

     Other notation ? polylog(n,x) == Li_n(x)
                      polylog(1,x) = -ln(1-x)
                      int( polylog(n,t)/t, t=0..x) = polylog(n+1,x)
                      int(ln(1-x)/x) = -polylog(2,x)
*/

dilog :=
proc(x)
  option noDebug;
  local fx;
begin
  if args(0) = 0 then
    error("no arguments given")
  elif x::dom::dilog<>FAIL then
    return(x::dom::dilog(args()))
  elif args(0)<>1 then
    error("wrong number of arguments")
  end_if;

  case type(x)
    of DOM_SET do
    of "_union" do
      return(map(x, dilog))
    of DOM_FLOAT do
      return(dilog::float(x)):
    of DOM_COMPLEX do
      if type(op(x,1)) = DOM_FLOAT or type(op(x,2)) = DOM_FLOAT then
        return(dilog::float(x)):
      end_if:
  end_case;
  
  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      if type(x)=Dom::ImageSet then
        return(map(x, dilog));
      else
        return(Dom::ImageSet(dilog(#x), #x, x));
      end_if;
    end_if;
    error("argument must be of 'Type::Arithmetical'")
  end_if;

  if testtype(x, Type::Numeric) then
  // Simplifications for exact numeric input: only return symbolic
  // calls with arguments in the right semi-circle Re(x)>=0, abs(x)<1
    fx:= float(x):
    if op(fx,1)<0  // i.e., Re(x) < 0
    then // reflect from Re(x)<0 to Re(x)>1
         return(PI^2/6 - dilog(1-x) - ln(x)*ln(1-x))
    end_if:
    if specfunc::abs(fx) > 1 
    then // use Landen-identity to reduce abs(x)>1 to abs(x)< 1
         // Note that this identity is valid for all arguments,
         // but has an exceptional form on the left real semi-axis:
         if testtype(x, Type::Real) and x < 0 
         then // this case should not arise, because Re(x)<0)
              // is reflected to Re(x)>1 above
              return(-dilog(1/x) - ln(x)^2/2 - 2*PI*I*ln(1-1/x) )
         else return(-dilog(1/x) - ln(x)^2/2 )
         end_if;
    end_if:
  end_if:

  procname(x)
end_proc:

dilog := prog::remember(dilog, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("dilog")]):

dilog(0):=PI^2/6:
dilog(1/2):=PI^2/12-ln(2)^2/2:
dilog(1):=0:
dilog(2):=-PI^2/12:
dilog(I):= PI^2/16 - I*CATALAN - I*PI/4*ln(2):
dilog(-I):= PI^2/16 + I*CATALAN + I*PI/4*ln(2):
dilog(1-I):= -PI^2/48 + I*CATALAN:
dilog(1+I):= -PI^2/48 - I*CATALAN:

dilog(infinity):=-infinity:
dilog(-infinity):=-infinity - I*infinity:
dilog(RD_INF):= RD_NINF:
dilog(RD_NINF):= RD_NINF + RD_NINF*I:
dilog(RD_NAN):= RD_NAN:

dilog:=funcenv(dilog):
dilog::print := "dilog":
dilog::info :=
  "dilog -- the dilogarithm function dilog(x) = int(ln(t)/(1-t),t=1..x)":
dilog::type := "dilog":

dilog::Content := stdlib::genOutFunc("Cdilog", 1):

dilog::diff :=
  proc()
    local op1;
  begin
    op1 := op(args(1));
    ln(op1)*diff(op1, args(2))/(1-op1)
 end_proc:

dilog::complexDiscont :=
    loadproc(dilog::complexDiscont, pathname("STDLIB","DISCONT"), "dilog"):
dilog::realDiscont := {}:
dilog::undefined := {}:

/* We only have a library routine for float(polylog). So
   use dilog(x) = polylog(2, 1-x), i.e. call
   polylog::float() provided by D. Duparc, 1998.05.31 */

dilog::float :=
  proc(z)
    local result;
  begin
    z:= float(z);
    if not has({DOM_FLOAT,DOM_COMPLEX}, domtype(z)) then
      return(hold(dilog)(z))
    end_if;
    if specfunc::abs(1-I-z)<1/2 or specfunc::abs(1+I-z)<1/2 then
      result:= float(PI^2/6-ln::float(z)*ln::float(1-z)) -
               polylog::float(2, z);
    else
      result:= polylog::float(2, 1-z);
    end_if;
    // Problem: for real z >= 0 the result is real. However,
    // the code above produces tiny imaginary round-off trash
    // for z > 2 (since ln(1 - z) is imaginary)
    // The zero imaginary part of the result is achieved
    // only via cancellation. Repair this by killing the
    // imaginary trash explicitly:
    if domtype(z) = DOM_FLOAT and z>=0 then
      return(Re(result));
    else return(result);
    end_if;
  end_proc:

dilog::series := loadproc(dilog::series, pathname("SERIES"), "dilog"):
dilog::rectform:= loadproc(dilog::rectform, pathname("STDLIB", "RECTFORM"),
                           "dilog"):
dilog::getprop := proc( xpr )
begin
  if is(op(xpr)>0, Goal=TRUE) then return( R_ ); end_if;
  C_
end_proc:
