/*++
        dirac -- the Dirac delta function

        dirac(x)
        dirac(x, n)   = n-th derivative of dirac(x)

        x - expression

	dirac(x) := diff(heaviside(x),x)
++*/

dirac :=
  proc(x)
    local n, X, a, b, s;
    option noDebug;
  begin
    if args(0) = 0 then error("no arguments given")
    elif x::dom::dirac <> FAIL then return(x::dom::dirac(args()))
    elif args(0) > 2 then error("expecting at most two arguments")
    end_if;

    n:= 0:
    if args(0) = 2 then
      n:= args(2);
      if n = 0 then
         return(dirac(x));
      end_if;
      if testtype(n, Type::Numeric) then
        if type(n) <> DOM_INT or n < 0 then
          error("second argument must be a nonnegative integer");
        end_if;
      end_if;
    end_if;

    case type(x)
      of DOM_SET do
      of "_union" do
         if args(0) = 1
            then return(map(x, dirac))
            else return(map(x, dirac, n))
          end_if;
    end_case;

    if domtype(x) = DOM_FLOAT then
       if iszero(x) then
          if args(0) = 1 then
            return(procname(float(0)));
          else
            return(procname(float(0), n));
          end_if;
       else return(float(0))
       end_if;
    end_if;

    if not testtype(x,Type::Arithmetical) then
      /* generic handling of sets */
      if testtype(x, Type::Set) then
        if type(x)=Dom::ImageSet then
          return(map(x, dirac, n));
        else
          return(Dom::ImageSet(dirac(#x, n), #x, x));
        end_if;
      end_if;
      error("argument must be of 'Type::Arithmetical'")
    end_if;

    if domtype(float(x))=DOM_COMPLEX then
      return(undefined);
    end_if;
    if (is(x<0) = TRUE or
        is(x>0) = TRUE or
        (is(x, Type::Real) = TRUE and
         is(x <> 0) = TRUE and
         bool(x <> float(0)) = TRUE )
       ) then
      return(0);
    end_if;
    // Dirac(a*X + b, n) = sign(a)/a^(n+1) * Dirac(X + b/a)
    // Identify symbolic X and numerical a, b
    // Use numeric::indets rather than indets, because
    // indets(2*x[1]+3) -> {x}, but one needs
    // numeric::indets(2*x[1]+3) -> {x[1]}
    // We can do this simplification only if there is
    // exactly one indeterminate X. Otherwise we do
    // not know, which indeterminate is to represent
    // the distribution variable:
    X:= numeric::indets(float(x)):
    if nops(X)=1 then
       X:= op(X);
       a:= diff(x, X):
       // Extract real numerical coefficients only.
       // Well, not has(a, I) is a bit crude ..
       if (not has(a, X)) and
          (not iszero(a)) and
          (not has(a, I)) then
          b:= subs(x, X=0, EvalChanges);
          // x = a * X + b;
          if n = 0 then
             return(sign(a)/a*procname(X + b/a));
          else
             return(sign(a)/a^(n+1)*procname(X + b/a, n));
          end_if;
       end_if;
    end_if;
    [s, x]:= stdlib::normalizesign(x);
    if s = 1 then
       return(procname(x, args(2..args(0))));
    else // s = - 1
       return((-1)^n*procname(x, args(2..args(0)))) ;
    end_if;
  end_proc:

dirac := prog::remember(dirac, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("dirac")]):

dirac:= funcenv(dirac):
dirac::type:= "dirac":
dirac::info:= "dirac -- the Dirac function dirac(x) = diff(heaviside(x),x)":
dirac::print:= "dirac":
dirac::float:= () -> dirac(float(args(1)), args(2..args(0))):

dirac::Content := stdlib::genOutFunc("Cdirac", 1, 2):

dirac::TeX :=
(d, data, prio) -> _concat("\\delta",
                           if nops(data) = 2 then
                             "^{(",
                             generate::tex(op(data, 2), output::Priority::Noop),
                             ")}"
                           end_if,
                           "\\!\\left(",
                           generate::tex(op(data, 1), output::Priority::Noop),
                           
                           "\\right)"):

dirac::diff:=
  proc(f, x)
    local y, n;
  begin
    y:= op(f, 1); // f=dirac(y, n)
    if nops(f)=1 then
      return(hold(dirac)(op(f,1), 1)*diff(y, x));
    else
      n:= op(f, 2);
      if has(n, x) then
        return(hold(diff)(f, x))
      end_if;
      return(hold(dirac)(y, n+1)*diff(y,x))
    end_if;
  end_proc:

/*
dirac::int:=
  proc(f,x)
    local t;
  begin
    t:=op(f);
    //if op(f)=x then heaviside(x) else FAIL end_if
    if not has(t-x,x) then
      heaviside(t)
    else
      FAIL
    end_if
  end_proc:
*/

dirac::simplify:= proc(f)
begin
  if type(f) = "dirac" and
     is(op(f, 1) <> 0) = TRUE then
        return(0)
  end_if;
  f;
end_proc:

dirac::Simplify:= dirac::simplify:

dirac::series:= loadproc(dirac::series, pathname("SERIES"), "dirac"):

dirac::sign:=
  proc(x)
  begin
    case is(op(x,1)=0)
      of TRUE do
        return(1)
      of FALSE do
        return(0)
      of UNKNOWN do
        return(hold(sign)(x))
    end_case;
    error("Unexpected result returned by 'is'")
  end_proc:

dirac::"transform::laplace":= loadproc(dirac::"transform::laplace",
                                       pathname("TRANS","LAPLACE"), "L_dirac"):

dirac(0) := hold(dirac)(0):

// end of file
