ellipticE:=
proc()
begin
  case args(0)
  of 1 do
    return(ellipticE::complete(args()));
  of 2 do
    return(ellipticE::incomplete(args()));
  otherwise
    error("expecting 1 or 2 arguments");
  end;
end_proc:

ellipticE := prog::remember(ellipticE, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("ellipticE")]):


ellipticE:=funcenv(ellipticE):
ellipticE::print:="ellipticE":
ellipticE::type:="ellipticE":
ellipticE::info:="ellipticE -- elliptic integral of the 2nd kind":


ellipticE::Content := stdlib::genOutFunc("CellipticE", 1, 2):

ellipticE::undefined:={}:

ellipticE::discont:=
proc(f,x,F)
begin
  case nops(f)
  of 1 do
    if args(0) > 2 and F=Real then
      return({});
    else
      return(Dom::Interval(1, infinity));
    end_if;
  of 2 do
    return(discont(sqrt(1-op(f,2)*sin(op(f,1))^2), x, F));
  otherwise
    return(FAIL);
  end_case;
end_proc:

ellipticE::complete:=
proc(m)
  local i;
begin
  case type(m)
  of DOM_FLOAT do
    return(ellipticE::float(m));

  of DOM_COMPLEX do
    if domtype(op(m,1))=DOM_FLOAT or domtype(op(m,2))=DOM_FLOAT then
      return(ellipticE::float(m));
    end_if;
    break;
		
  of DOM_SET do
  of "_union" do
    return(map(m, ellipticE));

  of "_plus" do
    for i from 1 to nops(m) do
      if op(m,i)=1 then
        return(ellipticCE(-subsop(m, i=0)));
      end_if;
    end_for;
    break;
  end_case;

  if not testtype(m,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(m, Type::Set) then
      if type(m)=Dom::ImageSet then
        return(map(m, ellipticE::complete));
      else
        return(Dom::ImageSet(ellipticE::complete(#m), #m, m));
      end_if;
    end_if;
    error("argument must be of 'Type::Arithmetical'")
  end_if;

  return(hold(ellipticE)(m)):
end_proc:

ellipticE::complete(0):=PI/2:
ellipticE::complete(1):=1:


ellipticE::incomplete:=proc(phi, m)
local s;
begin
  if iszero(phi) then
    return(0);
  elif iszero(m) then
    return(phi);
  elif iszero(m-1) and is(Re(phi)<=PI/2)=TRUE and is(Re(phi)>=-PI/2)=TRUE then
    return(sin(phi));
  end;

  s:=phi*2/PI;
  if testtype(s, Type::Integer)=TRUE then
    return(s*ellipticE(m));
  end;
 
  if type(phi)=DOM_FLOAT or (type(phi)=DOM_COMPLEX and (type(op(phi,1))=DOM_FLOAT or type(op(phi,2))=DOM_FLOAT)) or
     type(m)=DOM_FLOAT or (type(m)=DOM_COMPLEX and (type(op(m,1))=DOM_FLOAT or type(op(m,2))=DOM_FLOAT)) then
    return(ellipticE::float(phi,m));
  end_if;
 
  if not testtype(phi,Type::Arithmetical) then
    if testtype(phi, Type::Set) then
      if testtype(m, Type::Set) and not testtype(m,Type::Arithmetical) then
        return(Dom::ImageSet(ellipticE::incomplete(#phi, #m), [#phi, #m], [phi, m]));
      else
        return(Dom::ImageSet(ellipticE::incomplete(#phi, m), [#phi], [phi]));
      end_if;
    end_if;

    error("first argument must be of 'Type::Arithmetical'")
  end_if;
  if not testtype(m,Type::Arithmetical) then
    if testtype(m, Type::Set) then
      return(Dom::ImageSet(ellipticE::incomplete(phi, #m), [#m], [m]));
    end_if;

     error("second argument must be of 'Type::Arithmetical'")
  end_if;

  return(hold(ellipticE)(phi,m));
end:

//----------------------------------------------------

ellipticE::diff:=proc(f)
local phi, m;
begin
  case nops(f)
  of 1 do // complete elliptic integral
    m:=op(f, 1);
    return(diff(m, args(2..args(0))) * (ellipticE(m)/(2*m) - ellipticK(m)/(2*m)));
  of 2 do // incomplete elliptic integral
    phi:=op(f, 1);
    m:=op(f, 2);
    return(
      diff(phi, args(2..args(0))) * sqrt(1-m*sin(phi)^2) +
      diff(m, args(2..args(0))) * (ellipticE(phi,m)/2/m-ellipticF(phi,m)/2/m));
  otherwise
    error("expecting 1 or 2 arguments");
  end;
end_proc:


//----------------------------------------------------

ellipticE::series:=loadproc(ellipticE::series, pathname("SERIES"), "ellipticE"):

//----------------------------------------------------

ellipticE::recursiveKE:=proc(m, epsilon)
local k1, k1p1, KE;
begin
  if specfunc::abs(m)<epsilon then
     return(float(PI * (1/2 + m*(1/8 + m*(9/128 + m*(25/512 + m*1225/32768))))),
            float(PI * (1/2 - m*(1/8 + m*(3/128 + m*( 5/512 + m* 175/32768))))));
  else
    k1:=sqrt(1-m);
    k1p1:= 1 + k1;
    KE:=ellipticE::recursiveKE(m^2/k1p1^4, epsilon);

    return( KE[1]*2/k1p1,
            KE[2]*k1p1 - KE[1]*2*k1/k1p1 );
  end_if;
end_proc:


// Carlson's elliptic integral of the 2nd kind
// See note on ellipticF::RF in ellipticF.mu.
ellipticE::RD:=proc(x0,y0,z0)
local x,y,z,xx,yy,zz,A,A0,Q,P,lambda,alpha,E2,E3,E4,E5;
begin
  if Re(x0)<0 then x0:=-x0; end;
  if Re(y0)<0 then y0:=-y0; end;
  if Re(z0)<0 then z0:=-z0; end;

  if iszero(x0) and iszero(y0) then
    error("singularity");
  end;

  A0:=(x0^2+y0^2+3*z0^2)/5;

  x:=x0;
  y:=y0;
  z:=z0;
  A:=A0;

  Q:=1.5 * float(10^(DIGITS/6)) * max(specfunc::abs(A-x^2), specfunc::abs(A-y^2), specfunc::abs(A-z^2));

  alpha:=0;
  P:=1;

  while P*specfunc::abs(A)<Q do
    lambda:=x*y + x*z + y*z;

    alpha:=alpha + 1/P/z/(z^2+lambda);

    A:=(A+lambda)/4;
    x:=sqrt(x^2+lambda)/2;
    y:=sqrt(y^2+lambda)/2;
    z:=sqrt(z^2+lambda)/2;

    P:=P*4;
  end;

  xx:=(A0-x0^2)/P/A;
  yy:=(A0-y0^2)/P/A;
  zz:=-(xx+yy)/3;
  
  E2:=xx*yy-6*zz^2;
  E3:=(3*xx*yy-8*zz^2)*zz;
  E4:=3*(xx*yy-zz^2)*zz^2;
  E5:=xx*yy*zz^3;

  return((1 - 3/14*E2 + E3/6 + 9/88*E2^2 - 3/22*E4 - 9/52*E2*E3 + 3/26*E5) * A^(-3/2) / P + 3*alpha);
end_proc:


ellipticE::float:=proc()
local phi, m, ln_m, sqrt_m, cs, sn, k, EE;
begin
  if args(0)<>1 and args(0)<>2 then
    error("expecting 1 or 2 arguments");
  end;
  
  m:=float(args(1));

  if type(m)<>DOM_FLOAT and type(m)<>DOM_COMPLEX then
    return(hold(ellipticE)(args()));
  end;

  if args(0)=1 then // complete elliptic integral
    if iszero(m-1) then
      return(float(1));
    elif specfunc::abs(m) > float(10^DIGITS) then
      ln_m:=ln(-16*m);
      sqrt_m:=(-m)^(1/2);

      return(sqrt_m + (ln_m+1)/(4*sqrt_m));
    end;

    return(ellipticE::recursiveKE(m, float(10^(-DIGITS/5)))[2]);
  end;

  phi:=m;
  m:=float(args(2));
  
  if type(m)<>DOM_FLOAT and type(m)<>DOM_COMPLEX then
    return(hold(ellipticE)(args()));
  end;

  if iszero(phi) then
    return(float(0));
  end;

  cs:=cos(phi);
  sn:=sin(phi);

  k:=round(Re(phi)/PI);
  EE:=sn*(ellipticF::RF(cs, sqrt(1-m*sn^2), 1) - m/3*sn^2*ellipticE::RD(cs, sqrt(1-m*sn^2), 1));
  
  if k=0 then
    return(EE);
  elif modp(k,2)=0 then
    return(2*k*ellipticE::float(m) + EE);
  else
    return(2*k*ellipticE::float(m) - EE);
  end;
end_proc:


