/*++
exp -- the exponential function

exp(x)

x - expression
++*/

exp:=
proc(x)
  local y, lny, c, i, n;
  option noDebug;
begin
  if args(0) = 0 then error("expecting one argument")
  elif x::dom::exp <> FAIL then return(x::dom::exp(args()))
  elif args(0) <> 1 then error("expecting one argument")
  end_if;

  case type(x)
    of DOM_SET do
    of "_union" do
      return(map(x, exp))
  end_case;

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      return(Dom::ImageSet(exp(#x), #x, x));
    end_if;

     error("argument must be of 'Type::Arithmetical'")
  end_if;

  case type(x)
    of DOM_FLOAT do
      return(exp::float(x));

    of DOM_COMPLEX do
      if domtype(op(x,1)) = DOM_FLOAT or
         domtype(op(x,2)) = DOM_FLOAT
      then return(exp::float(x))
      end_if;
      break;

    of DOM_INTERVAL do
      return(exp::hull(x));

    of "_mult" do
      // handle case exp(y*PI*I) for rational y
      y:=x/PI/I;
      if testtype(y, DOM_INT) then
        return((-1)^y)
      end_if;
      if testtype(y, DOM_RAT) then
        if y>=1 or y<0 then
          // reduce to case exp(y*PI*I) with 0<=y<1
          /* recursive call is necessary to use the
            predefined values below */
          y:=2*frac(y/2); // this is >=0 and < 2, and its difference to the
                         // previous y is a multiple of 2
          if %<1 then
            return(exp(PI*I*%))
          else
            // exp(y*PI*I) = exp(-PI*I)*exp((y-1)*PI*I)
            //             =   (-1)    *exp((y-1)*PI*I)
            return(-exp(PI*I*(%-1)))
          end_if;
        else
          // do not determine the function value except for the
          // predefined cases below
          break
        end_if
      end_if;

      // exp(ln(y)*Constant) = y^Constant
      if has(x, hold(ln)) then
         // split x = ln(y)*Constant
         lny:= select(x, testtype, "ln");
         // select can return 1 --> test again
         if testtype(lny, "ln") then
            // x = ln(y)* Constant ??
            c:= x/lny;
            if testtype(c, Type::Constant) then
               return(op(lny,1)^c)
            end_if;
         end_if;
      end_if:

      // exp(I*arg(z)*n) = sign(z)^n, n in Z_
      if hastype(x, "arg") then 
        if type(x/I) = "arg" then // x = I*arg(y)
           y:= op(x/I);   
         //return(sign(y));
           return(y/abs(y));
        elif type(x/I) = "_mult" then
           n:= op(x/I, nops(x/I));
           if testtype(n, Type::Real) and
              type(x/I/n) = "arg" then // x = n*I*arg(y)
              y:= op(x/I/n);              
         //   return(sign(y)^n);
              return(y^n/abs(y)^n);
           end_if;
        end_if;
      end_if;

      break;

    of "_plus" do
      // handle exp(y + integer*PI*I)
      if has(x, PI) and has(x, I) then
         for y in x do
             if testtype(y/2/PI/I, DOM_INT) then
                return(exp(x-y))
             end_if;
             if testtype(y/PI/I, DOM_INT) then
                return(-exp(x-y))
             end_if;
         end_for:
      end_if:

      // look for the 1st term in the sum that is
      // of the form I*arg(y) or I*arg(y)*real_factor,
      // split this off and simplify this factor using
      // exp(arg(y)*I*c) = sign(y)^c, c in R_:
      for i from 1 to nops(x) do
        if hastype(op(x, i), "arg") then
           y:= op(x, i)/I;
           if type(y) = "arg" then 
              x:= eval(subsop(x, i = null()));
              y:= op(y);
            //return(sign(y)*exp(x));
              return(y/abs(y)*exp(x));
           elif type(y) = "_mult" then
              n:= op(y, nops(y));
              if testtype(n, Type::Real) and
                 type(y/n) = "arg" then
                 x:= eval(subsop(x, i = null()));
                 y:= op(y/n);
              // return(sign(y)^n*exp(x));
                 return(y^n/abs(y)^n*exp(x));
              end_if;
            end_if;
        end_if;
      end_for;

      break;

    of "ln" do
      return(op(x,1));
    of "lambertW" do
      assert(nops(x) = 2);
      // W(c, y) * exp(W(c, y)) = y
      // we may divide this by W(c, y) unless c=y=0
      // if the denominator is singular (and this has not been recognized),
      // we remove the singularity; this works since 0/lambertw(1,0) evaluates
      // to zero
      [c, y]:= [op(x)];
      // we have to introduce simplifyCondition here because
      // is(2*exp(x) = 0) does not return FALSE :-(
      case simplify::simplifyCondition(c = 0 and y = 0)
        of TRUE do
          return(1)
        of FALSE do
          return(y/x)
      end_case;
  end_case;

  procname(x)
end_proc:

exp := prog::remember(exp, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("exp")]):

// standard simplifications

exp(infinity):= infinity:
exp(-infinity):= 0:
exp(I*infinity):= undefined:
exp(-I*infinity):= undefined:
exp(I*infinity + infinity) := undefined: // oscillates between -inf and inf
exp(-I*infinity + infinity):= undefined:
exp(-I*infinity -infinity):= 0:
exp(I*infinity -infinity ):= 0:
exp(0):= 1:
exp(I*PI):= -1:

exp(I*PI  /2):=  I:

exp(I*PI  /3):= 1/2+ 1/2*I*3^(1/2):
exp(I*PI*2/3):= -1/2 + 1/2*I*3^(1/2):

exp(I*PI  /4):= (1+I)/2^(1/2):
exp(I*PI*3/4):= -(1-I)/2^(1/2):

exp(I*PI  /5):=  (5^(1/2)+1)/4 + (5-5^(1/2))^(1/2)*2^(1/2)/4*I:
exp(I*PI*2/5):=  (5^(1/2)-1)/4 + (5+5^(1/2))^(1/2)*2^(1/2)/4*I:
exp(I*PI*3/5):= (-5^(1/2)+1)/4 + (5+5^(1/2))^(1/2)*2^(1/2)/4*I:
exp(I*PI*4/5):= -(5^(1/2)+1)/4 + (5-5^(1/2))^(1/2)*2^(1/2)/4*I:

exp(I*PI   /6):=  3^(1/2)/2 + I/2:
exp(I*PI* 5/6):= -3^(1/2)/2 + I/2:

exp(I*PI   /8):= (2+2^(1/2))^(1/2)/2 + (2-2^(1/2))^(1/2)/2*I:
exp(I*PI* 3/8):= (2-2^(1/2))^(1/2)/2 + (2+2^(1/2))^(1/2)/2*I:
exp(I*PI* 5/8):=-(2-2^(1/2))^(1/2)/2 + (2+2^(1/2))^(1/2)/2*I:
exp(I*PI* 7/8):=-(2+2^(1/2))^(1/2)/2 + (2-2^(1/2))^(1/2)/2*I:

exp(I*PI   /10):=  (5+5^(1/2))^(1/2)*2^(1/2)/4 + (5^(1/2)-1)/4*I:
exp(I*PI* 3/10):=  (5-5^(1/2))^(1/2)*2^(1/2)/4 + (5^(1/2)+1)/4*I:
exp(I*PI* 7/10):= -(5-5^(1/2))^(1/2)*2^(1/2)/4 + (5^(1/2)+1)/4*I:
exp(I*PI* 9/10):= -(5+5^(1/2))^(1/2)*2^(1/2)/4 + (5^(1/2)-1)/4*I:

exp(I*PI   /12):=  (1-I)/4*2^(1/2) + (1+I)/4*6^(1/2):
exp(I*PI* 5/12):= -(1-I)/4*2^(1/2) + (1+I)/4*6^(1/2):
exp(I*PI* 7/12):=  (1+I)/4*2^(1/2) - (1-I)/4*6^(1/2):
exp(I*PI*11/12):= -(1+I)/4*2^(1/2) - (1-I)/4*6^(1/2):

// function attributes

exp:= funcenv(exp, op(specfunc::exp, 2)):
exp::print:= "exp":
exp::info:= "exp -- the exponential function":
exp::type:= "exp":
exp::float:=specfunc::exp:

exp::hull:= DOM_INTERVAL::exp@hull:
exp::inverse:= "ln":

exp::minprop:= hold(Type::NonZero):
exp::getprop := proc(xpr, options)
  local res;
begin
  res := property::_getpropRec(op(xpr), options);
  case type(res)
    of solvelib::BasicSet do
      if res=C_ then return( C_ minus {0} ); end_if;
      return( Dom::Interval( 0, infinity ) );
    of Dom::ImageSet do
      return( Dom::ImageSet::map( res, exp ) );
    of Dom::Interval do
      return( Dom::Interval::exp( res ) );
  end_case;
  C_ minus {0};
end_proc:

exp::abs:= x -> exp(Re(x)):
exp::sign:=     loadproc(exp::sign,      pathname("SPECFUNC","SIGN"),   "exp"):
exp::expand:=   loadproc(exp::expand,    pathname("STDLIB","EXPAND"),   "exp"):
exp::conjugate:=
                loadproc(exp::conjugate, pathname("STDLIB","CONJ"),     "exp"):
exp::rectform:= loadproc(exp::rectform,  pathname("STDLIB","RECTFORM"), "exp"):
exp::simplify:= loadproc(exp::simplify,  pathname("STDLIB","SIMPLIFY"), "exp"):
exp::series:=   loadproc(exp::series,    pathname("SERIES"),            "exp"):
exp::Re:=       loadproc(exp::Re,        pathname("STDLIB","RE"),       "exp"):
exp::Im:=       loadproc(exp::Im,        pathname("STDLIB","IM"),       "exp"):
exp::diff:=     loadproc(exp::diff,      pathname("STDLIB","DIFF"),     "exp"):

exp::"transform::laplace":=
    loadproc( exp::"transform::laplace",
        pathname("TRANS","LAPLACE"), "L_exp"):

// typesetting and MathML generation
exp::MMLContent:= (Out, arg1) ->
               if nops(arg1) <> 1 then
                 return(Out::stdFunc(arg1));
               elif op(arg1) = 1/2 then
                 Out::Capply(Out::Croot, Out(exp(1)))
               elif op(arg1) = 1 then
                 Out::Cexponentiale()
               elif generate::isNeg(op(arg1)) then
                 Out(hold(_invert)(hold(exp)(-op(arg1))))
               else
                 Out::Capply(Out::Cexp, Out(op(arg1)))
               end_if:

exp::Content:= proc(Out, arg1)
                 local power;
               begin
                 if nops(arg1) <> 1 then
                   return(Out::stdFunc(arg1));
                 elif op(arg1) = 1/2 then
                   Out::Capply(Out::Croot, Out(exp(1)))
                 elif op(arg1) = -1/2 then
                   Out::Capply(Out::Cdivide, Out(1),
                               Out::Capply(Out::Croot, Out(exp(1))))
                 elif op(arg1) = 1 then
                   Out::Cexponentiale()
                 else
                   // use all _power-beautifications
                   power := Out(hold(_power)(#e, op(arg1)));
                   if Out::typString(power) = "power" or Out::typString(power) = "powerBigExpo" then
                     Out::Capply(Out::Cexp, op(power, 2));
                   elif generate::isNeg(op(arg1)) then
                     Out(hold(_invert)(hold(exp)(-op(arg1))))
                   else
                     Out::Capply(Out::Cexp, Out(op(arg1)))
                   end_if;
                 end_if;
               end_proc:
exp::findRelations:= 
proc(subst: DOM_SET, options: DOM_TABLE)
  local l;
begin
  subst:= select(subst, equ -> _lazy_and(type(op(equ, 2)) = "exp", not iszero(op(equ, [2, 1]))));
  // in {X1 = exp(a1), X2 = exp(a2), ...}, replace every exp(ai) by #E()^ai 
  // note that we take #E() instead of #E in order to produce an irrational object
  subst:= map(subst, equ -> op(equ, 1) = #E() ^ op(equ, [2, 1]));
  l:= [rationalize::findPowerRelationsBaseFixed(#E(), subst, options)];
  l:= subs(l, #E() = exp(1), EvalChanges);
  l[1], l[2], {}
end_proc:  
               
               
exp::TeX :=
  proc(e, data, priority)
    local expo;
  begin
    expo := op(data, 1);
    if expo = 1/2 then
      "\\sqrt{\\mathrm{e}}"
    elif expo = 1 then
      "\\mathrm{e}"
    else
      generate::TeXquote("\\mathrm{e}^{".
                         generate::tex(expo, output::Priority::Noop)."}",
                         priority, output::Priority::Power);
    end_if;
  end_proc:
// end of file
