/*++
fresnelC -- the Fresnel cosine function

fresnelC(x) = int( cos(PI/2*t^2), t = 0.. x)
            = (1 - I)/4* (    erf(sqrt(PI)/2*(1+I)*x)
                          + I*erf(sqrt(PI)/2*(1-I)*x));

Properties:
  fresnelC(-x) = -fresnelC(x)
  fresnelC(conjugate(x)) = conjugate(fresnelC(x))

Parameters:
  x - an arithmetical expression

TODO: there is no hull attribute
++*/

fresnelC:=
proc(x)
  option noDebug;
  local s;
begin
  if args(0) <> 1 then 
     error("expecting one argument")
  end_if;
  if x::dom::fresnelC <> FAIL then 
     return(x::dom::fresnelC(args()));
  end_if;

  case type(x)
    of DOM_FLOAT do
      return(fresnelC::float(x));
    of DOM_COMPLEX do
      if domtype(op(x,1)) = DOM_FLOAT or
         domtype(op(x,2)) = DOM_FLOAT then
        return(fresnelC::float(x))
      end_if;
      break;
   of DOM_INT do
   of DOM_RAT do
      s:= specfunc::sign(x);
      return(s*procname(s*x));
/*
   of DOM_INTERVAL do
      return(fresnelC::hull(x))
*/
    of DOM_SET do
    of "_union" do
      return(map(x, fresnelC))
  end_case;

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      if type(x)=Dom::ImageSet then
        return(map(x, fresnelC));
      else
        return(Dom::ImageSet(fresnelC(#x), #x, x));
      end_if;
    end_if;

    error("argument must be of 'Type::Arithmetical'")
  end_if;

  procname(args())
end_proc:

fresnelC := prog::remember(fresnelC, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("fresnelC")]):

fresnelC(0):= 0:
fresnelC(infinity):= 1/2:
fresnelC(-infinity):= -1/2:
fresnelC(I*infinity):= I/2:
fresnelC(-I*infinity):= -I/2:

fresnelC:= funcenv(fresnelC):
fresnelC::print := "fresnelC":
fresnelC::type := "fresnelC":
fresnelC::info := "fresnelC -- the Fresnel cosine integral":

fresnelC::float := proc(x)
  local y, r, fPI;
  begin
     x:= float(x):
     if contains({DOM_FLOAT, DOM_COMPLEX}, domtype(x)) and
        specfunc::abs(x) <= 10.0^(-DIGITS/5) then
           // numerical stabilization:
           return(x);
     end_if;
     fPI:= float(PI):
     y:= fPI^(1/2)*x/2:
     if domtype(y) = DOM_FLOAT then
        r := (1 - I)/4*(erf((1 + I)*y) + I*erf((1 - I)*y));
        return(op(r, 1)); // ignore imginary trash
     end_if;
     if domtype(y/I) = DOM_FLOAT then
        r := (1 - I)/4*(erf((1 + I)*y) + I*erf((1 - I)*y));
        return(I*op(r/I, 1)); // ignore real trash
     end_if;
     if domtype(y) = DOM_COMPLEX then
        r := (1 - I)/4*(erf((1 + I)*y) + I*erf((1 - I)*y));
        return(r);
     end_if;
     hold(fresnelC)(x);
  end_proc:

//------------------------------
fresnelC::conjugate:= proc(x)
 begin
   x:= conjugate(x);
   if type(x) = "conjugate" then
      hold(fresnelC)(x)
   else
      fresnelC(x)
   end_if;
end_proc:

//-------------------------------------------
// fresnelC(z) is real for z on the real line.
// fresnelC(z) is imaginary for z on the imaginary axis.
//-------------------------------------------
fresnelC::Re := proc(z)
  begin
   if is(z in R_) = TRUE then
      return(hold(fresnelC)(z))
   end_if;
   if is(z/I in R_) = TRUE then
      return(0):
   end_if;
   return(hold(Re)(hold(fresnelC)(z)));
end_proc:

fresnelC::Im := proc(z)
  begin
   if is(z in R_) = TRUE then
      return(0)
   end_if;
   if is(z/I in R_) = TRUE then
      return(fresnelC(z/I))
   end_if;
   return(hold(Im)(hold(fresnelC)(z)));
end_proc:

fresnelC::sign := proc(z)
  begin
   // z == fresnelC(z)
   z:= op(z, 1):
   if is(z > 0) = TRUE then
      return(1)
   end_if;
   if is(z < 0) = TRUE then
      return(-1)
   end_if;
   if is(z/I > 0) = TRUE then
      return(I)
   end_if;
   if is(z/I < 0) = TRUE then
      return(-I)
   end_if;
   return(hold(sign)(hold(fresnelC)(z)));
end_proc:

//------------------------------
fresnelC::simplify := proc(a)
local s, x;
begin
  x:= simplify(op(a)); // a = fresnelC(x)
  [s, x]:= stdlib::normalizesign(x);
  return(s*hold(fresnelC)(x));
end_proc:

//------------------------------
fresnelC::Simplify := fresnelC::simplify:

//------------------------------
fresnelC::diff := proc(f, x)
local y;
begin
  y:= op(f):
  cos(PI/2*y^2)*diff(y, x);
end_proc:

/*------------------------------
fresnelC::inverse := "???":
------------------------------*/
fresnelC::undefined := {}:
fresnelC::realDiscont := {}:
fresnelC::complexDiscont := {}:

/* ----------------------------------------------------------- 
  The following version traces series back to erf.
   Problem: erf cannot be expanded around z = (1 +-I)*infinity
            With finite expansion points x0, the zero order term
            is not frsenelC(x0), but involves(erf((1+-I)*z0)!

fresnelC::series := proc(y, x) // series(fresnelC(y), x)
local z;
begin
  z:= sqrt(PI)/2*y;
  series((1 - I)/4*(erf((1 + I)*z) + I*erf((1 - I)*z)), args(2..args(0)));
end_proc:
--------------------------------------------------------------*/
// we need to reimplement the series attribute:

fresnelC::series:= loadproc(fresnelC::series, pathname("SERIES"), "fresnelC"):

fresnelC::Content := stdlib::genOutFunc("CfresnelC",1):

// end of file 
