/*++
The incomplete Gamma function

igamma(a, x) = int( exp(-t) * t^(a-1), t=x..infinity )

a, x - expressions

Properties:

 igamma(  a , x) = x^(a-1)*exp(-x) + (a-1) * igamma(a-1, x);

 igamma( -1 , x) = Ei(2, x)/x = -Ei(1,x) + exp(-x)/x;
 igamma(-1/2, x) = -2 * sqrt(PI) * erfc(sqrt(x)) + 2*exp(-x)/sqrt(x);
 igamma(  0 , x) = Ei(1, x);
 igamma( 1/2, x) = sqrt(PI) * erfc(sqrt(x));
 igamma(  1 , x) = exp(-x);

 igamma(a, infinity) = 0

++*/

igamma :=
proc(a, x)
  option noDebug;
  local ceila, floora, fraca, t1, t2, am1, k, absx, result;
  save DIGITS;
begin 
  if args(0) < 2 then error("2 arguments expected"); end_if;
  if a::dom::igamma <> FAIL then return(a::dom::igamma(args())) end_if;
  if x::dom::igamma <> FAIL then return(x::dom::igamma(args())) end_if;

  case type(x)
    of DOM_SET do
    of "_union" do
      return(map(x, u-> igamma(a,u)))
  end_case;

  if not testtype(a,Type::Arithmetical) then
    if testtype(a, Type::Set) then
      if testtype(x, Type::Set) and not testtype(x, Type::Arithmetical) then
        return(Dom::ImageSet(igamma(#a, #x), [#a, #x], [a, x]));
      else
        return(Dom::ImageSet(igamma(#a, x), [#a], [a]));
      end_if;
    end_if;
    error("first argument must be of 'Type::Arithmetical'");
  end_if;

  if not testtype(x,Type::Arithmetical) then
    if testtype(x, Type::Set) then
      return(Dom::ImageSet(igamma(a, #x), [#x], [x]));
    end_if;

    error("second argument must be of 'Type::Arithmetical'")
  end_if;

  //-----------------------------------------------
  // special cases 
  if x = infinity then
     if a <> infinity then
          return(0)
     else // e.g., igamma(x, x) -> infinity as x -> infinity
          return(procname(args()));
     end_if;
  end_if;
  if x=0 then return(gamma(a)) end_if;
  if a=0 then return(Ei(1, x)) end_if;
  if a=1 then return(exp(-x)) end_if;
  if x=float(0) then return(gamma(float(a))) end_if;
  if a=float(0) then return(Ei(1, float(x))) end_if;
  if a=float(1) then return(exp(-float(x))) end_if;

  // igamma::float handles a=1/2 only for real x>0,
  // so use erfc for float-evaluation
  if a = 1/2 then 
     if has({DOM_FLOAT, DOM_COMPLEX}, domtype(x))
     and x = float(x) // do not float for exact DOM_COMPLEX
                      // such as x = 1 + I etc. !
       then return(float(PI^(1/2))*erfc(x^(1/2)))
       else return(      PI^(1/2) *erfc(x^(1/2)))  
     end_if;
  end_if;
  if a = float(1/2) then 
    return(float(PI^(1/2))*erfc(float(x)^(1/2)))
  end_if;

  //-----------------------------------------------
  // Try float evaluation, if either x or a is
  // a float or a complex numbe using floats.
  if domtype(x) = DOM_FLOAT or 
    (domtype(x) = DOM_COMPLEX and domtype(Re(x)) = DOM_FLOAT) or
    domtype(a) = DOM_FLOAT or
    (domtype(a) = DOM_COMPLEX and domtype(Re(a)) = DOM_FLOAT) then
    result := igamma::float(a, x);
    if domtype(result) in {DOM_FLOAT, DOM_COMPLEX} then
      return(result);
    end_if;
  end_if;

  //-----------------------------------------------
  // normalization: use recursion formula to reduce
  // index a to strip 0<= a < 1. Do not do this for
  // complex a, because in this case float evaluation
  // is not possible, anyway.
  // Restrict this to |a| <= some limit, because this
  // would be much too expensive for huge |a|

  if testtype(a, Type::Real) and 
     specfunc::abs(a) <= Pref::autoExpansionLimit() then

     if domtype(a) = DOM_FLOAT then 
        absx:= specfunc::abs(float(x));
        if domtype(absx) = DOM_FLOAT then
        // Warning: dangerous float evaluation via recursion!
        // This is likely to be subject to cancellation, so
        // we need to boost DIGITS !
        // This is just a quick hack!!! (W. Oevel)
          if a>1 then                     // 2.30258.. = ln(10.0)
             absx:= absx^absx/gamma(absx+1):
             DIGITS:= max(DIGITS,round( ln(absx)/2.30258509299)); 
          end_if:
        // The following does not help much and definitely needs
        // to be improved:
          if a<0 then
             absx:= max(1, gamma(1-a)*absx^a);
             DIGITS:= max(DIGITS,round(-ln(absx)/2.30258509299)); 
          end_if;
        end_if;
        x:= float(x);
     end_if;

     if 1 < a then
        // normalize a by shifting the first argument
        // to the region 0 < a < 1.
        // Use the recursion:  
        //     igamma(a, x) = (a-1)*igamma(a-1, x) + x^(a-1)*exp(-x)
        //  = (a-1)*(a-2)*..*(a-k)*igmma(a - k, x) + exp(-x)*(x^(a-1) + (a-1)*x^(a-2) + ..)

        /* the following special case a : DOM_INT is covered by the code 
          further down below:
          if domtype(a) = DOM_INT then 
             return(fact(a-1)*exp(-x)*_plus(x^k/fact(k)$k=0..a-1));
          end_if:
        */
        fraca:= frac(a):
        floora:= floor(a):
        am1:= a - 1:
        if iszero(fraca) then 
             t2:= x^(round(am1)) // if a is a float, produce integer exponents
        else t2:= x^(am1)
        end_if;:
        t1:= am1:
        result:= x^(am1):
        // further terms only occur for a > 2:
        for k from 2 to floora do
            t2:= x^(a - k);// t2 = x^(a - k)
            result:= result + t1*t2: // x^(a-1) + (a-1)*x^(a-2) + ...
            t1:= (a - k)*t1: // t1 =  (a - 1)*(a - 2)* .. * (a - k)
        end_for:
        // Now:
        //  result = x^(a-1) +(a-1)*x^(a-2) + (a-2)*(a-1)*x^(a-3) + ....
        if iszero(fraca) then
             result:= exp(-x)*result:
        else
             result:= igamma(fraca, x)*t1 + exp(-x)*result:
        end_if;
        if domtype(a) = DOM_FLOAT then
           result:= float(result);
        end_if;
        return(result);
     end_if;

     if a < 0 then
        // normalize a by shifting the first argument
        // to the region 0 < a < 1:

        /* the following special case a : DOM_INT is covered by the code 
          further down below:
        if domtype(a) = DOM_INT then 
           a:= -a;
           return((-1)^a/fact(a)*
                  (Ei(1, x) + exp(-x)*_plus(fact(k)/(-x)^(k+1) $k=0..a-1)));
        end_if:
        */
        // use recursion:  igamma(a, x) = igamma(a + 1, x)/a - x^a/a*exp(-x)
        //  = gamma(a + k)/a/(a + 1)/../(a+k) - exp(-x)*(x^a/a + x^(a+1)/a/(a+1) + ..)
        fraca:= frac(a):
        ceila := ceil(a):
        if iszero(fraca) then
           ceila:= ceila + 1:
        end_if;
        t1:= a:
        if domtype(x) = DOM_FLOAT then
           t1:= float(t1):
        end_if:
        if iszero(fraca) then 
             t2:= x^round(a)  // if a is a float, produce integer exponents
        else t2:= x^a
        end_if;:
        result:= -t2/t1;
        for k from 1 to -ceila do
            t1:= t1*(a + k); // t1 =  a*(a+1)*..*(a + k);
            t2:= t2*x;       // t2 = x^(a + k)
            result:= result - t2/t1;
        end_for:
        // now, result = x^a/a + x^(a+1)/a/(a+1) + ....
        result:= igamma(fraca, x)/t1 + exp(-x)*result;
        if domtype(a) = DOM_FLOAT then
           result:= float(result);
        end_if;
        return(result);
     end_if;
  end_if;
  procname(args());
end_proc:

igamma := prog::remember(igamma,
                         () -> [property::depends(args()), DIGITS,
                                Pref::autoExpansionLimit(),
                                slotAssignCounter("igamma")]):

igamma:= funcenv(igamma):
igamma::print:= "igamma":
igamma::type:= "igamma":
igamma::info:=
  "igamma -- the incomplete Gamma function [try ?igamma for details]":
igamma::float:= // specfunc::igamma:
    loadproc(igamma::float, pathname("SPECFUNC"), "igamma_float"):

igamma::conjugate:=
    loadproc(igamma::conjugate, pathname("STDLIB", "CONJ"), "igamma"):

igamma::diff:=
    loadproc(igamma::diff, pathname("STDLIB", "DIFF"), "igamma"):

igamma::discont:=
    loadproc(igamma::discont, pathname("STDLIB", "DISCONT"), "igamma"):

igamma::rectform:=
    loadproc(igamma::rectform, pathname("STDLIB", "RECTFORM"), "igamma"):
igamma::series:=
    loadproc(igamma::series, pathname("SERIES"), "igamma"):

igamma::Content := stdlib::genOutFunc("Cigamma", 2):

igamma::TeX := (g, data, prio) -> generate::TeXseq("\\Gamma\\!\\left(",
						  "\\right)",
						  op(data)):

igamma::expand:=
proc(i)
  local a, x, r;
begin
   [a, x]:= [op(i)];
   r:= Pref::autoExpansionLimit();
   if testtype(a, Type::Real) and 
      specfunc::abs(a) > r then
      Pref::autoExpansionLimit(ceil(abs(a)) + 10);
      i:= eval(i);
      Pref::autoExpansionLimit(r);
      return(i);
   end_if:
   return(igamma(expand(a), expand(x)))
end_proc:

igamma::expand := prog::remember(igamma::expand,
  () -> [property::depends(args()), Pref::autoExpansionLimit()]):

// end of file 
