//---------------------------------------------------------------------------------------
// This file contains the float attribute of igamma together with
// various utility functions igamma::asymptotic, .. , igamma::shift
//---------------------------------------------------------------------------------------
// igamma::asymptotic = asymptotic expansion  z^(a-1)*exp(-z)*(1 + (a-1)/z + (a-1)*(a-2)/z^2 + ...)
// igamma::legendre = Legendre's continued fraction expansion 
// igamma::gauss = Gauss' continued fraction expansion 
// igamma::taylor2 = gamma(a) - exp(-z)*z^a*gamma(a)*sum(z^k/gamma(a + 1 + k), k = 0, 1, ...);
// igamma::series3 = gamma(a) - z^a * sum( (-z)^k/(a+k)/k!, k = 0, 1, ...)
// igamma::shift = igamma(a + 1, z)/a - z^a/a*exp(-z)  (shift a to the right)
//                 implemented via 
//             igamma(a, z) = z^a*exp(-z)*G[m] with
//             eps = a + floor(1/2 - Re(a))   (-1/2 <= Re(eps) <= 1/2)
//             G[n] = (1 - z*G[n-1])/(n - eps), 
//             G[0] = z^(-eps)*exp(z)*igamma(eps, z)
//---------------------------------------------------------------------------------------

//------------------------------------------------------------------
// Asymptotic expansion:
// igamma::asymptotic = (a, z) ->  = z^(a-1)*exp(-z)*(1 + (a-1)/z + (a-1)*(a-2)/z^2 + ...)
//------------------------------------------------------------------
igamma::asymptotic:= proc(a, z)
local s, A, t, k, kmax, last_t;
begin
  [a, z]:= float([a, z]);
  s:= 1:
  A:= a;
  t:= 1;
  kmax:= 1000*DIGITS;
  for k from 1 to RD_INF do
    A:= A - 1; 
    last_t:= t;
    t:= t*A/z;
    s:= s + t;
    if specfunc::abs(t) <= /* specfunc::abs(s) * */ 10.0^(-DIGITS) then
       break;
    end_if:
    if k > kmax or
      (k > 10 and specfunc::abs(t) >= 2*specfunc::abs(last_t)) then
       return(FAIL);
    end_if;
  end_for; 
  return(z^(a-1)*exp(-z)*s);
end_proc:

/* ===============================================================
//----------------------------------------------------------------
// Backwards recursion of the Legendre contfrac using a-priori
// estimate of the required number of steps.
// Based on the paper by Serge Winitzki, "Computing the incomplete
// Gamma function to arbitrary precision" (google for it!)
// Use the contfrac igamma(a, z) 
//  = exp(-z)*z^a*(1/(z+(1-a)/(1+ 1/(z+(2-a)/(1 + 2/...))
//  = exp(-z)*z^(a-1)* (..(1+(1-a)*v/(1+v/(1+(2-a)*v/(1+2*v/1 + ..)..)
//  = exp(-z)*z^(a-1)*(a0+b0/(a1+b1/(a2+b2/(a3 + ..) .. )
// with v = 1/z, a0 = 0, b0 = 1 and
//    a.k = 1 for k = 1,2,3, ...
//    b.(2*k-1) = (k - a)*v for k = 1,2,3,...
//    b.(2*k) = k*v
//----------------------------------------------
igamma::Legendreback:= proc(a, z)
local A, B, kmax, F, k;
save DIGITS;
begin
  [a,z]:= float([a,z]):
  A:= ln(4*float(PI)*10^DIGITS) + Re(z + (3/2 -a)*ln(abs(z)) - lngamma(1-a)):
  B:= 4*Re(sqrt(z)):
  kmax:= max(2, A/B):
  kmax:= max(2, (A + ln(kmax))/B):
  kmax:= max(2, (A + ln(kmax))/B):
  kmax:= max(kmax^2);
  kmax:= ceil(kmax);
  F:= 1/2*(1 + (a-1)/z + sqrt(4*kmax/z + (1 - (a-1)/z )^2 ) ):
  for k from kmax-1 downto 1 do
      if iszero(F) then
         F:= 1:
      else
       //F:= 1 + k/z/F:
       //F:= 1 + (k-a)/z/F:
         F:= (k + (z - a + k)*F)/(k + z*F);
      end_if:
  end_for:
  // The very last step for k = 1 is F:= 1/F;
  return(z^(a-1)*exp(-z)/F);
end_proc:
============================================= */

// =====================================================
// Using Gauss' contfrac 
// igamma(a, z) = gamma(a) - exp(-z)*z^a*
//       1/(a - a*z/(a+1 + z/(a+2 - (a+1)*z/(a+3 + 2*z/(a+4 - (a+2)*z/(a+5 + 3*z/(a+6 + ...)...)
// i.e., for (a - a*z/(a+1 + z/(a+2 - ...)..):
// f = b1 + a2/ (b2 + a3/(b3 + a4/(b4 + ..)..)
//     b1 = a, 
//     a2 = -(a+0)*z,     b2 = a+1, 
//     a3 =     z,        b3 = a+2, 
//     a4 = -(a+1)*z,     b4 = a+3, 
//     a5 =   2*z,        b5 = a+4, 
//     a6 = -(a+2)*z,     b6 = a+5, 
// Start with
// p0:= 1: q0:= 0:
// p1:= b1: q1:= 1:

igamma::gauss:= proc(a, z)
local p0, q0, p1, q1, k, kmax, ak, bk, tmp, result,
      origDIGITS, extraDIGITS;
save DIGITS;
begin
    [a, z]:= float([a, z]);
    if iszero(a) then
       return(specfunc::Ei(z));
    end_if:
    // Special case: a is a negative integer.
    // Move the first parameter a bit and interpolate:
    if domtype(a) = DOM_FLOAT and 
       iszero(frac(a)) and
       a < 0 then
      tmp:= 10^(-DIGITS); 
      return((igamma::gauss(a*(1 + tmp), z)
             +igamma::gauss(a*(1 - tmp), z))/2);
    end_if:
    origDIGITS:= DIGITS;
    extraDIGITS:= 3:
    kmax:= 1000*DIGITS;
    while TRUE do // loop to control DIGITS
      DIGITS:= origDIGITS + extraDIGITS:
      bk:= a;
      p0:= 1; q0:= 0;
      p1:= a; q1:= 1;
      k:= 0;
      while TRUE do // evaluate the contfrac 
         ak:= -(a + k)*z; 
         bk:= bk + 1;     // bk = a + 2*k + 1;
         [p1, p0]:= [bk * p1 + ak* p0, p1];
         [q1, q0]:= [bk * q1 + ak* q0, q1];
         k:= k+1:
         ak:= k*z;
         bk:= bk + 1;
         [p1, p0]:= [bk * p1 + ak* p0, p1];
         [q1, q0]:= [bk * q1 + ak* q0, q1];
         // two subsequent approximations of the result
         // are given by p0/q0 and p1/q1 with a difference
         // of p1/q1 - p0/q0 = (p1*q0 - p0*q1)/(q0*q1). The 
         // relative error is |p1*q0-p0*q1|/(q0*q1)/(p1/q1) 
         // = |p1*q0-p0*q1|/(q0*p1):
         if specfunc::abs(p1*q0 - p0*q1) <= 10^(-DIGITS)*specfunc::abs(p1*q0) then
            break;
         end_if;
         if k > kmax then
            break;
            return(FAIL); // no convergence
         end_if;
      end_while:
      tmp:= gamma(a);
      result:= tmp - exp(-z)*z^a*q1/p1;
      if specfunc::abs(result) <= 10^(2 -extraDIGITS)*specfunc::abs(tmp) then
         // Beware: what to do if iszero(result)??
         extraDIGITS:= extraDIGITS + ceil(log(10.0, specfunc::abs(tmp/result)));
   //    if extraDIGITS > 20 then 
   //      print("(gauss) boosting DIGITS by ".expr2text(extraDIGITS).
   //            ", [a,z] = ".expr2text([a,z]));
   //    end_if:
         next;
      end_if;
      return(result);
   end_while; // end of loop for controlling DIGITS
end_proc:

//----------------------------------------------------------------------------------------
// Using Legendre's contfrac
// igamma(a, z) = exp(-z)*z^a*(1/(1-a+z + (a-1)/(3-a+z+ 2*(a-2)/(5-a+ z + 3*(a-3)/...)...)
// i.e., b1 = 0,
//     a2 = 1-a+z, b2 = 1+(1-a),
//     a3 = 3-a+z, b3 = 2*(a-2),
//     a4 = 5-a+z, b4 = 3*(a-3),
//
// Backwards recursion:
//      R.(k-1) = (2*k - 1)*z + k*(a-k)/R.k , k = ..., 3, 2, 1.
//      igamma(a, z) = exp(-z)*z^a/R.0
// Alternative: r.k = R.k - k satisfies 
//      r.(k-1) = z + (k-a)/(1 + k/r.k) 

//------------------------------------------------------------------------------------------
igamma::legendre:= proc(a, z)
local p0, q0, p1, q1, k, kmax, ak, bk,
      origDIGITS, extraDIGITS, boost;
save DIGITS;
begin
    [a, z]:= float([a, z]);
    origDIGITS:= DIGITS:
    extraDIGITS:= 3:
    p0:= 0; q0:= 1;
    bk:= float(1 - a + z);
    p1:= 1; q1:= bk;
    [a, z]:= float([a, z]);
    k:= 1;
    kmax:= 1000*DIGITS;
    while TRUE do
       DIGITS:= origDIGITS + extraDIGITS;
       boost:= FALSE:
       ak:= k*(a - k);
       bk:= bk + 2;
       [p1, p0]:= [bk * p1 + ak* p0, p1];
       [q1, q0]:= [bk * q1 + ak* q0, q1];
       if abs(p1) <= 10^(2 - extraDIGITS)*specfunc::abs(bk*p0) then
          extraDIGITS:= extraDIGITS + ceil(log(10.0, specfunc::abs(bk*p0)/specfunc::abs(p1)));
          boost:= TRUE;
       end_if:
       if abs(q1) <= 10^(2 - extraDIGITS)*specfunc::abs(bk*q0) then
          extraDIGITS:= extraDIGITS + ceil(log(10.0, specfunc::abs(bk*q0)/specfunc::abs(q1)));
          boost:= TRUE;
       end_if:
       if boost then
          DIGITS:= origDIGITS + extraDIGITS;
          p0:= 0; q0:= 1;
          bk:= float(1 - a + z);
          p1:= 1; q1:= bk;
          k:= 1:
          next;
       end_if:
       // two subsequent approximations of the result
       // are given by p0/q0 and p1/q1 with a difference
       // of p1/q1 - p0/q0 = (p1*q0 - p0*q1)/(q0*q1). The
       // relative error is |p1*q0-p0*q1|/(q0*q1)/(p1/q1)
       // = |p1*q0-p0*q1|/(q0*p1):
       if specfunc::abs(p1*q0 - p0*q1) <= 10^(2 -origDIGITS - extraDIGITS)*specfunc::abs(p1*q0) then
          break;
       end_if;
       k:= k + 1;
       if k > kmax then
          break;
          return(FAIL); // no convergence
       end_if;
    end_while:
    return(exp(-z)*z^a*p1/q1);
end_proc:


//----------------------------------------------------------------
// taylor2 = gamma(a) - exp(-z)*z^a* sum(z^n*gamma(a)/gamma(a + 1 + n), n = 0..infinity);
//----------------------------------------------------------------
igamma::taylor2:= proc(a, z)
local s, aa, A, Z, k, result, origDIGITS, extraDIGITS, mm,
      a_is_negative_int, tmp, k0;
save DIGITS;
begin
  [a, z]:= float([a, z]);

  // special case a = 0: 
  if iszero(a) then
     return(float(Ei(1, z)))
  end_if:
  // special case a = -1: 
  if iszero(a + 1) then
     return(exp(-z)/z - Ei(1, z))
  end_if:

  a_is_negative_int:= bool(_lazy_and(domtype(a) = DOM_FLOAT, 
                                     iszero(frac(a)),
                                     a <= 0));
  origDIGITS:= DIGITS:
  extraDIGITS:= 3:
  while TRUE do
    DIGITS:= origDIGITS + extraDIGITS:
    if a_is_negative_int then
       k0:= round(-a):
    end_if:

    s:= float(1):
    aa:= a+1:
    A:= aa;
    Z:= z:
    mm:= specfunc::abs(Z/A);
    for k from 1 to RD_INF do
      if a_is_negative_int = TRUE and k = k0 then
         break;
      end_if:
      if specfunc::abs(Z/A) <= 10^(-DIGITS)*specfunc::abs(s) then
        break;
      end_if:
      mm:= max(mm, specfunc::abs(Z/A));
      s:= s + Z/A;
      aa:= aa + 1;
      A:= A*aa;
      Z:= Z*z;
    end_for:
    // for a = -2 and z = 1 we end up with s = 0.0 here:
    if specfunc::abs(s) <= 10^(2 - extraDIGITS)*mm then
       if not iszero(s) then
         extraDIGITS:= extraDIGITS + ceil(log(10.0, mm/specfunc::abs(s)));
         next;
       end_if;
    end_if;

    if a_is_negative_int then
      tmp:= (-1)^(k0)/gamma(1-a)*Ei(1, z);
    else
      tmp:= gamma(a);
    end_if: 
    result:= tmp - exp(-z)*z^a/a * s;
    // check for numerical cancellation and boost DIGITS if necessary
    if iszero(result) then
       extraDIGITS:= extraDIGITS + 10:
       next;
    end_if:
    if specfunc::abs(result) <= 10^(2-extraDIGITS)*specfunc::abs(tmp) then
       extraDIGITS:= extraDIGITS + ceil(log(10.0, specfunc::abs(tmp/result)));
       next;
    end_if;
    break;
  end_while;
  return(result);
end_proc:

//------------------------------------------------------------------------
// igamma(a, z) = gamma(a) - z^a*sum((-1)^k*z^k/(a+k)/k!, k = 0..infinity)
//------------------------------------------------------------------------
igamma::series3:= proc(a, z)
local a_is_negative_int,
      origDIGITS, extraDIGITS, k0, s, 
      mm, ds, Z, K, k;
save DIGITS;
begin
  [a, z]:= float([a, z]);
  if iszero(a) then
     return(float(Ei(1, z)));
  end_if:
  a_is_negative_int:= bool(_lazy_and(domtype(a) = DOM_FLOAT, 
                                     iszero(frac(a)),
                                     a <= 0));
  origDIGITS:= DIGITS:
  extraDIGITS:= 3:
  while TRUE do  // loop to control DIGITS
    DIGITS:= origDIGITS + extraDIGITS:
    if a_is_negative_int then
       k0:= round(-a):
       //-------------------------------------------------
       // in the expansion igamma(a, z) = 
       // gamma(a) - z^a*sum((-1)^k*z^k/(a+k)/k!, k = 0..infinity)
       // the singularities gamma(a) and -(-1)^k*z^a*z^k/(a+k)/k!
       // cancel as a tends to -k = negative integer. Note that 
       //        gamma(a)          = (-1)^k/(a+k)/k! + (-1)^k*psi(k+1)/k!
       // -(-1)^k*z^a*z^k/(a+k)/k! = (-1)^k/(a+k)/k! - (-1)^k*ln(z)/k! 
       //-------------------------------------------------
       s:= (-1)^(-a)*(psi(1-a) - ln(z))/gamma(1-a);
    else
       s:= gamma(a);
    end_if:
    mm:= specfunc::abs(s); // mm stores the maximal term of the sum s
    ds:= z^a/a;
    s:= s - ds;
    mm:= max(specfunc::abs(ds), mm):
    Z:= z^a;
    K:= float(1);
    for k from 1 to RD_INF do
      Z:= -Z*z;
      K:= k*K;   // K = k!
      if a_is_negative_int and k = k0 then
         // omit this term in the series (it cancels the
         // singularity in gamma(a))
         next;
      end_if;
      ds:= Z/K/(a + k); // = the next term in the sum
      s:= s - ds;
      // control cancellation by keeping track
      // of the size mm of the terms in the sum
      mm:= max(specfunc::abs(ds), specfunc::abs(s), mm):
      if // check that the term is small enough 
         specfunc::abs(ds) <= 10^(-DIGITS)*specfunc::abs(s) and
         // check that the following terms in the sum are decreasing
         specfunc::abs(z*(a + k)) < specfunc::abs((k+1)*(a + k + 1)) then
         break;
      end_if:
    end_for;

    // If there was cancellation, try again with increased DIGITS
    if abs(s) <= 10^(2 - extraDIGITS)*mm then
       extraDIGITS:= extraDIGITS + ceil(log(10.0, mm/specfunc::abs(s)));
       next;
    end_if;
    // If no cancellation was detected, break the DIGITS control loop
    break;
  end_while; // end of loop to control DIGITS
  return(s);
end_proc:

/*
//-------------------------------------------------------------------
// use recursion:  igamma(a, z) = igamma(a + 1, z)/a - z^a/a*exp(-z)
//  = gamma(a + k)/a/(a + 1)/../(a+k) - exp(-z)*(z^a/a + z^(a+1)/a/(a+1) + ..)
//-------------------------------------------------------------------
igamma::shift:= proc(a, z)
local m, eps, zG, G, n, dobreak, extraDIGITS, origDIGITS;
save DIGITS;
begin
   [a, z]:= float([a, z]);
   origDIGITS:= DIGITS;
   extraDIGITS:= 3:
   while TRUE do  // boost DIGITS if necessary
     DIGITS:= origDIGITS + extraDIGITS;
     m:= ceil(1/2 - Re(a));
     if Re(m + a) < -0.5 then m:= m + 1; end_if;
     if Re(m + a) >  0.5 then m:= m - 1; end_if;
     eps:= m + a;
     G:= exp(z)/z^eps*specfunc::igamma(eps, z):
     dobreak:= FALSE;
     for n from 1 to m do
       zG:= z*G:
       if specfunc::abs(1 - zG) <= 10^(2 - extraDIGITS) then
          extraDIGITS:= extraDIGITS + max(10, ceil(log(10.0, 1/specfunc::abs(1-zG))));
          dobreak:= TRUE;
          break;
       end_if;
       G:= (1 - zG)/(n - eps);
     end_for:
     if dobreak then 
        next;
     end_if:
     return(z^a*exp(-z)*G);
   end_while;
end_proc:
*/
      
//------------------------------------------------
// The driver routine calling a suitable workhorse
// from igamm::asymptotic .. igamma::shift:
// igamma::asymptotic = asymptotic expansion z^(a-1)*exp(-z)*(1 + (a-1)/z + (a-1)*(a-2)/z^2 + ...)
// igamma::legendre = Legendre's continued fraction expansion 
// igamma::gauss = Gauss's continued fraction expansion 
// igamma::taylor2 = gamma(a) - exp(-z)*z^a*gamma(a) * sum(z^k/gamma(a + 1 + k), k = 0, 1, ...);
// igamma::series3 = gamma(a) - z^a * sum( (-z)^k/(a+k)/k!, k = 0, 1, ...)
// igamma::shift = igamma(a + 1, z)/a - z^a/a*exp(-z)  (shift a to the right)
//             implemented via 
//             igamma(a, z) = z^a*exp(-z)*G[m] with
//             eps = a + floor(1/2 - Re(a))   (-1/2 <= Re(eps) <= 1/2)
//             G[n] = (1 - z*G[n-1])/(n - eps), 
//             G[0] = z^(-eps)*exp(z)*igamma(eps, z)
igamma::float:= proc(a, z)
local absz, absa, Rea, Ima, origDIGITS, ga, zp1,
      lower, n0, n1, n2, n3, nn3, A, B, algo; 
save DIGITS;
begin
  zp1:= float(z+1);
  [a, z]:= float([a, z]):
  if iszero(a) then
     return(Ei(1, z));
  elif iszero(a-1) then
     return(exp(-z));
  end_if:
  if domtype(a) = DOM_FLOAT and a >= 0 and
     domtype(z) = DOM_FLOAT and z >= 0 and
     specfunc::igamma <> FAIL then
     // call the kernel function for speed
     return(specfunc::igamma(a, z));
  end_if:
  if not (contains({DOM_FLOAT, DOM_COMPLEX}, domtype(a)) and
          contains({DOM_FLOAT, DOM_COMPLEX}, domtype(z))) then
     return(hold(igamma)(a, z));
  end_if:

  // Take care of the special case igamma(2, -1) = 0 
  // that slips through all the cases further down below
  if iszero(a - 2) then
   //return(exp(-z)*(1 + z));
     return(exp(-z)*zp1);
  end_if;

  origDIGITS:= DIGITS:
  DIGITS:= 10:
  absa:= specfunc::abs(a):
  Rea:= Re(a):
  Ima:= Im(a):
  absz:= specfunc::abs(z):
  lower:= FALSE;
  if absz < 1.2*absa then
     if (not (domtype(a) = DOM_FLOAT and
              iszero(frac(a)) and 
              a < 0
             ) // a must not be a negative integer
        ) then
        // consider taylor2(a, z) = 
        // gamma(a)*( 1 - z^a*exp(-z)*sum(z^n/gamma(a + 1 + n), n >= 0))
        // The terms z^n/gamma(a + 1 + n) will typically decrease until
        // n = n1 = -Re(a) - sqrt(absz^2 - specfunc::abs(Ima)^2 ),
        // then increase until
        // n = n2 = -Re(a) + sqrt(absz^2 - specfunc::abs(Ima)^2 ),
        // The biggest term in the sum taylor2 is given by n = n2 with the
        // following n2 = solve(|z|/|a+n| = 1, n):
        n2:= -Re(a) + sqrt(absz^2 - specfunc::abs(Ima)^2 ):
        if domtype(n2) <> DOM_FLOAT or n2 < 0 then
           // the terms in the sum are decreasing monotonically
           n2:= 0
        end_if: 
        // A is the size of the largest term in the sum:
        if n2 <> 0 then
           A:= max(specfunc::abs(z^(a+n2)*exp(-z)/gamma(a + 1 + n2)),
                   specfunc::abs(z^a     *exp(-z)/gamma(a + 1))):
        else
           A:= specfunc::abs(z^(a+n2)*exp(-z)/gamma(a + 1 + n2)):
        end_if:
        if A <= 10^(-origDIGITS) then
           userinfo(3, "using shortcut, returning gamma(a)"):
           DIGITS:= origDIGITS:
           // return the first terms of taylor2(a,z):
           return(gamma(a) - z^a*exp(-z)/a);
        elif A <= 10^(-3) then
           // The final result is igamma(a, z) = gamma(a) + O(A).
           // This calls for an algorithm for the lower incomplete
           // gamma function. Set a flag 'lower = TRUE' indicating 
           // not to use Legendre's contfrac:
           lower:= TRUE;
        end_if:
     end_if:
  end_if:

  //---------------------------------------------------------------
  // Compute (rough) estimates for the number of terms in the various
  // methods. Note for the Legendre contfrac: "rough" means "very rough"!
  //---------------------------------------------------------------
  //---------------------------------------------------------------
  // n0 == asymptotic series
  // igamma(a, z) = z^(a-1)*exp(-z)* ( 1 + (a-1)/z + (a-1)*(a-2)/z^2 + ...
  //---------------------------------------------------------------
  if specfunc::abs(a - 1) <  absz/10 and
     specfunc::abs(a - origDIGITS) < absz/10  then
     // The n0-th term of the sum is smaller 10^(-origDIGITS)
     n0:= origDIGITS;
  elif absz > absa and absz >= abs(Ima) then
     // the n0-th term is the minimal term in the sum:
     n0:= max(10, Rea + sqrt(absz^2 - Ima^2)); 
     n0:= ceil(n0);
     // If this term in the sum is small enough, then improve the estimate of n0.
     // Otherwise, estimate n0 as infinity:
     if specfunc::abs(pochhammer(a - n0, n0)) >= 10^(-3-origDIGITS)*absz^n0 then
        n0:= infinity;
     else // improve the estimate
        if Rea > 0 and absz > 1.01*absa then
          // pochhammer(a - n0, n0)/z^n  approx (|a|/|z|)^n0 = 10^(-origDIGITS)
          n0:= max(10, origDIGITS* 2.3025 / ln(absz/absa));
          n0:= ceil(n0);
        else
          // The previous estimate of n0 for the smallest term was too large.
          // Reduce it to a realistic size:
          repeat 
            n0:= max(5, round(n0/2.0)):
          until specfunc::abs(pochhammer(a - n0, n0)) > 10^(-3-origDIGITS)*absz^n0
          or n0 = 5
          end_repeat:
          n0:= 2*n0:
        end_if:
     end_if:
  else // no chance 
     n0:= infinity;
  end_if:

  //-------------------------------------------
  // n1 == number of terms in Legendre contfrac
  //-------------------------------------------
  if lower then
    n1:= infinity;
  else
    n1:= infinity;
    if specfunc::abs(a - z) <= absa*0.5 and
      (A:= specfunc::abs((a-z)^2/a)) >= ln(10.0)*origDIGITS then
       // If |a| >> |a-z| and |a-z|^2/|a| >> 1 the error estimate 
       // for the n-th step (with small n) is
       // (p1*q0 - p0*q1)/(p1*q0) == 
       //      (-1)^n*n!/((a-z)^2/a)^n * (1 - c.n/A+ O(1/A^2))
       // where c.n approx (n+1)^2/2. Determine n1 such that 
       //      n1! approx (n1/exp(1))^n1 <= 10^(-origDIGITS)*A^n1.
       // Note that origDIGITS*ln(10) <= A, i.e., the argument of the
       // lambertW below is >= exp(-1).
       // There are 2 real solutions of (n/exp(1))^n = A^n/10^origDIGITS.
       // The smaller is given by the following formula using lamberW(-1, ..).
       // For the bigger one replace lambertW(-1, ..) by lambertW(0, ..):
       // For n within this values the error estimate
       //      n! approx (n/exp(1))^n <= 10^(-origDIGITS)*A^n
       // is satisfied.
       n1:= ln(10.0)*origDIGITS ; 
       n1:= -n1/lambertW(-1, -n1/exp(1.0)/A);
       n1:= max(2, n1):
       // A posteriori check that the leading order of the error
       //   = (-1)^n*n!/((a-z)^2/a)^n * (1 - c.n/A + O(1/A^2))
       // with c.n approx (n+1)^2/2 was indeed correct. If not, forget
       // n1 by setting it to infinity and proceed to the next check.
       if (n1 + 1)^2 >= 0.2 * A then
          n1:= infinity;
       end:
    end_if:
    if n1 = infinity then
      // The following uses the asymptotic error estimate in the 
      // paper by Serge Winitzki, "Computing the incomplete
      // Gamma function to arbitrary precision" (google for it!)
      if domtype(a) = DOM_FLOAT and
        iszero(frac(1-a)) and 1 - a <= 0 then
        // 1 - a is a nonpositive integer, i.e., gamma(1-a) is singular.
        ga:= RD_INF;
      else
        ga:= gamma(1 - a);
      end_if:
      B:= 4*Re(sqrt(z)):
      if iszero(B) then
        // We are on the negative real semi-axes. The Legendre contfrac 
        // does not converge. Still, the approximations may be sufficiently
        // good for a small number of steps such as n1 = 10, say:
        n1:= 10.0: // some wild guess
        if 6.283185307*abs(z^(1-a)*exp(z))*n1*exp(-4*sqrt(n1)*Re(sqrt(z))) 
           <= 10^(-origDIGITS)*abs(ga) then
           n1:= 10.0: // accept the guess n1 = 10.0
        else
           n1:= infinity: // do not accept the guess n1 = 10.0
        end_if:
      else
        // solve the equation for
        //  2*PI*abs(z^(1-a)*exp(z))*n1*exp(-4*sqrt(n1)*Re(sqrt(z)))
        //        = 10^(-origDIGITS)*abs(gamma(1 - a))
        // i.e.,  
        //      n1*exp(-B*sqrt(n1)) = A, 
        //
        // where B = 4*Re(sqrt(z))) was defined above and:
        if ga = RD_INF then
           A:= RD_INF;
        else
           A:= 10^(-origDIGITS)*abs(ga) / float(2*PI*abs(z^(1-a)*exp(z)));
        end_if:
/*
        // Solve sqrt(n)*exp(-B*n) = AA by 2 steps of a fixpoint iteration 
        // starting with n1 = 1:
        AA:= ln(6.283185307*10^origDIGITS) + Re(z + (1-a)*ln(z) - ln(ga)):
        n1:= max(5.0, AA/B):
        n1:= max(5.0, (AA + 2*ln(n1))/B):
        n1:= n1^2;
*/
        // Note that f(n1) = n1*exp(-B*sqrt(n1)) satisfies f(0) = 0 and
        // f(infinity) = 0 and has exactly one maximum n1 = 4/B^2 with
        // f(4/B^2) = 4/B^2 * exp(-2):
        // 
        // Since we solve f(n1) = A for n1, there won't be a real solution
        // unless A < = 4/B^2*exp(-2). 

        if A*B^2 > 0.541342 /*(= 4*exp(-2))*/ then
           // there is no real solution: the Legendre contfrac will not converge
           n1:= infinity: 
        else
           // If A < = 4/B^2*exp(-2), then there are two positive solutions:
           // The smaller solution: 4/B^2*lambertW(0,  A^(1/2)*B/2)^2.
           // The larger  solution: 4/B^2*lambertW(-1,-A^(1/2)*B/2)^2. 
           // We need the larger one:
 
           n1:= float(4/B^2*lambertW(-1,-A^(1/2)*B/2)^2);
 
           assert(domtype(n1) = DOM_FLOAT);
 
           // If n1 is very close to zero, the asymptotic estimate would 
           // not be valid. Thus, assume that n1 is larger than some 
           // reasonably large value: 
           n1:= max(25, n1):
 
        end_if:
      end_if: // of 'if izero(B) ...'
    end_if: // of 'if n1 = infinity ...'
  end_if:  // of 'if not lower'

  //------------------------------------------------------
  // n2 == Taylor-like expansion
  // igamma(a, z) = gamma(a) - z^a*exp(-z)*sum(gamma(a)/gamma(a+1+n)*z^n, n = 0..infinity)
  //------------------------------------------------------
  if absz < 0.999*abs(Ima) then
     n2:= max(10, origDIGITS*2.3026/ln(absa/absz)); // (|z|/|a|)^n2 <= 10^(-origDIGITS)
  else
     // n2 = number of terms until |z|/|a+n| = 1 (this is the largest term in the sum).
     n2:= max(10, -Rea + sqrt(max(0, absz^2 - Ima^2))); 
     // We need further sqrt(abs(100*z^2) - Im(a)^2) - sqrt(abs(z^2) - Im(a)^2)
     // approx 99*abs(z)^2/sqrt(abs(100*z^2)) approx 10*abs(z) terms until |z|/|a+n| < 1/10.
     // We need origDIGITS of such terms to ensure that their product is smaller than 10^(-origDIGITS).
     n2:= n2 + 3.7*absz + origDIGITS;
  end_if:

  //------------------------------------------------------
  // n3 == Taylor expansion
  // igamma(a, z) = gamma(a) - z^a*sum((-z)^n/n!/(a+n), n = 0..infinity)
  //------------------------------------------------------
  // The largest term in the sum is given by n = nn3 = |z|
  nn3:= specfunc::abs(z);
  // n3 = number of terms until |z|^n/n! = 10^(-origDIGITS)
  A:= ln(10.0^(origDIGITS)/2/3.1414):
  n3:= max(10.0, float(A/lambertW(A/absz/exp(1.0)))); 
  // If the quotient between the largest term (n = nn3) and sufficiently
  // small terms (n = n3) is large, we expect major cancellation problems.
  // If the quotient is > 10^origDIGITS, we need to boost DIGITS at least
  // by a factor of 2. Increase n3 dramatically to make the Taylor expansion
  // unattractive:
  if n3 > nn3 and
     Re(z) >= 0 and
     specfunc::abs((exp(1.0)*z)^(nn3-n3)*n3^n3/nn3^nn3) > 10^origDIGITS then
     n3:= 100*n3:
  end_if:
  n3:= round(n3);

  //------------------------------------------------------
  // n4 == Gauss' continued fraction
  // Alas, we do not have an estimate for the number of terms in gauss!
  //------------------------------------------------------
  
  // ===========================================================
  // We have estimates n0, n1, n2, n3 for the different methods.
  userinfo(3, 
           "predicted number of terms to achieve the desired precision".
           ": asymptotics = ".expr2text(round(n0)).
           ", legendre = ".expr2text(round(n1)).
           ", taylor2 = ".expr2text(round(n2)).
           ", series3 = ".expr2text(round(n3))):
  // Pick the fastest method
  if lower then 
     // choose between algo 0, 1, 2, 3 and gauss
     algo:= min(n0, n2, n3, 50*origDIGITS);
  elif Re(z) < 0 then
     // choose between algo 0, 1, 2, 3 and gauss
     algo:= min(n0, n1, n2, n3, 50*origDIGITS);
  else
     // choose between algo 0, 1, 2, 3
     algo:= min(n0, n1, n2, n3);
  end_if:

  DIGITS:= origDIGITS;
  case algo
  of n0 do  
            userinfo(3, "using asymptotics");
            return(igamma::asymptotic(a, z));
            break;
  of n1 do  
            userinfo(3, "using Legendre's continued fraction");
            return(igamma::legendre(a, z));
            break;
  of n2 do  userinfo(3, "using Taylor expansion around the origin");
            return(igamma::taylor2(a, z));
            break;
  of n3 do  userinfo(3, "using series expansion around the origin");
            return(igamma::series3(a, z));
            break;
  otherwise 
            userinfo(3, "using Gauss' continued fraction");
            return(igamma::gauss(a, z));
  end_case;
end_proc:
